# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import collections
import time
import traceback
import deal_dha_info

from cliFactory import cli
from exportInfo import exportInfo
from util import config
from util import util
from java.io import File
from java.lang import Exception as JException
from com.huawei.ism.tool.obase.exception import ToolException

COLLECT_ITEM = {
    "smart": config.COLLECT_TYPE_SMART,
    "dha": config.COLLECT_TYPE_DHA
}

QueryCmdResult = collections.namedtuple('QueryCmdResult', ['err_msg', 'log_path', 'failed_list'])


class ExportDepthCollectItemsInfo:
    def __init__(self, dev_obj, logger, item_type, result_dict):
        self.dev_obj = dev_obj
        self.logger = logger
        self.item_type = item_type
        result_dict[item_type] = {}
        self.result_dict = result_dict
        self.progress_lock = dev_obj.get("progress_lock")
        # 是否选择节点信息，ip框节点 ，控制器节点
        self.has_nodes, self.modules_ids, self.controller_ids = util.get_all_nodes_id(dev_obj)

    def execute_collect(self):
        self.logger.info(self.dev_obj, "start export info {} by bdm".format(self.item_type))
        is_success = True
        try:
            # 收集前检查，失败即停止当前检查项收集
            is_success_pre_check, err_msg = self.pre_collect_check()
            util.refresh_depth_collect_process(self.dev_obj, 10, self.progress_lock)
            if not is_success_pre_check:
                self.logger.error(self.dev_obj, "pre-collection check failed.")
                self.post_collect(err_msg)
                return False
            # 收集项信息收集
            if self.item_type == config.COLLECT_TYPE_DHA:
                is_success, err_msg, failed_list = self.collect_dha_info()
            else:
                is_success, err_msg, failed_list = self.collect_item_info()
            util.refresh_depth_collect_process(self.dev_obj, 50, self.progress_lock)
            # 收集后处理
            self.post_collect(err_msg, is_success, failed_list)
        except (Exception, JException) as e:
            self.logger.error(self.dev_obj, "export info {} by bdm failed. {} {}".format(
                self.item_type, e, str(traceback.format_exc())))
            self.post_collect(util.getMsg(self.dev_obj, "failed.collect.information"))
            is_success = False
        self.logger.info(self.dev_obj, "result_dict is {}".format(self.result_dict))
        return is_success

    def collect_item_info(self):
        # 打包
        is_success_packing, err_msg = self.packing_logs_info()
        if not is_success_packing:
            return False, err_msg, ""
        util.refresh_depth_collect_process(self.dev_obj, 15, self.progress_lock)
        # 查询
        is_success_polling, query_cmd_result = self.poll_packaging_progress()
        if not is_success_polling:
            # 收集过程发生错误，或者收集失败，清理文件
            self.delete_file()
            return False, query_cmd_result.err_msg, ""
        util.refresh_depth_collect_process(self.dev_obj, 40, self.progress_lock)
        # 下载
        is_success_download, err_msg = self.download_file(query_cmd_result.log_path)
        if not is_success_download:
            return False, err_msg, ""
        return True, "", query_cmd_result.failed_list

    def pre_collect_check(self):
        """
        收集前检查
        :return: boolean 检查是否通过
        """
        self.logger.info(self.dev_obj, "start pre-collection check...")
        # 检查系统是否正常
        is_system_normal, err_msg = util.check_system_normal(self.dev_obj)
        if not is_system_normal:
            self.logger.error(self.dev_obj, "The system status is abnormal.")
            return False, err_msg
        # 检查用户级别是否满足收集信息
        is_user_privilege, err_msg = util.check_user_privilege_by_role_id(self.dev_obj)
        if not is_user_privilege:
            self.logger.error(self.dev_obj, "The user level is too low.")
            return False, err_msg
        return True, ""

    def packing_logs_info(self):
        self.logger.info(self.dev_obj, "start packing {} logs info...".format(self.item_type))
        is_success = True
        err_msg = ""
        for _ in range(4):
            is_success, ret, err_msg = self.issue_packing_cmd()
            if is_success:
                break
            # 其他设备正在导出，直接返回失败。
            if exportInfo.is_export_file(ret):
                return False, util.getMsg(self.dev_obj, "system.is.exporting.files")
            # 打包失败，清理残留文件
            self.delete_file()
        return is_success, err_msg

    def issue_packing_cmd(self):
        packing_cmd = self.splice_nodes_info("change disk disk_log log_type={}".format(self.item_type))
        is_success, ret = cli.executeCmd(self.dev_obj, packing_cmd)
        self.logger.info(self.dev_obj, "packing {} logs info and the cliRet is {}".format(self.item_type, ret))
        if not is_success:
            return False, ret, util.getMsg(self.dev_obj, "dev.conn.failure")
        if config.CLI_EXECUTE_CMD_SUCCESS not in ret:
            return False, ret, util.getMsg(self.dev_obj, "cli.excute.failure", ret)
        return True, ret, ""

    def splice_nodes_info(self, cmd):
        if not self.has_nodes:
            return cmd
        if self.controller_ids:
            cmd = "{} controller_id={}".format(cmd, self.controller_ids)
        if self.modules_ids:
            cmd = "{} enclosure_id={}".format(cmd, self.modules_ids)
        return cmd

    def delete_file(self):
        delete_cmd = self.splice_nodes_info("delete disk disk_log log_type={}".format(self.item_type))
        is_success, ret = cli.executeCmd(self.dev_obj, delete_cmd)
        self.logger.info(self.dev_obj, "delete {} file and the cliRet is {}".format(self.item_type, ret))
        if not is_success:
            self.logger.error(self.dev_obj, "delete file failed because cli execute failed")

    def poll_packaging_progress(self):
        self.logger.info(self.dev_obj, "start poll {} packaging progress...".format(self.item_type))
        time_out = config.PARALLEL_EXPORT_INFO_TIMEOUT[COLLECT_ITEM.get(self.item_type)]
        # 轮询次数
        repeat_times = time_out * 2
        ret = ""
        while repeat_times != 0:
            repeat_times -= 1
            is_success, ret, err_msg = self.issue_polling_cmd()
            if not is_success:
                return False, QueryCmdResult._make([err_msg, '', ''])
            total_result, log_path, failed_list = self.get_poll_progress_info(ret)
            if total_result == "Collecting":
                time.sleep(30)
                continue
            if total_result == "Failed":
                return False, QueryCmdResult._make([util.getMsg(self.dev_obj, "cli.excute.failure", ret), '', ''])
            if "Successful" in total_result:
                return True, QueryCmdResult._make(['', log_path, failed_list])
        return False, QueryCmdResult._make([util.getMsg(self.dev_obj, "cli.execute.failure.on.allContr", ret), '', ''])

    def issue_polling_cmd(self):
        poll_progress_cmd = self.splice_nodes_info("show disk disk_log log_type={}".format(self.item_type))
        is_success, ret = cli.executeCmd(self.dev_obj, poll_progress_cmd)
        self.logger.info(self.dev_obj, "poll {} packaging progress and the cliRet is {}".format(
            self.item_type, ret))
        if not is_success:
            return False, ret, util.getMsg(self.dev_obj, "dev.conn.failure")
        if "Error:" in ret or "Total Result" not in ret:
            return False, ret, util.getMsg(self.dev_obj, "cli.excute.failure", ret)
        return True, ret, ""

    def get_poll_progress_info(self, ret):
        ret_dict_list = util.getVerticalCliRet(ret)
        for ret_dict in ret_dict_list:
            self.logger.info(self.dev_obj, "ret dict is: {}".format(ret_dict))
            return ret_dict.get("Total Result"), ret_dict.get("Log Path"), ret_dict.get("Failed List")

    def download_file(self, log_path):
        sftp = util.getSftp(self.dev_obj)
        file_dict = self.get_remote_path_and_name(log_path)
        name = file_dict.get("name")
        local_path = util.getLocalInfoPathByType(self.dev_obj, COLLECT_ITEM.get(self.item_type)) + name
        try:
            download_file = File(local_path)
            sftp.getFile(file_dict.get("path"), download_file, None)
            self.logger.info(self.dev_obj, "download {} success".format(name))
            return True, ""
        except (ToolException, Exception) as e:
            self.logger.error(self.dev_obj, "download {} file failed. {} {}".format(
                self.item_type, e, str(traceback.format_exc())))
            if "local free space is not enough" in str(e).lower():
                return False, util.getMsg(self.dev_obj, "failed.export.log.insufficient.space")
            return False, util.getMsg(self.dev_obj, "downLoad.file.failure")
        finally:
            self.delete_file()

    def get_remote_path_and_name(self, log_path):
        path = log_path.replace('"', '')
        file_dict = {"path": path, "name": path.split("/")[-1]}
        self.logger.info(self.dev_obj, "file dict is {}".format(file_dict))
        return file_dict

    def post_collect(self, err_msg, is_success=False, failed_list=""):
        if not is_success:
            self.delete_local_file()
            self.result_dict[self.item_type]["err_msg"] = err_msg
            self.result_dict[self.item_type]["status"] = "Failed"
            self.result_dict[self.item_type]["collect_res"] = False
            return
        if is_success and not failed_list and not err_msg:
            self.result_dict[self.item_type]["err_msg"] = ""
            self.result_dict[self.item_type]["status"] = "Successful"
            self.result_dict[self.item_type]["collect_res"] = True
            return
        self.result_dict[self.item_type]["status"] = "Partially Successful"
        self.result_dict[self.item_type]["collect_res"] = True
        if err_msg:
            self.result_dict[self.item_type]["err_msg"] = err_msg
            return
        self.result_dict[self.item_type]["err_msg"] = self.get_failed_nodes(failed_list)

    def get_failed_nodes(self, failed_list):
        failed_nodes = failed_list.split(",")
        failed_controller_ids = ""
        failed_modules_ids = ""
        for node in failed_nodes:
            if "DAE" in node:
                failed_modules_ids = failed_modules_ids + node + " "
                continue
            failed_controller_ids = failed_controller_ids + node + " "
        err_msg = ""
        if failed_controller_ids:
            err_msg = util.getMsg(self.dev_obj, "controllers.failed.information", failed_controller_ids)
        if failed_modules_ids:
            err_msg = err_msg + util.getMsg(self.dev_obj, "ip.enclosures.failed.information", failed_modules_ids)
        return err_msg

    def collect_dha_info(self):
        loop_times = 0
        down_load_list = []
        try:
            while loop_times < config.COLLECT_DHA_MAX_LOOP_TIMES:
                loop_times += 1
                self.logger.info(self.dev_obj, "{}th, collect dha info....".format(loop_times))
                is_success, err_msg, failed_list = self.collect_item_info()
                if not is_success:
                    self.logger.error(self.dev_obj, "export dha logs error. loop times is {}".format(loop_times))
                    return False, util.getMsg(self.dev_obj, deal_dha_info.get_dha_err_msg_key(
                        "dha.process.failure", loop_times)), ""
                is_check_success, err_msg, is_loop_required = deal_dha_info.check_dha_result_catalog(
                    self.dev_obj, down_load_list, loop_times)
                if not is_check_success:
                    return False, err_msg, ""
                if not is_loop_required:
                    self.logger.info(self.dev_obj, "The file has been fully exported, quit...")
                    return True, "", failed_list
                self.logger.info(self.dev_obj, "The file has been partly exported, looping...")
            # 达到收集次数上限
            self.logger.info(self.dev_obj, "the DHA file looping reached to max loop times, quit...")
            return True, util.getMsg(self.dev_obj, "dha.export.reach.to.loop.limit.info"), ""
        except (Exception, ToolException) as exception:
            self.logger.error(self.dev_obj, "collect dha log except: {} {}".format(
                exception, str(traceback.format_exc())))
            return False, util.getMsg(self.dev_obj, "dha.process.failure"), ""

    def delete_local_file(self):
        self.logger.info(self.dev_obj, "start delete {} local file...".format(self.item_type))
        local_dir = util.getLocalInfoPathByType(self.dev_obj, self.item_type)
        try:
            is_delete_success = util.cleanDir(local_dir)
            if not is_delete_success:
                self.logger.error(self.dev_obj, "del {} file failed.".format(self.item_type))
        except Exception as e:
            self.logger.error(self.dev_obj, "del {} file error. {} {}".format(
                self.item_type, e, str(traceback.format_exc())))
