from java.io import File
from com.huawei.ism.tool.obase.exception import ToolException
import os
import shutil
import re
from commonUtils import MsgInfo, Log, CommonUtils
from sshTools import SSHToolUtils
from common.util import *
from config import config
import traceback

class FltDiskLgCollector():
    
    '''
    @summary: this class differs from collecting fault disk log from the main controller 
    #Here we do looping in connecting to all controller using SSH connection directly, 
    #due to missing authorization in entering in mini-system on current IP addr.
    # 
    '''
    
    def __init__(self, devObject, commonUtil, logger, messageInfo, disklogConfig, collectAllLog = True):
        '''
        @summary: __init__
        '''
        self.devObj = devObject
        self.util = commonUtil
        self.log = logger
        self.msgInfo = messageInfo
        self.disklogConf = disklogConfig
        self.successedTimes = 0
        self.totalCtlr = 0
        self.isNeedHotPatch = False
        self.collectAll = collectAllLog
        self.log.info('fault disk log collection initiating...')
        self.util.initPyDetailMsg()
        
        
    def collectFltDiskLg(self):
        '''
        @summary: collection entrance
        @param : vacant
        @return:  returnFlag , returnMsg : (boolean, str)
        '''
        isCmdSuccess, errMsgKey = self.util.checkSystemNormal()
        if not isCmdSuccess:
            self.util.setPyDetailMsg(errMsgKey)
            return False, ""
        isQrySuccess, enginNum = self.util.getEnginNum()
        if not isQrySuccess:
            self.util.setPyDetailMsg("query.dev.is.6U4CTRL.dev.failed")
            return False, ""
        isSuccess, controllerIpLists = self.util.getControllerIpList()
        if not self.collectAll:
            controllerIpLists = [controllerIpLists[0]]
        #set the total controller number as the length of the controllerIP list length
        self.totalCtlr = len(controllerIpLists)
        
        if len(controllerIpLists) < 1 :
            self.util.setPyDetailMsg("failed.query.ctrlr.ip.lists")
            return False, ""
        currentProcess = 5
        percentNumber = 80.0 / len(controllerIpLists)
        
        for currentCtlrIp in controllerIpLists:
            self.log.info("collecting fault disk log on ctrlr%s"%str(currentCtlrIp))
            self.getCurentCtrlrFltDiskLg(currentCtlrIp)
            
            if self.isNeedHotPatch:
                self.log.info("no permission to copy the fault disk logs, quit!")
                break
            currentProcess = util.refreshProcessByStep(self.devObj, currentProcess, percentNumber)

        return self.calculateStatus(enginNum)

    def getCurentCtrlrFltDiskLg(self, ip):
        '''
        @summary: connect to specified controller, build new ssh connection and  collect fault disk log 2 local 
        @param : ip : ipV4 pattern 
        @return:  vacant
        '''
        sshToolUtil = SSHToolUtils(self.devObj,self.log)
        ssh = None
        sftpTrans = None
        try:
            
            self.log.info('Building SSH connection on ctrlr %s ... ' % ip)
            isQrySuccess, ssh = sshToolUtil.newSshUsingCrntUsrCredential(ip)
            if not isQrySuccess:
                raise Exception, "failed.build.ssh.connection.on.ctrlr" + "||" +str(ip)
            
            self.log.info('Checking current controller: %s status . ' % ip)
            #if pointed ctlr is not ready or is upgrading ,raise exception
            self.util.checkPointedSystemAvailable(ssh, ip)

            #change current SSH connection user mode into mini-system
            self.log.info('Entering current controller: %s minisystem mode ... ' % ip)
            self.util.enterMiniSystemModOnHost(ssh, ip)

            # get local fault disk log directory, make one if not exist 
            currentFaultDisklogDir = self.getCurrentDiskLogDir(ip)
            self.log.info('Current controller: %s local fault disk log dir: %s ... ' % (ip, currentFaultDisklogDir))
            # get SFTP object
            self.log.info('Deriving SFTP transporter on  controller: %s  ... ' % ip)
            isQrySuccess, sftpTrans = sshToolUtil.deriveSftpTransporter( ssh)
            if not isQrySuccess :
                raise   Exception , "failed.create.sftp.connection.on.ctlr" + "||" + str(ip)
              
            self.log.info("start downloading fault disk log using SFTP")
            self.downloadFltLog2lcl(sftpTrans, ip, currentFaultDisklogDir)
        
        except Exception , e :
            e = str(e)
            self.log.error("exception occurred while processing getCurentCtrlrFltDiskLg -> FltDiskLgCollector %s"%e)
            if re.match("([a-z|A-z]+[.]+)+[a-z|A-Z]+(\|\|)(\S)+", e):
                
                self.log.info( "error type is known with params, sealing return message...")
                errMsg = e.split("||")[0]
                msgParam = e.split("||")[1]
                self.util.addPyDetailMsg(errMsg, msgParam)
                
            elif re.match("([a-z|A-z]+[.]+)+[a-z|A-Z]+", e):
                
                self.log.info( "error type is known, sealing return message...")
                self.util.addPyDetailMsg(str(e))

            else:
                self.log.info("unknown exception, using unified err message:%s"%str(e))
                self.util.addPyDetailMsg("failed.collect.fault.disk.log.conn.err", ip)
                
        finally:
            
            self.log.info("cleaning & reseting...")
            self.util.tryExitMiniSystemOnHost(ssh, ip)    
            sshToolUtil.closeSftpConnection( sftpTrans)
            sshToolUtil.closeSshConnection( ssh)
            

    def downloadFltLog2lcl(self, sftpTrans, controllerIp, currentFaultDisklogDir):
        '''
        @summary: download remote files to local dir 
        @param sftpTrans:  sftp instance
        @param controllerIp:  controllerIp IPV4
        @param currentFaultDisklogDir: current Fault Disk log Directory 
        '''
        
        try:
            self.log.info( "now start downloading fault disk log on %s"%controllerIp)
            
            sftpTrans.getDirRecurse(self.disklogConf.DISK_FAULT_LOG_REMOTE_ROOT_DIR, File(currentFaultDisklogDir))
            self.log.info('Collect current controller: %s successfully . ' % controllerIp)
            self.successedTimes += 1
            
            if not self.collectAll:
                self.postProcess4RecentLogDownload(currentFaultDisklogDir)
        except ToolException, te:
            self.log.error('ToolException occurred while downloading fault-disk-log:' + unicode(te))
            if 'PERMISSION' in te.getMessage().upper():
                self.log.info( "permission denied , tip user to patch")
                self.isNeedHotPatch = True
                raise Exception, self.tipHotPatchRquired()
                
            elif sftpTrans.isDirExist(self.disklogConf.DISK_FAULT_LOG_REMOTE_ROOT_DIR):
                self.log.info( "fault disk log exists but failed to download")
                self.msgInfo.writeMsgLine('Controller ' + unicode(controllerIp) + ' fault disk log directory download exception.',
                         self.disklogConf.DISK_LOG_COLLECT_ERRMSG_FILE_NAME)
                raise  Exception, "download.other.ctrl.fault.disk.log.failed" + "||" + str(controllerIp)
                
            else:
                self.log.info('Fault disk log directory does not exist.')
                self.msgInfo.writeMsgLine('Controller ' + unicode(controllerIp) + ' fault disk log directory not exist.',
                             self.disklogConf.DISK_LOG_COLLECT_ERRMSG_FILE_NAME)
                
                shutil.rmtree(currentFaultDisklogDir, True)
                self.successedTimes += 1
                
        except Exception, e:
            self.log.error('Caught Exception when down loading fault disk log directory:' + self.disklogConf.DISK_FAULT_LOG_REMOTE_ROOT_DIR + ', exception:' + unicode(e))
            raise Exception , e
        
    
    def tipHotPatchRquired(self):
        '''
        @summary: hint what kinds of cause causes dir with no permission
        @return:  the errMsg info
        '''
        deviceVersion = self.util.getDeviceVersion()
        self.log.info("current device version: %s"%str(deviceVersion))
        isHotPatchAvailable  = True if deviceVersion in self.disklogConf.HAS_ISOLATED_DISKLOG_PATCH_LIST else False
        self.log.info("checking whether current version is supported for patching %s"%isHotPatchAvailable)
        
        if isHotPatchAvailable:
            return "install.hotpatch.collect.fault.disk.log"
        else:
            return "current.version.does.not.support.fault.disk.log"
        
                
    def getCurrentDiskLogDir(self, iP):
        '''
        @summary: find current disk log dir ,if not exists , make one
        @return: directory :string, absolute path
        '''
        self.log.info("collectRetDir:%s" % self.devObj["collectRetDir"])
        lclDiskClctRootDir = self.util.getLocalInfoPathByType("Disklog")
        currentFaultDisklogDir = os.path.join(lclDiskClctRootDir, self.disklogConf.ISOLATED_DISKLOG_DIR_NAME, iP)
             
        if not os.path.exists(currentFaultDisklogDir):
            self.log.info("no dir for current controller, create one ...")
            os.makedirs(currentFaultDisklogDir)
        return currentFaultDisklogDir
    
    def calculateStatus(self, enginNum):
        '''
        @summary: while process finished, calculate collection status
        @param enginNum: engine number
        @return: status, errMsg
        '''

        if self.successedTimes > 0 :
            self.log.info("enginNum:%d"%enginNum)
            self.log.info("successedTimes:%d"%self.successedTimes)
            self.log.info("totalCtlr:%d"%self.totalCtlr)
            self.util.setCollectAllInfo(self.successedTimes == self.totalCtlr)
            return True, ""
        else :
            self.log.info("All failed in  collecting fault disk log ")
            return False, "" 
    def postProcess4RecentLogDownload(self, logDir):
        '''
        @summary: post_Process for collect recent fault disk log
        '''
        try:
            self.common_RemoveFile(os.path.join(logDir, "hdd"))
            self.common_RemoveFile(os.path.join(logDir, "hssd"))
            interLogDir = os.path.join(logDir, "internal_log")
            if not os.path.exists(interLogDir):
                self.log.error("[postProcessCtrlLog] internal_log dir not exist")
                return
            fileList = self.getRecentFileList(interLogDir)
            if not fileList:
                self.log.error("[postProcessCtrlLog] no internal_log")
                return
            totalSizeCount = 0L
            for fileTmp in fileList:
                fileTmpPath = os.path.join(interLogDir, fileTmp)
                fileTmpSize = os.path.getsize(fileTmpPath)
                if (totalSizeCount + fileTmpSize) > config.RECENT_FAULTLOG_MAX_SIZE:
                    self.log.error("[postProcessCtrlLog] The file %s size %s is too large, need delete!" % (fileTmpPath, str(fileTmpSize)))
                    self.common_RemoveFile(fileTmpPath)
                    continue
                
                totalSizeCount += fileTmpSize
            return
        except:
            self.log.error("[postProcessCtrlLog]error: " + traceback.format_exc())
            return
    def common_RemoveFile(self, filePath):
        '''
        @summary: common function for file or directory move action
        '''
        try:
            if not os.path.exists(filePath):
                return
            if os.path.isdir(filePath):
                shutil.rmtree(filePath, ignore_errors = True)
            elif os.path.isfile(filePath):
                os.remove(filePath)
            return
        except:
            self.log.error("[common_RemoveFile]error: " + traceback.format_exc())
            return
        
    def getRecentFileList(self, fileDir):
        '''
        @summary: common function for file sort
        '''
        fileTmpList = os.listdir(fileDir)
        fileTmpList.sort(reverse = True)
        return fileTmpList
