# -*- coding: UTF-8 -*-

import traceback
import time
import json

from common.util import util
from common.no_check_exception import NoCheckException

from cbb.frame.rest.restUtil import (
    get_container_rest,
    release_container_rest,
    ContainerRestService,
    CommonRest,
)

from java.lang import Exception as JException
import java.util.HashMap as HashMap


class ContainerCollect:
    """
    容器信息收集
    """
    def __init__(self, dev_obj, uri, logger, local_dir):
        self.dev_obj = dev_obj
        self.lang = dev_obj.get("lang")
        self.dev_node = self.dev_obj.get("devNode")
        self.uri = uri
        self.logger = logger
        self.local_dir = local_dir
        self.rest = None

    def execute_collect(self):
        """
        使用get的方式下载文件
        :return:
        """
        try:
            flag, msg = self.init_rest_connection()
            if not flag:
                return flag, msg
            self.execute_package()
            self.execute_query_progress()
            self.execute_download()
            self.delete_remote_file()
            return True, ''
        except NoCheckException as e:
            self.logger.error(self.dev_obj, str(traceback.format_exc()))
            return False, e.error_msg
        except (JException, Exception):
            self.logger.error(self.dev_obj, str(traceback.format_exc()))
            err_msg = util.getMsg(self.dev_obj, "failed.collect.information")
            return False, err_msg
        finally:
            if self.rest:
                release_container_rest(self.dev_node)

    def delete_remote_file(self):
        param_dict = {"type": "remove"}
        self._execute_get(param_dict)

    def execute_package(self):
        """
        触发打包
        :return:
        """
        param_dict = {"type": "package"}
        flag, ctrl_id = util.getController(self.dev_obj)
        if flag:
            self.logger.info(
                self.dev_obj, "need filter ctrl:{}".format(ctrl_id))
            param_dict["nodeName"] = ctrl_id
        return self._execute_get(param_dict)

    def execute_query_progress(self):
        """
        查询进度
        :return:
        """
        sleep_time = 5
        time_out = 3600
        param_dict = {"type": "package_progress"}
        package_progress = 1
        err_msg = util.getMsg(self.dev_obj, "failed.collect.information")
        while time_out > 0:
            params_str = CommonRest.getParamsJsonStr(param_dict, False)
            json_data = ContainerRestService.get_for_big(
                self.rest, self.uri, params_str)
            flag, msg = ContainerRestService.check_response(
                json_data, self.lang)
            error_code = self._get_error_code(json_data, msg)
            if error_code == '-1':
                raise NoCheckException(msg)

            if error_code != '0':
                self.dev_obj["collectAllInfo"] = False
                self.dev_obj["py_detail"] = msg

            record = json_data.get("data")
            if not record or not isinstance(record, list):
                raise NoCheckException(err_msg)
            # record 返回的是list，只有一个元素
            for data in record:
                package_progress = data.get("progress")

            time.sleep(sleep_time)
            time_out -= sleep_time
            if str(package_progress) == '100':
                return True
        raise NoCheckException(util.getMsg(
            self.dev_obj, "failed.collect.information"))

    @staticmethod
    def _get_error_code(json_data, msg):
        """
        获取错误码
        :param json_data: json数据
        :param msg: 默认错误消息
        :return:
        """
        error_info = json_data.get("error")
        if not error_info:
            raise NoCheckException(msg)
        return str(error_info.get("errId"))

    def _execute_get(self, param_dict):
        """
        传入不同参数，执行触发打包，或查询进度。
        :param param_dict:
        :return:
        """
        err_msg = util.getMsg(self.dev_obj, "failed.collect.information")
        params_str = CommonRest.getParamsJsonStr(param_dict, False)
        record = ContainerRestService.get_for_big(
            self.rest, self.uri, params_str)
        flag, msg = ContainerRestService.check_response(record, self.lang)
        if flag is not True:
            raise NoCheckException(msg)
        data_list = record.get("data")
        if not data_list:
            raise NoCheckException(err_msg)
        return data_list

    def execute_download(self):
        # 下载信息到本地
        req_param = HashMap()
        req_param.put("type", "download")
        ContainerRestService.get_file(
            self.rest, self.uri, req_param, self.local_dir)

    def collect_alarm(self):
        """
        收集容器告警信息-因使用post下载，所以需在框架侧增加post接口execPostDownloadFile。
        :return:
        """
        try:
            flag, msg = self.init_rest_connection()
            if not flag:
                return flag, msg
            req_param = HashMap()
            response_info = self.rest.execPostDownloadFile(
                self.rest.getBaseUrl() + self.uri, req_param,
                self.local_dir, None, None
            )
            return self.check_response(response_info)
        except NoCheckException as e:
            self.logger.error(self.dev_obj, str(traceback.format_exc()))
            return False, e.error_msg
        except (JException, Exception):
            self.logger.error(self.dev_obj, str(traceback.format_exc()))
            err_msg = util.getMsg(self.dev_obj, "failed.collect.information")
            return False, err_msg
        finally:
            if self.rest:
                release_container_rest(self.dev_node)

    def check_response(self, response_info):
        """
        检查告警信息是否收集成功
        :param response_info:
        :return:
        """
        error_msg = util.getMsg(self.dev_obj, "failed.collect.information")
        data_string = response_info.getContent()
        # io 流返回成功的话，响应为空
        if not data_string:
            return True, ''

        response_dict = json.loads(data_string)
        error_code = response_dict.get("errorCode")
        if error_code and str(error_code) == '0':
            return True, ''

        msg = response_dict.get("errorMessage")
        if not msg:
            msg = error_msg
        return False, msg

    def init_rest_connection(self):
        self.rest = get_container_rest(self.dev_node)
        if self.rest:
            return True, ''
        return False, util.getMsg(self.dev_obj, "failed.establish.rest.connection")
