# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
import time
import threading

from com.huawei.ism.tool.base.utils import FileUtils
from collect import collectAllLog
from collect import collectSmart
from collect import collectDHA
from common.export_depth_collect_items_info import ExportDepthCollectItemsInfo
from common.config import config
from common.dealSmartInfo import dealSmartInfo
from common.util import log
from common.util import util

ALL_FAILED = True


def depth_concurrent_collect(dev_obj):
    """
    深度问题分析并发收集入口
    :param dev_obj: dev对象
    :return: 是否收集成功，错误信息
    """
    progress_lock = threading.Lock()
    dev_obj["progress_lock"] = progress_lock
    log.info(dev_obj, "start concurrent collect depth items.")
    util.refresh_depth_collect_process(dev_obj, 5, progress_lock)
    util.initPyDetailMsg(dev_obj)
    flag = collect_depth_items(dev_obj)
    util.refresh_depth_collect_process(dev_obj, 100, progress_lock)
    return flag, ""


def collect_more_all_log(dev_obj, result_dict):
    res, msg = collectAllLog(dev_obj)
    if res:
        global ALL_FAILED
        ALL_FAILED = False
    result_dict["all_log"] = {"collect_res": res}


def parse_result_dict(dev_obj, result_dict):
    log.info(dev_obj, "start parse result dict... {}".format(result_dict))
    result_info_map = dev_obj.get("resultInfoMap")
    is_all_logs = dev_obj.get("collectAllLogs")
    if result_dict.get("all_log"):
        item_name = "all_logs" if is_all_logs else "all_log"
        all_log_result = result_info_map.get(item_name)
        all_log_result.put("is_success", result_dict.get("all_log").get("collect_res"))
        all_log_result.put("err_msg", dev_obj.get("py_detail"))
    global ALL_FAILED
    if result_dict.get("smart") and result_dict.get("smart").get("collect_res"):
        smart_result = result_info_map.get("smart")
        smart_result.put("is_success", True)
        smart_result.put("err_msg", result_dict.get("smart").get("err_msg"))
        ALL_FAILED = False
    if result_dict.get("dha") and result_dict.get("dha").get("collect_res"):
        dha_result = result_info_map.get("dha")
        dha_result.put("is_success", True)
        dha_result.put("err_msg", result_dict.get("dha").get("err_msg"))
        ALL_FAILED = False


def add_result(dev_obj, collect_item, is_success, err_msg):
    if is_success:
        global ALL_FAILED
        ALL_FAILED = False
    result = dev_obj.get("resultInfoMap").get(collect_item)
    result.put("is_success", is_success)
    result.put("err_msg", err_msg)


def collect_depth_items(dev_obj):
    log.info(dev_obj, "start concurrent collect depth items.")
    result_dict = {}
    thread_list = []
    no_retry_item_type = ["all_log", "all_logs"]
    origin_collect_method_dict = {
        "smart": collectSmart,
        "dha": collectDHA,
    }
    collect_concurrent_item_info(dev_obj, no_retry_item_type, result_dict, thread_list)
    util.refresh_depth_collect_process(dev_obj, 60, dev_obj.get("progress_lock"))

    parse_result_dict(dev_obj, result_dict)
    # smart信息收集成功或者部分成功需要做处理
    if config.COLLECT_TYPE_SMART in result_dict.keys() \
            and "Successful" in result_dict.get(config.COLLECT_TYPE_SMART).get("status"):
        is_success_parse, err_msg_key = dealSmartInfo(dev_obj, True)
        tmp_dir = os.path.join(dev_obj[config.COLLECT_INFO_LOCAL_PATH], "temp")
        util.refresh_depth_collect_process(dev_obj, 70, dev_obj.get("progress_lock"))
        if os.path.exists(tmp_dir):
            FileUtils.deleteFile(tmp_dir)
        if not is_success_parse:
            add_result(dev_obj, config.COLLECT_TYPE_SMART, is_success_parse, util.getMsg(dev_obj, err_msg_key))

    progress = 70
    # 判断结果 走老逻辑单独串行收集: 主要是判断smart 和 dha 是否需要重试老逻辑
    for collect_item, collect_res in result_dict.items():
        if not collect_res.get("collect_res", False) and collect_item not in no_retry_item_type:
            util.initPyDetailMsg(dev_obj)
            is_success, msg = origin_collect_method_dict.get(collect_item)(dev_obj)
            add_result(dev_obj, collect_item, is_success, dev_obj.get("py_detail"))
        progress += 10
        util.refresh_depth_collect_process(dev_obj, progress, dev_obj.get("progress_lock"))

    return not ALL_FAILED


def collect_concurrent_item_info(dev_obj, no_retry_item_type, result_dict, thread_list):
    collect_item_list = dev_obj.get("depthCollectItemsName")
    for collect_type in collect_item_list:
        if collect_type in no_retry_item_type:
            check_thread = threading.Thread(
                target=collect_more_all_log, args=(dev_obj, result_dict,)
            )
            check_thread.setName("collect_thread_{}".format(collect_type))
            thread_list.append(check_thread)
        else:
            collect_item = ExportDepthCollectItemsInfo(dev_obj, log, config.COLLECT_ITEM.get(collect_type), result_dict)
            check_thread = threading.Thread(
                target=collect_item.execute_collect,
            )
            check_thread.setName("collect_thread_{}".format(collect_type))
            thread_list.append(check_thread)
    start_thread(thread_list)


def start_thread(thread_list):
    # 启动所有线程
    for check_thread in thread_list:
        time.sleep(0.01)
        check_thread.start()

    # 主线程中等待所有子线程退出
    for check_thread in thread_list:
        check_thread.join()
