# -*- coding: UTF-8 -*-
import os
import re
import traceback
import time
import datetime
from common.util import log, util, systemMode, device
from common.config import config
from common.cliFactory import cli

CREAT_EVENT_NAME_LIST = ["Succeeded In Creating A Thin LUN", "Succeeded In Creating A THICK LUN",
                         "Succeeded In Creating A Snapshot", "Succeeded In Creating A File System",
                         "Succeeded In Creating A File System Snapshot"]
DELETE_EVENT_NAME_LIST = ["Succeeded In Deleting A LUN", "Succeeded In Deleting A Snapshot",
                          "Succeeded In Deleting A File System", "Succeeded In Deleting A File System Snapshot"]
CACHE_CHECK_RISK_VERSION = ["V300R003C20", "V300R003C20SPC100", "V300R003C20SPC200", "V300R005C00", "V300R005C00SPC300",
                            "V300R005C01", "V300R006C00", "V300R006C00SPC100", "V300R006C01", "V300R006C10",
                            "V300R006C10SPC100",
                            "V300R006C20", "V500R007C00", "V500R007C00SPC100", "V500R007C10",
                            "V500R007C10SPC100", "V300R006C20SPC100","V500R007C10SPC100", "V300R006C20SPC100"]


CACHE_CHECK_RISK_VERSION_TO_PATCH = {
    "V300R005C00SPC300": "V300R005C00SPH308",
    "V300R006C00SPC100": "V300R006C00SPH111",
    "V500R007C00SPC100": "V500R007C00SPH105",
    "V300R006C10SPC100": "V300R006C10SPH105",
    "V500R007C10": "V500R007C10SPH006",
    "V300R006C20": "V300R006C20SPH006",
    "V500R007C10SPC100": "V500R007C10SPH115",
    "V300R006C20SPC100": "V300R006C20SPH115",
    "V300R003C20SPC200": "V300R003C20SPH217",
}


'''
收集前准备
删除用户选择的路径中的temp目录
'''


def preCollect(devObj):
    try:
        util.refreshProcess(devObj, 1)
        loginUserName = devObj.get("username")
        log.info(devObj, "loginUserName is %s" % str(loginUserName))
        util.refreshProcess(devObj, 20)
        # 判断设备是否处在升级状态
        isUpgrading = util.isSystemUpgrading(devObj)
        if isUpgrading:
            log.error(devObj, "The system is upgrading.")
            return (False, "")
        # 检查用户名称是否非法（部分关键字作为用户名将影响CLI执行结果判断）
        isPass = checkLoginUserName(loginUserName)
        if not isPass:
            util.setPyDetailMsg(devObj, "loginUser.name.check.failure")
            return (False, "")
        util.refreshProcess(devObj, 40)
        productModel = device.getDeviceType(devObj)
        # 兼容OEM后的设备型号
        if devObj.get("isOEMTool"):
            productModel = devObj.get("devNode").getOemModel()
        util.refreshProcess(devObj, 60)
        # 除去产品型号中的空格
        productTime = util.getCurrentDate()
        localFileFinal = "Data_%s_%s_%s" % (str(productModel).replace(" ", ""), str(productTime),
                                            str(devObj.get("devNode").getDeviceSerialNumber()))
        localDir = util.getLocalDir(devObj) + os.path.sep + localFileFinal + os.path.sep + "DataCollect"
        flag = util.check_contain_chinese(unicode(localDir).encode("utf-8"))
        if flag:
            util.setPyDetailMsg(devObj, "route.exist.ch")
            return False, ""
        if not os.path.exists(localDir):
            os.makedirs(localDir)
        util.refreshProcess(devObj, 80)
        # 将保存的路径通知框架，做显示
        devObj["collectRetFileName"] = localFileFinal
        devObj[config.COLLECT_INFO_LOCAL_PATH] = localDir
        # DNS服务器探测配置检查
        isPass, errMsg = checkHotPatchVersionIsRisk(devObj)
        if not isPass:
            util.setPyDetailMsg(devObj, errMsg)
            return (False, "")
        util.refreshProcess(devObj, 90)
        # cache对象数量检查
        checkResult, errMsg, errArgs = CheckCreateAndDeleteLUN(devObj)
        if checkResult != True:
            util.setPyDetailMsg(devObj, errMsg, errArgs)
            return (checkResult, "")
        util.refreshProcess(devObj, 99)
        return (True, "")
    except Exception:
        log.error(devObj, "tool: pre_collect except trace back:" + str(traceback.format_exc()))
        return (False, "")
    finally:
        util.refreshProcess(devObj, 100)


USER_NAME_BLACK_LIST = ["developer", "diagnose", "error", "password", "upgrade", "minisystem", "storage"]


def checkLoginUserName(loginUserName):
    '''
    @summary: 检查工具登录用户名称是否为特殊用户名
    '''

    loginUserName = loginUserName.lower()

    for keyWords in USER_NAME_BLACK_LIST:
        if loginUserName.endswith(keyWords):
            return False

    return True


def checkHotPatchVersionIsRisk(devObj):
    '''
    2. 在信息收集 precollect 检查的时候 增加DNS服务器地址数量的检查。
        操作步骤：
                    步骤1执行如下命令：show upgrade package，查询系统版本与热补丁版本；
        步骤2执行如下命令：show dns_server general，获取当前配置DNS Server探测配置列表 IP Address List的地址个数 。
        
        判断标准：
        1、    当系统版本为V300R003C20、V300R003C20SPC200、V300R006C00SPC100时，检查补丁版本如果低于V300R003C20SPH001、V300R003C20SPH203、V300R006C00SPH105之一。则检查DNS server的个数是否大于1，如果大于1则不允许信息收集。
        2、    当系统版本为V300R003C20SPC100、V300R006C00、Dorado V300R001C00时，检查DNS server的个数是否大于1，如果大于1则不允许信息收集。如果小于等于1，则允许进行信息收集
    '''
    cmd = "show dns_server general"
    # 获取版本及补丁
    issucess, softwareVersionList, hotPatchVersionList = device.parse_upgradePackage(devObj)
    # 判断是否含异常场景
    if issucess != True:
        return (False, "")

    # 检查版本及补丁是否为风险版本
    flag = isRiskVersion(devObj, softwareVersionList, hotPatchVersionList)

    if flag != True:
        # 获取回显中"IP Address List"的中IP地址的个数
        flag, errMsg = checkDnsServerIp(devObj, cmd)
        if flag != True:
            return (False, errMsg)
        return (True, "")

    return (True, "")


def isRiskVersion(devObj, softwareVersionList, hotPatchVersionList):
    '''
    @summary: 判断当前阵列版本和补丁版本是否存在风险，如果阵列版本在riskVersionDict中且补丁版本低于如果阵列版本在riskVersionDict中
    补丁版本，则返回False。如果阵列版本在riskVersionDict中补丁版本为--，直接返回False
    @param softwareVersionList: 当前阵列的版本所有控制器的版本列表
    @param hotPatchVersionList: 当前阵列的版本所有控制器的补丁版本列表
    retrun：False：表示是风险版本
            True：表示不是风险版本
    '''
    # 系统版本及热补丁风险版本
    riskVersionDict = {'V300R003C20': 'V300R003C20SPH001',
                       'V300R003C20SPC100': '--',
                       'V300R003C20SPC200': 'V300R003C20SPH203',
                       'V300R006C00': '--',
                       'V300R006C00SPC100': 'V300R006C00SPH105',
                       'V300R001C00': '--'}
    # 检查系统版本
    flag, currentSoftVersion = device.getCurrentVersion(devObj, softwareVersionList)
    if not flag:
        return False

    currentSoftVersion = currentSoftVersion.strip()

    # 检查热补丁版本
    flag, currentHotpatchVersion = device.getCurrentVersion(devObj, hotPatchVersionList)
    if not flag:
        return False

    # 对版本型号及热补丁版本进行适配,如果系统版本不涉及则通过
    if currentSoftVersion not in riskVersionDict:
        return True

    hotpatchVersion = riskVersionDict.get(currentSoftVersion)
    # 如果热补丁版本匹配则通过
    if hotpatchVersion != '--' and currentHotpatchVersion.strip() >= hotpatchVersion:
        return True

    # 如果热补丁版本不匹配
    log.info(devObj, "hot patch version is lower than request")
    return False


def checkDnsServerIp(devObj, cmd):
    '''
    @summary: 提取出IP列表中地址数量
    @param cliRet: 收集到的信息回显
    @return: flag , errMsg
    '''

    flag, cliRet = cli.executeCmdWithTimoutAndRetry(devObj, cmd, 120)
    if not flag:
        return (True, "")

    ipaddList = 'IP Address List'
    switchStatus = 'Check Switch'
    ipAddressStr = ""
    checkSwitch = ""
    dnsServerCliRet = systemMode.getVerticalCliRet(cliRet)
    try:
        if len(dnsServerCliRet) != 0:
            if switchStatus in dnsServerCliRet[0]:
                checkSwitch = dnsServerCliRet[0].get(switchStatus)
                if checkSwitch == "Off":
                    return (True, "")
            ipAddressStr = dnsServerCliRet[0].get(ipaddList)
            # 如果存在关键字
            if ipAddressStr is not None:
                ipList = ipAddressStr.split()
                if len(ipList) > 1:
                    errMsg = util.getMsg(devObj, "dns.server.disposelist.overstep", len(ipList))
                    return (False, errMsg)
                else:
                    return (True, "")
        else:
            # 没有关键字的情况下
            return (True, "")

    except Exception:
        log.error(devObj, "tool: pre_collect except trace back:" + str(traceback.format_exc()))
        return (False, "")


def CheckCreateAndDeleteLUN(devObj):
    '''
    检查是存在创删LUN，文件系统，快照动作。
    :param cli:
    :return:
    '''
    errMsg = ""
    needHotPatchVersion = ""
    isCheck = False
    # 检查用户是否勾选系统日志
    selectItems = devObj.get("collectInfo").getCollectConfInfo().getSelectItems()
    # 如果没有勾选系统日志收集项则不做检查
    for selectItem in selectItems:
        log.info(devObj, "Get select item id:%s" % str(selectItem.getId()))
        if str(selectItem.getId()) in ["log", "all_log"]:
            isCheck = True
            break
    if isCheck == False:
        return (True, "", needHotPatchVersion)
    # 检查是否风险版本
    flag, softwareVersionList, hotPatchVersionList = device.parse_upgradePackage(devObj)
    if flag != True:
        return (False, "cannot.get.product.version.info", needHotPatchVersion)
    flag, softwareVersion = device.getCurrentVersion(devObj, softwareVersionList)
    if flag != True:
        return (False, "cannot.get.product.version.info", needHotPatchVersion)
    flag, hotPatchVersion = device.getCurrentVersion(devObj, hotPatchVersionList)
    if flag != True:
        return (False, "cannot.get.hotpatch.version.info", needHotPatchVersion)
    log.info(devObj, "Get softwareVersion:[%s] hotPatchVersion:[%s]" % (softwareVersion, hotPatchVersion))

    if softwareVersion not in CACHE_CHECK_RISK_VERSION:
        return (True, "", needHotPatchVersion)
    needHotPatchVersion = CACHE_CHECK_RISK_VERSION_TO_PATCH.get(softwareVersion, "")
    if needHotPatchVersion != "" and hotPatchVersion != "--" and hotPatchVersion >= needHotPatchVersion:
        return (True, "", needHotPatchVersion)

    flag, curTime, errMsg = getCurTime(devObj)
    if flag != True:
        return (False, errMsg, needHotPatchVersion)
    curTime_match = re.match(
        "[1-9]\d{3}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])/(20|21|22|23|[0-1]\d):[0-5]\d:[0-5]\d", curTime)
    if not curTime_match:
        return (False, "cannot.get.system.time.info", needHotPatchVersion)
    curTime = curTime_match.group()
    flag, computableCurTime = Time2computableTime(devObj, curTime, "%Y-%m-%d/%H:%M:%S")
    if flag != True:
        return (False, "cannot.get.system.time.info", needHotPatchVersion)
    fromOneHourTime = (
            computableCurTime - datetime.timedelta(hours=1)).strftime(
        "%Y-%m-%d/%H:%M:%S"
    )
    # 检查1小时前到现在的事件
    flag, deleteEventList, errMsg = get_del_event_by_time(
        devObj, fromOneHourTime
    )
    if flag != True:
        return (False, errMsg, needHotPatchVersion)
    if len(deleteEventList) <= 5:
        return (True, "", needHotPatchVersion)
    if needHotPatchVersion != "":
        log.info(devObj, "Check no pass and need install hot patch.")
        return (False, "check.remain.obj.install.patch", needHotPatchVersion)
    else:
        log.info(devObj, "Check no pass and no need to install hot patches.")
        return (False, "check.remain.obj.no.pass", needHotPatchVersion)


def getCurTime(devObj):
    '''
    获取系统时间
    :param cli:
    :return:
    '''
    sysTime = ""
    cmd = "show system general"
    flag, cliRet = cli.executeCmdWithRetry(devObj, cmd)
    if not flag:
        return (False, sysTime, "cannot.get.system.time.info")
    cliRetDictList = util.getVerticalCliRet(cliRet)
    if len(cliRetDictList) == 0:
        return (False, "", "cannot.get.system.time.info")
    cliRetDict = cliRetDictList[0]
    sysTime = cliRetDict.get("Time")
    return (True, sysTime, "")


def get_del_event_by_time(devObj, startTime):
    '''
    :param startTime:开始时间
    :return: flag, deleteEnvenList
    '''
    errMsg = ""
    deleteEventList = []
    cmd = "show event level=informational from_time=%s" % startTime
    flag, ret = cli.executeCmdWithRetry(devObj, cmd)
    if not flag:
        return False, deleteEventList, "cannot.get.event.info"
    eventDictList = util.getHorizontalCliRet(ret)
    for enventDict in eventDictList:
        if enventDict.get("Name") in DELETE_EVENT_NAME_LIST:
            deleteEventList.append(enventDict)
    return True, deleteEventList, ""


def Time2computableTime(devObj, timeStr, timeFormat):
    '''
    将字符串时间转换成可计算的时间Time format
    :param timeStr: 待转换的时间字符串
    :param timeFormat: 带转换的时间字符串格式
    :return: （成功标志，可计算的时间）
    '''
    try:
        struct_time = time.strptime(timeStr, timeFormat)
        computableTime = datetime.datetime(struct_time.tm_year, struct_time.tm_mon, struct_time.tm_mday,
                                           struct_time.tm_hour, struct_time.tm_min, struct_time.tm_sec)
        return (True, computableTime)
    except:
        log.error(devObj, str(traceback.format_exc()))
        return (False, None)
