# coding:utf-8

import os
import shutil
import traceback
import subprocess
import stat
import time
from common.util import log, util
from common.constants import COMLOG9000FILE, COLLECT_MODULE_LIST


def getDeviceSn(dev_object):
    """
                获取设备SN号
    """

    devNode = dev_object.get("devNode")
    if None == devNode:
        return ""
    sn = devNode.getDeviceSerialNumber()
    return sn


def move_file(orgin_path, moved_path):
    dir_files = os.listdir(orgin_path)
    for file1 in dir_files:
        file_path = os.path.join(orgin_path, file1)
        if os.path.isfile(file_path):
            shutil.move(file_path, moved_path)


def retry_delete_file(dev_object, source_file):
    try_time_list = [1, 2, 3]
    for index in try_time_list:
        try:
            os.remove(source_file)
        except Exception as error:
            log.error(dev_object, "Failed to delete file:%s" % source_file)
            time.sleep(5)

        if os.path.exists(source_file):
            break


def clean_temp_dir(dev_object, input_file_path):
    """
    清理临时目录，当文件太大时，第一次删除因windows内部处理问题导致清理会失败，
    文件残留，需要重试。
    :return:
    """

    if not os.path.exists(input_file_path):
        log.info("already delete success! dir=%s ." % input_file_path)
        return
    time_out = 60
    add_permission(dev_object, input_file_path)
    while time_out > 0:
        try:
            shutil.rmtree(input_file_path, ignore_errors=False)
            log.info(dev_object, "delete success! dir=%s ." % input_file_path)
            break
        except Exception:
            log.error(dev_object,
                      "clean_temp_dir Failed! dir=%s ." % input_file_path)
        time_out -= 1
        time.sleep(5)


def add_permission(dev_object, input_path):
    """
    递归增加目标目录下所有权限
    :param input_path:
    :return:
    """

    if os.path.isdir(input_path):
        file_dir_list = os.listdir(input_path)
        # 开始解压缩迭代及写入硬盘信息
        for file_dir in file_dir_list:
            file_path = os.path.join(input_path, file_dir)
            add_permission(dev_object, file_path)
    else:
        try:
            os.chmod(input_path, stat.S_IWRITE)
        except Exception:
            log.error(dev_object,
                      "add permission failed! %s " % input_path)


def decompress_common_log(dev_object, data_path, tmp_data_path, tmp_collect_path):
    cmd7z = dev_object.get("7z_command")

    curComLogFile = os.path.join(dev_object.get("collectRetDir"),
                                 COMLOG9000FILE)
    if not os.path.exists(curComLogFile):
        return (True, "")
    # System_log目录
    systemlogPath = os.path.join(dev_object.get("collectRetDir"), "System_log")

    strCmd = "\"%s\" x \"%s\" -o\"%s\"" % (cmd7z,
                                           curComLogFile, tmp_data_path)

    log.info(dev_object, "decompress normal start")

    p = subprocess.Popen(strCmd,
                         stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE,
                         shell=False)
    uncompressRes = p.stdout.read()

    # 解压成功才删除压缩文件，否则保留压缩文件
    if uncompressRes and uncompressRes.find("Everything is Ok") != -1:
        if os.path.isdir(tmp_collect_path):
            shutil.move(tmp_collect_path, data_path)
            retry_delete_file(dev_object, curComLogFile)
            if os.path.exists(curComLogFile):
                log.error(dev_object, "failed to delete file:%s" % curComLogFile)
                return (False, "Failed to delete file DataCollect_OceanStor9000.zip.")

            # 将System_log日志MV到tmp目录下
            if os.path.isdir(systemlogPath):
                move_file(systemlogPath,
                          os.path.join(data_path, "System_log"))
                clean_temp_dir(dev_object, systemlogPath)
        else:
            log.error(dev_object, "decompress Fail,tmp_collect_path=%s"
                      % tmp_collect_path)
            return (False, "create data collect directory failed.")
    else:
        # 解压失败，将temp目录重命名
        log.error(dev_object,
                  "decompress normal Failed,file=%s. Res=%s."
                  % (curComLogFile, uncompressRes))
        return (False, "decompress normal Failed.")
    return (True, "")


def decompress_other_log(dev_object, data_path, tmp_data_path, tmp_collect_path):
    cmd7z = dev_object.get("7z_command")

    if not os.path.exists(data_path):
        isSuccess = util.createDir(dev_object, data_path)
        if not isSuccess:
            log.error(dev_object, "create dir Failed, data_path=%s" % data_path)
            return (False, "create directory Failed.")

    # 重新创建临时文件
    if os.path.isdir(tmp_data_path):
        clean_temp_dir(dev_object, tmp_data_path)
    isSuccess = util.createDir(dev_object, tmp_data_path)
    if not isSuccess:
        log.error(dev_object, "create dir Failed,tmp_data_path=%s" % tmp_data_path)
        return (False, "create data tmp directory Failed.")

    for key in COLLECT_MODULE_LIST:
        log.info(dev_object, "compress other start: %s" % key)
        otherPathName = os.path.join(dev_object.get("collectRetDir"),
                                     COLLECT_MODULE_LIST.get(key))
        if os.path.exists(otherPathName):
            strCmd = "\"%s\" x \"%s\" -o\"%s\"" % (cmd7z,
                                                   otherPathName,
                                                   tmp_data_path)
            p = subprocess.Popen(strCmd,
                                 stdin=subprocess.PIPE,
                                 stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE,
                                 shell=False)
            decompRes = p.stdout.read()
            if decompRes and decompRes.find("Everything is Ok") != -1:
                log.info(dev_object,
                         "decompress other success,file=%s. ret=%s. cmd=%s."
                         % (otherPathName, decompRes, strCmd))
            else:
                log.error(dev_object,
                          "decompress other Failed,file=%s. ret=%s. cmd=%s."
                          % (otherPathName, decompRes, strCmd))

            if not os.path.exists(os.path.join(tmp_data_path, key)):
                shutil.move(tmp_collect_path, os.path.join(data_path, key))
            else:
                shutil.move(os.path.join(tmp_data_path, key),
                            os.path.join(data_path, key))
            
            retry_delete_file(dev_object, otherPathName)
            if os.path.exists(otherPathName):
                log.error(dev_object, "failed to delete file:%s" % otherPathName)
                return (False, "Failed to delete file:%s." % otherPathName)
    
            log.info(dev_object,
                     "start compress key:%s--tempPath=%s,src=%s,dest=%s"
                     % (key,
                        COLLECT_MODULE_LIST.get(key),
                        os.path.join(tmp_data_path, key),
                        os.path.join(data_path, key)))


def create_tmp_dir(dev_object, tmp_collect_data_dir, tmp_data_path):
    # 创建DataCollect目录
    isSuccess = util.createDir(dev_object, tmp_collect_data_dir)
    if not isSuccess:
        log.error(dev_object, "create dir Failed,tmp_collect_data_dir=%s" % tmp_collect_data_dir)
        return (False, "create data collect directory Failed.")

    # 创建解压存放临时文件目录
    isSuccess = util.createDir(dev_object, tmp_data_path)
    if not isSuccess:
        log.error(dev_object, "create dir Failed,tmp_data_path=%s" % tmp_data_path)
        return (False, "create directory Failed.")
    return (True, "")


def delete_tmp_dir(dev_object, tmp_collect_data_dir, tmp_data_path):
    # 删除临时文件
    if os.path.isdir(tmp_data_path):
        clean_temp_dir(dev_object, tmp_data_path)

    if os.path.isdir(tmp_collect_data_dir):
        clean_temp_dir(dev_object, tmp_collect_data_dir)


def remove_data(dev_object, data_path):
    if os.path.isdir(data_path):
        try:
            clean_temp_dir(dev_object, data_path)
        except Exception:
            rm_cmd = "rd/s/q " + data_path
            if os.path.isdir(data_path):
                subprocess.call(rm_cmd)

    elif os.path.isfile(data_path):
        retry_delete_file(dev_object, data_path)


def cutting_log(dev_object, unzip_data_dir):
    # System_log目录
    systemlogPath = os.path.join(dev_object.get("collectRetDir"), "System_log")
    # isTiming直接删除System_log日志
    if os.path.exists(systemlogPath):
        clean_temp_dir(dev_object, systemlogPath)

    if os.path.isdir(unzip_data_dir):
        # \DataCollect
        for dir_name in os.listdir(unzip_data_dir):
            if dir_name != "disksmartinfo":
                remove_data(dev_object, os.path.join(unzip_data_dir, dir_name))


def filter_disk_valid_info(dev_object, z_path):
    def zip_extract_all(z_path, unzip_dir):
        tar_class = dev_object.get("PYENGINE.PY_ZIP")
        tar_class.decompressZipFile(z_path, unzip_dir)

    mylog = dev_object.get("logger")
    mylog.info("the Timing collect start the log cutting")
    if os.path.exists(z_path):
        unzip_dir = os.path.dirname(z_path)
        unzip_data_dir = os.path.join(unzip_dir, "DataCollect")
        remove_data(dev_object, unzip_data_dir)

        zip_extract_all(z_path, unzip_dir)
        retry_delete_file(dev_object, z_path)

        # Cutting log
        cutting_log(dev_object, unzip_data_dir)

        # repack
        z_exe_path = dev_object.get("7z_command")
        cmd = z_exe_path + " a " + z_path + " " + unzip_data_dir
        subprocess.call(cmd)
        remove_data(dev_object, unzip_data_dir)
    mylog.info("the Timing collect End the log cutting")


def execute(dev_object):
    log.info(dev_object, "compress start")

    collect_ret_dir = dev_object.get("collectRetDir")
    log.info(dev_object, "Collect Return Directory=%s" % collect_ret_dir)
    curComLogFile = os.path.join(dev_object.get("collectRetDir"), COMLOG9000FILE)
    cmd7z = dev_object.get("7z_command")
    log.info(dev_object, "The path of 7z excutable=%s" % cmd7z)
    if not os.path.exists(cmd7z):
        log.error(dev_object, "The 7z excutable=%s Not exist." % cmd7z)
        return (False, "The 7z excutable does not exist.")
    if dev_object.get("isTiming"):
        filter_disk_valid_info(dev_object, curComLogFile)
        log.info(dev_object, "decompress success,ret=%s." % curComLogFile)
        return (True, "")

    try:
        tmp_collect_data_dir = os.path.join(dev_object.get("collectRetDir"), "DataCollect")
        # 解压各个模块的信息后，移动文件到此目录,最后打包目录
        dataDir = os.path.join(tmp_collect_data_dir, "DataCollect")
        # 解压目录：tmpDataCollect；创建临时目录：create_tmp_dir
        tmpDirPath = os.path.join(dev_object.get("collectRetDir"), "temp")
        tmpDataCollect = os.path.join(tmpDirPath, "DataCollect")

        isSuccess, errinfo = create_tmp_dir(dev_object, tmp_collect_data_dir, tmpDirPath)
        if not isSuccess:
            return (False, errinfo)

        # 解压常用日志（common_log）、其他模块日志（other_log）
        isSuccess, errinfo = decompress_common_log(dev_object, dataDir,
                                                   tmpDirPath, tmpDataCollect)
        if not isSuccess:
            return (False, errinfo)

        decompress_other_log(dev_object, dataDir, tmpDirPath, tmpDataCollect)

        create_package_info(tmp_collect_data_dir, dev_object)

        # 压缩后最终生成的压缩包名
        localFileFinal = os.path.join(dev_object.get("collectRetDir"),
                                      dev_object.get("collectRetFileName"))
        strCmd = "\"%s\" a \"%s\" \"%s\\*\" -y -mx0" % (cmd7z, localFileFinal, tmp_collect_data_dir)
        archiving_process = subprocess.Popen(strCmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, shell=False)
        subcompressRes = archiving_process.stdout.read()

        if subcompressRes and subcompressRes.find("Everything is Ok") != -1:
            log.info(dev_object, "sub compress success,file=%s.ret=%s. cmd=%s."
                     % (localFileFinal, subcompressRes, strCmd))
            return (True, "")
        else:
            log.error(dev_object, "sub compress Failed,file=%s.ret=%s. cmd=%s."
                      % (localFileFinal, subcompressRes, strCmd))
            return (False, "")

    except Exception as e:
        log.error(dev_object, "catch except when compress file,errinfo=%s."
                  % str(e).encode("utf-8"))
        log.error(dev_object, "catch except when compress file,info:%s"
                  % str(traceback.format_exc()).encode("utf-8"))
        dev_object["py_detail"] = "catch except when compress file."
        return (False, "")
    finally:
        delete_tmp_dir(dev_object, tmp_collect_data_dir, tmpDirPath)


def create_package_info(root_path, dev_obj):
    '''
    @summary: 创建与datacollect同级的目录pkginfo，用于表示该包是信息收集的包。
        pkginfo下有两个空文件
            type_collect: 信息收集的包
            domain_storage: 存储设备
    '''
    info_path = os.path.realpath(root_path + os.sep + "pkginfo")
    type_file = info_path + os.sep + "type_collect"
    domain_file = info_path + os.sep + "domain_storage"
    desc_file = info_path + os.sep + "desc"
    sn_string = "deviceSN=${%s}\n"
    time_string = "fileCreateTime=%s"
    try:
        # 如果pkginfo已存在，说明工具侧已经创建好目录
        if os.path.exists(info_path):
            log.info(dev_obj, "[creatPkginfo] pkginfo path exists!")
        else:
            os.mkdir(info_path)
        # 先创建type文件，然后拷贝type并重命名为domain文件
        with open(type_file, "w") as f:
            f.write("")
        shutil.copy(type_file, domain_file)
        # 创建Desc文件并写入SN号和时间（UTC）
        # 获取SN号和时间毫秒数
        sn_info = sn_string % str(getDeviceSn(dev_obj))
        time_info = time_string % str(long(time.time() * 1000))
        desc_file = open(desc_file, "w")
        seq = [sn_info, time_info]
        desc_file.writelines(seq)
        desc_file.close()
        log.info(dev_obj, "[creatPkginfo] Creat the pkginfo succeed!")
    except Exception:
        if os.path.exists(info_path):
            shutil.rmtree(info_path, True)
        log.error(dev_obj, "[creatPkginfo] Creat the pkginfo fail {}".format(
            str(traceback.format_exc())))
