# -*- coding: UTF-8 -*-

import re
import shutil
import os


from defusedxml import ElementTree


#信息收集项配置参数
G_COLLECT_TYPE_ITEMS = {

"ParseXML":     [
                 "verType",     #配置信息对应的版本类型
                 "fileName",    #需要解析的XML文件（配置所有收集命令）
                 "resultFile"   #收集项结果文件名称
                 ],
"ExportCmd":    [
                 "verType",     #配置信息对应的版本类型
                 "exportCmd",   #需要执行的Cli主命令
                 "inputFileName",   #export命令的-f参数: 有些只需要输入路径、有些只需要输入文件名、有的需要输入路径和文件名
                 "DownLoadDir", #文件下载路径
                 "DownLoadFile",    #通过SFTP下载文件的文件名开头（系统自动生成的文件，由关键字+时间戳构成）
                 "resultFile"   #收集项结果文件名称
                 ],
"ExecCmd":      [
                 "verType",     #配置信息对应的版本类型
                 "execCmd",     #执行单条收集命令
                 "cmdModel",    #命令执行模式（cli、debug、mml）
                 "DownLoadFile", #文件下载路径（包含文件路径+文件名称）
                 "resultFile"   #收集项结果文件名称
                 ],
"ExecShell":      [
                 "verType",     #配置信息对应的版本类型
                 "shellCmd",     #执行shell脚本命令
                 "DownLoadFile",    #通过执行脚本生成的文件（路径 + 文件名）
                 "resultFile",   #收集项结果文件名称
                 "loginPeer",    #到对端执行命令，获取文件到本端
                 "renameByIp"    #以IP重新组织收集结果
                 ],
"DownRemoteFile": [
                   "verType",     #配置信息对应的版本类型
                   "fileName"     #配置文件的路径
                   ]
}

# **************************************************************************** #
# 函数名称: createCollectConfigDict
# 功能说明: 创建收集项的配置信息字典(解析所有配置项文档，如collectItem_S5000.xml)
# 输入参数: configFile, collectType
# 输出参数: True, False
# **************************************************************************** # 
#    <!--收集系统信息-->
#    <CollectItem name="SYSTEM_INFO">
#        <!--收集方式：ExecCmd（执行简单命令）、ExportCmd（执行导出命令）、ParseXML（执行XML中配置的所有命令）-->
#        <Method type="ParseXML">
#            <Item Keyword="fileName">config/S5000/System_S5000.xml</Item>
#            <Item Keyword="resultFile">System_information.txt</Item>
#        </Method>
#        <!--收集方式：ExecCmd（执行简单命令）、ExportCmd（执行导出命令）、ParseXML（执行XML中配置的所有命令）-->
#        <Method type="ExportCmd">
#            <Item Keyword="exportCmd">exportalm</Item>
#            <Item Keyword="inputFileName">False</Item>    <!--export命令的-f参数是否需要输入目标文件名（基于export命令的不兼容性）：True、False-->
#            <Item Keyword="DownLoadFile">hisalarm</Item>
#            <Item Keyword="resultFile"></Item>
#        </Method>
#    </CollectItem>
# **************************************************************************** # 
#解析结果:
#{
#'SYSTEM_INFO': 
#   [
#      {'ParseXML': {'resultFile': 'System_information.txt', 'fileName': 'config/S5000/System_S5000.xml'}}, 
#      {'ExportCmd': {'exportCmd': 'exportalm', 'DownLoadFile': 'hisalarm', 'inputFileName': 'False', 'resultFile': 'None'}}
#   ]
#}
# **************************************************************************** # 
def createCollectConfigDict(configFile, collectType):
    
    #所有收集项配置信息的字典
    collectItemConfigDict = {}
    
    #参数有效性校验
    if not collectType:
        return {}

    #获取所有的element
    root = ElementTree.parse(configFile)
    collectItemList = root.getiterator("CollectItem")
    #获取第一层信息：CollectItem列表
    for itemConfig in collectItemList:
        itemName = itemConfig.attrib["name"]
        
        #收集项名称匹配
        if itemName != collectType:
            continue

        #初始化单个配置项的详细信息(每一个collectItem可能包含多个Method)
        itemConfigDataDictList = []
        
        #填充单个收集项的配置map信息
        collectMethodList = itemConfig.getchildren()
        #获取第二层信息：Method列表
        for collectMethod in collectMethodList:
            typeKey = collectMethod.attrib["type"]
            
            #获取sub参数信息
            itemConfigDataDict = {}
            subItemDict = {}
            #获取第三层信息：item列表
            for subItem in collectMethod.getchildren():
                
                subItemKey = str(subItem.attrib["Keyword"])
                subItemValue = str(subItem.text)
                
                #记录所有Item配置
                subItemDict[subItemKey] = subItemValue
            
            #将每一个method配置以字典的形式保存,key值为type
            itemConfigDataDict[typeKey] = subItemDict

            #为避免两个method均为同一种收集方式,此处用列表保存所有method配置
            itemConfigDataDictList.append(itemConfigDataDict)
            
        #记录所有CollectItem配置
        collectItemConfigDict[itemName] = itemConfigDataDictList

    return collectItemConfigDict
        
# **************************************************************************** #
# 函数名称: checkConfigDictValid
# 功能说明: 检查配置信息字典的有效性(检查所有配置项文档，如collectItem_S5000.xml)
# 输入参数: ConfigDict
# 输出参数: True, False
# **************************************************************************** # 
#    备注：
#    type为"ParseXML"
#        参数（1）：fileName：需要解析的XML文件（配置所有收集命令）
#        参数（2）：resultFile：收集项结果文件名称
#    type为"ExportCmd"
#        参数（1）：exportCmd：需要执行的Cli主命令
#        参数（2）：inputFileName：export命令的-f参数是否需要输入目标文件名（基于export命令的不兼容性）：True、False
#        参数（3）：DownLoadFile：通过SFTP下载文件的文件名开头（系统自动生成的文件，由关键字+时间戳构成）
#        参数（4）：resultFile：收集项结果文件名称
#    type为"ExecCmd"
#        参数（1）：execCmd：执行单条收集命令
#        参数（2）：resultFile：收集项结果文件名称
# **************************************************************************** # 
# #解析结果:
#{
#'SYSTEM_INFO': 
#   [
#      {'ParseXML': {'resultFile': 'System_information.txt', 'fileName': 'config/S5000/System_S5000.xml'}}, 
#      {'ExportCmd': {'exportCmd': 'exportalm', 'DownLoadFile': 'hisalarm', 'inputFileName': 'False', 'resultFile': 'None'}}
#   ]
#}
# **************************************************************************** # 
def checkConfigDictValid(ConfigDict):

    for key in ConfigDict:
        #字典的列表
        collectMethodDictList  = ConfigDict.get(key)
        for collectMethodDict in collectMethodDictList:
            methodKey = collectMethodDict.keys()[0]
            #配置了不存在的收集类型
            if not G_COLLECT_TYPE_ITEMS.has_key(methodKey):
                return False

            #获取所有sub参数列表，并排序
            methodConfigDict = collectMethodDict.get(methodKey)
            configItemsList = methodConfigDict.keys()
            configItemsList.sort()
            
            #参数完整性校验
            standardTypeItemsList = G_COLLECT_TYPE_ITEMS.get(methodKey)
            #参数列表排序
            standardTypeItemsList.sort()
            
            #参数个数和类型必须完全匹配
            if standardTypeItemsList != configItemsList:
                return False
        
    return True

# **************************************************************************** #
# 函数名称: checkSymbolItemDict
# 功能说明: 检查symbol的参数是否正确
# 输入参数: itemDict
# 输出参数: True, False
# **************************************************************************** # 
def checkSymbolItemDict(itemDict):

    #命令参数为空，配置错误
    if not itemDict:
        return False
    
    #参数有效性检查
    if not itemDict.has_key("name"):
        return False
    
    #name为空判断
    symbolName = itemDict.get("name")
    if not symbolName:
        return False
    
    #参数为区间值（每一个值都要执行）
    if itemDict.has_key("min") and itemDict.has_key("max"):
        try:
            symbolMin = itemDict.get("min")
            symbolMax = itemDict.get("max")
            
            tempMin = int(symbolMin)
            tempMax = int(symbolMax)
            if tempMin >= tempMax:
                return False
            else:
                return True
        except:
            return False
    #例：参数为showlun查询到的所有ID，再查询详细信息
    elif itemDict.has_key("loop"):
        #loop关键字不允许为空
        if not itemDict.get("loop"):
            return False
        else:
            return True
    #参数为枚举值，如A/B控制器的场景
    elif  itemDict.has_key("rangeList"):
        #rangeList关键字必须包含|
        if itemDict.get("rangeList").find("|") == -1:
            return False
        else:
            return True
    #默认True，兼容后续新增配置类型
    else:
        return True          

# **************************************************************************** #
# 函数名称: getCmdListByParseFile
# 功能说明: 通过解析XML文件获取命令执行列表
# 输入参数: xmlFile
# 输出参数: collectCmdConfigDictList
# **************************************************************************** # 
def getCmdListByParseFile(xmlFile, logger):

    #所有收集命令
    collectCmdConfigDictList = []
    
    #获取所有的element
    root = ElementTree.parse(xmlFile)
    cmdConfigList = root.getiterator("command")
    
    #获取第一层信息：CollectItem列表
    for cmdConfig in cmdConfigList:
        
        #初始化字典结构
        collectCmdConfigDict = {"cmdBase":"", "items": []}
        
        #保存基础命令（无参数）
        cmdName = cmdConfig.attrib["name"]
        collectCmdConfigDict["cmdBase"] = cmdName
        
        #保存Cli命令的多个参数
        subConfigDataDict = {}
        subConfigDataDictList = []
        
        #以symbol作为迭代
        cmdSubConfigList = cmdConfig.getiterator("symbol")
        
        #存在三种情况：
        #（1）sub参数为空，则认为是不需要参数的命令，如showport
        #（2）命令存在一个参数，如showlun -lun
        #（3）命令存在多条参数，如：showiscsiroute -c a -p 00
        for subConfig in cmdSubConfigList:
            
            #获取参数字典
            subConfigDataDict = subConfig.attrib
            
            #参数有效性检查
            if not checkSymbolItemDict(subConfigDataDict):
                logger.error("[getCmdListByParseFile]: Configured data error:" + str(cmdName) + ", " + str(subConfigDataDict))
                #命令中只要一条参数配置信息出错，则整条丢弃
                break
            
            #有效的配置数据保存到字典中
            subConfigDataDictList.append(subConfigDataDict)

        #带参数的命令保存
        collectCmdConfigDict["items"] = subConfigDataDictList
    
        #所有命令汇总(数据结构定位列表是为了保持命令执行顺序与配置文件相同)
        collectCmdConfigDictList.append(collectCmdConfigDict)

    return collectCmdConfigDictList

# **************************************************************************** #
# 函数名称: getFileListByParseFile
# 功能说明: 通过解析XML文件获取下载文件列表
# 输入参数: xmlFile
# 输出参数: 
# **************************************************************************** # 
def getFileListByParseFile(xmlFile, logger):
    fileInfoDictList = []
    
    #获取所有的element
    root = ElementTree.parse(xmlFile)
    fileInfoList = root.getiterator("file")
    
    #获取第一层信息：CollectItem列表
    for fileInfo in fileInfoList:
        #初始化字典结构
        fileInfoDict = {"name":"", "isLatest":""}
        #
        fileName = fileInfo.attrib["name"]
        fileInfoDict["name"] = fileName
        
        isLatest = fileInfo.attrib["isLatest"]
        fileInfoDict["isLatest"] = isLatest
        
        fileInfoDictList.append(fileInfoDict)
    
    return fileInfoDictList
    
    


