# -*- coding: UTF-8 -*-
import re

#枚举类型
START_INDEX = 0
END_INDEX = 1

#常用匹配规则定义

#连续=分隔符
patternEqu = re.compile("^\s*=+\s*$")

#连续-分隔符
patternSub = re.compile("^\s*-+\s*$")

#连续*分隔符
patternStar = re.compile("^\s*\*+\s*$")

#空格分隔符
patternBlank = re.compile("^\s*$")

#回显结束标志
patternEnd = re.compile("^.*>.*$")

#行中不包含字母和数字
patternSymbol = re.compile("^\W*$")

# *****************************************************************************************#
# 函数名称: _isSkipLine(line)
# 功能说明: 符号行需要跳过（符号行即此行不包含任何字母或数字的情况）
# 输入参数: line
# 输出参数: True or False
# *****************************************************************************************#
def _isSkipLine(line):
    return not re.search("[A-Za-z0-9]", line) or "The system is in single controller status" in line

# *****************************************************************************************#
# 函数名称: _isEndLine(line)
# 功能说明: 是否为结束行
# 输入参数: line
# 输出参数: True or False
# *****************************************************************************************#
def _isEndLine(line):
    if re.match("^.*>\s*$", line):
        return True
    
    if re.match("^.*~\s*#\s*$", line):
        return True
    
    if re.match("^.*spu ui>\s*$", line):
        return True

class cTypeList:
    """
    # *****************************************************************************************#
    # 类名称: cTypeList（通用类）
    # 功能说明: 对于CLI回显，一行为关键字，下面所有行为其对应的value值格式，格式化为一个关键字对应
    #           value值的list(dict)结果，value值的行数即为list的长度
    # 输入参数: cliRet：开始某行为key值，下方为对应value值CLI回显格式
    #           fstKeyEndIndex：第一个key值对应的结束位置（用于判断value值是否存在换行的情况）
    #           keyPosInfo：传入list，单个元素为字典结构，key值对应value的起始，结束位置
    #           newLineSign：对于value值存在多行的情况，默认为直接连接
    # 传入格式说明：cliRetList行为key值对应的value行，handle函数对外提供
    #                 (0, 0)  Normal           FC             Double                             37  
    #                 (0, 1)  Normal           FC             Double                             38  
    #                 (0, 2)  Normal           FC             Double                             41  
    #                 (0, 3)  Normal           FC             Double                             36  
    # 输出参数: list(dict)
    # *****************************************************************************************# 
    """
    def __init__(self, cliRetList, fstKeyEndIndex, keyPosInfo, newLineSign=""):
        self.cliRetList = cliRetList
        self.fstKeyEndIndex = fstKeyEndIndex
        self.keyPosInfo = keyPosInfo
        self.newLineSign = newLineSign
    
    #入参检查
    def __checkParams(self):
        if not self.cliRetList:
            return False
        if - 1 == self.fstKeyEndIndex:
            return False
        if int != type(self.fstKeyEndIndex):
            return False
        if dict != type(self.keyPosInfo):
            return False
        return True
    
    #通用处理函数，格式化为一个关键字对应value值的list(dict)结果，value值的行数即为list的长度
    def handle(self, isAlign=False):
        result = []
        oneLineDict = {}
        
        if not self.__checkParams():
            return result

        for valInfoLine in self.cliRetList:
            
            if _isSkipLine(valInfoLine):
                continue
            
            if patternEnd.match(valInfoLine):
                break
            
            #当前行不为空，需要把上次的值赋到oneLineDict中，完成一行的赋值操作
            if valInfoLine[:self.fstKeyEndIndex].split():
                if oneLineDict:
                   #单行value值计算完成后，与key值合并成dict添加到result结果集中  
                   result.append(oneLineDict)
                   oneLineDict = {}
                       
                #keyPosInfo传入字典，单个元素key值对应value的起始位置
                for key in self.keyPosInfo.keys():
                    #为了兼容不同版本关键字的修复，中间可以传入不存在的关键字，对应返回None
                    if [-1, -1] != self.keyPosInfo[key]: 
                        oneLineDict[key] = self.__getValue(valInfoLine, self.keyPosInfo[key], isAlign)
                    else:
                        oneLineDict[key] = None
            
            #第一列ID为空，需要把相应位置的值增加到上一次的key值对应的value中
            else:
                #keyPosInfo传入字典，单个元素key值对应value的起始位置
                for key in self.keyPosInfo.keys():
                    #为了兼容不同版本关键字的修复，中间可以传入不存在的关键字，对应返回None
                    if [-1, -1] != self.keyPosInfo[key]: 
                        oneLineDict[key] += self.newLineSign + self.__getValue(valInfoLine, self.keyPosInfo[key])
                    else:
                        oneLineDict[key] = None
        #最后一行加入  
        if oneLineDict:
            result.append(oneLineDict)
            
        return result
 
    #有些产品value可能没有对齐于key，需要补充起始位置之外value是否还有相连的字母
    def __getValue(self, valInfoLine, keyPosInfo, isAlign=False):
        
        #Value行的值已经严格对齐了,不需要调整
        if isAlign:
            return valInfoLine[keyPosInfo[START_INDEX]:keyPosInfo[END_INDEX]].strip()
            
        valuePosInfo = [-1, -1]
        
        valuePosInfo[START_INDEX] = valInfoLine[0:keyPosInfo[START_INDEX] + 1].rfind(" ")
        endIndex = valInfoLine[keyPosInfo[END_INDEX] - 1:].find(" ")
        if - 1 == endIndex:
            valuePosInfo[END_INDEX] = len(valInfoLine)
        else:
            valuePosInfo[END_INDEX] = keyPosInfo[END_INDEX] - 1 + endIndex

        retVal = valInfoLine[valuePosInfo[START_INDEX]:valuePosInfo[END_INDEX]].strip()
        if '  ' in retVal:
            retVal = retVal.split('  ')[0]
            
        return retVal 
        
    #测试接口
    def getCliRet(self):
        cliRet = ""
        for lineInfo in self.cliRetList:
            cliRet += lineInfo + "\n"
        return cliRet
    
    def getFstKeyEndIndex(self):
        return self.fstKeyEndIndex
    
    def getKeyPosInfo(self):
        return self.keyPosInfo
    
        
class cTypeDict():
    """
    # *****************************************************************************************#
    # 函数名称: cTypeDict（通用类）
    # 功能说明: 对于CLI回显，左侧为key值，右测为value值的情况
    # 输入参数: cliRet：CLI回显
    #           titleNameAreaList：单个元素结构[titleName,keyValStartId, keyValEndId]
    #           keyWordList：key值列表，指定需要获取的key值，不指定默认获取全部
    #           withTitle 为 True：默认套用两层字典，即是否带titleKey信息
    #           defaultSep：默认分隔符为"|"
    #           newLineSign：对于value值存在多行的情况，默认为直接连接
    # 传入格式说明：cliRet为一段下述结构，handle函数对外提供
    #        ======================================================
    #                          System Information
    #        *-----------------------------------------------------*
    #         System Name           |  
    #         Device Model          |  OceanStor S5600
    #         Current System Mode   |  Normal Dual-Controller Mode
    #         Mirror Path           |  Normal
    #         Location              |  
    #         System Time           |  2014-05-04 07:36:11
    #         Device Serial Number  |  210231G407Z09C000013
    #         Product Version       |  V100R002C01
    #         iSCSI Device Name     |  --
    #        ======================================================
    # 输出参数: withTitle 为 True时：返回一个list,单个元素为只有一个键值的字典，字典里面键值对应为
    #           titleKey，titleKey对应的value为多个key值关联value的字典（list(dict(dict))）
    #           withTitle 为 False时：返回一个list,不包含titleKey信息，list单个元素为一个字典，相
    #           当于只包含具体信息的字典（list(dict)）
    # *****************************************************************************************# 
    """
    def __init__(self, cliRet, titleNameAreaList, keyWordList=None, withTitle=False, defaultSep="|", newLineSign=""):
        self.cliRet = cliRet
        self.titleNameAreaList = titleNameAreaList
        self.keyWordList = keyWordList
        self.withTitle = withTitle
        self.defaultSep = defaultSep
        self.newLineSign = newLineSign

    #入参检查
    def __checkParams(self):
        if not self.cliRet:
            return False
        self.cliRet = self.cliRet.encode('utf8')
        if not self.defaultSep or str != type(self.defaultSep):
            return False
        #分隔符不能只为数字和空格
        if  re.compile("^[\w\s]*$").match(self.defaultSep):
            return False
        if str != type(self.newLineSign):
            return False
        return True
    
    #通用处理函数，对于typeDict类回显，若是一段段结构，格式化为list(dict(dict))结构
    def handle(self):
        
        result = []
        
        if not self.__checkParams():
            raise ParseException("Base class param error(cliRet=" + str(self.cliRet) + \
                            ",titleNameAreaList=" + str(self.titleNameAreaList) + \
                            ",keyWordList=" + str(self.keyWordList) + ",withTitle=" + str(self.withTitle) + \
                            ",defaultSep=" + str(self.defaultSep) + ",newLineSign=" + str(self.newLineSign) + ")")
        
        cliRetList = self.cliRet.splitlines()
        for oneTitleArea in self.titleNameAreaList:
            titleDict = {}
            currentCliRetList = cliRetList[oneTitleArea[1]:oneTitleArea[2]]
            titleValue = self.__generateDict(currentCliRetList)
            if self.withTitle:
                titleDict[oneTitleArea[0]] = titleValue
                result.append(titleDict)
            else:
                result.append(titleValue)
        return result
     
    #获取指定key指对应的字典结构，默认获取全部   
    def __generateDict(self, currentCliRetList):
        
        aimInfoDict = {}
        allInfoDict = self.__generateAllDict(currentCliRetList)
        #重新组合，只获指定key值的value
        if self.keyWordList:
            for key in self.keyWordList:
                if allInfoDict.has_key(key):
                    aimInfoDict[key] = allInfoDict[key]
                elif key and str == type(key):
                    aimInfoDict[key] = None
            return aimInfoDict
        else:
            return allInfoDict
    
    #对于某一段typeDict类回显，格式化为key对应的value式字典结构     
    def __generateAllDict(self, currentCliRetList):
        titleInfoDict = {}
        
        if not currentCliRetList:
            return titleInfoDict
        
        key = None
        value = None
        lastIndex = -1
        
        for i in xrange(0, len(currentCliRetList)):
            
            keyValInfo = currentCliRetList[i]
            
            #需要跳过的行
            if _isSkipLine(keyValInfo):
                continue
            
            if patternEnd.match(keyValInfo):
                break
            
            index = keyValInfo.find(self.defaultSep)
            if index > 0:
                lastIndex = index
                #添加上一行的key及value值时，当前行key位置不能无效，否则视为上一个key的value值   
                if keyValInfo[0:index].strip():
                    #key不为空，说明不为第一行
                    if key:
                        titleInfoDict[key] = value
                        
                        #当前行的值先保存
                        key = keyValInfo[0:index].strip()
                        value = keyValInfo[index + 1:].strip()
                    else:
                        #当前行的值先保存
                        key = keyValInfo[0:index].strip()
                        value = keyValInfo[index + 1:].strip() 
                    
                else:
                    value += self.newLineSign + keyValInfo[index + 1:].strip()
            
            #此分支处理只包含value值的行，前面没有默认的分隔符情况     
            else:
                if - 1 == lastIndex:
                    continue
                
                #不存在分隔符的场景只存在于只有value值的情况
                if not keyValInfo[0:lastIndex].strip():
                    value += self.newLineSign + keyValInfo[lastIndex + 1:].strip()
                    
        if key:
            titleInfoDict[key] = value
        return titleInfoDict
    
class cTypeListWithSep:
    """
    # *****************************************************************************************#
    # 类名称: cTypeListWithSep（通用类）
    # 功能说明: 对于CLI回显，key值行和value行都是由固定的defaultSep分隔开,并且第一位和最后一位
    #           都是defaultSep的情况
    # 输入参数: cliRet：defaultSep分隔的CLI回显格式
    #           defaultSep：分隔符
    #           areaIndex：处理CLI回显的区域,一定要含有key值行
    # 传入格式说明：cliRet为defaultSep分隔的CLI回显格式，handle函数对外提供
    #|Port|LossOfSignal|Bad RX Char|Loss of Sync|Link Fail|RxEOFa|Dis Frame|Bad CRC|Proto error|
    #|  30|           0|          0|           0|       10|     0|        0|      0|         0|
    #|  31|           0|          0|           0|        0|     0|        0|      0|         0|
    #|  20|           0|          0|           0|        0|     0|        0|      0|         0|
    #|  21|           0|          0|           0|        0|     0|        0|      0|         0|
    # 输出参数: list(dict)
    # *****************************************************************************************# 
    """
    def __init__(self, cliRet, defaultSep="|", areaIndex=[None, None]):
        cliRet = cliRet.encode('utf8')
        if not self.__checkParams(cliRet, defaultSep, areaIndex):
            raise ParseException("Input param error(defaultSep=" + str(defaultSep) + \
                        ",areaIndex=" + str(areaIndex) + ")")
          
        cliRet = cliRet.encode('utf8') 
        self.cliRetList = self.__getCliRetDoArea(cliRet, areaIndex)
        self.defaultSep = defaultSep

    #入参检查  
    def __checkParams(self, cliRet, defaultSep, areaIndex):
        if not cliRet:
            return False
        if str != type(defaultSep):
            return False
        if list != type(areaIndex) and 2 == len(areaIndex):
            return False
        return True  
 
    #获取需要处理区域的CLI回显，请传入一个表头+N个value行的结构     
    def __getCliRetDoArea(self, cliRet, areaIndex=[None, None]):
        
        cliRetList = cliRet.splitlines()
        if [None, None] == areaIndex:
            return cliRetList
        elif areaIndex[0] > 0 and None == areaIndex[1]:
            return  cliRetList[areaIndex[0]:len(cliRetList)] 
        elif [None, None] != areaIndex:
            if 2 != len(areaIndex) or areaIndex[0] >= areaIndex[1]:
                return None
            return  cliRetList[areaIndex[0]:areaIndex[1]]
        else:
            return cliRetList
    
    def __getKeyInfo(self):
        keyWordPos = -1
        
        keyWordLine = ""
        keyWordList = []
        
        for i in xrange(len(self.cliRetList)):
            lineInfo = self.cliRetList[i].strip()
            if len(lineInfo) < 2:
                continue
            if self.defaultSep == lineInfo[0] and self.defaultSep == lineInfo[-1]:
                keyWordPos = i
                keyWordLine = lineInfo
                break
        
        tmpKeySplit = keyWordLine.split(self.defaultSep)
        for key in tmpKeySplit:
            if key:
                keyWordList.append(key.strip())
        
        return keyWordPos, keyWordList
    
    def __getValList(self, lineInfo):
        valList = []
        
        tmpValSplit = lineInfo.split(self.defaultSep)
        
        for value in tmpValSplit:
            if value:
                valList.append(value.strip())
        return valList
                   
    def handle(self):
        result = []
        oneLineDict = {}
        
        keyWordPos, keyWordList = self.__getKeyInfo()
        if - 1 == keyWordPos:
            return result
        
        for i in xrange(keyWordPos + 1, len(self.cliRetList)):
            valueLineInfo = self.cliRetList[i].strip()
            if len(valueLineInfo) < 2:
                continue
            
            if self.defaultSep == valueLineInfo[0] and self.defaultSep == valueLineInfo[-1]: 
                oneLineValList = self.__getValList(valueLineInfo)
                if len(keyWordList) == len(oneLineValList):
                    result.append(dict(zip(keyWordList, oneLineValList)))
        return result
        
class cFastTypeDict():
    """
    # *****************************************************************************************#
    # 函数名称: cFastTypeDict（通用类）
    # 功能说明: 快速查找键值对，对于CLI回显，左侧为key值，右测为value值的情况
    # 输入参数: cliRet：CLI回显
    #           defaultSep：默认分隔符为"|"
    # 传入格式说明：cliRet为多段下述结构，handle函数对外提供
    #        ======================================================
    #                          System Information
    #        *-----------------------------------------------------*
    #         System Name           |  
    #         Device Model          |  OceanStor S5600
    #         Current System Mode   |  Normal Dual-Controller Mode
    #         Mirror Path           |  Normal
    #         Location              |  
    #         System Time           |  2014-05-04 07:36:11
    #         Device Serial Number  |  210231G407Z09C000013
    #         Product Version       |  V100R002C01
    #         iSCSI Device Name     |  --
    #        ======================================================
    # 输出参数:list(dict)
    # 约束条件:key和value值一定为defaultSep分开
    # *****************************************************************************************#
   """  
    
    def __init__(self, cliRet, defaultSep="|", newLineSign=""):
        self.cliRet = cliRet
        self.defaultSep = defaultSep
        self.newLineSign = newLineSign
    
    #入参检查  
    def __checkParams(self):
        if not self.cliRet:
            return False
        self.cliRet = self.cliRet.encode('utf8')
        if  str != type(self.defaultSep):
            return False
        #分隔符不能只为数字和空格
        if  re.compile("^[\w\s]*$").match(self.defaultSep):
            return False
        if str != type(self.newLineSign):
            return False
        return True
        
    def handle(self):
        cliRetList = self.cliRet.splitlines()
        result = []
        secKeyWordList = []
        secInfoDict = {}
        
        lastIndex = -1
        key = None
        value = ""
        
        if not self.__checkParams():
            raise ParseException("Base class param error(cliRet=" + str(self.cliRet) + \
                            ",defaultSep=" + str(self.defaultSep) + \
                            ",newLineSign=" + str(self.newLineSign) + ")")
        
        for lineInfo in cliRetList:
            
            #需要跳过的行
            if _isSkipLine(lineInfo):
                continue
            
            if patternEnd.match(lineInfo):
                continue 
     
            index = lineInfo.find(self.defaultSep) 
            if index > 0:
                lastIndex = index
                #添加上一行的key及value值时，当前行key位置不能无效，否则视为上一个key的value值   
                if lineInfo[0:index].strip():
                    #key不为空，说明不为第一行
                    if key:
                        #一个分段开始
                        if key in secKeyWordList:
                            result.append(secInfoDict)
                            
                            secKeyWordList = []
                            secKeyWordList.append(key)
                            
                            secInfoDict = {}
                            secInfoDict[key] = value
                        else:
                            secKeyWordList.append(key)
                            secInfoDict[key] = value
                            
                        #当前行的值先保存
                        key = lineInfo[0:index].strip()
                        value = lineInfo[index + 1:].strip()  
                            
                    else:
                        #当前行的值先保存
                        key = lineInfo[0:index].strip()
                        value = lineInfo[index + 1:].strip() 
                    
                else:
                    value += self.newLineSign + lineInfo[index + 1:].strip()
            
        if secInfoDict:
            secInfoDict[key] = value
            result.append(secInfoDict)   
        return result

class ParseException(Exception):
     pass 
