﻿#coding: UTF-8

import re
import shutil
import time
import os
import traceback
import modelManager
import common
from common import TimeOutException

#存储需要的最少保险箱盘数量
G_MIN_COFF_DISK_NUM = 2

#所有数据存放顶层目录名称
G_ALL_DATA_DIRNAME = "DataCollect"

#文件信息分隔符(独行20个*，用于不用硬盘smart信息和基本信息的分隔)
G_INFO_SEPERATOR = os.linesep + "********************" + os.linesep

#smart信息存放目录名称
G_SMART_DATA_DIRNAME = "disksmartinfo"

#单盘smart信息收集超时值（秒）
G_DISK_SMART_TIMEOUT = 120

#远端告警文件所在目录
G_REMOTE_ALARM_DIR_DEFAULT = r"/OSM/"
G_REMOTE_ALARM_DIR_S2600V1R1 = r"/"#虽然这里写成”/“，但实际在S2600V1R1中生成的告警文件是在”/OSM/“下

#导出类型：1代表告警，2代表日志
G_EXPORT_TYPE = "1"

#原始告警文件名
ALARM_FILE_NAME_ORIGINAL = "local_alm_file.txt"

#重命名后告警文件名
ALARM_FILE_NAME_RENAMED = "event.txt"

# modified 20140513 begin
#常规信息收集收集是否成功，默认成功
isGeneralInfoCollSuc = True

#设备型号
#S5000系列
DeviceModelList4S5000 = ["S5300", "S5500", "S5600", "S6800E"]
#S2600系列
DeviceModelList4S2600 = ["S2600"]
#S2000系列
DeviceModelList4S2000 = ["S2100", "S2300", "S2300E"]

#产品版本
V1R1 = "V100R001"
V1R2 = "V100R002"
V1R5 = "V100R005"


# **************************************************************************** #
# 函数名称: handleBeforeErrorExit
# 功能说明: 收集错误退出前处理
# 输入参数: 设备对象，初始系统模式
# 输出参数: 无
# **************************************************************************** # 
def handleBeforeErrorExit(devObj, oldSysMode):
        
    #记录错误
    logger = devObj.get("logger")
    logger.info("[SIC] Collection exited with the error!")
        
    try:
        #恢复系统模式（不关注是否退出成功）
        modelManager.changeMode(devObj, oldSysMode, False)
    except:
        logger.error("[SIC] Failed to change mode before exit:" + traceback.format_exc())
        
    #清理资源，忽略异常
    tmpDataCollectDir = str(devObj.get("tmpDataCollectDir"))
    #常规信息收集失败或未进行常规信息收集
    if not isGeneralInfoCollSuc:
        if "" != tmpDataCollectDir and os.path.exists(tmpDataCollectDir):
            shutil.rmtree(tmpDataCollectDir, True)
    else:
        if "" != tmpDataCollectDir and os.path.exists(tmpDataCollectDir):
            disksmartinfoDir = tmpDataCollectDir + os.sep + G_ALL_DATA_DIRNAME + os.sep + G_SMART_DATA_DIRNAME
            shutil.rmtree(disksmartinfoDir, True)
    return
    
    
# **************************************************************************** #
# 函数名称: getDeviceModel
# 功能说明: 获取设备型号
# 输入参数: 设备对象，命令回显
# 输出参数: 设备型号
# **************************************************************************** #
def getDeviceModel(devObj, cmdRet):
    logger = devObj.get("logger")
    cmdRetList = cmdRet.splitlines()
    for line in cmdRetList:
        if re.search("Device Model", line, re.IGNORECASE):
            lineList = line.split("|")
            if len(lineList) < 2:
                return ""
            devModel = lineList[1].strip()
            logger.info("[SIC] Device Model:" + devModel)
            devModelList = devModel.split()
            if len(devModelList) < 2:
                return ""
            return devModelList[1].strip()
    return ""


# **************************************************************************** #
# 函数名称: getProductVersion
# 功能说明: 获取产品版本（只获取到R版本，比如：V100R005）
# 输入参数: 设备对象，命令回显
# 输出参数: 产品版本
# **************************************************************************** #
def getProductVersion(devObj, cmdRet):
    logger = devObj.get("logger")
    cmdRetList = cmdRet.splitlines()
    for line in cmdRetList:
        if re.search("Product Version", line, re.IGNORECASE):
            lineList = line.split("|")
            if len(lineList) < 2:
                return ""
            pdtVer = lineList[1].strip()
            logger.info("[SIC] Product Version:" + pdtVer)
            return pdtVer[:8]#返回前8位
    return ""
    
    
# **************************************************************************** #
# 函数名称: getSystemInfo
# 功能说明: 获取showsys信息
# 输入参数: 设备对象，信息文件路径
# 输出参数: 成功or失败，设备型号，产品版本
# **************************************************************************** # 
def getSystemInfo(devObj, infoFilePath):
    
    ssh = devObj.get("SSH")
    
    #获取showrg命令结果并保存
    temp = common.execCmd(ssh, "showsys")
    
    devModel = getDeviceModel(devObj, temp)
    pdtVer = getProductVersion(devObj, temp)
    if not devModel or not pdtVer:
        if common.isChinese(devObj):
            devObj["py_detail"] = u"[错误] 获取设备型号或产品版本号失败。"
        else:
            devObj["py_detail"] = "[ERROR] Failed to get device model or product version."
        return (False, "", "")
    
    temp = temp + G_INFO_SEPERATOR
    errInfo = common.writeInfoToFile(infoFilePath, temp, False)
    if "" != errInfo:
        devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, infoFilePath, errInfo)
        return (False, "", "")
    
    return (True, devModel, pdtVer)
    
        
# **************************************************************************** #
# 函数名称: getRaidGroupInfo
# 功能说明: 获取raid组信息
# 输入参数: 设备对象，信息文件路径
# 输出参数: True or False
# **************************************************************************** # 
def getRaidGroupInfo(devObj, infoFilePath):
    
    ssh = devObj.get("SSH")
    #获取showrg命令结果并保存
    temp = common.execCmd(ssh, "showrg")
    temp = temp + G_INFO_SEPERATOR
    errInfo = common.writeInfoToFile(infoFilePath, temp, True)
    if "" != errInfo:
        devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, infoFilePath, errInfo)
        return False
        
    return True
    

# **************************************************************************** #
# 函数名称: getDiskPhysicInfo
# 功能说明: 获取硬盘的物理信息
# 输入参数: 设备对象，信息文件路径，概要信息文件路径
# 输出参数: 是否成功，硬盘物理信息位置列表
# **************************************************************************** # 
def getDiskPhysicInfo(devObj, infoFilePath, summaryFilePath):
    diskPhysicInfoLocationList = []
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    
    #获取showdisk -p命令结果并保存
    temp = common.execCmd(ssh, "showdisk -p")
    
    #处理回文
    try:
        #在showdisk -p命令回文中添加一条vendor信息
        execRet, diskInfoWithVendor = addVendor2DiskInfo(devObj, temp)
        if not execRet:
            logger.error("[getDiskPhysicInfo] Add vendor failed!")
        
        #针对SN和Model换行场景，需要将换行后的SN和Model拼接上,并重新替换成正确的值
        checkFlag, SNAndModeldict, modelMaxLen, snMaxlen = getDiskSNAndModel(temp)
        logger.info("[getDiskPhysicInfo] modelMaxLen=" + str(modelMaxLen))
        logger.info("[getDiskPhysicInfo] snMaxlen=" + str(snMaxlen))
        logger.info("[getDiskPhysicInfo] len(SNAndModeldict)=" + str(len(SNAndModeldict)))
        if checkFlag:
           diskInfoWithVendor = changeDiskInof2Right(diskInfoWithVendor, SNAndModeldict, modelMaxLen, snMaxlen)     
        
        temp = diskInfoWithVendor + G_INFO_SEPERATOR  
    
    #处理过程有误，则直接按原流程走，并记录日志
    except:
        logger.info("[getDiskPhysicInfo] except: " + traceback.format_exc())
        temp = temp + G_INFO_SEPERATOR
        
    errInfo = common.writeInfoToFile(infoFilePath, temp, True)
    if "" != errInfo:
        devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, infoFilePath, errInfo)
        return (False, diskPhysicInfoLocationList)   
    #获取硬盘总数并保存
    lineList = temp.splitlines()
    totalDiskNum = 0
    for line in lineList:
        index = line.find(")")
        if index >= 0:
            #包含硬盘槽位信息
            totalDiskNum += 1
            location = line[:index + 1]
            diskPhysicInfoLocationList.append(location.replace(" ", ""))
        else:
            continue
        
    logger.info("[SIC] diskPhysicInfoLocationList=" + str(diskPhysicInfoLocationList))
            
    #如果硬盘总数小于2则报错
    if totalDiskNum < G_MIN_COFF_DISK_NUM:
        if common.isChinese(devObj):
            devObj["py_detail"] = u"[错误] 从CLI命令showdisk -p中获取的硬盘总数为" + str(totalDiskNum) + u", 小于" + str(G_MIN_COFF_DISK_NUM)
        else:
            devObj["py_detail"] = "[ERROR] The total disk number that got from the CLI command showdisk -p is " + str(totalDiskNum) + ", less than " + str(G_MIN_COFF_DISK_NUM)
        return (False, diskPhysicInfoLocationList)
    
    #记录硬盘总数（采用覆盖方式，清除之前的残留）
    temp = "total=" + str(totalDiskNum) + os.linesep
    errInfo = common.writeInfoToFile(summaryFilePath, temp, False)
    if "" != errInfo:
        devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, infoFilePath, errInfo)
        return (False, diskPhysicInfoLocationList)
    
    return (True, diskPhysicInfoLocationList)
    


# ************************************************************************************************ #
# 函数名称: getLocalSmartInfo
# 功能说明: 查询当前控制器下的硬盘Smart信息（调用者保证：调用此接口前，ssh在debug模式下）
# 输入参数: 设备对象，设备硬盘字典，硬盘物理信息列表，smart收集相关文件路径
# 输出参数: True or False
# ************************************************************************************************ # 
def getLocalSmartInfo(devObj, devDiskDict, diskPhyInfoList, filesPath):
    smartSavePath = filesPath[0]
    failFilePath = filesPath[1]
    summaryFilePath = filesPath[2]
    
    gotDiskNum = 0  #获取到信息的硬盘数量    
    dataFileId = 0  #当前可用下标：smart#.txt的下标从0开始
    diskCntr = 0    #硬盘计数器
    smartInfoStr = ""  #smart信息
    failSmartStr = ""  #smart收集失败信息
    indexInfoStr = ""  #smart文件存放的硬盘索引：smart0=0-19
    tmpStr = ""        #临时smart信息
    failedDisk = ""
    
    #遍历获取列表中所有硬盘的smart信息
    ssh = devObj.get("SSH")
    logger = devObj.get("logger")
    
    for diskLocation in diskPhyInfoList:
        #在mml下的命令dev diskinfo中未找到
        if not devDiskDict.has_key(diskLocation):
            failSmartStr += "Disk " + diskLocation + " has not been found, collecting smart information is failed!" + G_INFO_SEPERATOR
            failedDisk += diskLocation + ","
            continue
        else:
            disk = devDiskDict.get(diskLocation).strip()
            if "" == disk:#无盘符的
                failSmartStr += "Disk " + diskLocation + " has no name, collecting smart information is failed!" + G_INFO_SEPERATOR
                failedDisk += diskLocation + ","
                continue
            
            #使用disktool获取硬盘smart信息
            tmpStr = common.execCmdNoLogTimout(ssh, "disktool -f a /dev/" + disk, G_DISK_SMART_TIMEOUT,timeOutLimit=5)
            
            #真实的硬盘smart中含有"Serial Number"字段
            if tmpStr.find("Serial Number") >= 0:
                smartInfoStr = smartInfoStr + tmpStr + G_INFO_SEPERATOR
                diskCntr += 1
                gotDiskNum += 1
            else:
                failSmartStr = failSmartStr + tmpStr + G_INFO_SEPERATOR
                failedDisk += diskLocation + ","
                continue
            
            #每20个盘的smart信息存一个文件
            if diskCntr >= 20:
                #将本次数据存入文件
                errInfo = common.writeInfoToFile(smartSavePath.replace("@FID", str(dataFileId)), smartInfoStr, False)
                if "" != errInfo:
                    devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, smartSavePath, errInfo)
                    return False
                
                #刷新文件信息
                indexInfoStr = indexInfoStr + "smart" + str(dataFileId) + "=" + str(gotDiskNum - diskCntr) + "-" + str(gotDiskNum - 1) + os.linesep
                dataFileId += 1
                
                #重置循环变量
                diskCntr = 0
                smartInfoStr = ""
    
    #保存最后一部分未保存的smart信息，并刷新文件信息
    if diskCntr > 0:
        errInfo = common.writeInfoToFile(smartSavePath.replace("@FID", str(dataFileId)), smartInfoStr, False)
        if "" != errInfo:
            devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, smartSavePath, errInfo)
            return False
        indexInfoStr = indexInfoStr + "smart" + str(dataFileId) + "=" + str(gotDiskNum - diskCntr) + "-" + str(gotDiskNum - 1) + os.linesep
    
    #判断部分收集成功
    if "" != failedDisk:
        failedDisk = failedDisk[:-1]
        logger.info("[SIC] Collection is not full, the disks which were not collected are " + failedDisk)
        devObj["collectAllInfo"] = False
        if common.isChinese(devObj):
            devObj["py_detail"] = u"硬盘" + failedDisk + u"收集Smart失败，请按照如下建议处理：\
                \n（1）收集失败的原因可能是：系统软件版本过低、硬盘单链路、响应慢、故障或者休眠、控制框或是硬盘框ID被修改。请先按照标准处理方法修复，修复完成后重新收集一次；\
                \n（2）如果仍然收集失败，请尝试手动执行命令收集单块硬盘；\
                \n（3）如果有任何疑问，请联系技术支持工程师协助处理。"
        else:
            devObj["py_detail"] = "Collecting smart information of disks " + failedDisk + " failed, please handle them according to the suggestion:\
            \n(1)The reason of failure may be: the system software is of an earlier version, the disk is single path, the response of disk is slow, the disk is fault or in spin down status, or the ID of enclosure has been modified. Please repair it according to the standard treatment, then collect again;\
            \n(2)If collecting failed again, please try to collect the smart information manually;\
            \n(3)If you have any questions, please contact technical support engineers for further handling."
    
    #如果存在smart收集失败的信息，则存入文件
    if "" != failSmartStr.strip():
        errInfo = common.writeInfoToFile(failFilePath, failSmartStr, False)
        if "" != errInfo:
            devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, failFilePath, errInfo)
            return False
    
    #在summary.ini中追加块设备信息，收集完成时间和smart存放的硬盘索引信息
    tmpStr = "success=" + str(gotDiskNum) + os.linesep
    curTime = str(time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time())))
    tmpStr = tmpStr + "date=" + curTime + os.linesep
    tmpStr = tmpStr + indexInfoStr
    errInfo = common.writeInfoToFile(summaryFilePath, tmpStr, True)
    if "" != errInfo:
        devObj["py_detail"] = common.getDisplayFailedInfoInWritingFile(devObj, summaryFilePath, errInfo)
        return False
    if 0 == gotDiskNum:
        devObj["collectAllInfo"] = True#为防止影响后面其他收集，这里做初始化
        if common.isChinese(devObj):
            devObj["py_detail"] = u"[错误] 没有收集到任何硬盘Smart信息，请按照如下建议处理：\
                \n（1）收集失败的原因可能是：系统软件版本过低、硬盘单链路、响应慢、故障或者休眠、控制框或是硬盘框ID被修改。请先按照标准处理方法修复，修复完成后重新收集一次；\
                \n（2）如果仍然收集失败，请尝试手动执行命令收集单块硬盘；\
                \n（3）如果有任何疑问，请联系技术支持工程师协助处理。"
        else:
            devObj["py_detail"] = "[ERROR] Failed to get smart information of any disk, please handle them according to the suggestion:\
                \n(1)The reason of failure may be: the system software is of an earlier version, the disk is single path, the response of disk is slow, the disk is fault or in spin down status, or the ID of enclosure has been modified. Please repair it according to the standard treatment, then collect again;\
                \n(2)If collecting failed again, please try to collect the smart information manually;\
                \n(3)If you have any questions, please contact technical support engineers for further handling."
        return False
    return True


# **************************************************************************** #
# 函数名称: clearRemoteTmpFile
# 功能说明: 清除阵列上产生的临时文件
# 输入参数: 设备对象，临时文件名，阵列告警目录
# 输出参数: 成功与否，错误消息
# **************************************************************************** # 
def clearRemoteTmpFile(devObj, fileName, remoteAlmDir):
    logger = None 
    try:
        dbgPwd = devObj.get("debug")
        ssh = devObj.get("SSH")
        logger = devObj.get("logger")
        errMsg = ""
        if dbgPwd == None or dbgPwd == "":
            if common.isChinese(devObj):
                errMsg = u"[错误] 由于debug密码不存在，无法将系统模式切换到debug。"
            else:
                errMsg = "[Error] Changing system mode to debug failed since the debug password is not existed."
            return (False, errMsg)
        logger.info("[SIC]debugPassword:******")
        common.execCmd(ssh, "debug")
        pwdResult = common.execCmdNoLog(ssh, dbgPwd)
        pwdResult = pwdResult.upper()
        if("ERROR" in pwdResult):
            if common.isChinese(devObj):
                errMsg = u"[错误] debug密码错误。"
            else:
                errMsg = "[Error] debug password is invalid."
            return (False, errMsg)
        common.execCmd(ssh, "rm -f " + remoteAlmDir + fileName)
        common.execCmd(ssh, "exit")#从debug模式回到CLI模式
        return (True, "")
    except BaseException, e:
        if None != logger:
            logger.info("[SIC] It occurs exception when clearing remote temporary file, the details are as follows:" + str(e))
    except:
        if None != logger:
            logger.info("[SIC] It occurs exception when clearing remote temporary file:" + traceback.format_exc())


# **************************************************************************** #
# 函数名称: decompressAndDel
# 功能说明: 解压下载后的压缩包，然后删除该包，最后重命名
# 输入参数: 设备对象，本地目录，压缩包名
# 输出参数: 成功与否，错误消息
# **************************************************************************** # 
def decompressAndDel(devObj, localHostDir, fileName):
    logger = devObj.get("logger")
    pkgPath = localHostDir + os.path.sep + fileName
    logger.info("[SIC] pkgPath=" + pkgPath)
    errMsg = ""
    
    #判断告警文件结果包是否存在
    if not os.path.exists(pkgPath):
        if common.isChinese(devObj):
            errMsg = u"[错误] 告警文件结果包" + pkgPath + u"不存在。"
        else:
            errMsg = "[ERROR] The alarm package " + pkgPath + " is not exist."
            return (False, errMsg)
    
    #解压告警文件结果包
    excRet = common.decompressPkg(pkgPath, localHostDir)
    isSucc = excRet[0]
    errMsg = excRet[1]
    if not isSucc:
        logger.info("[SIC] Is decompressing package successful: " + str(errMsg))
        errMsg = "Failed to decompress file " + pkgPath + ":\n" + errMsg
        return (False, errMsg)
    
    #删除该压缩包
    try:
        os.remove(pkgPath)
        logger.info("[SIC] The file " + pkgPath + " has been removed successfully!")
    except:
        logger.warn("[SIC] Failed to delete file " + pkgPath)
    
    #重命名文件
    oldName = localHostDir + os.path.sep + ALARM_FILE_NAME_ORIGINAL
    newName = localHostDir + os.path.sep + ALARM_FILE_NAME_RENAMED
    os.rename(oldName, newName)
    logger.info("[SIC] The file " + oldName + " has been renamed to " + newName)
    
    return (True, "")




# **************************************************************************** #
# 函数名称: initDataSaveDir
# 功能说明: 初始化数据存放目录
# 输入参数: 设备对象
# 输出参数: ""（初始化失败） 或 smart信息存放路径
# **************************************************************************** # 
def initDataSaveDir(devObj):
    #获取或构造临时数据存放目录路径
    tmpDataCollectDir = devObj.get("tmpDataCollectDir")
    curTime = str(time.strftime('%Y%m%d%H%M%S', time.localtime(time.time())))
    tmpRootDir = devObj.get("collectRetDir") + os.sep + "tmpdir@" + curTime

    global isGeneralInfoCollSuc
    if None == tmpDataCollectDir or "" == tmpDataCollectDir:
        #临时目录不存在时则创建并记入设备对象
        tmpDataCollectDir = tmpRootDir
        isGeneralInfoCollSuc = False
        devObj["tmpDataCollectDir"] = tmpDataCollectDir
    
    #保证临时数据收集目录存在
    if not os.path.exists(tmpDataCollectDir):
        os.mkdir(tmpDataCollectDir)
        if not os.path.exists(tmpDataCollectDir):
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 创建数据收集目录" + tmpDataCollectDir + u"失败。"
            else:
                devObj["py_detail"] = "[ERROR] Creating data collect directory " + tmpDataCollectDir + " failed."
            return ""
    
    #生成数据存放根目录
    tmpDir = tmpDataCollectDir + os.sep + G_ALL_DATA_DIRNAME
    if not os.path.exists(tmpDir):
        #如果目录不存在则创建
        os.mkdir(tmpDir)
        #如果创建不成功则返回
        if not os.path.exists(tmpDir):
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 创建数据收集目录" + tmpDir + u"失败。"
            else:
                devObj["py_detail"] = "[ERROR] Creating data save directory " + tmpDir + " failed."
            return ""
    
    #生成smart数据存放目录
    tmpDir = tmpDir + os.sep + G_SMART_DATA_DIRNAME
    if not os.path.exists(tmpDir):
        #如果目录不存在则创建
        os.mkdir(tmpDir)
        #如果创建不成功则返回
        if not os.path.exists(tmpDir):
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 创建smart数据收集目录" + tmpDir + u"失败。"
            else:
                devObj["py_detail"] = "[ERROR] Creating the smart data save directory " + tmpDir + " failed."
            return ""
    
    return tmpDir
    
    
# **************************************************************************** #
# 函数名称: getDevDiskDict
# 功能说明: 根据产品类型获取设备硬盘信息
# 输入参数: 设备对象，设备型号，产品版本
# 输出参数: 设备硬盘字典
# **************************************************************************** # 
def getDevDiskDict(devObj, devModel, pdtVer):
    #老5000V1R1的设备sda不是系统盘，需要对sda进行smart收集
    logger = devObj.get("logger")
    logger.info("[SIC] devModel=" + devModel)
    logger.info("[SIC] pdtVer=" + pdtVer)
    if devModel in DeviceModelList4S5000 and pdtVer == V1R1:
        return getDevDiskinfo4common(devObj)
    elif devModel in DeviceModelList4S2600 and pdtVer == V1R1:
        return getDevDiskinfo4common(devObj)
    elif devModel in DeviceModelList4S2000 and pdtVer == V1R1:
        return getDevDiskinfo4common(devObj)
    elif devModel in DeviceModelList4S5000 and pdtVer == V1R2:
        return getDevDiskinfo4common(devObj)
    elif devModel in DeviceModelList4S2600 and pdtVer == V1R2:
        return getDevDiskinfo4S2600V1R2(devObj)
    elif devModel in DeviceModelList4S5000 and pdtVer == V1R5:
        return getDevDiskinfo4S__00V1R5(devObj)
    elif devModel in DeviceModelList4S2600 and pdtVer == V1R5:
        return getDevDiskinfo4S__00V1R5(devObj)
    else:
        logger.info("[SIC] The device model or product version is out of support type scope.")
        return None
    
    
# **************************************************************************** #
# 函数名称: getDevDiskinfo4S2600V1R2
# 功能说明: 获取S2600V1R2系列设备硬盘信息
# 输入参数: 设备对象
# 输出参数: 设备硬盘字典
# **************************************************************************** # 
def getDevDiskinfo4S2600V1R2(devObj):
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    
    #发送命令
    cmdRet = common.execCmd(ssh, "dev disktoolinfo")
    cmdRetList = cmdRet.splitlines()
    
    devDiskInfoDict = {}
    for line in cmdRetList[7:]:
        #找到"Debugging finish"则表示有效行结束
        if - 1 != line.find("Debugging finish"):
            break
        lineGroup = line.split("|")
        
        #未避免后面取值下标越界
        if len(lineGroup) < 3:
            continue
        frameId = lineGroup[0].strip()
        slotId = lineGroup[1].strip()
        location = "(" + frameId + "," + slotId + ")"
        deviceName = lineGroup[2].strip()
        devDiskInfoDict[location] = deviceName
    logger.info("[SIC] devDiskInfoDict=" + str(devDiskInfoDict))
    return devDiskInfoDict
    
    
    
# **************************************************************************** #
# 函数名称: getDevDiskinfo4S__00V1R5
# 功能说明: 获取S2600V1R5、S5000V1R5系列设备硬盘信息
# 输入参数: 设备对象
# 输出参数: 设备硬盘字典
# **************************************************************************** # 
def getDevDiskinfo4S__00V1R5(devObj):
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    #发送命令
    cmdRet = common.execCmd(ssh, "dev diskinfo")
    
    devDiskInfoDict = {}
    #获取硬盘位置（框号，槽位号）
    keys = re.findall('\(\d+,\d+\)', cmdRet, re.IGNORECASE)
    #获取盘符
    vals = re.findall(',Name (.*?),', cmdRet, re.IGNORECASE)
    
    if len(keys) != len(vals):
        logger.info("[SIC] keys=" + str(keys))
        logger.info("[SIC] vals=" + str(vals))
        return devDiskInfoDict
    #用于判断部分收集成功
    devDiskInfoDict = dict(zip(keys, vals))
    
    logger.info("[SIC] devDiskInfoDict=" + str(devDiskInfoDict))
    return devDiskInfoDict
    
    
# **************************************************************************** #
# 函数名称: getDevDiskinfo4common
# 功能说明: 获取S5000V1R1、S5000V1R2、S2600V1R1或者S2000V1R1系列设备硬盘信息
# 输入参数: 设备对象
# 输出参数: 设备硬盘字典
# **************************************************************************** # 
def getDevDiskinfo4common(devObj):
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    #发送命令
    ret = common.execCmd(ssh, "dev diskinfo 99")

    retList = ret.splitlines()
    devDiskInfoDict = {}
    location = ""
    for line in retList:
        line = line.strip()
        #获取硬盘位置
        #方式为：首先找到如Disk[的字符串，然后截取[0:0]，最后修改为(0,0)
        if line.startswith("Disk["):
            location = line[line.find("[") : line.find("]") + 1]
            location = location.replace("[", "(").replace(":", ",").replace("]", ")")
            devDiskInfoDict[location] = ""#这里赋初始值，在下一行如果有盘符（sd开头）会重新赋值
        #获取sdX并加入列表
        elif line.startswith("sd"):
            disk = line[:line.find("(")]
            
            devDiskInfoDict[location] = disk
    
    logger.info("[SIC] devDiskInfoDict=" + str(devDiskInfoDict))
    return devDiskInfoDict

# **************************************************************************** #
# 函数名称: getSubrackIdList
# 功能说明: 获取subrackID列表
# 输入参数: 设备对象
# 输出参数: subrackIdList
# **************************************************************************** # 
def getSubrackIdList(devObj):
    subrackIdList = []
    
    ssh = devObj.get("SSH")

    subrackInfo = common.execCmd(ssh, "showsubrack")
    subrackInfoLine = subrackInfo.splitlines()
    if "Subrack ID" not in subrackInfo or len(subrackInfoLine) < 3:
        return subrackIdList
    
    for line in subrackInfoLine[2:-1]:
        if len(line.strip()) == 0:
            continue
        subrackId = line.split()[0].strip()
        subrackIdList.append(subrackId)
        
    return subrackIdList
        
# **************************************************************************** #
# 函数名称: getDiskvendor
# 功能说明: 获取subrack信息
# 输入参数: 设备对象, subrackIdList
# 输出参数: diskVendorDict
# **************************************************************************** # 
def getDiskvendor(devObj, subrackIdList):
    diskVendorDict = {}
    
    ssh = devObj.get("SSH")
    
    if not subrackIdList:
        return diskVendorDict
    
    cmd = "showdiskinfo -s %s"
    for subrackId in subrackIdList:
        vendorInfo = common.execCmd(ssh, cmd % subrackId)
        vendorInfoLine = vendorInfo.splitlines()
        
        if "DiskID" not in vendorInfo or len(vendorInfoLine) < 4:
            return diskVendorDict
        
        for line in vendorInfoLine[3:-1]:
            if len(line.strip()) == 0:
                continue
            diskId = line.split()[0].strip()
            vendor = " "
            if len(line.split()) == 2:
                vendor = line.split()[1]
            diskLocation = '(%s, %s)' % (subrackId, diskId)
            diskVendorDict[diskLocation] = vendor
            
    return diskVendorDict
    
    
# **************************************************************************** #
# 函数名称: addVendor2DiskInfo
# 功能说明: 将厂商信息添加到disk信息中
# 输入参数: 设备对象， disk信息
# 输出参数: diskInfoWithVendor, 添加vendor后的disk信息
# **************************************************************************** # 
def addVendor2DiskInfo(devObj, diskinfo):
    #现将硬盘信息头加入该信息中
    diskInfoWithVendor = diskinfo.splitlines()[0]
    diskInfoWithVendor += os.linesep + diskinfo.splitlines()[1] + " Vendor"
    
    #获取subrackID列表
    subrackIdList = getSubrackIdList(devObj)
    if not subrackIdList:
        return False, diskinfo
    
    #获取厂商
    diskVendorDict = getDiskvendor(devObj, subrackIdList)
    if not diskVendorDict:
        return False, diskinfo
    
    #将厂商信息添加到对应硬盘后
    diskInfoLines = diskinfo.splitlines()
    for line in diskInfoLines[2:-1]:
        #查找loaction
        locationStart = line.find('(')
        locationEnd = line.find(')')
        if locationStart == -1 or locationEnd == -1:
            continue
        
        location = line[locationStart:locationEnd + 1]
            
        if diskVendorDict.has_key(location):
            line += " %s" % diskVendorDict[location]   
        else:
            line += " "
            
        diskInfoWithVendor += os.linesep + line
        
    #写temp信息尾部
    diskInfoWithVendor += os.linesep + diskinfo.splitlines()[-1]

    return True, diskInfoWithVendor


def changeDiskInof2Right(diskinfo, SNAndModeldict, modelMaxLen, snMaxlen):
    '''
    @summary: 从新组装disk信息，将SN和Model写在一行
    '''
    #查找Sn和Model字段的位置
    getFlag, ModelPosStar, ModelPosEnd, SNPosStar, SNPosEnd  = getDiskSNAndModelPos(diskinfo)
    #获取失败，原样返回
    if not getFlag:
        return diskinfo
    
    newDiskInfo = ""
    
    #替换头部,拼接的字段长度+2，使前后字段分隔
    newDiskInfo +=  diskinfo.splitlines()[0]
    newModel = "%-*s" % (modelMaxLen + 2, "Model")
    newSN = "%-*s" % (snMaxlen + 2, "Serial Number")
    
    #Model在前
    line = diskinfo.splitlines()[1]
    if ModelPosStar < SNPosStar:  
        line = line[:ModelPosStar] + newModel + line[ModelPosEnd:SNPosStar] + newSN + line[SNPosEnd:]
    #Model在后
    else:
        line = line[:SNPosStar] + newSN + line[SNPosEnd:ModelPosStar] + newModel + line[ModelPosEnd:]
    
    newDiskInfo += os.linesep + line
    
    #替换SN和Model
    for line in diskinfo.splitlines()[2:-1]:
        #获取loaction及正确的SN和Model
        locationStart = line.find('(')
        locationEnd = line.find(')')
        if locationStart == -1 or locationEnd == -1:
            continue
        
        location = line[locationStart:locationEnd + 1]
        modelValue = "%-*s" % (modelMaxLen + 2, SNAndModeldict[location][0])
        snValue = "%-*s" % (snMaxlen  + 2, SNAndModeldict[location][1])
        
        #拼接新行
        #Model在前
        if ModelPosStar < SNPosStar:  
            line = line[:ModelPosStar] + modelValue + line[ModelPosEnd:SNPosStar] + snValue + line[SNPosEnd:]
        #Model在后
        else:
            line = line[:SNPosStar] + snValue + line[SNPosEnd:ModelPosStar] + modelValue + line[ModelPosEnd:]
        
        newDiskInfo += os.linesep + line
    #结束行
    newDiskInfo += os.linesep + diskinfo.splitlines()[-1]

    return newDiskInfo
    
def getDiskSNAndModelPos(diskinfo):
    '''
    @summary: 获取硬盘Sn和model的位置信息
    @return: (True/False, ModelStar,ModeEnd,SnStar,SnEnd)
    '''  
    #定义字段，每个字段后面都应该至少有一个空格
    SN = "Serial Number "
    Model = "Model "
    SNAndModelDict = {}
    
    #获取标题行，并查找Sn和Model字段的位置
    keyLine = diskinfo.splitlines()[1]
    ModelPosStar = keyLine.find(Model)
    SNPosStar = keyLine.find(SN) 
    #如果这两个关键字没找到，则可以认为回显存在问题，直接退出
    if ModelPosStar == -1 or SNPosStar == -1:
        return (False, -1, -1, -1,-1)
    
    #字段结束位置 
    ModelPosEnd = ModelPosStar + len(Model)
    SNPosEnd = SNPosStar + len(SN)
    #直到下一个字段开始，该字段才结束
    for chr in keyLine[ModelPosEnd:]:
        if chr == " ":
            ModelPosEnd += 1
        if chr != " ":
            break
        
    for chr in keyLine[SNPosEnd:]:
        if chr == " ":
            SNPosEnd += 1
        if chr != " ":
            break
            
    return (True, ModelPosStar, ModelPosEnd, SNPosStar, SNPosEnd)
   
    
def getDiskSNAndModel(diskinfo):
    '''
    @summary: 获取硬盘的SN号和Model，组成列表
    @return: 返回字段：(True/False, “{location:[Model, SN]”}, ModelMaxLen, SnMaxLen)    
    '''
    lastSn = ""
    lastModel = ""
    lastLocation = ""
    SNAndModelDict = {}
    
    #查找Sn和Model字段的位置
    getFlag, ModelPosStar, ModelPosEnd, SNPosStar, SNPosEnd  = getDiskSNAndModelPos(diskinfo)
    if not getFlag:
        return (False, SNAndModelDict, 0, 0)
    
    #设置初始长度，应该为标题的长度
    snMaxLen = SNPosEnd - SNPosStar 
    modelMaxLen = ModelPosEnd - ModelPosStar
    
    #查找每个盘的Sn和Model
    for line in diskinfo.splitlines()[2:-1]:
        #忽略空行
        if len(line.strip()) == 0:
            continue
       
        #查找loaction
        locationStart = line.find('(')
        locationEnd = line.find(')')
        
        #换行场景,数据还是上块盘的
        if locationStart == -1 or locationEnd == -1:
            location = lastLocation
        #下一块盘的数据开始，上块盘的信息要清除
        else:  
            location = line[locationStart:locationEnd + 1]
            lastLocation = location
            lastModel = ""
            lastSn = ""
        
        #获取对应的值
        snValue = lastSn + line[SNPosStar:SNPosEnd].strip()
        modelValue = lastModel + line[ModelPosStar:ModelPosEnd].strip()
        SNAndModelDict[location] = [modelValue, snValue]

        #设置最长字段的长度
        lastSn = snValue
        lastModel = modelValue
        snLen = len(snValue)
        modelLen = len(modelValue)
        if snLen >= snMaxLen:
            snMaxLen = snLen
        if modelLen >= modelMaxLen:
            modelMaxLen = modelLen
        
    return (True, SNAndModelDict, modelMaxLen, snMaxLen)
    
# **************************************************************************** #
# 函数名称: execute
# 功能说明: 脚本执行入口
# 输入参数: 设备对象
# 输出参数: 无
# **************************************************************************** # 
def execute(devObj):
    oldSysMode = ""#这里做定义为防止try里初始化时出异常后在except里引用错误
    try:
        
        #获取日志记录器:SIC = Smart Info Collection
        logger = devObj.get("logger")
        logger.info("[SIC] smart info collect begin.")
        
        #获取当前的系统模式
        oldSysMode = modelManager.getCurSysMode(devObj.get("SSH"))
        if not oldSysMode:
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 查询系统模式失败。请确保设备工作状态正常。"
            else:
                devObj["py_detail"] = "[Error] Failed to query system mode. Please make sure the device works in normal status."
            return (False, devObj["py_detail"])
        
        #当前登录用户必须为admin
        devUserName = devObj.get("username")
        if "admin" != devUserName:
            handleBeforeErrorExit(devObj, oldSysMode)
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 收集smart信息时，设备的登录用户名必须为admin，当前用户名" + devUserName + u"无效。"
            else:
                devObj["py_detail"] = "[ERROR] The login username must be admin when collecting smart information. The current username " + devUserName + " is invalid."
            return (False, devObj["py_detail"])
        
        #初始化本地smart信息目录内容
        localHostDir = initDataSaveDir(devObj)
        if "" == localHostDir:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        infoFilePath = localHostDir + os.path.sep + "basicinfo.txt"
        summaryFilePath = localHostDir + os.path.sep + "summary.ini"
        smartFilePath = localHostDir + os.path.sep + "smart@FID.txt"
        failFilePath = localHostDir + os.path.sep + "failsmart.txt"
        
        isSucc = True
        
        #先切换到CLI模式
        logger.info("[SIC] login to cli.")
        isSucc = modelManager.changeMode2CLI(devObj)
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        
        #收集基本信息
        #showsys信息        
        logger.info("[SIC] get showsys info.")
        excRet = getSystemInfo(devObj, infoFilePath)
        isSucc = excRet[0]
        devModel = excRet[1]
        pdtVer = excRet[2]
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        #raid组信息
        logger.info("[SIC] get raid group info.")
        isSucc = getRaidGroupInfo(devObj, infoFilePath)
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        #硬盘的物理信息
        logger.info("[SIC] get disk physic info.")
        excRet = getDiskPhysicInfo(devObj, infoFilePath, summaryFilePath)
        isSucc = excRet[0]
        diskPhyInfoList = excRet[1]
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        
        #切换到mml模式
        logger.info("[SIC] login to mml.")
        #如果切换失败再重试2次，总共最多3次
        retryTimes = 2
        while retryTimes >= 0:
            isSucc = modelManager.changeMode2MML(devObj)
            if not isSucc:
                retryTimes -= 1
                #如果失败则休眠30s后重试
                time.sleep(30)
                continue
            else:
                break
        if not isSucc:
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 已尝试三次从" + oldSysMode + u"切换到MML均失败，原因可能是：\n" + \
                u"（1）debug密码不正确，请确认是否已修改debug密码；\n" + \
                u"（2）设备业务压力大，请在业务压力较小时重试；\n" + \
                u"（3）设备状态异常，请确保设备工作状态正常。"
            else:
                devObj["py_detail"] = "[ERROR] Switching from the " + oldSysMode + " to the MML failed for three times. Possible causes:\n" + \
                "(1)The debug password is incorrect. Please check whether the debug password has been changed.\n" + \
                "(2)The device workload is heavy. Please retry when the workload is small.\n" + \
                "(3)The device is abnormal. Please ensure that the running status of the device is normal."
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        # modified 20140513 end
        
        #获取设备硬盘信息
        logger.info("[SIC] get the device disk list.")
        devDiskDict = getDevDiskDict(devObj, devModel, pdtVer)
        if not devDiskDict:
            if common.isChinese(devObj):
                devObj["py_detail"] = u"[错误] 未查到设备硬盘信息。"
            else:
                devObj["py_detail"] = "[ERROR] Disk information is not detected."
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        
        #切换到debug模式
        logger.info("[SIC] change to debug.")
        isSucc = modelManager.changeMode2Debug(devObj)
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        
        #收集本端控制器smart信息，并记录其概要信息到summary.ini以及失败信息（如果有的话）到failsmart.txt中
        logger.info("[SIC] get the smart info of all disk block.")
        filesPath = (smartFilePath, failFilePath, summaryFilePath)
        isSucc = getLocalSmartInfo(devObj, devDiskDict, diskPhyInfoList, filesPath)
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
        
        #重新回到CLI模式
        logger.info("[SIC] logout to cli.")
        isSucc = modelManager.changeMode2CLI(devObj)
        if not isSucc:
            handleBeforeErrorExit(devObj, oldSysMode)
            return (False, devObj["py_detail"])
            
        
        logger.info("[SIC] " + devModel + " series smart info collect finished.")
        return (True, "")
    except TimeOutException:
        logger.error("[SIC] Execute command timeout:" + traceback.format_exc())
        return (False, common.getCmdTimeOutErrMsg(devObj))
    except:
        try:
            handleBeforeErrorExit(devObj, oldSysMode)
        except:
            logger.error("[SIC] Failed to remove temporary directory:" + traceback.format_exc())
        if common.isChinese(devObj):
            devObj["py_detail"] = u"[错误] 执行数据收集脚本异常。"
        else:
            devObj["py_detail"] = "[ERROR] An exception occurred during executing data collecting script."
        return (False, devObj["py_detail"])
