# -*- coding: UTF-8 -*-


from java.io import File

import re
import socket
import os
import shutil
import zipfile
import traceback
import tarfile
import codecs
import cTV1R1
import time

import XMLParseCom
import modelManager
import common
import fileOperation
from common import TimeOutException

#命令或参数无效的回文
G_INVALID_CMD_RESULT ="The command name is invalid|The command format is wrong|does not exist"

#命令间隔符
G_INFORMATION_SPLIT_STR = "\n********************\n"

#所有数据存放顶层目录名称
G_ALL_DATA_DIRNAME = "DataCollect"

#信息收集子项类型
class CollectType:
        
    SYSTEM_INFO  = "SYSTEM_INFO"
    CONFIG_INFO  = "CONFIG_INFO"
    ALARM_INFO   = "ALARM_INFO"
    RUNNING_LOG  = "RUNNING_LOG"
    DEBUG_INFO   = "DEBUG_INFO"
    OTHER_INFO   = "OTHER_INFO"
    SYSTEM_STAT  = "SYSTEM_STAT"
    BACKUP_INFO  = "BACKUP_INFO"
    BST_INFO     = "BST_INFO"
    CURRENT_LOG = "CURRENT_LOG"
    

#信息收集方式类型
class CollectMethod:
        
    ParseXML  = "ParseXML"
    ExportCmd  = "ExportCmd"
    ExecCmd   = "ExecCmd"
    ExecShell  = "ExecShell"
    DownRemoteFile  = "DownRemoteFile"
    
    
#执行命令模式
class CmdExecuteModel:
    
    Cli = "cli"
    Debug = "debug"
    Mml = "mml"
    
#基于设备类型和版本号，将设备分为几种类型
#相同的类型，对于同一个method的处理方式相同
class DevVersionType:
    
    TypeUnKnown = ""
    #S2000型号
    S2000OSM  = "S2000_OSM"    #OSM版本,导出告警和日志是不同的命令,无参数-t
    S2000ISM  = "S2000_ISM"    #ISM版本,导出告警和日志是一条命令,要加参数-t
    
    #S2600型号
    S2600OLD   = "S2600_OLD"  #S2600 R2 的1.04.02.108.T01之前的版本, 对于告警和日志,使用txt后缀
    S2600NEW   = "S2600_NEW"  #S2600 R2 的1.04.02.108.T01及后续版本, 对于告警和日志,使用TAR后缀  
    
    #S5000型号
    S5000OSM  = "S5000_OSM"    #OSM版本,导出告警和日志是不同的命令,无参数-t
    S5000ISM  = "S5000_ISM"    #ISM版本,导出告警和日志是一条命令,要加参数-t

#收集项类型
G_COLLCCT_TYPE_DICT = {
    CollectType.ALARM_INFO: "1",    #导告警参数
    CollectType.RUNNING_LOG: "2"    #导出日志参数   
}

# *************************************************************#
# 函数名称: commonFileWrite
# 功能说明: 文件写入操作（通用函数）
# 其 他   :  无
# *************************************************************#  
def commonFileWrite(fileName, writeInfoStr, ifAddFlag, devObj):

    flag = True
    try:
        lang = devObj.get("lang")
        logger = devObj.get("logger")

        #文件不存在，新创建文件
        if not os.path.exists(fileName):
            localFile = codecs.open(fileName, "w", "utf-8")
        #向文件中追加内容
        elif ifAddFlag:
            localFile = codecs.open(fileName, "a", "utf-8")
        #覆盖文件内容
        else:
            localFile = codecs.open(fileName, "w", "utf-8")
            
        localFile.write(writeInfoStr)
        localFile.close()
    except:
        logger.error("[commonFileWrite] except trace back:" + str(traceback.format_exc()))
        if lang == "zh":
            devObj["py_detail"] = u"收集信息写入文件失败"
        else:
            devObj["py_detail"] = "Failed to write collection information to file"
        flag = False
    
    return flag

# **************************************************************************** #
# 函数名称：downloadFileAndDelete
# 功能说明：下载文件后，删除远端阵列临时文件
# 传入参数：downloadDict对象
# 返  回  值：True:执行成功；False:执行失败
# **************************************************************************** #
def downloadFileAndDelete(downloadDict):
    
    try:
        #获取方法
        sftp = downloadDict.get("SFTP")
        logger = downloadDict.get("logger")
        lang = downloadDict.get("lang")
        sourceFileDir = downloadDict.get("DownRemoteFileDir")
        sourceFileName = downloadDict.get("DownRemoteFileName")
        destFileDir = downloadDict.get("DownLocalFileDir")
        destFileName = downloadDict.get("DownLocalFileName")
        
        #判断远端阵列文件是否存在
        if not sftp.isFileExist(sourceFileDir, sourceFileName):
            logger.error("[downloadFileAndDelete] remote file does not exist:" + str(sourceFileDir) + str(sourceFileName))
            if lang == "zh":
                downloadDict["py_detail"] = u"远端阵列文件不存在，可能的原因：\n（1）收集命令执行失败；\n（2）当前版本不支持此项收集。"
            else:
                downloadDict["py_detail"] = "The remote file does not exist, the reason of failure may be: \n(1) The collection command failed to be executed; \n(2) The current version does not support this collection item."
            return False
        
        #判断本地保存文件夹是否存在，不存在则创建
        if os.path.exists(destFileDir) != True:
            os.mkdir(destFileDir)

        #通过SFTP下载文件
        remoteFile = sourceFileDir + sourceFileName
        localFile = destFileDir + os.path.sep + destFileName
        file = File(localFile)
        sftp.getFile(remoteFile, file, None)

        #不判断返回值：失败了会打印错误信息，但不影响最终收集结果
        deleteRemoteFile(downloadDict, remoteFile)
        return True

    except:
        logger.error("[downloadFileAndDelete] except trace back:" + str(traceback.format_exc()))
        if lang == "zh":
            downloadDict["py_detail"] = u"通过SFTP下载文件失败"
        else:
            downloadDict["py_detail"] = "Downloading remote file by SFTP failed"
        return False

# **************************************************************************** #
# 函数名称：deleteRemoteFile
# 功能说明：删除远端阵列上的文件
# 传入参数：devObj对象；remoteFile远端阵列文件
# 返  回  值：True:执行成功；False:执行失败
# **************************************************************************** #
def deleteRemoteFile(devObj, remoteFile):
    try:
        sftp = devObj.get("SFTP")
        logger = devObj.get("logger")
        #使用sftp自带接口删除远端临时文件
        sftp.deleteFile(remoteFile)
        return True
    except:
        logger.error("[deleteRemoteFile] except trace back:" + str(traceback.format_exc()))
        return False

# **************************************************************************** #
# 函数名称: execute_system_info
# 功能说明: 收集系统信息
# 输入参数: devObj
# **************************************************************************** # 
def execute_system_info(devObj):

    devObj["collectType"] = CollectType.SYSTEM_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)
    
# **************************************************************************** #
# 函数名称: execute_config_info
# 功能说明: 收集配置信息
# 输入参数: devObj
# **************************************************************************** # 
def execute_config_info(devObj):
    
    devObj["collectType"] =  CollectType.CONFIG_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)
    
# **************************************************************************** #
# 函数名称: execute_alarm_info
# 功能说明: 收集告警信息
# 输入参数: devObj
# **************************************************************************** #   
def execute_alarm_info(devObj):

    devObj["collectType"] =  CollectType.ALARM_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)
        
# **************************************************************************** #
# 函数名称: execute_running_log
# 功能说明: 收集运行日志
# 输入参数: devObj
# **************************************************************************** # 
def execute_running_log(devObj):

    devObj["collectType"] =  CollectType.RUNNING_LOG
    #调用公共处理方法
    return collectCommon_function(devObj)
    
# **************************************************************************** #
# 函数名称: execute_debug_info
# 功能说明: 收集debug信息
# 输入参数: devObj
# **************************************************************************** #     
def execute_debug_info(devObj):

    devObj["collectType"] =  CollectType.DEBUG_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)

# **************************************************************************** #
# 函数名称: execute_other_info
# 功能说明: 收集其他信息
# 输入参数: devObj
# **************************************************************************** # 
def execute_other_info(devObj):

    devObj["collectType"] =  CollectType.OTHER_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)

# **************************************************************************** #
# 函数名称: execute_system_stat
# 功能说明: 收集系统状态
# 输入参数: devObj
# **************************************************************************** #     
def execute_system_stat(devObj):

    devObj["collectType"] =  CollectType.SYSTEM_STAT
    #调用公共处理方法
    return collectCommon_function(devObj)

# **************************************************************************** #
# 函数名称: execute_backup_info
# 功能说明: 收集备份信息
# 输入参数: devObj
# **************************************************************************** #     
def execute_backup_info(devObj):

    devObj["collectType"] =  CollectType.BACKUP_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)
    
# **************************************************************************** #
# 函数名称: execute_bst_info
# 功能说明: 收集BST信息
# 输入参数: devObj
# **************************************************************************** #     
def execute_bst_info(devObj):

    devObj["collectType"] =  CollectType.BST_INFO
    #调用公共处理方法
    return collectCommon_function(devObj)

# **************************************************************************** #
# 函数名称: execute_current_log
# 功能说明: 收集当前日志
# 输入参数: devObj
# **************************************************************************** #     
def execute_current_log(devObj):

    devObj["collectType"] =  CollectType.CURRENT_LOG
    #调用公共处理方法
    return collectCommon_function(devObj)

# **************************************************************************** #
# 函数名称: collect_bst_info
# 功能说明: 收集BST信息
# 输入参数: devObj
# **************************************************************************** #     
def collect_bst_info(devObj):
    
    cmdExeResult = ''
    logger = devObj.get("logger")
    lang = devObj.get('lang')
    logger.info("[collect_BST_function] start to collect BST information.")
    LocalSaveDir = devObj.get('LocalSaveDir')
    bstInfoFileName = 'BST_information.txt'
    bstInfoFile = LocalSaveDir + os.sep + bstInfoFileName
    bstInfoFileObj = ''
    try:
        ssh = devObj.get("SSH")
        changeFlag = modelManager.changeMode2MML(devObj)        #切换到MML模式
        if not changeFlag:
            return False
        cmdExeResult = common.execCmd(ssh,"bst enable 3")      #查询BST开关
        cmdExeResult += common.execCmd(ssh,"bst showbst")      #查询BST信息
        bstInfoFileObj = open(bstInfoFile, 'w')         #打开文件
        bstInfoFileObj.write(cmdExeResult)              #将BST信息写入文件
        return True
    except TimeOutException:
        logger.error("[collect_BST_function] Execute command timeout:" + traceback.format_exc())
        common.getCmdTimeOutErrMsg(devObj)
        return False
    except Exception, e:
        logger.error("[collect_BST_function] Failed to collect BST information." + cmdExeResult)
        logger.error("[collect_BST_function] catch except of arguement:" + str(e.args))
        logger.error("[collect_BST_function] catch except of messages:" + str(e))
        logger.error("[collect_BST_function] catch except of trace back:" + str(traceback.format_exc()))
        if lang == "zh":
            devObj["py_detail"] = u"[错误] 获取BST信息失败。"
        else:
            devObj["py_detail"] = "[Error] Failed to get BST information."
        return False
    finally:
        try:
            modelManager.changeMode2CLI(devObj, False)     #切换到CLI模式
        except:
            pass
        if bstInfoFileObj:
            bstInfoFileObj.close()      #关闭文件
    
# **************************************************************************** #
# 函数名称: collectCommon_function
# 功能说明: 收集公共处理方法
# 输入参数: devObj
# **************************************************************************** #     
def collectCommon_function(devObj):
    
    try:
        devNode = devObj.get("devNode")
        collectType = devObj.get("collectType")
        lang = devObj.get("lang")
        logger = devObj.get("logger")
        flag = True
        logger.info("[collectCommon_function] start to collect: " + collectType)
    
        """步骤1：获取设备公共信息"""
        #获取设备类型，并归类
        ssh = devObj.get("SSH")
        deviceType =  common.getDeviceType(ssh)
        deviceType = common.switchDeviceType(deviceType)
    
        #获取设备版本类型(有些配置项并不满足版本类型)
        existMeetVerTypeCollect = False
        versionType = common.checkDeviceVersionType(devObj)
        logger.info("[collectCommon_function] Current device version type:" + str(versionType))
        if versionType == DevVersionType.TypeUnKnown:
            if lang == "zh":
                devObj["py_detail"] = u"获取设备版本类型失败"
            else:
                devObj["py_detail"] = "Getting device version type failed"
            return (False, "")
        devObj["deviceVerType"] = versionType
    
        """步骤2：创建本地数据保存目录"""
        #获取或构造临时数据存放目录路径
        tmpDataCollectDir = devObj.get("tmpDataCollectDir")
        if not tmpDataCollectDir:
            #临时目录不存在时则创建并记入设备对象
            curTime = str(time.strftime('%Y%m%d%H%M%S',time.localtime(time.time())))
            tmpDataCollectDir = devObj.get("collectRetDir") + os.sep + "tmpdir@" + curTime
            if not os.path.exists(tmpDataCollectDir):
                os.mkdir(tmpDataCollectDir)
            devObj["tmpDataCollectDir"] = tmpDataCollectDir
    
        #如果本地收集结果目录不存在，则创建
        parentSaveDir = devObj.get("tmpDataCollectDir") + os.path.sep + G_ALL_DATA_DIRNAME
        if not os.path.exists(parentSaveDir):
            os.mkdir(parentSaveDir)
        
        localSaveDir = parentSaveDir + os.path.sep + "Information"
        if not os.path.exists(localSaveDir):
            os.mkdir(localSaveDir)
        
        #目录结构调整,（非历史日志需要增加一级目录）
        if collectType not in [CollectType.BACKUP_INFO]:
            localSaveDir = localSaveDir + os.path.sep + "KeyInformation"
            if not os.path.exists(localSaveDir):
                os.mkdir(localSaveDir)
            	devObj["KeyInformation"] = localSaveDir

            if collectType == CollectType.CURRENT_LOG:
                localSaveDir = localSaveDir + os.path.sep + "currentLog"
                if not os.path.exists(localSaveDir):
                	os.mkdir(localSaveDir)
        devObj["LocalSaveDir"] = localSaveDir
        
        #查询BST信息
        if collectType == CollectType.BST_INFO:
            flag = collect_bst_info(devObj)
            return (flag, '')
        
        """步骤3：获取设备对应的指定收集项配置信息"""
        #获取配置文件路径
        fileName = "collectItem_" + deviceType + ".xml"
        scriptPath = os.path.split(os.path.realpath(__file__))[0]
        configFile = scriptPath + os.sep + "config" + os.sep + fileName
        logger.info("[collectCommon_function] configure file name : " + configFile)
        
        #获取单个收集项的配置信息
        collectItemConfigDict = XMLParseCom.createCollectConfigDict(configFile, collectType)
        logger.info("[collectCommon_function] configure data : " + str(collectItemConfigDict))
        if not collectItemConfigDict.has_key(collectType):
            if lang == "zh":
                devObj["py_detail"] = u"获取收集项配置信息失败，不能执行收集操作"
            else:
                devObj["py_detail"] = "Parsing configuration file failed, can not support to collect"
            return (False, "")
        
        #获取配置信息
        if not XMLParseCom.checkConfigDictValid(collectItemConfigDict):
            if lang == "zh":
                devObj["py_detail"] = u"获取收集项配置信息失败，不能执行收集操作"
            else:
                devObj["py_detail"] = "Parsing configuration file failed, can not support to collect"
            return (False, "")
        
        """步骤4：执行收集项的收集：循环执行多个method"""
        #一个收集项可能配置多种收集method:返回列表
        collectMethodDictList = collectItemConfigDict.get(collectType)
        #单个收集method字典数据结构:{'ParseXML': {'resultFile': 'System_information.txt', 'fileName': 'config/S5000/System_S5000.xml'}}
        for collectMethodDict in collectMethodDictList:
            #只有一个key
            collectMethodKey = collectMethodDict.keys()[0]
            #保存单个method的参数信息
            devObj["methodConfigDict"] = collectMethodDict.get(collectMethodKey)
            
            #获取配置method的版本类型
            ConfigVerType = devObj["methodConfigDict"].get("verType")
            #查看配置数据与当前设备版本类型是否匹配
            checkFlag = checkVerTypeConfigure(versionType, ConfigVerType)
            if not checkFlag:
                #不符合当前设备版本类型的数据，直接丢弃，报通过（配置文件中做兼容性处理，配置多种类型的数据结构，如OSM和ISM均要配置）
                logger.info("[collectCommon_function] Current configure data is invalid, drop it: " + str(devObj["methodConfigDict"]))
                continue
            
            #存在满足当前版本的收集method
            existMeetVerTypeCollect = True
            
            if collectMethodKey == CollectMethod.ParseXML:
                #执行ParseXML的通用处理流程
                iRet = execType_ParseXML(devObj)
                if not iRet:
                    flag = False
    
            elif collectMethodKey == CollectMethod.ExportCmd:
                #执行ExportCmd的通用处理流程
                iRet = execType_ExportCmd(devObj)
                if not iRet:
                    flag = False
    
            elif collectMethodKey == CollectMethod.ExecCmd:
                #执行ExecCmd的通用处理流程
                iRet = execType_ExecCmd(devObj)
                if not iRet:
                    flag = False
    
            elif collectMethodKey == CollectMethod.ExecShell:
                #执行ExecShell的通用处理流程
                iRet = execType_ExecShell(devObj)
                if not iRet:
                    flag = False
            #新增        
            elif collectMethodKey == CollectMethod.DownRemoteFile:
                #执行DownRemoteFile的通用处理流程
                iRet = execType_DownRemoteFile(devObj)
                if not iRet:
                    flag = False
    
            else:
                flag = False
        
        """步骤5：收集项执行结果返回"""
        #未执行任何信息收集method
        if not existMeetVerTypeCollect:
            if lang == "zh":
                devObj["py_detail"] = u"收集项配置信息错误，未执行任何收集命令"
            else:
                devObj["py_detail"] = "Configuration information error, can not execute collection"
            return (False, "")
    
        logger.info("[collectCommon_function] Succeed to collect: " + collectType)
        #执行结果返回    
        return (flag, "")
    except TimeOutException:
        logger.error("[collectCommon_function] Execute command timeout:" + traceback.format_exc())
        return (False, common.getCmdTimeOutErrMsg(devObj))
    except:
        logger.error("[collectCommon_function] except trace back:" + str(traceback.format_exc()))
        #设置错误返回信息
        if lang == "zh":
            devObj["py_detail"] = u"执行信息收集失败。"
        else:
            devObj["py_detail"] = "Collection executed failed."
        return (False, "")   

# **************************************************************************** #
# 函数名称: insertSingleItemCmds
# 功能说明: 将单个参数的列表插入到单个基础命令对应的的总体命令列表中
# 输入参数: singleCmdAllItemList：单条基础命令对应的的所有命令列表（待更新）
#         needSaveSingleItemList：单个配置参数的所有命令列表（待保存）
#         cmdBase:基础命令,第一次存入时使用
# **************************************************************************** #    
def insertSingleItemCmds(singleCmdAllItemList, insertSingleItemList, cmdBase):

    #录入列表为空
    if not insertSingleItemList:
        return singleCmdAllItemList
    
    #判断是否为第一次录入
    if not singleCmdAllItemList:
        #第一次录入，需要带上基础命令
        for cmdItemSub in insertSingleItemList:
            newInsertCmd = cmdBase + " " + cmdItemSub
            singleCmdAllItemList.append(newInsertCmd)
    else:
        #创建临时信息结构
        editCmdAllItemListTmp = []
        #非第一次录入，在源Base命令+原参数基础上,重新加上新的参数,生成新的命令
        for singleCmdAllSub in singleCmdAllItemList:
            for cmdItemSub in insertSingleItemList:
                oldCmdCombineNewItem = singleCmdAllSub + " " + cmdItemSub
                editCmdAllItemListTmp.append(oldCmdCombineNewItem)
        #命令列表信息更新
        singleCmdAllItemList = editCmdAllItemListTmp

    #返回更新后的列表
    return singleCmdAllItemList

# **************************************************************************** #
# 函数名称: createSingleCmdList
# 功能说明: 通过解析文件获取所有命令执行列表
# 输入参数: CmdConfigDict：命令配置文件名
#         ssh：有些配置需要执行命令获取参数值
#        logger:日志方法
# **************************************************************************** #
#备注：命令参数类型
#（1）不需要任何参数：items：[]
#（2）需要一个或多个参数，参数数值为整数：
#    cmdBase：showdisk 
#    items：[{'max': '23', 'name': '-sl', 'min': '0'}, {'max': '99', 'name': '-s', 'min': '0'}]
#（3）需要一个或多个参数，参数数值为枚举：items：[{'rangeList': 'A|B', 'name': '-c'}, {'rangeList': '00|01|10|11', 'name': '-p'}]
#    cmdBase：showiscsiroute 
#    items：[{'rangeList': 'A|B', 'name': '-c'}, {'rangeList': '00|01|10|11', 'name': '-p'}]     
#（4）需要一个或多个参数，参数依赖于各自基础命令查询到的key值(例子为虚构)
#    cmdBase：showlun 
#    items：[{'loop': 'ID', 'LoopBaseCmd': 'showlun', 'name': '-i '}, {'loop': 'RAID ID', 'LoopBaseCmd': 'showlun', 'name': '-rg '}, {'loop': 'ID', 'LoopBaseCmd': 'showrg', 'name': '-mm '}]
# **************************************************************************** #    
def createSingleCmdList(CmdConfigDict, ssh, logger):
    
    #有参数列表的命令，需要组合出来命令的列表
    singleCmdAllItemList = []
    
    #所有Loop参数对应的基础命令信息列表
    AllLoopBaseCmdDict = {}
    
    cmdBase = CmdConfigDict.get("cmdBase")
    
    cmdConfItemList = CmdConfigDict.get("items")
    #基础命令为空
    if not cmdBase:
        return singleCmdAllItemList

    """
    loop属性的参数，处理方式与其他有所不同：如果一个命令配置了两个loop参数，则两个参数的值
          并不是任意组合的，而是相匹配的：如一个Lun有唯一的ID和RAID ID，此处要把一个命令的所有loop参数记录，最终再处理
    """
    singleCmdLoopKeyDictList = []
    
    #（1）不需要任何参数：items：[],只保存基础命令
    if not cmdConfItemList:
        singleCmdAllItemList.append(cmdBase)
        return singleCmdAllItemList

    #需要一个或多个参数
    #每个循环处理一个参数（单条命令可能存在不同类型的参数）
    for cmdItemDict in cmdConfItemList:
        #参数配置错误
        if not cmdItemDict.has_key("name"):
            return singleCmdAllItemList
        
        #当前参数需要录入的命令列表
        needSaveSingleItemList = []

        #参数为基础命令查询到的字段取值
        if cmdItemDict.has_key("loop"):
            #保存所有loop参数的配置,暂时先不处理,后续统一处理
            singleCmdLoopKeyDictList.append(cmdItemDict)
            continue

        #参数取值为整形数区间值
        elif cmdItemDict.has_key("min") and cmdItemDict.has_key("max"):
            #min和max：整形数的有效性判断在getCmdListByParseFile中已经处理，此处不需要再判断
            rangeMinInt = int(cmdItemDict.get("min"))
            rangeMaxInt = int(cmdItemDict.get("max"))

            #min和max都包含边界值
            for index in range(rangeMinInt, rangeMaxInt+1):
                singleItemStr = cmdItemDict.get("name") + " " + str(index)
                needSaveSingleItemList.append(singleItemStr)
        
        #参数取值为枚举值
        elif cmdItemDict.has_key("rangeList"):
            #获取枚举值列表
            rangeListStr = cmdItemDict.get("rangeList")
            valueRangeList = rangeListStr.split("|")
            
            #将所有枚举值保存一遍
            for key in valueRangeList:
                singleItemStr = cmdItemDict.get("name") + " " + key
                needSaveSingleItemList.append(singleItemStr)
        else:
            #配置错误，返回空
            return []
       
        #保存单个配置项的命令
        singleCmdAllItemList = insertSingleItemCmds(singleCmdAllItemList, needSaveSingleItemList, cmdBase)
    
    #不存在loop属性的参数，直接返回
    if not singleCmdLoopKeyDictList:
        return singleCmdAllItemList
        
    #单独处理包含loop属性参数的命令
    loopKeysItemList = []
    logger.info("singleCmdLoopKeyDictList:" + str(singleCmdLoopKeyDictList))
    
    #创建所有loop参数的基础命令字典(注：每一个参数的loop值可能由不同的baseCmd命令获取)
    for loopDict in singleCmdLoopKeyDictList:
        subItemName = loopDict.get("name")
        loopKey  = loopDict.get("loop")
        LoopBaseCmd = loopDict.get("LoopBaseCmd")
        
        #字典中去除LoopBaseCmd，保存到新的数据结构
        #要用copy方法，防止原数据结构被破坏
        NameKeyDict = loopDict.copy()
        NameKeyDict.pop("LoopBaseCmd")

        #基础命令第一次录入
        if not AllLoopBaseCmdDict.has_key(LoopBaseCmd):
            AllLoopBaseCmdDict[LoopBaseCmd] = []
            AllLoopBaseCmdDict[LoopBaseCmd].append(NameKeyDict)
        #基础命令已存在，新的key值录入
        else:
            cmdKeyList = AllLoopBaseCmdDict.get(LoopBaseCmd)
            if loopKey not in AllLoopBaseCmdDict.get(LoopBaseCmd):
                AllLoopBaseCmdDict[LoopBaseCmd].append(NameKeyDict)

    #基础命令及key值对应列表
    logger.info("AllLoopBaseCmdDict:" + str(AllLoopBaseCmdDict))
    for baseCmd in AllLoopBaseCmdDict:
        nameKeyDictList = AllLoopBaseCmdDict.get(baseCmd)
        #同一个基础命令对应的所有key值的命令列表
        baseCmdKeysItemList = []
        
        #执行基础命令，获取key值对应的列表
        cliRet = common.execCmd(ssh,baseCmd)
        #没有对应的记录，需要丢弃所有的loop参数（注：基础命令仍然可以在后面录入）
        if len(cliRet.splitlines()) < 4:
            break
        #格式化处理命令回文     
        formatFunction = cTV1R1.cHandleTypeList(cliRet)
        DictList = formatFunction.handle()
        for dict in DictList:
            #获取当前字典中所包含所有的key值
            singleItemStr = ""
            #nameKeyDict数据结构：{'loop': 'ID', 'name': '-i '}
            for nameKeyDict in nameKeyDictList:
                subItemName = nameKeyDict.get("name")
                loopKey = nameKeyDict.get("loop")
                keyValue = dict.get(loopKey)
                #在同一行命令回文中获取所有key
                singleItemStr += subItemName + " " + keyValue + " "
            #sub命令录入
            baseCmdKeysItemList.append(singleItemStr)
        #每一个baseCmd录入一次
        singleCmdAllItemList = insertSingleItemCmds(singleCmdAllItemList, baseCmdKeysItemList, cmdBase)

    #存在loopKeys的情况,需要将基础命令插入列表最前面
    #防止前面的insertSingleItemCmds处理会将基础命令拼装,
    #在处理最终将所有的基础命令插入到总命令列表中 
    for loopDict in singleCmdLoopKeyDictList:
        logger.info("loopDict:" + str(loopDict))
        LoopBaseCmd = loopDict.get("LoopBaseCmd")
        singleCmdAllItemList.insert(0, LoopBaseCmd)

    #创建成功,返回
    return singleCmdAllItemList

# **************************************************************************** #
# 函数名称: createAllCmdList_ParseXML
# 功能说明: 通过解析文件获取所有命令执行列表
# 输入参数: cmdXmlFileName：命令配置文件名
#         ssh：有些配置需要执行命令获取参数值
#        logger:日志方法
# **************************************************************************** #   
#备注：命令参数类型
#（1）不需要任何参数：items：[]
#（2）需要一个或多个参数，参数数值为整数：
#    cmdBase：showdisk 
#    items：[{'max': '23', 'name': '-sl', 'min': '0'}, {'max': '99', 'name': '-s', 'min': '0'}]
#（3）需要一个或多个参数，参数数值为枚举：items：[{'rangeList': 'A|B', 'name': '-c'}, {'rangeList': '00|01|10|11', 'name': '-p'}]
#    cmdBase：showiscsiroute 
#    items：[{'rangeList': 'A|B', 'name': '-c'}, {'rangeList': '00|01|10|11', 'name': '-p'}]     
#（4）需要一个或多个参数，参数依赖于基础命令查询到的key值(例子为虚构)
#    cmdBase：showlun 
#    items：[{'name': '-i ', 'loop': 'ID'}, {'name': '-rg ', 'loop': 'RAID ID'}]
# **************************************************************************** # 
def createAllCmdList_ParseXML(cmdXmlFile, ssh, logger):

    #所有需要输入的命令（包括基础命令+带多个参数的命令）列表
    AllNeedExeCmdList = []
   
    #解析文件获取命令信息
    CmdConfigDictList = XMLParseCom.getCmdListByParseFile(cmdXmlFile, logger)
    if not CmdConfigDictList:
        logger.error("[createAllCmdList_ParseXML] Getting collect commands information failed.")
        if lang == "zh":
            devObj["py_detail"] = u"获取收集项配置命令失败，不能执行收集操作"
        else:
            devObj["py_detail"] = "Getting collect commands information failed"
        return False
    
    #获取单个命令配置的命令列表:基础命令+参数配置
    for CmdConfigDict in CmdConfigDictList:
        #调用公共处理函数
        singleBaseCmdsList = createSingleCmdList(CmdConfigDict, ssh, logger)
        if singleBaseCmdsList:
            AllNeedExeCmdList.extend(singleBaseCmdsList)
    
    #去除重复的命令，并保持原有命令的顺序
    newAllNeedExeCmdList = list(set(AllNeedExeCmdList))
    newAllNeedExeCmdList.sort(key=AllNeedExeCmdList.index)

    return newAllNeedExeCmdList


# **************************************************************************** #
# 函数名称: execType_ParseXML
# 功能说明: 执行ParseXML收集方式
# 输入参数: devObj
# **************************************************************************** #     
def execType_ParseXML(devObj):

    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    
    #部分收集成功标志
    partCollectFlag  = False
    #文件路径信息
    localSaveDir = devObj["LocalSaveDir"]    
    flag = True

    #获取参数：参数的有效性，本函数的调用者保证
    methodConfigDataDict = devObj["methodConfigDict"]

    #命令写入文件路径
    resultFile = methodConfigDataDict.get("resultFile")
    logger.info("resultFile:" + resultFile)
    cmdWriteResultFile = localSaveDir + os.path.sep + resultFile
    logger.info("cmdWriteResultFile:" + cmdWriteResultFile)
    
    #获取命令行配置文件路径
    cmdXmlFileName = methodConfigDataDict.get("fileName")
    scriptPath = os.path.split(os.path.realpath(__file__))[0]
    cmdXmlFile = scriptPath + os.sep + cmdXmlFileName
    
    #所有需要输入的命令（包括基础命令+带多个参数的命令）列表
    AllNeedExeCmdList = createAllCmdList_ParseXML(cmdXmlFile, ssh, logger)
    if not AllNeedExeCmdList:
        if lang == "zh":
            devObj["py_detail"] = u"需要收集的命令列表为空"
        else:
            devObj["py_detail"] = "The counter of commands is 0"
        return False

    #命令实时执行
    cmdExecuteResultStr = ""
    cmdExecuteCounter = 0
    logger.info("AllNeedExeCmdList:" + str(AllNeedExeCmdList))
    logger.info("[execType_ParseXML]: need collect command counter: " + str(len(AllNeedExeCmdList)))
    
    #收集成功命令个数
    successCounter = 0
    for cmd in AllNeedExeCmdList:
        #命令之间间隔100ms
        time.sleep(0.1)
        cliResult = common.execCmdNoLog(ssh,cmd)
        if re.search(G_INVALID_CMD_RESULT, cliResult, re.IGNORECASE):
            continue
        #开启性能统计开关的命令执行后，需要睡眠10秒
        if "chgstatswitch -o" in cmd:
            time.sleep(10)
        #收集成功命令个数
        successCounter += 1
        #每20条命令写入文件一次
        cmdExecuteResultStr += cliResult + G_INFORMATION_SPLIT_STR
        cmdExecuteCounter += 1
        if cmdExecuteCounter >= 20:
            boolRet = commonFileWrite(cmdWriteResultFile, cmdExecuteResultStr, True, devObj)
            if not boolRet:
                return False
        
            #初始化
            cmdExecuteResultStr = ""
            cmdExecuteCounter = 0
    
    #剩余命令写入文件
    if cmdExecuteResultStr:
        boolRet = commonFileWrite(cmdWriteResultFile, cmdExecuteResultStr, True, devObj)
        if not boolRet:
            return False
    
    #打印收集成功数
    logger.info("[execType_ParseXML]: collect success command counter: " + str(successCounter)) 
    if successCounter == 0:
        logger.error("[execType_ParseXML]: collect success command counter is 0, Failed")
        if lang == "zh":
            devObj["py_detail"] = u"未收集到任何命令信息"
        else:
            devObj["py_detail"] = "Did not get any command information"
        return False

    return flag
            
# **************************************************************************** #
# 函数名称: checkVerTypeConfigure
# 功能说明: 查看版本配置类型是否匹配
# 输入参数: deviceVerType, configVerType
# **************************************************************************** #
def checkVerTypeConfigure(deviceVerType, configVerType):
    
    #查看配置数据与当前设备版本类型是否匹配
    if configVerType == "None":
        return True
    
    if deviceVerType.endswith(configVerType):
        return True
    else:
        return False

# **************************************************************************** #
# 函数名称: checkIfNeedType
# 功能说明: 查看当前设备是否需要-t参数
# 输入参数: collectType：收集项
#        versionType：设备版本类型
# **************************************************************************** #
def checkIfNeedType(collectType, versionType):
    
    flag = False
    
    #非收集告警和运行日志，不需要-t参数
    if collectType not in [CollectType.ALARM_INFO, CollectType.RUNNING_LOG]:
        return flag
    
    #S5000和S2000的ISM版本+S2600的新版本+S2600老版本，需要-t参数
    if versionType in [DevVersionType.S2000ISM, DevVersionType.S5000ISM, DevVersionType.S2600OLD, DevVersionType.S2600NEW]:
        flag = True
    #S5000和S2000的OSM版本，不需要-t参数
    elif versionType in [DevVersionType.S2000OSM, DevVersionType.S5000OSM]:
        flag = False
    else:
        pass
    
    return flag

# **************************************************************************** #
# 函数名称: execType_ExportCmd
# 功能说明: 执行ExportCmd收集方式
# 输入参数: devObj
# **************************************************************************** #
def execType_ExportCmd(devObj):

    ssh = devObj.get("SSH")
    sftp = devObj.get("SFTP")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    devNode = devObj.get("devNode")
    
    #获取全局临时信息
    collectType = devObj["collectType"]
    versionType = devObj["deviceVerType"]
    
    #设备IP使用默认值
    ctrlIp = "127.0.0.1"
    UserName = devObj.get("username")
    loginPwd = devObj.get("password")

    flag = True

    #获取参数：参数的有效性，本函数的调用者保证
    methodConfigDataDict = devObj["methodConfigDict"] 
    ConfigVerType = methodConfigDataDict.get("verType")
    exportBaseCmd = methodConfigDataDict.get("exportCmd")
    inputFileName = methodConfigDataDict.get("inputFileName")
    DownLoadFileKey = methodConfigDataDict.get("DownLoadFile")
    resultFile = methodConfigDataDict.get("resultFile")
    
    #远端阵列生成文件保存路径
    DownLoadDir = methodConfigDataDict.get("DownLoadDir")
    devObj["DownRemoteFileDir"] = DownLoadDir
    
    #文件路径信息
    localSaveDir = devObj["LocalSaveDir"]

    CmdCombine = exportBaseCmd + " -i " + ctrlIp + " -u " + UserName + " -p " + loginPwd + " -f " + inputFileName
    #需要-t参数的命令
    if checkIfNeedType(collectType, versionType):
        #获取-t参数
        CmdCombine += " -t " + G_COLLCCT_TYPE_DICT.get(collectType)
    
    #执行命令
    cliRet = common.execCmdNoLog(ssh,CmdCombine)
    logger.info("export command exec result:" + cliRet.replace(loginPwd, "******"))
    #命令是否执行成功
    if re.search("failed|error", cliRet, re.IGNORECASE):
        if lang == "zh":
            devObj["py_detail"] = exportBaseCmd + u"命令执行失败。"
        else:
            devObj["py_detail"] = "The command " + exportBaseCmd + " execution failed."
        
        #提示需要连接主控
        if re.search("not the primary controller|master controller", cliRet, re.IGNORECASE):
            if lang == "zh":
                devObj["py_detail"] += u"\n请尝试连接主控执行信息收集。"
            else:
                devObj["py_detail"] += u"\nPlease try to connect to the primary controller to collect information."
        return False
    
    #export命令中未输入文件名，认为文件名称由阵列来生成
    if inputFileName.endswith("/"):
        #keyWord是一个列表：runlog,cmd_log
        fileStartKeyList = DownLoadFileKey.split(",")
        for fileStartKey in fileStartKeyList:
            #查看remote路径下是否存在以keyWord为开头的文件
            remoteFileList = sftp.listFiles(DownLoadDir)
            logger.info("remoteFileList:" + str(remoteFileList))
            for fileName in remoteFileList:
                if fileName.startswith(fileStartKey):
                    #将文件下载到本地（文件名称与阵列源文件相同）
                    remoteFileName = fileName
                    localFileName = fileName
                    #保存download临时参数
                    devObj["DownRemoteFileName"] = remoteFileName
                    devObj["DownLocalFileDir"] = localSaveDir
                    devObj["DownLocalFileName"] = localFileName
                    #存在文件下载失败
                    if not downloadFileAndDelete(devObj):
                        flag = False
                    continue
    #文件名称已指定
    else:
        remoteFileName = DownLoadFileKey
        localFileName = resultFile
        #保存download临时参数
        devObj["DownRemoteFileName"] = remoteFileName
        devObj["DownLocalFileDir"] = localSaveDir
        devObj["DownLocalFileName"] = localFileName
        #存在文件下载失败
        if not downloadFileAndDelete(devObj):
            flag = False    
    
    return flag

# **************************************************************************** #
# 函数名称: execType_ExecCmd
# 功能说明: 执行单条命令
# 输入参数: devObj
# **************************************************************************** #     
def execType_ExecCmd(devObj):

    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")

    #文件路径信息
    localSaveDir = devObj["LocalSaveDir"]
    flag = True

    #获取参数：参数的有效性，本函数的调用者保证
    methodConfigDataDict = devObj["methodConfigDict"]
    executeCmd = methodConfigDataDict.get("execCmd")
    cmdModel = methodConfigDataDict.get("cmdModel")
    
    #获取文件下载路径
    DownLoadFileFullPath = methodConfigDataDict.get("DownLoadFile")
    
    #命令写入文件路径
    resultFile = methodConfigDataDict.get("resultFile")
    cmdWriteResultFile = localSaveDir + os.path.sep + resultFile
    
    cmdExeResult = ""
    #Cli模式下
    if cmdModel == CmdExecuteModel.Cli:
        #命令模式切换到Cli
        changeFlag = modelManager.changeMode2CLI(devObj)
        if not changeFlag:
            return False
        cmdExeResult = common.execCmd(ssh,executeCmd)
    #debug模式下
    elif cmdModel == CmdExecuteModel.Debug:
        #命令模式切换到debug
        changeFlag = modelManager.changeMode2Debug(devObj)
        if not changeFlag:
            return False
        cmdExeResult = common.execCmd(ssh,executeCmd)
    #mml模式下
    elif cmdModel == CmdExecuteModel.Mml:
        #命令模式切换到debug
        changeFlag = modelManager.changeMode2MML(devObj)
        if not changeFlag:
            return False
        cmdExeResult = common.execCmd(ssh,executeCmd)
    else:
        logger.error("Command execute model error:" + str(cmdModel))
        return False    
    
    #命令切换回Cli
    changeFlag = modelManager.changeMode2CLI(devObj)
    if not changeFlag:
        return False
    
    #解析生成文件名称和路径
    remoteFileName = DownLoadFileFullPath.split("/")[-1]
    remoteFileDir = DownLoadFileFullPath[0:DownLoadFileFullPath.index(remoteFileName)]
    logger.info("remoteFileDir:" + remoteFileDir)
    logger.info("remoteFileName:" + remoteFileName)
    #保存download临时参数
    devObj["DownRemoteFileDir"] = remoteFileDir
    devObj["DownRemoteFileName"] = remoteFileName
    devObj["DownLocalFileDir"] = localSaveDir
    devObj["DownLocalFileName"] = resultFile

    #存在文件下载失败
    if not downloadFileAndDelete(devObj):
        flag = False

    return flag

# **************************************************************************** #
# 函数名称: execType_ExecShell
# 功能说明: 执行脚本命令，生成文件下载到本地
# 输入参数: devObj
# **************************************************************************** #     
def execType_ExecShell(devObj):

    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")

    #文件路径信息
    localSaveDir = devObj["LocalSaveDir"]
    flag = True
    #脚本执行命令的sub命令：防止脚本运行时，打印类似命令结束符而框架认为命令已经结束
    KILL_CMD_END_SUB = " >/dev/null 2>&1"

    #获取参数：参数的有效性，本函数的调用者保证
    methodConfigDataDict = devObj["methodConfigDict"]
    exeShellCmd = methodConfigDataDict.get("shellCmd")
    DownLoadFileFullPath = methodConfigDataDict.get("DownLoadFile")
    resultFile = methodConfigDataDict.get("resultFile")
    
    #获取所有控制器的IP，用于创建目录
    ipList = fileOperation.getCtrlIp(ssh)
    if ipList == []:
        logger.error("[getAllNodeFile] Ip list is empty")
        return False
    
    #获取当前的IP和对端IP
    curIp = devObj.get("devIp")
    peerIP = "peerController"
    for ip in ipList:
        if ip != curIp:
            peerIP = ip
    
    #命令模式切换到debug
    changeFlag = modelManager.changeMode2Debug(devObj)
    if not changeFlag:
        return False
    
    #根据loginPeer字段来判断是否需要到对端执行脚本
    if methodConfigDataDict["loginPeer"].lower() == "true":
        beatIp = fileOperation.getHeartbeatIp(ssh, devObj)
        if beatIp == "":
            return False
        
        #执行脚本,可能存在连接不上对端的情况
        exeShellCmdAll = 'ssh admin@%s "%s >/dev/null 2>&1"' % (beatIp, exeShellCmd)
        result = loginPeerExecCmd(devObj, exeShellCmdAll)
        if not result:
            #如果获取到的IP列表只有一个，则默认为当前为单控设备，不报错
            if len(ipList) > 1:
                devObj["collectAllInfo"] = False
                if common.isChinese(devObj):
                    devObj["py_detail"] = u"[错误] 无法连接到对端控制器，收集对端控制器的信息失败。"
                else:
                    devObj["py_detail"] = "[Error] Failed to collect information about the peer controller, because the peer controller cannot be connected."  
                    
            #命令切换回Cli           
            changeFlag = modelManager.changeMode2CLI(devObj)
            if not changeFlag:
                return False
            
            return True
        
        #拷贝文件到本端
        devicePeerList = []
        devicePeerList.append(DownLoadFileFullPath)
        deviceTmpDir = os.path.dirname(DownLoadFileFullPath)
        fileOperation.getPeerFile(ssh, devObj, beatIp, devicePeerList, deviceTmpDir)
        if methodConfigDataDict["renameByIp"].lower() == "true":
            resultFile = resultFile % peerIP
    else:
        if methodConfigDataDict["renameByIp"].lower() == "true":
            resultFile = resultFile % curIp
        #执行脚本命令
        exeShellCmdAll = exeShellCmd + KILL_CMD_END_SUB
        common.execCmd(ssh,exeShellCmdAll)
        
    #命令切换回Cli
    changeFlag = modelManager.changeMode2CLI(devObj)
    logger.info("changeMode2CLI:" + str(changeFlag))
    if not changeFlag:
        return False

    #解析生成文件名称和路径
    remoteFileName = DownLoadFileFullPath.split("/")[-1]
    remoteFileDir = DownLoadFileFullPath[0:DownLoadFileFullPath.index(remoteFileName)]
    logger.info("remoteFileDir:" + remoteFileDir)
    logger.info("remoteFileName:" + remoteFileName)
    #保存download临时参数
    devObj["DownRemoteFileDir"] = remoteFileDir
    devObj["DownRemoteFileName"] = remoteFileName
    devObj["DownLocalFileDir"] = localSaveDir
    devObj["DownLocalFileName"] = resultFile

    #存在文件下载失败
    if not downloadFileAndDelete(devObj):
        flag = False

    return flag

# **************************************************************************** #
# 函数名称: loginPeerExecCmd
# 功能说明: 登录远端执行命令
# 输入参数: devObj， cmd
# **************************************************************************** # 
def loginPeerExecCmd(devObj, cmd):
    ssh = devObj.get("SSH")
    itemCliRet = common.execCmd(ssh, cmd)
    cliRet = itemCliRet
    if re.search("Connection timed out", itemCliRet, re.IGNORECASE):
        return False
    
    if re.search("Are you sure you want to continue connecting (yes/no)?", itemCliRet, re.IGNORECASE):
        itemCliRet = common.execCmd(ssh, "yes")
        cliRet += itemCliRet
        
    if re.search("Password:", cliRet, re.IGNORECASE):
        #获取密码，用于登录对端
        pwd = devObj.get("password")
        cliRet = common.execCmdNoLog(ssh, pwd)

    return True
    

    
# **************************************************************************** #
# 函数名称: execType_DownRemoteFile
# 功能说明: 下载阵列文件到本地
# 输入参数: devObj
# **************************************************************************** #  
def execType_DownRemoteFile(devObj):
    
    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    sftp = devObj.get("SFTP")
    
    #获取命令行配置文件路径
    currentLogConfigFile = devObj["methodConfigDict"]["fileName"]
    scriptPath = os.path.split(os.path.realpath(__file__))[0]
    cmdXmlFile = scriptPath + os.sep + currentLogConfigFile
    
    #解析xml文件，获取需要下载文件的列表
    fileInfoDictList = XMLParseCom.getFileListByParseFile(cmdXmlFile, logger)
    
    #获取当前时间，用于阵列创建临时目录
    curTime = common.getCurTime()
    arrayTempDir = "/tmp/collect/" + curTime + "/currentLog"
    
    #获取文档
    result = fileOperation.getAllNodeFile(ssh, devObj, fileInfoDictList, arrayTempDir)
    return result
        
    
    
