# -*- coding: UTF-8 -*-
'''
@summary: datacollect module func:collect basic(CLI) information, executed in cli mode on current connection
'''
import os
import shutil
from common import util
from common import dataConstants
from common.dataCollectResultRecorder import DataCollectResultRecorder
from time import sleep
import modelManager

class DeviceBasisCollector():
    '''
    @summary: collect Admin mode command echo, collect in one file named cliCmd_info.txt
    '''
    def __init__(self, devObject, opztMngIp, localCtrlrPriority):
        '''
        @summary: initializing parameters
        '''
        self.devObj = devObject
        self.logger = self.devObj.get("logger")
        self.logger.info("[basicInfo] initiating BasicDeviceInfoCollector...")
        self.localMngIp = self.devObj.get("devIp")
        self.opztMngIp = opztMngIp
        self.localCtrlrPriority = localCtrlrPriority
        self.cmdEchos = ""
        self.errLog = "/********Admin information error********/\n"

    def doCollect(self):
        '''
        @summary: entrance
        '''
        self.closePaginationFunc()
        currentCtrlrDir, oppositeCtrlrDir = self.tryMkLocalDir()
        self.executeCliCmdAndClctEcho(currentCtrlrDir, oppositeCtrlrDir)
        return

    def tryMkLocalDir(self):
        '''
        @summary: try to make 'other' directory, if it does not exist, create one.
        @return: local directory corresponding to  current controller
        @return: local directory corresponding to  opposite controller
        '''
        localBaseDir = self.devObj.get("collectRetDir", "")
        self.logger.info("current local base file Path is " + unicode(localBaseDir))
        localMainCtlrLocation = os.path.join(self.devObj["localMainCtlrLocation"],
             dataConstants.COLLECT_TYPE_OTHERS)
        if not os.path.exists(localMainCtlrLocation) and localMainCtlrLocation:
            os.makedirs(localMainCtlrLocation)
        localOpztCtlrLocation = ""
        if self.opztMngIp :
            localOpztCtlrLocation = os.path.join(self.devObj["localOpztCtlrLocation"],
                                                 dataConstants.COLLECT_TYPE_OTHERS)
            if not os.path.exists(localOpztCtlrLocation) and localOpztCtlrLocation:
                os.makedirs(localOpztCtlrLocation)

        return localMainCtlrLocation, localOpztCtlrLocation

    def closePaginationFunc(self):
        '''
        @summary: set 'chgpagination' switch to off , which brings '--more--' interaction disabled
        '''
        cmd = "chgpagination off"
        cmdEchoList = util.executeCmdWithTimeout(self.devObj, cmd)
        return cmdEchoList[0]


    def copyLocalFile(self, sourceFile, destFile):
        '''
        @summary: copy local file(s) from sourceFolder to destFolder
        @param sourceFolder (str): destFolder(str)
        @return: boolean
        '''
        if not os.path.exists(sourceFile) :
            self.logger.info("try to copy files to destination but found source folder not exist!")
            return False
        try:
            shutil.copy(sourceFile, destFile)
            return True
        except EnvironmentError, envEr:
            self.logger.error("Environment Error occurred while copying dirs: %s" % str(envEr))
            return False
        except BaseException, excpt:
            self.logger.error("error occurred while copying dirs: %s" % str(excpt))
            return False

    def executeCliCmdAndClctEcho(self, currentCtrlrDir, oppositeCtrlrDir):
        '''
        @summary: execute command on admin mode and collect command echo to specified file
        @note: cliCmdInfo.txt
        '''
        exportFile = os.path.join(currentCtrlrDir, dataConstants.CLI_CMD_ECHO_DESTINATION)
        self.logger.info("now try to export system settings to %s..." % str(exportFile))
        hasSucceeded, hasFailed = self.runCmdAndCltEcho()
        try:
            appenderObj = open(exportFile, "a")
            appenderObj.write(self.cmdEchos)
            appenderObj.close()
        except BaseException, excp:
            self.logger.info("failed to append result to file : %s" % str(excp))
            self.errLog += ("~failed to write admin mode collection result to cliCmdInfo.txt.\n")
        if self.opztMngIp:
            self.copyLocalFile(exportFile, oppositeCtrlrDir)
        self.recordCrntModuleStatus(hasSucceeded, hasFailed)
        return hasSucceeded, hasFailed

    def runCmdAndCltEcho(self):
        '''
        @summary: collect command echo
        @return: collect status, errLog, cmdEchos
        '''

        hasFailed = False
        hasSucceeded = False
        try:
            #collecting simple command：
            for checkItem in dataConstants.COMMON_CLI_CMD_LIST:
                isSucc, cmdEcho = util.executeCmdWithTimeout(self.devObj, checkItem)
                if not isSucc:
                    self.cmdEchos += ("\n" + checkItem + " command executed failed \n")
                    hasFailed = True
                    self.errLog += ("~" + checkItem + " command executed failed \n" + cmdEcho + ".\n")
                else:
                    self.cmdEchos += ("\n" + cmdEcho + "\n")
                    hasSucceeded = True
            #collect host information
            failedOcurd, successOcurd = self.clctHostInfos()
            hasFailed = hasFailed or failedOcurd
            hasSucceeded = hasSucceeded or successOcurd
            #collecting IO status command  for 3 times
            if not modelManager.changeCli2Developer(self.devObj):
                modelManager.changeAnyModel2Cli(self.devObj.get("SSH"), self.devObj.get("SFTP"))
                self.errLog += ("~failed to enter developer mode, unable to execute showio command! \n")
                hasFailed = True
                return hasSucceeded, hasFailed

            for times in ["first", "second", "third"]:
                self.logger.info("enter developer mode to execute showio command!")
                self.cmdEchos += ("\n processing showio command for the %s time: \n" % str(times))
                isSucc, cmdEcho = util.executeCmdWithTimeout(self.devObj, dataConstants.SHOW_IO_CMDS[0])
                self.cmdEchos += ("\n" + cmdEcho + "\n")
                if self.opztMngIp:
                    isSucc, cmdEcho = util.executeCmdWithTimeout(self.devObj, dataConstants.SHOW_IO_CMDS[1])
                    self.cmdEchos += ("\n" + cmdEcho + "\n")

                sleep(dataConstants.IO_SLEEP_INTERVAL)

            modelManager.changeAnyModel2Cli(self.devObj.get("SSH"), self.devObj.get("SFTP"))
            return hasSucceeded, hasFailed
        except BaseException, expn:
            self.logger.info("[basicInfo] try to fetch admin mode info but found exception:%s." % str(expn))
            self.errLog += ("~try to collect Admin mode information but found exception: %s.\n" % str(expn))
            hasFailed = True
        return hasSucceeded, hasFailed

    def clctHostInfos(self):
        '''
        @summary: collect device host information:include host information list and specific host information
        '''
        hasFailed, hasSuccessed = False, False
        try:
            isSucc, cmdEcho = util.executeCmdWithTimeout(self.devObj, dataConstants.SHOW_HOST_CMD)
            if not isSucc:
                hasFailed = True
                self.logger.info("failed to execute command %s" % str(dataConstants.SHOW_HOST_CMD))
                self.cmdEchos += ("\n" + dataConstants.SHOW_HOST_CMD + " command executed failed \n")
                self.errLog += ("~" + dataConstants.SHOW_HOST_CMD + " command executed failed. \n" + cmdEcho + "\n")
                self.cmdEchos += ("\n" + dataConstants.SHOW_HOST_DETAIL_CMD + \
                                  " command executed failed due to failure of showhost command\n")
                self.errLog += ("~" + dataConstants.SHOW_HOST_DETAIL_CMD + \
                                  " command executed failed due to failure of showhost command.\n")
            else:
                hasSuccessed = True
                self.logger.info("Successfully executed command %s" % str(dataConstants.SHOW_HOST_CMD))
                self.cmdEchos += ("\n" + cmdEcho + "\n")

            self.logger.info("start to fetch for specific host ID list...")
            hostIds = self.parse4HostIdList(cmdEcho)
            if not hostIds or len(hostIds) <= 0:
                self.logger.info("found no data about linked hostIds")
                self.errLog += ("~no data found about hosts corresponding to the device.\n")
                return hasFailed, hasSuccessed

            for hostId in hostIds:
                if not hostId:
                    continue
                cmd = dataConstants.SHOW_HOST_DETAIL_CMD + " " + hostId
                isSucc, cmdEcho = util.executeCmdWithTimeout(self.devObj, cmd)
                if not isSucc:
                    self.cmdEchos += ("\n" + cmd + " command execute failed \n")
                    self.errLog += ("~" + cmd + " command executed failed :" + cmdEcho + ".\n")
                    hasFailed = True
                    continue
                else:
                    hasSuccessed = True
                    self.cmdEchos += ("\n" + cmd + "command executed failed.\n")

            return hasFailed, hasSuccessed
        except:
            raise

    def parse4HostIdList(self, cmdEcho):
        '''
        @summary: parse for showhost command's echo, fetching for all linked host id list
        @return: host id lists
        '''
        self.logger.info("start to fetch for linked hosts... ")
        hostIds = []
        concatPoi = 0
        idPoi = 0
        if len(cmdEcho.splitlines()) < 6:
            return []
        for line in cmdEcho.splitlines()[4:-2]:
            if "===" in line or "---" in line :
                pass
            elif "Host ID" in line:
                idPoi = line.find("Host ID")
                titleLen = len("Host ID")
                for char in line[idPoi + titleLen:]:
                    if not char.isspace():
                        concatPoi = idPoi + line[idPoi + titleLen:].index(char) + titleLen
                        break
            else:
                hostId = line[idPoi:concatPoi].strip()
                self.logger.info("found one host: %s" % str(hostId))
                hostIds.append(hostId)
        hostIds = list(set(hostIds))
        if "" in hostIds:
            hostIds.remove("")
        return hostIds

    def recordCrntModuleStatus(self, hasSucceeded, hasFailed):
        '''
        @summary: write into info.txt to record current item status
        '''
        result = ""
        if hasFailed and hasSucceeded:
            result = dataConstants.IS_COLLECT_ITEM_PARTLY_SUCCESS
        elif hasFailed:
            result = dataConstants.IS_COLLECT_ITEM_FAILED
        else:
            result = dataConstants.IS_COLLECT_ITEM_SUCCESS
        resultRecorder = DataCollectResultRecorder(self.devObj)
        itemName = dataConstants.CLI_CMD_ECHO_DESTINATION
        resultRecorder.recordCrntModuleStatus(self.devObj["localMainCtlrLocation"], itemName, result, self.errLog)
        if self.opztMngIp:
            resultRecorder.recordCrntModuleStatus(self.devObj["localOpztCtlrLocation"], itemName, result, self.errLog)
