# -*- coding: UTF-8 -*-

from common import deviceInfo, code

# **************************************************************************** #
# 函数名称: checkLimit4MemLeak
# 功能说明: 检查消息节点泄漏完成后无法登陆问题
# 输入参数: devObj
# **************************************************************************** # 
def checkLimit4MemLeak(devObj):
    
    CHECK_TYPE_LIST = ['S2600T', 'S5600T', 'S5500T', 'S2200T', 'S5800T', 'S6800T']
    SYS_PATCH_VER_MAP = {'V100R005C01SPC700':'V100R005C01SPH705',
                         'V100R005C00SPC700':'V100R005C00SPH705',
                         'V100R005C01SPC900':'V100R005C01SPH905',
                         'V100R005C00SPC900':'V100R005C00SPH905',
    }
    
    #不限制收集的版本
    NoLimitVersion = "V100R005C02SPC300"

    lang = devObj.get("lang")
    logger = devObj.get("logger")

    #判断设备型号和版本
    sysSpcVer = deviceInfo.getSystemVersion(devObj)
    logger.info("[checkLimit4MemLeak] system version:" +  sysSpcVer)
    if sysSpcVer >= NoLimitVersion:
        return (True, "")
    if sysSpcVer not in SYS_PATCH_VER_MAP:
        return (False, code.getError(lang, "versionLimit"))
    
    #判断设备类型
    deviceType = deviceInfo.getDeviceType(devObj)
    logger.info("[checkLimit4MemLeak] Device type:" +  deviceType)
    if deviceType not in CHECK_TYPE_LIST:
        return (False, code.getError(lang, "versionLimit")) 
    
    #判断热补丁
    queryOk, hotPatchVer = deviceInfo.getHotpatchVersion(devObj)
    logger.info("[checkLimit4MemLeak] Hotpatch version:" +  hotPatchVer)
    if not queryOk:
        return (False, code.getError(lang, "queryError"))
    
    #获取需要安装的热补丁
    needHotPatchVer = SYS_PATCH_VER_MAP.get(sysSpcVer)
    if not hotPatchVer or hotPatchVer < needHotPatchVer:
        return (False, code.getError(lang, "hotPatchLimit", [needHotPatchVer]))
    return (True, "")

# **************************************************************************** #
# 函数名称: checkLimit4InfModule
# 功能说明: 检查4口FCoE或TOE接口卡，低概率发生OOPS的问题
# 输入参数: devObj
# **************************************************************************** # 
def checkLimit4InfModule(devObj):

    SYS_PATCH_VER_MAP = {
                         'V100R005C30SPC500':'V100R005C30SPH501',
                         'V100R005C30SPC700':'V100R005C30SPH701',
    }

    #限制收集的开始版本
    LimitStartVersion = "V100R005C02SPC300"
    #已经解决的版本
    SolvedStartVersion = "V100R005C30SPC800"

    lang = devObj.get("lang")
    logger = devObj.get("logger")

    #判断设备版本
    sysSpcVer = deviceInfo.getSystemVersion(devObj)
    logger.info("[checkLimit4InfModule] system version:" +  sysSpcVer)
    if sysSpcVer < LimitStartVersion or sysSpcVer >= SolvedStartVersion:
        return (True, "")
    
    queryOk, existFlag = deviceInfo._ifExistT4InfModule(devObj)
    logger.info("[checkLimit4InfModule] Exist T4 Interface Module:" +  str(existFlag))
    if not queryOk:
        return (False, code.getError(lang, "queryError"))    
    if not existFlag:
        #不存在4口FCoE或TOE接口卡，不限制
        return (True, "")

    if sysSpcVer not in SYS_PATCH_VER_MAP:
        return (False, code.getError(lang, "versionLimit"))

    #判断热补丁
    queryOk, hotPatchVer = deviceInfo.getHotpatchVersion(devObj)
    logger.info("[checkLimit4InfModule] Hotpatch version:" +  hotPatchVer)
    if not queryOk:
        return (False, code.getError(lang, "queryError"))
    
    #获取需要安装的热补丁
    needHotPatchVer = SYS_PATCH_VER_MAP.get(sysSpcVer)
    if not hotPatchVer or hotPatchVer < needHotPatchVer:
        #V100R005C30SPC700版本定制化错误
        if sysSpcVer == "V100R005C30SPC700":
            return (False, code.getError(lang, "hotPatchLimit_SPC700", [needHotPatchVer]))
        else:
            return (False, code.getError(lang, "hotPatchLimit", [needHotPatchVer]))
    return (True, "")

# **************************************************************************** #
# 函数名称: ifCanCollect
# 功能说明: 检查是否允许执行收集
# 输入参数: devObj
# **************************************************************************** # 
def ifCanCollect(devObj):
    
    #检查消息节点泄漏完成后无法登陆问题
    canCollect, errMsg = checkLimit4MemLeak(devObj)
    if not canCollect:
        return (canCollect, errMsg)
    
    #检查4口FCoE或TOE接口卡，低概率发生OOPS的问题
    canCollect, errMsg = checkLimit4InfModule(devObj)
    if not canCollect:
        return (canCollect, errMsg)
    return (True, "")
