# -*- coding: UTF-8 -*-

import re

# **************************************************************************** #
# 函数名称: _switchHotPatchVersion
# 功能说明: 将热补丁版本号转化成有效的版本号（V100R002C00SPC011SPHb04转变成V100R002C00SPHb04）
# 输入参数: version
# 输出参数: 无
# 返 回 值: version
# **************************************************************************** # 
def _switchHotPatchVersion(version):
    
    startIndex = version.index("V100")
    #提取有效的版本号信息
    version = version[startIndex:]

    #老版本热补丁版本不规范：V100R002C00SPC011SPHb04
    if "SPC" in version:
        startIndex = version.index("SPC")
        endIndex = startIndex + 6
        delStr = version[startIndex: endIndex]
        version = version.replace(delStr, "")
    return version

# **************************************************************************** #
# 函数名称: _switchValidVersion
# 功能说明: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
# 输入参数: version
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def _switchValidVersion(version):

    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  #格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  #格式：V100R005C02
    
    #提取有效的版本号信息
    temp = version[startIndex: endIndex]
    
    return temp

# *************************************************************#
# 函数名称: getSystemVersion
# 功能说明: 获取当前系统SPC版本号
# 其 他   :  无
# *************************************************************#
def getSystemVersion(devObj):
    
    if devObj.get("SystemVersion") is None:
        #查询系统版本号
        devNode = devObj.get("devNode")
        sysSpcVersion = devNode.getProductVersion()
        devObj["SystemVersion"] = _switchValidVersion(sysSpcVersion)
    return devObj.get("SystemVersion")

# *************************************************************#
# 函数名称: getDeviceType
# 功能说明: 获取当前系统型号
# 其 他   :  无
# *************************************************************#
def getDeviceType(devObj):
    #查询系统型号
    devNode = devObj.get("devNode")
    return str(devNode.getDeviceType())

# *************************************************************#
# 函数名称: getHotpatchVersion
# 功能说明: 查询当前设备已安装热补丁
# 其 他   :  无
# *************************************************************#
def getHotpatchVersion(devObj):
    
    if devObj.get("HotpatchVersion") is not None:
        return (True, devObj.get("HotpatchVersion"))
    #初始化为空
    devObj["HotpatchVersion"] = ""

    #查询当前设备已安装热补丁
    ssh = devObj.get("SSH")
    curHotPatchVer = ""

    hotPatchVerInfo = ssh.execCmd("showupgradepkginfo -t 3")
    #不支持热补丁，检查通过
    if re.search("not exist.|Error : Invalid argument", hotPatchVerInfo, re.IGNORECASE):
        return (True, '')

    #没有安装任何热补丁，返回空
    if re.search("command operates successfully", hotPatchVerInfo, re.IGNORECASE):
        return (True, '')

    #导入了热补丁包但是没有激活
    if not re.search("Running", hotPatchVerInfo, re.IGNORECASE):
        return (True, '')
    
    lineList = hotPatchVerInfo.splitlines()
    #信息少于7行，查询信息失败
    if len(lineList) < 7:
        return (False, '')
    
    #正常情况下获取Running的热补丁版本
    for line in lineList:
        if re.search("A|B", line) and re.search("Running", line, re.IGNORECASE):
            field = line.split()
            if len(field) >= 2:
                curHotPatchVer = _switchHotPatchVersion(field[1])
                break
        else:
            continue 
    
    #保存获取成功的版本号
    devObj["HotpatchVersion"] = curHotPatchVer
    #检查结果返回
    return (True, curHotPatchVer)

# **************************************************************************** #
# 函数名称: _ifExistT4InfModule
# 功能说明: 检查设备上是否存在4口FCoE或TOE接口卡
# 输入参数: cli
# **************************************************************************** # 
def _ifExistT4InfModule(devObj):
    
    existFlag = False
    ssh = devObj.get("SSH")

    infModuleInfo = ssh.execCmd("showifmodule")
    lineList = infModuleInfo.splitlines()
    #信息少于7行，查询信息失败
    if len(lineList) < 7:
        if "command operates successfully" in infModuleInfo:
            return (True, existFlag)
        return (False, existFlag)
    for line in lineList:
        if not line.strip().startswith("Type"):
            continue
        if "4 x 10G FCoE Interface Module" in line or \
            "4 x 10GE Interface Module" in line:
            existFlag = True
            return (True, existFlag)
    return (True, existFlag)
    