﻿# -*- coding: UTF-8 -*-
import ast
import os
import re
import common
import time
import modelManager

#获取信息分隔符
G_INFORMATION_SPLIT_STR = "\n********************\n"

CLI_EXEC_FINISH = ["Root Ckg Idx larger than Max, please check","Error:","Current controller is not the pool work controller"]
POOL_NUM = 0
BASE_PATH = ""
TAIL_MSG = ""

# **************************************************************************** #
# 函数名称: _init(devObj, plInfoListDict)
# 功能说明: 初始化
# 输入参数: devObj，plInfoListDict
# 输出参数: 无
# **************************************************************************** # 
def _init(devObj, plInfoListDict):
    _initBasePath(devObj)
    _initPoolInfo(devObj,plInfoListDict)
    
# **************************************************************************** #
# 函数名称: _initBasePath(devObj)
# 功能说明: 初始化语言环境及设置文件保存基本路径
# 输入参数: devObj
# 输出参数: 无
# **************************************************************************** # 
def _initBasePath(devObj):
    global BASE_PATH
    
    BASE_PATH = ""
    logger = devObj.get("logger")
        
    fileBaseDir = devObj.get("collectRetDir")+os.path.sep+"temp"+os.path.sep
    subFileDir = fileBaseDir + "poolCkInfo"

    logger.info("[collect] Create fileBaseDir is :" + subFileDir)
    if not os.path.exists(subFileDir):
        os.makedirs(subFileDir)
    devObj['ckInfoDir'] = subFileDir
    BASE_PATH = subFileDir
    return

# **************************************************************************** #
# 函数名称: _initPoolInfo(devObj,plInfoListDict)
# 功能说明: 获取POOL总个数及状态不正常POOL等信息
# 输入参数: devObj,plInfoListDict
# 输出参数: True or False
# **************************************************************************** #  
def _initPoolInfo(devObj,plInfoListDict):
    flag = True
    global POOL_NUM
    global TAIL_MSG
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    
    POOL_NUM = 0
    TAIL_MSG = ""
    for poolInfo in plInfoListDict:
        POOL_NUM +=1
        if "Normal"  != poolInfo.get("Health Status") and "Degraded"  != poolInfo.get("Health Status"):
            flag = False
            TAIL_MSG += "POOL("+str(poolInfo.get("ID"))+") health status is abnoraml("+ \
            poolInfo.get("Health Status")+"), can't collect it CK information\n"
    return          

# **************************************************************************** #
# 函数名称: _isCkgInfoQueryFinish(cliRet)
# 功能说明: 判断查询rootCkg&expandCKg命令是否结束
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False
# **************************************************************************** # 
def _isCkgInfoQueryFinish(cliRet):
    for endTag in CLI_EXEC_FINISH:
        if re.search(endTag, cliRet, re.IGNORECASE):
            return True
    return False

# **************************************************************************** #
# 函数名称: _getCtrlNum(cliRet)
# 功能说明: 获取控制器个数
# 输入参数: CliRet（CLI回显）
# 输出参数: True or False
# **************************************************************************** # 
def _getCtrlNum(cliRet):
    ctrNum = 0
    
    if None == cliRet or ""==cliRet:
        return ctrNum
    
    lineList = cliRet.splitlines()
    
    for line in lineList:
        line = line.replace(" ","")
        if line.startswith("Controller:"):
            ctrNum += 1
    
    return ctrNum

# **************************************************************************** #
# 函数名称: _getMetaLastCkgId(cliRet)
# 功能说明: 获取metaCkg最后的Ckg Id信息
# 输入参数: cliRet（CLI回显）
# 输出参数: ckgId
# **************************************************************************** # 
def _getMetaLastCkgId(cliRet):
    ckgId = "0xffffffff"
    splitCli = common.splitByblanckLine(cliRet)
    if len(splitCli)<2:
        return ckgId
    
    metaDataListDict = common.formatDict(splitCli[-2])
    if not metaDataListDict:
        return ckgId
    
    ckgId = metaDataListDict.get("Ckg Id")
    
    if not ckgId:
        ckgId = "0xffffffff"
    return ckgId

# **************************************************************************** #
# 函数名称: _getFilePath(poolId)
# 功能说明: 根据POOLID获取文件路径
# 输入参数: poolId
# 输出参数: 文件路径
# **************************************************************************** # 
def _getFilePath(poolId):
    return BASE_PATH +os.path.sep+"POOL_"+str(poolId)+".log"

# **************************************************************************** #
# 函数名称: _getPoolListDict(devObj)
# 功能说明: 获取阵列POOL ID列表，包含ID及健康状态信息
# 输入参数: devObj
# 输出参数: cliRet, ListDict
# **************************************************************************** # 
def _getPoolListDict(devObj):
    
    listDict = []
    ssh = devObj.get("SSH")
    cliRet = ssh.execCmd("show storage_pool general")
    
    #命令是否执行成功
    if common.isExecSuccess(cliRet):
        return cliRet,listDict
    
    poolInfo = common.formatList(cliRet)
    if not poolInfo:
        return cliRet, listDict
    
    key = ["ID","Health Status"]   
    for dic in poolInfo:
        val =[]
        for k in key:
            val.append(dic.get(k))     
        listDict.append(dict(zip(key,val)))
    
    return cliRet, listDict

# **************************************************************************** #
# 函数名称: _getCtrlAIpAddrList(ipInfoListDict)
# 功能说明: 获取A控对应的IP地址
# 输入参数: devObj
# 输出参数: ip地址列表
# **************************************************************************** # 
def _getCtrlAIpAddrList(ipInfoListDict):
    ctrAIp = []
    for ipDict in ipInfoListDict:
        if "A" == ipDict.get("Controller"):
            ctrAIp.append(ipDict.get("IPv4 Address"))
            ctrAIp.append(ipDict.get("IPv6 Address"))
            
    return ctrAIp

# **************************************************************************** #
# 函数名称: _getIpListDict(devObj)
# 功能说明: 获取Ip信息列表
# 输入参数: devObj
# 输出参数: Ip信息列表
# **************************************************************************** # 
def _getIpListDict(devObj):
    ssh = devObj.get("SSH")
    cliRet = ssh.execCmd("show controller ip")
    
    listDict = common.formatDict(cliRet)
    return cliRet, listDict
    
# **************************************************************************** #
# 函数名称: _getCtrlPoolList(loginIp,ipInfoListDict,poolListDict,isLocal,ctrNum)
# 功能说明: 获取对应端阵列正常的POOL ID列表，单控时都在本端执行
# 输入参数: loginIp,ipInfoListDict,poolListDict,isLocal,ctrNum
# 输出参数: List列表
# **************************************************************************** # 
def _getCtrlPoolList(loginIp,ipInfoListDict,poolListDict,isLocal,ctrNum):
    evenList = []
    oddList = []
    for poolInfo in poolListDict:
        #状态异常时不收集
        if "Normal"  == poolInfo.get("Health Status") or \
         "Degraded"  == poolInfo.get("Health Status"):
            if 0 == int(poolInfo.get("ID")) % 2:
                evenList.append((poolInfo.get("ID")))
            else:
                oddList.append((poolInfo.get("ID")))
    
    evenList.sort()
    oddList.sort()
    
    #双控按照规则来
    if 2 == ctrNum:
        if loginIp in _getCtrlAIpAddrList(ipInfoListDict):
            if isLocal:
               return evenList 
            else:
               return oddList  
        else:
            if isLocal:
               return oddList 
            else:
               return evenList 
           
    #单控都在本端收集
    else:
        if isLocal:
            listAll = list(set(evenList+oddList))
            listAll.sort()
            return listAll
        else:
            return []
           
# **************************************************************************** #
# 函数名称: _createSummaryInfo(devObj, plInfoCliRet, ipInfoCliRet, successNum, partlyNum)
# 功能说明: 保存收集描述信息到summary.init中
# 输入参数: devObj, plInfoCliRet, successNum, partlyNum
# 输出参数: True or False
# **************************************************************************** #  
def _createSummaryInfo(devObj, plInfoCliRet, ipInfoCliRet, successNum, partlyNum):
    logger = devObj.get("logger")
    loginIp = devObj.get("devIp")
    
    #summary.ini文件操作变量
    configDataStr = G_INFORMATION_SPLIT_STR[1:]         
    
    configDataStr += "total=" + str(POOL_NUM) + "\n"
    
    #记录成功数到信息增加
    configDataStr += "success=" + str(successNum) + "\n"
    
    if 0 != partlyNum:
        configDataStr += "partly=" + str(partlyNum) + "\n"

    #记录收集完成时间
    curTime = time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(time.time()))
    configDataStr += "date=" + str(curTime)+"\n"
    configDataStr += "loginIp=" + str(loginIp)
    configDataStr += G_INFORMATION_SPLIT_STR
    if ipInfoCliRet:
        configDataStr += ipInfoCliRet+"\n"
    configDataStr += plInfoCliRet
    
    configDataStr += G_INFORMATION_SPLIT_STR
    configDataStr += TAIL_MSG
    
    #文件写入
    filePath = _getFilePath("summary")[:-3]+"ini"
    filePath = filePath.replace("POOL_","")
    if not common.writeFile(filePath, configDataStr, False, logger):
        return False
    return True   

# **************************************************************************** #
# 函数名称: _createPoolBaseInfo(plInfoCliRet, poolId)
# 功能说明: 创建POOL_XXX.log信息，写入show storage_pool general信息
# 输入参数: plInfoCliRet, poolId
# 输出参数: True or False
# **************************************************************************** #  
def _createPoolBaseInfo(plInfoCliRet, poolId):
    return common.writeFile(_getFilePath(str(poolId)), plInfoCliRet, True, None)

# **************************************************************************** #
# 函数名称: _getAndSavePoolInfo(devObj, poolId)
# 功能说明: 获取并保存单个POOL描述信息
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False
# **************************************************************************** # 
def _getAndSavePoolInfo(devObj, poolId):
    cliRet = ""
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    
    logger.info("[collect] Execute cmd is:"+"spa showrootckg "+str(poolId))
    cliRet = ssh.execCmdNoLog("spa showrootckg "+str(poolId))
   
    #文件写入
    if not common.writeFile(_getFilePath(poolId), cliRet, True, logger):
        return False
    return True

# **************************************************************************** #
# 函数名称: _getAndSaveCkgInfo(devObj, poolId)
# 功能说明: 获取并保存rootCkg&expandCKg信息
# 输入参数: devObj, poolId
# 输出参数: True or False
# **************************************************************************** # 
def _getAndSaveCkgInfo(devObj, poolId):
    flag = True
    index = 0
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    
    cmd = "spa showrootckg "+str(poolId)+" " 
    while(True):
        logger.info("[collect] Execute cmd is:"+cmd+str(index))
        cliRet = ssh.execCmdNoLog(cmd+str(index))
    
        #文件写入
        if not common.writeFile(_getFilePath(poolId), cliRet, True, logger):
            flag = False
            
        #判断查询ckgInfo信息是否结束
        if(_isCkgInfoQueryFinish(cliRet)):
            break

        index += 50
    return flag
    
# **************************************************************************** #
# 函数名称: _getAndSaveMetaCkgInfo(devObj, poolId)
# 功能说明: 获取并保存metaCKg信息
# 输入参数: devObj, poolId
# 输出参数: True or False
# **************************************************************************** # 
def _getAndSaveMetaCkgInfo(devObj, poolId):
    flag = True
    index = 0
    logger = devObj.get("logger")
    ssh = devObj.get("SSH")
    
    cmd = "spa showckg -p "+str(poolId)+" -z 0 -k 12 -n 100 "
    
    #遍历完三层Tier
    for tierId in range(3):
        startId = 0
        cliRet = ""
        while(True):
            logger.info("[collect] Execute cmd is:"+cmd+"-t "+str(tierId)+" -s "+str(startId))
            cliRet = ssh.execCmdNoLog(cmd+"-t "+str(tierId)+" -s "+str(startId))
            
            #文件写入
            if not common.writeFile(_getFilePath(poolId), cliRet, True, logger):
                flag = False
                
            startId = _getMetaLastCkgId(cliRet)
            if("0xffffffff" == startId):
                break
            else:
                startId = hex(ast.literal_eval(startId)+1)
            
    return flag

# **************************************************************************** #
# 函数名称: _collectLocalCkInfo(devObj, plInfoCliRet, localIdList)
# 功能说明: 收集本端ckInfo信息
# 输入参数: devObj，localIdList（本地归属POOL ID）
# 输出参数: True or False:
# **************************************************************************** # 
def _collectLocalCkInfo(devObj, plInfoCliRet, localIdList): 
    return _collectCkInfo(devObj, plInfoCliRet, localIdList)
    

# **************************************************************************** #
# 函数名称: _collectRemoteCkInfo(devObj, plInfoCliRet, localIdList)
# 功能说明: 收集远端ckInfo信息
# 输入参数: devObj, plInfoCliRet, localIdList（远端归属poolIdList）
# 输出参数: True or False:
# **************************************************************************** # 
def _collectRemoteCkInfo(devObj, plInfoCliRet, localIdList):
    return _collectCkInfo(devObj, plInfoCliRet, localIdList)


# **************************************************************************** #
# 函数名称: _collectCkInfo(devObj, plInfoCliRet, poolIdList)
# 功能说明: 收集ckInfo信息主函数
# 输入参数: devObj, plInfoCliRet, poolIdList
# 输出参数: successNum收集成功的次数,partly部分成功的次数
# **************************************************************************** # 
def _collectCkInfo(devObj, plInfoCliRet, poolIdList):    
    flag = True
    successNum = 0
    partlyNum = 0 
    logger = devObj.get("logger")
    global TAIL_MSG
    
    for poolId in poolIdList:
        countNum = 4
        flag = True 
        #创建文件并写入POOL整体信息
        if  not _createPoolBaseInfo(plInfoCliRet, poolId):
            logger.error("[collect] Create file failed :"+_getFilePath(str(poolId)))
            countNum -= 1
            pass
        
        #写入单个POOL描述信息
        if not _getAndSavePoolInfo(devObj, poolId): 
            logger.error("[collect] Writing spa info failed") 
            countNum -= 1
            pass
        
        #写入rootCkg及expandCkg信息
        if not _getAndSaveCkgInfo(devObj, poolId):
            logger.error("[collect] Writing rootCkg&expandCkg info failed") 
            countNum -= 1
            pass
        
        #写入metaCkg信息
        if not _getAndSaveMetaCkgInfo(devObj, poolId):
            logger.error("[collect] Writing metaCkg info failed") 
            countNum -= 1
            pass
        
        if 0 == countNum:
            TAIL_MSG += "POOL("+str(poolId)+") CK information collected failed\n"
        elif 4 == countNum:
            TAIL_MSG += "POOL("+str(poolId)+") CK information was successfully collected\n"
            successNum +=1    
        else:
            TAIL_MSG += "POOL("+str(poolId)+") part of CK information was successfully collected\n"
            partlyNum +=1
            
    return successNum, partlyNum
               
# **************************************************************************** #
# 函数名称: execute(devObj)
# 功能说明: 收集ckInfo入口
# 输入参数: devObj
# 输出参数: True or Flase
# **************************************************************************** # 
def execute(devObj):
    try:
        flag = True
        #记录最终收集概况
        global TAIL_MSG
        logger = devObj.get("logger")
        loginIp = devObj.get("devIp")
        successNum = 0
        partlyNum = 0
        ctrNum = 0
        plInfoCliRet = ""
        ipInfoCliRet = ""
        
        plInfoCliRet, plInfoListDict = _getPoolListDict(devObj)
        
        logger.info("[collect] Init base information")
        _init(devObj, plInfoListDict)
        
        #必须在超级用户下收集
        logger.info("[collect] Collecting Ck info begin....")
        if not modelManager.isSuperUser(devObj):
            TAIL_MSG = "Login user is not super admin, can't collect CK information"
            logger.error("[collect] Login user is not super admin, can't collect CK information")
            return False
        
        #没有任何可收集的POOL
        if not plInfoListDict:
            TAIL_MSG = "No pool information"
            logger.info("[collect] No pool information")
            return True
        
        ipInfoCliRet, ipInfoListDict = _getIpListDict(devObj)
        
        ctrNum = _getCtrlNum(ipInfoCliRet)
        if 1== ctrNum:
            tempMsg = TAIL_MSG
            TAIL_MSG = "System was working in single controller, all CK information was collected in local controller:\n" + tempMsg
            logger.info("[collect] System was working in single controller, all CK information was collected in local controller")      
        elif 2 != ctrNum:
            TAIL_MSG = "Can't collect CK information because of getting no controller information"
            logger.error("[collect] Can't collect CK information because of getting no controller information")
            return False
           
        logger.info("[collect] Local change to debug mode")
        flag = modelManager.changeCli2Developer(devObj)
        if not flag:
            TAIL_MSG = "Local enter to developer mode failed, can't collect CK information"
            logger.error("[collect] Local enter to debug mode failed, can't collect CK information")
            return False
    
        flag = modelManager.changeDeveloper2Debug(devObj)
        if not flag:
            TAIL_MSG = "Local enter to debug mode failed, can't collect CK information"
            logger.error("[collect] Local enter to debug mode failed, can't collect CK information")
            return False
        
        #双控情况下，偶数在A控执行
        logger.info("[collect] Getting local pool list")
        lclPlIdList = _getCtrlPoolList(loginIp,ipInfoListDict,plInfoListDict,True, ctrNum)
        
        logger.info("[collect] Collecting local CK information")
        successNum, partlyNum = _collectLocalCkInfo(devObj, plInfoCliRet, lclPlIdList)
        
        logger.info("[collect] Getting remote pool list")
        rmtPlIdList = _getCtrlPoolList(loginIp,ipInfoListDict,plInfoListDict,False, ctrNum)
        
        logger.info("[collect] Collecting local pool id list is: "+str(lclPlIdList)+\
                    ", remote pool id list is: "+str(rmtPlIdList))
        
        #判断是否需要切换到对端，单控及没有获取到对端归属POOL均不跳转
        if rmtPlIdList:
            #切换到对端  
            logger.info("[collect] Remote change to debug mode")
            flag = modelManager.connect2HeartBeat(devObj)
            if not flag:
                TAIL_MSG += "Connected to heart beat port failed:\n"
                for plId in rmtPlIdList:
                   TAIL_MSG += "Remote POOL("+str(plId)+") CK information collected failed\n" 
                logger.error("[collect] Remote connect to heart beat port failed")
                return False
            
            flag = modelManager.changeCli2Debug(devObj)
            if not flag:
                TAIL_MSG += "Remote enter to debug mode failed:\n"
                for plId in rmtPlIdList:
                   TAIL_MSG += "Remote POOL("+str(plId)+") CK information collected failed\n" 
                logger.error("[collect] Remote enter to debug mode failed")
                return False
            
            #奇数在B控执行
            logger.info("[collect] Collecting remote CK information")
            successTemp, partlyTemp = _collectRemoteCkInfo(devObj, plInfoCliRet, rmtPlIdList)
            successNum += successTemp
            partlyNum += partlyTemp
            
            #退回到CLI模式
            logger.info("[collect] Collecting CK information end....")
            if not modelManager.stopHeartBeat(devObj):
                return False
        return True      
    
    except BaseException, exception:
        logger.error("[collect] Collecting CK information catch exception:"+str(exception))
        TAIL_MSG += "Collecting CK information catch exception"
        return False
    
    finally:  
        modelManager.changeAnyModel2Cli(devObj)   
        if os.path.exists(BASE_PATH):
            _createSummaryInfo(devObj, plInfoCliRet, ipInfoCliRet, successNum, partlyNum)
  