﻿# -*- coding: UTF-8 -*-

import re
from time import sleep
from toolutilities import sshutils
import modelManager
import os
import time
import zipfile
import codecs
import stat
import shutil

#单次收集硬盘最大个数
CollectDiskInfo_MAXCounter = 20

#获取信息分隔符
G_INFORMATION_SPLIT_STR = "\n********************\n"

#系统支持的硬盘最小个数
G_SYS_DISK_NUMBER_MIN   = 2 

#cli命令回文行数判断
G_CLI_INFO_NORMAL_ROW_MIN = 5

#端口号
PORT_NUM = 22

#是否是HVS C99版本
G_IS_HVS_C99_VERSION = False

#C99版本硬盘表示为DAE还是DKE，默认为DAE，C99会根据实际回显更新，TV200R001不涉及
G_DISK_PREFIX = "DKE"

# **************************************************************************** #
# 函数名称: isConnected2StorageIp
# 功能说明: 判断是否登录的为阵列IP地址
# 输入参数: devObj,loginIp
# 输出参数: True:登录阵列IP，False:登录SVP IP
# **************************************************************************** # 
def isConnected2StorageIp(devObj, loginIp):

    #获取全局信息
    ssh = devObj.get("SSH")
    logger = devObj.get("logger")
    
    storageIpList = modelManager.getIpList(ssh)
    
    for ipInfo in storageIpList:
        if loginIp == ipInfo:
            return True
        
    return False

# **************************************************************************** #
# 函数名称: createSshConected
# 功能说明: 脚本自行创建SSH连接
# 输入参数: devObj，ipAddr
# 输出参数: 返回同阵列创建的SSH连接，使用后需要注销
# **************************************************************************** # 
def createSSHConnection(devObj, ipAddr):
     
    cliCon = None
    lang = devObj.get("lang")
    
    username = devObj.get("username")
    password = devObj.get("password")
    logger = devObj.get("logger")
    context = devObj.get("context")
     
    #限定必须为admin用户才能收集smart信息
    if not isSuperUser(devObj, username):
        return None
    
    try:
        cliCon = sshutils.createSSHConnection(ipAddr,username,password,PORT_NUM,context)
        logger.info("[Smart Info] Begin to connecting!")
        cliCon.connect()
    except:
        logger.error("[Smart Info] Collecting failed, the reason is: Create ssh ["+ipAddr+"] connection catch except!")
        if lang == "zh":
            devObj["py_detail"] = u"创建SSH["+ipAddr+u"]连接失败"
        else:
            devObj["py_detail"] = "create ssh ["+ipAddr+"] connection failed"  
        return None

    logger.info("[Smart Info] Create ssh ["+ipAddr+"] connection success!")        
    return cliCon
       
# **************************************************************************** #
# 函数名称: isSysNormal
# 功能说明: 判断系统是否正常
# 输入参数: devObj,cliRet
# 输出参数: True:系统正常，False:异常；show system general回显信息
# **************************************************************************** # 
def isSysNormal(devObj):
    Health = ""
    Running = ""
    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    
    #查看系统状态是否正常(设置5秒超时：系统未正常开工时CLI命令也能快速返回)
    cliRet = ssh.execCmdWithTimout("show system general", 5)
    
    if None != cliRet and "" != cliRet:
        list = cliRet.splitlines()
        for field in list:
            field = field.replace(" ","")
            if field.startswith("HealthStatus:"):
                Health = field.replace("HealthStatus:","")
                continue
            if  field.startswith("RunningStatus:"):
                Running = field.replace("RunningStatus:","")
                break
        if "Normal"==Health and "Normal" == Running:
            return (True,cliRet)
    
    
    logger.error("[Smart Info] Collecting failed, the reason is: system abnormal!")
    if lang == "zh":
        devObj["py_detail"] = u"系统状态不正常，不能收集Smart信息。"
    else:
        devObj["py_detail"] = "The status of system is abnormal, and collecting smart information failed."
    
    return (False,cliRet)

# **************************************************************************** #
# 函数名称: isAdminUser
# 功能说明: 判断用户名是否为超级管理员用户
# 输入参数: devObj,userName
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUser(devObj, userName):
    
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    if userName != "admin":
        logger.error("[Smart Info] Collecting failed, the reason is: login name ["+userName+"] is low-privileged!")    
        #设置错误信息
        if lang == "zh":
            devObj["py_detail"] = u"当前用户权限过低，不能够收集硬盘Smart信息。请使用admin用户收集。"
        else:
            devObj["py_detail"] = "The user logged in with is low-privileged and can not support to collect disk's smart information. " \
                    + "Please login with admin, then collect the disk's smart information."
        return False
    return True

# **************************************************************************** #
# 函数名称: setProductVersion
# 功能说明: 设置当前阵列是否为HVS C99版本，保存到全局变量G_IS_HVS_C99_VERSION中
# 输入参数: cliRet（show system general回显）
# 输出参数: True:HVS C99版本，False:否
# **************************************************************************** # 
def setProductVersion(cliRet):

    ProductVersion = ""
    global G_IS_HVS_C99_VERSION 
    
    if None == cliRet or "" == cliRet:
        return False
    
    list = cliRet.splitlines()
    for field in list:
        field = field.replace(" ","")
        if field.startswith("ProductVersion:"):
            ProductVersion = field.replace("ProductVersion:","")
            break
          
    if "V100R001C99" == ProductVersion:
        G_IS_HVS_C99_VERSION = True
    else:
        G_IS_HVS_C99_VERSION = False

# **************************************************************************** #
# 函数名称: diskSatisfyMinNum
# 功能说明: 判断盘数是否满足最小盘数要求
# 输入参数: devObj,diskCounter
# 输出参数: True:满足，False:不满足
# **************************************************************************** # 
def diskSatisfyMinNum(devObj, diskCounter, fileBaseDir):
    
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    
    if diskCounter < G_SYS_DISK_NUMBER_MIN:
        logger.error("[Smart Info] Collecting failed, the reason is: disk num "+str(diskCounter)+" is less than "+str(G_SYS_DISK_NUMBER_MIN)) 
        if lang == "zh":
            devObj["py_detail"] = u"通过硬盘物理信息获取硬盘个数失败。"
        else:
            devObj["py_detail"] = "Getting disk number by disk physic information failed." 
        
        #删除总收集文件夹
        try:
            shutil.rmtree(fileBaseDir, True)
        except:
            pass
        finally:
            return False
   
    #设置返回信息
    return True  
    
# **************************************************************************** #
# 函数名称: getLogicInfo
# 功能说明: 获取POOL对应的硬盘列表等信息
# 输入参数: ssh
# 输出参数: CLI命令回显信息
# **************************************************************************** # 
def getLogicInfo(ssh):
    i = 0
    cliRet1 = ssh.execCmd("show storage_pool general")
    cliRet = cliRet1
    list = cliRet1.splitlines()
    for field1 in list:
        i = i+1
        if i > 4:
            list1 = field1.split()
            if len(list1)<1:
                continue
            
            #Find to the end
            if("admin:/>" == list1[0]):
                break
            
            if("ID" == list1[0]) or ("--" == list1[0]):
                continue

            #Find right lines and show each POOL_ID
            if(len(list1)>=6):
                cliRet2 = ssh.execCmd("show storage_pool tier " + str(list1[0]))
                cliRet = cliRet + cliRet2
    return cliRet


# **************************************************************************** #
# 函数名称: writeDiskDetailInfo(devObj, sysBasicInfoFile)
# 功能说明: 写入具体的硬盘信息
# 输入参数: devObj, diskPhysicInfo, sysBasicInfoFile
# 输出参数: True or False
# **************************************************************************** #        
def writeDiskDetailInfo(devObj, diskPhysicInfo, sysBasicInfoFile):
    
    flag = True
    ssh = devObj.get("SSH")
    logger = devObj.get("logger")
    iRet = getDiskCounterAndList(diskPhysicInfo)
    diskLocationList = iRet[1]
    i = 0
    detailDiskInfo = ""
    
    for diskId in diskLocationList:
        i +=1
        detailDiskInfo += ssh.execCmd("show disk general "+str(diskId))
        
        if CollectDiskInfo_MAXCounter == i:
            if not commonFileWrite(sysBasicInfoFile, detailDiskInfo, True, logger):
                flag = False
            detailDiskInfo = ""
            i = 0
    
    if 0 != i:
        if not commonFileWrite(sysBasicInfoFile, detailDiskInfo, True, logger):
            flag = False
            
    commonFileWrite(sysBasicInfoFile, G_INFORMATION_SPLIT_STR, True, logger)    
    return flag

# **************************************************************************** #  
# 函数名称: createBasicFile
# 功能说明: 创建子文件夹并保存信息
# 输入参数: devObj，sysInfoStr
# 输出参数: fileBaseDir（创建文件路径），diskPhysicInfo（硬盘物理信息回文）
# **************************************************************************** # 
def createBasicFile(devObj, sysInfoStr):
    flag = True
    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    ipAddr = devObj.get("devIp")
    
    fileBaseDir = devObj.get("collectRetDir")+os.path.sep+"temp"+os.path.sep
    subFileDir = fileBaseDir + "disksmartinfo"

    logger.info("[Smart Info] Create fileBaseDir is :" + fileBaseDir)
    if not os.path.exists(subFileDir):
        os.makedirs(subFileDir)
    devObj['SmartInfoDir'] = subFileDir

    #保存系统基本信息
    basicInfoStr = ""
    basicInfoStr += "smart analyzelog device ip:" + ipAddr + G_INFORMATION_SPLIT_STR
    basicInfoStr += sysInfoStr + G_INFORMATION_SPLIT_STR
    basicInfoStr += getLogicInfo(ssh) + G_INFORMATION_SPLIT_STR
    diskPhysicInfo = ssh.execCmd("show disk general")
    basicInfoStr += diskPhysicInfo + G_INFORMATION_SPLIT_STR

    sysBasicInfoFile = subFileDir + os.sep + "basicinfo.txt"
    if not commonFileWrite(sysBasicInfoFile, basicInfoStr, False, logger):
        flag = False
    #追加具体硬盘信息
    if not writeDiskDetailInfo(devObj, diskPhysicInfo, sysBasicInfoFile):
        flag = False
    if not flag:    
        #删除总收集文件夹
        try:
            shutil.rmtree(fileBaseDir, True)
        except:
            pass
        finally:
            logger.error("[Smart Info] Collecting failed, the reason is: Create basic file failed!") 
            if lang == "zh":
                devObj["py_detail"] = u"创建basicinfo.txt文件失败。"
            else:
                devObj["py_detail"] = "Creating basicinfo.txt file failed." 
            return (None, None)
    
    return (fileBaseDir, diskPhysicInfo)

# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
# **************************************************************************** #
# 函数名称: getDiskCounter
# 功能说明: 通过硬盘物理信息计算硬盘个数和Location的列表
# 输入参数: diskPhysicInfo(硬盘物理信息CLI回显)
# 输出参数: diskCounter（硬盘个数），diskLocationList（硬盘对应位置列表）
# **************************************************************************** # 
def getDiskCounterAndList(diskPhysicInfo):
    
    diskCounter = 0
    diskLocationList = []
    global G_DISK_PREFIX
    lineList = diskPhysicInfo.splitlines()
    i = 0
    for line in lineList: 
        i = i+1
        field = line.split()
        if -1!=line.find("successfully"):
            diskCounter=0
            break
        
        if i > 4:     
            if len(field) < 6:
                continue
        
            if -1!=line.find("show disk general") or -1!=line.find("ID") or -1!=line.find("----") or -1!=line.find("admin:"):
                continue
            
            diskCounter+=1
            location = field[0]
            diskLocationList.append(location) 
    
    #HVS C99版本获取硬盘表示的前缀，保存到全局变量中
    if G_IS_HVS_C99_VERSION:
        for diskName in diskLocationList:
            G_DISK_PREFIX = diskName[0:3]
            break
            
    return (diskCounter, diskLocationList)
# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End

# *************************************************************#
# 函数名称: commonFileWrite
# 功能说明: 文件写入操作（通用函数）
# 其 他   :  无
# *************************************************************#  
def commonFileWrite(fileName, writeInfoStr, ifAddFlag, logger):

    flag = True
    try:
        #判断是否为追加内容
        if True == ifAddFlag:
            localFile = codecs.open(fileName, "a", "utf-8")
        else:
            localFile = codecs.open(fileName, "w", "utf-8")
            
        localFile.write(writeInfoStr)
        localFile.close()
    except Exception,e:
        logger.error("[Smart Info] Collecting failed, the reason is: Writing to file error:" + str(e))
        flag = False
    
    return flag



# **************************************************************************** #
# 函数名称: convertUserId2DisplayId
# 功能说明: 仅供HVS C99版本调用，用户框号转换为CLI命令下面显示的框号（如258对应转换为DAE002，C99存在DKE版本）
# 输入参数: strUserId（minisystem下面获取的用户框号userid，传入参数为字符串数字）
# 输出参数: 对于HVS C99版本，转为为用户可见的硬盘框号
# **************************************************************************** # 
def convertUserId2DisplayId(strUserId):
    
    iUserId = int(strUserId)
    displayId = G_DISK_PREFIX
    displayId += str((iUserId>>24)&0xFF)
    
    # ，优化：槽位号转换为DAEabc.X的函数某些场景有问题，modified 20131014 Begin
    #某些场景转化为DAEabc.X时，位置b处ID转换需要特殊处理
    locationB = str((iUserId>>16)&0xFF)
    if "20" == locationB:
        displayId += str(2)
    elif "21" == locationB:
        displayId += str(3)
    else:
        displayId += locationB 
    # ，优化：槽位号转换为DAEabc.X的函数某些场景有问题，modified 20131014 End
	
    displayId += str(iUserId&0xFF)
    return displayId

# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
# **************************************************************************** #
# 函数名称: getCollectDiskList_Minisystem
# 功能说明: 获取需要获取Smart信息的硬盘列表（调用者保证：调用此接口前，ssh在minisystem模式下）
# 输入参数: devObj，ssh
# 输出参数: 无
# **************************************************************************** # 
def getCollectDiskList_Minisystem(devObj, ssh):
    
    diskInfoDict = {"location":"", "name":""}
    userId = ""
    slotId = ""
    location = ""    
    needCollectDiskList = []
    diskLocationList = devObj["diskLocationList"]
    logger = devObj.get("logger")
    
    #查询所有硬盘信息，获取所有SAS和SATA盘的盘符列表
    temp = ssh.execCmd("showdiskinfo")
    if re.search("===========", temp, re.IGNORECASE):
        lineList = temp.splitlines()
        for line in lineList:
            if line.find("slotid") >= 0:
                #查询硬盘的location
                field = line.split(",")
                for temp in field:
                    if temp.find("userid") >= 0:
                        userId = temp[temp.index(":")+1:]
                        if G_IS_HVS_C99_VERSION:
                            userId = convertUserId2DisplayId(userId)
                            
                    elif temp.find("slotid") >= 0:
                        slotId = temp[temp.index(":")+1:temp.index(")")]
                if userId != "" and slotId != "":
                    location = userId + "." + slotId
                    #如果location不在硬盘的物理信息中，则认为是系统盘，不需要收集
                    if location not in diskLocationList:
                        location = ""
        
                #继续查询下一行
                continue
            
            if location != "" and line.find("Name") >= 0:
                #查询盘符信息
                field = line.split(",")
                for temp in field:
                    if temp.startswith("Name"):
                        #盘符可能为空，也需要记录
                        diskName = temp.replace("Name ", "")
                        #获取正常的硬盘盘符(注：sda为可能为系统盘的盘符，不需要收集)
                        if diskName != "sda":
                            diskInfoDict["location"] = location
                            diskInfoDict["name"] = diskName
                            
                            #记录硬盘信息，初始化变量
                            needCollectDiskList.append(diskInfoDict)
                            diskInfoDict = {"location":"", "name":""}
                            userId = ""
                            slotId = ""
                            location = ""
                            diskName = ""      
            else:
                continue
    
    return needCollectDiskList
# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End

# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
# **************************************************************************** #
# 函数名称: setCollectRetMsg
# 功能说明: 设置部分收集成功的相关状态
# 输入参数: devObj，collectFailDiskStr
# 输出参数: 无
# **************************************************************************** # 
def setCollectRetMsg(devObj, collectFailDiskStr):
    
    lang = devObj.get("lang")
    logger = devObj.get("logger")

    #部分收集成功
    if collectFailDiskStr != "":
        logger.error("[Smart Info] Collecting partly successful, end!")
        #设置返回信息
        if lang == "zh":
            errMsg = u"硬盘（" + collectFailDiskStr + u"）收集Smart失败，请按照如下建议处理：\
            \n（1）收集失败的原因可能是：硬盘单链路、响应慢、故障或者休眠。请先按照标准处理方法修复，修复完成后重新收集一次；\
            \n（2）如果仍然收集失败，请尝试手动执行命令收集单块硬盘；\
            \n（3）如果有任何疑问，请联系技术支持工程师协助处理。"
        else:
            errMsg = "Collecting smart information of disks (" + collectFailDiskStr + " ) failed, please handle them according to the suggestion:\
            \n(1)The reason of failure may be: the disk is single path, the response of disk is slow, the disk is fault or in spin down status. Please repair it according to the standard treatment, then collect again;\
            \n(2)If collecting failed again, please try to collect the smart information manually;\
            \n(3)If you have any questions, please contact technical support engineers for further handling."
            
        #设置部分收集成功
        devObj["collectAllInfo"] = False
        devObj["py_detail"] = errMsg
    else:
        logger.info("[Smart Info] Collecting successful, end!")
        devObj["collectAllInfo"] = True
# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End

# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
# **************************************************************************** #
# 函数名称: minisystemModelCollect
# 功能说明: 查询当前控制器下的硬盘Smart信息（minisystem模式下）
# 输入参数: devObj,ssh
# 输出参数: 无
# **************************************************************************** # 
def minisystemModelCollect(devObj, ssh):
    
    #获取全局信息
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    smartInfoSaveDir = devObj['SmartInfoDir']
    
    #收集成功的Smart信息
    diskSmartInfoStr = ""
    #收集失败的Smart信息
    collectFailSmartINfoStr = ""
    
    #计数器清零
    saveCounter = 0
    tempCounter = 0
    successCounter = 0
    
    #通过showdiskinfo命令获取硬盘盘符列表
    logger.info("[Smart Info] Begin to getting disk info in minisystem!")
    needCollectDiskList = getCollectDiskList_Minisystem(devObj,ssh)
    logger.info("[Smart Info] Need to collect disk number:" + str(len(needCollectDiskList)))
    
    #summary.ini文件操作变量
    configFileName = smartInfoSaveDir + os.sep + "summary.ini"
    configDataStr = ""
    indexInfoStr = ""
    physicDiskCounter = devObj["TotalDiskNumber"]
    configDataStr += "total=" + str(physicDiskCounter) + "\n"

    
    diskName = ""
    diskLocation = ""
    
    #收集失败的硬盘列表
    collectFailDiskStr = ""

    #收集所有盘的Smart信息
    for diskInfoTmp in needCollectDiskList:
        #获取硬盘信息
        diskName = diskInfoTmp["name"]
        diskLocation = diskInfoTmp["location"]

        #盘符为空，则直接将此硬盘设定为收集失败
        if diskName == "":
            #收集失败，需要将location写入失败列表
            if collectFailDiskStr == "":
                collectFailDiskStr = diskLocation
            else:
                collectFailDiskStr += "," + diskLocation
            
            diskInfoStrTmp = "Disk " + diskLocation + " has no name, collecting smart information failed!"
            collectFailSmartINfoStr += diskInfoStrTmp + G_INFORMATION_SPLIT_STR
            continue

        #盘符不为空，继续收集
        diskInfoStrTmp = ssh.execCmdNoLog("disktool -f a /dev/" + diskName)
        #判断收集是否成功。因硬盘健康分析需使用Device Model做分隔，所以必须判断此字段存在，否则可能导致分析数据错位。
        if re.search("Serial Number", diskInfoStrTmp, re.IGNORECASE) and "Device Model" in diskInfoStrTmp:
            tempCounter += 1
            successCounter += 1
            diskSmartInfoStr += diskInfoStrTmp + G_INFORMATION_SPLIT_STR
        else:
            #收集失败，需要将location写入失败列表
            if collectFailDiskStr == "":
                collectFailDiskStr = diskLocation
            else:
                collectFailDiskStr += "," + diskLocation
            collectFailSmartINfoStr += diskInfoStrTmp + G_INFORMATION_SPLIT_STR

        #每收集20块盘，保存一份文件
        if tempCounter >= CollectDiskInfo_MAXCounter:
            
            #新建文件，保存smart信息
            fileName = smartInfoSaveDir + os.sep + "smart" + str(saveCounter) + ".txt"
            flag = commonFileWrite(fileName, diskSmartInfoStr, False, logger)
            if not flag:
                return (False, collectFailDiskStr)
            
            logger.info("[Smart Info] Collect 20 disks and write the result to the new file:" + fileName)
            
            #增加index描述
            indexInfoStr += "smart" + str(saveCounter) + "=" + str(successCounter-tempCounter) + "-" + str(successCounter-1) + "\n"
            
            #计数加一，变量清空
            saveCounter += 1
            diskSmartInfoStr = ""
            tempCounter = 0
            
    #不满20块盘，则最终收集写入文件
    if tempCounter > 0:
        #新建文件，保存smart信息
        fileName = smartInfoSaveDir + os.sep + "smart" + str(saveCounter) + ".txt"
        flag = commonFileWrite(fileName, diskSmartInfoStr, False, logger)
        if not flag:
            return (False, collectFailDiskStr)
        
        logger.info("[Smart Info] Collect remaining disks and write the result to the new file:" + fileName)
        
        #增加index描述
        indexInfoStr += "smart" + str(saveCounter) + "=" + str(successCounter-tempCounter) + "-" + str(successCounter-1) + "\n"

    #是否收集失败
    if successCounter == 0:
        logger.error("The number of disks successfully collected is 0, Error!")
        return (False, collectFailDiskStr)   

    #记录成功数到信息增加
    configDataStr += "success=" + str(successCounter) + "\n"

    #记录收集完成时间
    curTime = time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(time.time()))
    configDataStr += "date=" + str(curTime) + "\n"
        
    #将所有的信息写入summary.ini
    configDataStr += indexInfoStr
    flag = commonFileWrite(configFileName, configDataStr, False, logger)
    if not flag:
        return (False, collectFailDiskStr)
    
    #如果存在收集失败的盘，写入failsmart.txt文件
    if collectFailSmartINfoStr != "":
        fileName = smartInfoSaveDir + os.sep + "failsmart.txt"
        flag = commonFileWrite(fileName, collectFailSmartINfoStr, False, logger)
        if not flag:
            return (False, collectFailDiskStr)

    #正确返回
    return (True, collectFailDiskStr)
# （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End

# **************************************************************************** #
# 函数名称: storageIpCollect
# 功能说明: 工具登录为阵列Ip时收集，SSH连接可以直接从devObj中获取
# 输入参数: devObj,fileBaseDir
# 输出参数: 无
# **************************************************************************** #
def storageIpCollect(devObj, fileBaseDir):
    
    ssh = devObj.get("SSH")
    logger = devObj.get("logger")
    lang = devObj.get("lang")
    
    #切换到developer模式下
    logger.info("[Smart Info] Begin to changing to developer model!")
    flag = modelManager.changeCli2Developer(devObj)
    
    #切换到developer模式失败，直接返回
    if not flag:
        logger.error("[Smart Info] Collecting failed, the reason is: Login to developer model failed!")
        if lang == "zh":
            devObj["py_detail"] = u"进入developer模式失败，smart信息收集失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
        else:
            devObj["py_detail"] = "Login to developer model failed, can not collect disk smart information.The reason of failure may be:\n" \
                    + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."
        return False
    
    #切换到minisystem模式
    logger.info("[Smart Info] Begin to changing to minisystem model!")
    flag = modelManager.changeDeveloper2Minisystem(ssh)
        
    if not flag:
        logger.error("[Smart Info] Collecting failed, the reason is: Login to minisystem model failed!")
        if lang == "zh":
            devObj["py_detail"] = u"进入minisystem模式失败，smart信息收集失败。失败的原因可能为：\n"\
                                   + u"（1）工具连接SVP IP的场景不支持收集smart信息。\n（2）设备异常，无法进入minisystem模式。"
        else:
            devObj["py_detail"] = "Login to minisystem model failed, can not collect disk smart information.The reason of failure may be:\n" \
                     + "(1) Used tool connecting to SVP IP.\n(2) The system is abnormal."        
        return False
    
    #Minisystem下开始收集
    logger.info("[Smart Info] Begin to collecting smart info!")
    # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
    iRet = minisystemModelCollect(devObj,ssh)
    
    if not iRet[0]:
        logger.error("[Smart Info] Collecting failed, the reason is: execute collecting failed!")
        #设置返回信息
        if lang == "zh":
            devObj["py_detail"] = u"在minisystem模式下收集硬盘Smart信息失败。"
        else:
            devObj["py_detail"] = "Collecting disk smart information failed by minisystem model!"  
        return False
    
    setCollectRetMsg(devObj,iRet[1])
    # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End
    
    #切回CLI模式
    logger.info("[Smart Info] Collecting smart info (storage) finished, back to cli model!")
    modelManager.changeAnyModel2Cli(devObj)
    return True

# **************************************************************************** #
# 函数名称: svpIpCollect
# 功能说明: 工具登录为阵列Ip时收集smart信息，ssh连接需要脚本自行创建，使用后必须得及时销毁
# 输入参数: devObj获取打印对象,
# 输出参数: 无
# **************************************************************************** #
def svpIpCollect(devObj, fileBaseDir):
    
    #获取阵列IP地址列表
    logger = devObj.get("logger")
    lang = devObj.get("logger")
    SSH = devObj.get("SSH")
    isConnected = False
    ipList = []
    masterIp = modelManager.getMasterIp(SSH)
    ipListAll =  modelManager.getIpList(SSH)
    
    if "--"!= masterIp and ""!=masterIp:
        ipList.append(masterIp)
    
    for ipInfos in ipListAll:
        if masterIp!=ipInfos:
            ipList.append(ipInfos)
 
    #判断是否获取到IP地址
    if 0 == len(ipList):
        if lang == "zh":
            devObj["py_detail"] = u"没有获取到有效的阵列Ip地址信息"
        else:
            devObj["py_detail"] = "Get of Invalid ip address"  
        return False  
    
    #脚本自行创建SSH连接，使用后需要释放
    for ipInfo in ipList:
        logger.info("[Smart Info] Begin to creating ["+ipInfo+"] ssh connection!")
        ssh = createSSHConnection(devObj, ipInfo)
        
        #选择其他IP创建，连接创建成功后，后面的切换即收集必须成功，否则直接退出
        if None == ssh:
            continue  
        
        isConnected = True
        
        #切换到developer模式下
        logger.info("[Smart Info] Begin to changing to developer model!")
        flag = modelManager.changeCli2Developer(devObj)
    
        #切换到developer模式失败，直接返回
        if not flag:
            logger.error("[Smart Info] Collecting failed, the reason is: Login to developer model failed!")
            if lang == "zh":
                devObj["py_detail"] = u"进入developer模式失败，smart信息收集失败。失败的原因可能为：\n" \
                        + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
            else:
                devObj["py_detail"] = "Login to developer model failed, can not collect disk smart information.The reason of failure may be:\n" \
                        + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."
    
            ssh.close() 
            return False
    
        #切换到minisystem模式
        logger.info("[Smart Info] Begin to changing to minisystem model!")
        flag = modelManager.changeDeveloper2Minisystem(ssh)
        
        if not flag:
            logger.error("[Smart Info] Collecting failed, the reason is: Login to minisystem model failed!!")
            if lang == "zh":
                devObj["py_detail"] = u"进入minisystem模式失败，smart信息收集失败。失败的原因可能为：\n"\
                                       + u"（1）设备异常，无法进入minisystem模式。\n（2）当前版本不支持进入minisystem模式。"
            else:
                devObj["py_detail"] = "Login to minisystem model failed, can not collect disk smart information.The reason of failure may be:\n" \
                         + "(1) The system is abnormal.\n(2) Current version not support minisystem mode."  
            ssh.close()          
            return False
    
        #minisystem模式下收集
        logger.info("[Smart Info] Begin to collecting smart info!")
        iRet = minisystemModelCollect(devObj,ssh)
        
        # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
        if not iRet[0]:
            logger.error("[Smart Info] Collecting failed, the reason is: execute collecting failed!")
            ssh.close()
            #设置返回信息
            if lang == "zh":
                devObj["py_detail"] = u"在minisystem模式下收集硬盘Smart信息失败。"
            else:
                devObj["py_detail"] = "Collecting disk smart information failed by minisystem model!"  
            return False
        
        setCollectRetMsg(devObj,iRet[1])
        # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End
        
        #切回CLI模式
        logger.info("[Smart Info] Collecting smart info (SVP) finished, back to cli model!")
        modelManager.changeAnyModel2Cli(devObj)
        ssh.close()
        #任意一个IP收集成功，则跳出循环
        break
    
    #没有任何的连接，直接退出
    if not isConnected:
        return False
        
    return True

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 脚本执行入口
# 输入参数: ssh
# 输出参数: 无
# **************************************************************************** # 
def execute(devObj):
    
    flag =True
    isHvsC99 = False
    ssh = devObj.get("SSH")
    lang = devObj.get("lang")
    logger = devObj.get("logger")
    userName = devObj.get("username")
    loginIp = devObj.get("devIp")
    
    #初始化smart收集状态标志，在execute函数返回值为True的前提下，设置为True为成功，False为部分收集成功
    devObj["collectAllInfo"] = True

    logger.info("[Smart Info] Begin to checking login name!")
    if not isSuperUser(devObj, userName):
        return False
    #系统是否正常
    logger.info("[Smart Info] Begin to checking system status!")
    iRet = isSysNormal(devObj)
    sysStats = iRet[0]
    sysInfoStr = iRet[1]
    if not sysStats:
        return False
    
    #创建子文件夹并保存信息
    logger.info("[Smart Info] Begin to creating system basic information file!")
    iRet = createBasicFile(devObj,sysInfoStr)
    fileBaseDir = iRet[0]
    diskPhysicInfo = iRet[1]
    if None == fileBaseDir:
        return False
    
    #设置是否为HVS C99版本的G_IS_HVS_C99_VERSION全局变量
    setProductVersion(sysInfoStr)
    
    # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 Begin
    #通过硬盘物理信息计算硬盘数
    logger.info("[Smart Info] Begin to getting disk list info!")
    iRet = getDiskCounterAndList(diskPhysicInfo)
    diskCounter = iRet[0]
    #保存供后续使用
    devObj["diskLocationList"] = iRet[1]
    # （同步老产品问题单） 收集失败打印硬盘位置信息  modified 20131014 End
    
    logger.info("[Smart Info] Getting disk number by disk physic information:" + str(diskCounter))
    if not diskSatisfyMinNum(devObj, diskCounter, fileBaseDir):
        return False
    
    #记录硬盘总个数
    devObj["TotalDiskNumber"] = diskCounter

    #S5000TV200R001产品直接收集
    if not G_IS_HVS_C99_VERSION:
        return storageIpCollect(devObj, fileBaseDir)
    #判断登录的IP地址是否为阵列IP 
    logger.info("[Smart Info] Begin to judging whether connecting to storage directly!")
    flag = isConnected2StorageIp(devObj, loginIp)
    if flag:
        logger.info("[Smart Info] Begin to collecting smart information (Storage)!")
        return storageIpCollect(devObj, fileBaseDir)
    
    else:
        logger.info("[Smart Info] Begin to collecting smart information (SVP)!")
        return svpIpCollect(devObj, fileBaseDir) 
