# -*- coding: UTF-8 -*-
import os
import stat
import gzip
import datetime
import traceback
import sys
from common.util import device

DISKID_START_FLAG = "Smart of Disk["
SMART_COLLECT_FAIED_FLAG = "Failed to collect"
OS_OPEN_MODES = stat.S_IWUSR | stat.S_IRUSR  # 拥有者具有读写权限

def format_traceback():
    '''
    @summary: 格式化输出错误堆栈信息
    '''
    new_extracted_list = []
    etype, value, tb = sys.exc_info()
    extracted_list = traceback.extract_tb(tb)
    for item in extracted_list:
        filename, lineno, name, line = item
        filename = os.path.basename(filename)
        new_extracted_list.append((filename, lineno, name, line))
        
    return "".join(traceback.format_list(new_extracted_list) + traceback.format_exception_only(etype, value))

def logException(log):
    '''
    @summary: 日志输出错误堆栈信息
    '''
    log.info(format_traceback())
    
def getDiskMapInfo(gzFilePath, diskMapInfo, log, diskSNList, devObj):
    '''
    @summary: 读取gz压缩包中文件内容，将信息记入字典。
        key为硬盘id，值为元组。元组第0个元素表示是否收集成功，第1个元素表示该硬盘对应的收集到的信息。
                      如果该硬盘在某个引擎上的信息收集成功，如果对应硬盘在该引擎上其他节点收集失败，仍然认为该硬盘信息收集成功（单链路场景）
    @param gzFilePath: gz文件
    @param diskMapInfo: key为硬盘id，值为元组。元组第0个元素表示是否收集成功，第1个元素表示该硬盘对应的收集到的信息。
    '''
    needUseSnTypeList = ["5300 V3", "5500 V3"]
    needUseSnVersion = ["V300R002C10"] 
    needUseSnType2800 = "2800 V3"
    needUseSnVersion2800 = ["V300R001C00", "V300R001C00SPC100", "V300R001C00SPC200"]
    diskSmartInfoFile = None
    try:
        diskSmartInfoFile = gzip.open(gzFilePath, 'rb')
        oneDiskInfo = ""
        while True:
            line = diskSmartInfoFile.readline()
            
            if getDiskID(line, log) or isEnd(line):
                diskID = getDiskID(oneDiskInfo, log)
                
                if diskID and (diskID not in diskMapInfo or diskMapInfo[diskID][0] != True):
                    #对系统盘的盘框一体的设备，会存在ID相同的情况，需要用SN识别, SN不在baseinfo.txt中的不记录
                    deviceType = str(device.getDeviceType(devObj)) #deviceType类型为java类型，需要转换
                    deviceVersion = device.getDeviceVersion(devObj)
                    if (deviceType == needUseSnType2800 and deviceVersion in needUseSnVersion2800) or \
                       (deviceType in needUseSnTypeList and deviceVersion in needUseSnVersion):
                        diskSN = getDiskSnBySmartInfo(oneDiskInfo)
                        if diskSN not in diskSNList:
                            oneDiskInfo = ""
                            oneDiskInfo += line
                            log.info("[parseDisk][getDiskMapInfo][not access system disk_id=%s]" % diskID)
                            continue
                    
                    # 缓存上一块硬盘的信息
                    if isCollectSucc(oneDiskInfo):
                        # 因硬盘健康分析需使用Device Model做分隔，所以必须判断此字段存在，否则可能导致分析数据错位。
                        if "Device Model" in oneDiskInfo:
                            diskMapInfo[diskID] = [True, oneDiskInfo]
                            log.info("[parseDisk][getDiskMapInfo][get disk %s success.]" % diskID)
                        else:
                            diskMapInfo[diskID] = [False, oneDiskInfo]
                            log.info("[parseDisk][getDiskMapInfo][the disk do not have Device Model (%s)!]" % diskID)
                    # 硬盘信息收集失败
                    else:
                        diskMapInfo[diskID] = [False, oneDiskInfo]
                        log.info("[parseDisk][getDiskMapInfo][get disk %s failed!]" % diskID)
                
                # 一块硬盘信息读取结束，开始缓存下一块硬盘的信息
                oneDiskInfo = ""
            
            oneDiskInfo += line
                
            # 读取结束
            if isEnd(line):
                log.info("[parseDisk][getDiskMapInfo][parse %s end.]" % gzFilePath)
                break
    except:
        logException(log)
        raise IOError
    finally:
        if diskSmartInfoFile is not None:
            diskSmartInfoFile.close()
    return diskMapInfo


def isEnd(line):
    """
    @summary: 判断文件读取是否结束
    """
    if not line:
        return True
    
    return False


def getDiskID(line, log):
    """
    @summary: 获取硬盘ID
    """
    if DISKID_START_FLAG not in line:
        return ""
    
    temp = line.split(DISKID_START_FLAG)
    if len(temp) < 2:
        return ""
    
    temp = temp[1]
    diskID = temp.split()[0]
    
    if not isDiskID(diskID, log):
        return ""
    
    return diskID


def isDiskID(diskID, log):
    """
    @summary: 判断是否为硬盘ID
    """
    if not diskID:
        return False
    
    if diskID.startswith("CTE") or diskID.startswith("ENG") or diskID.startswith("DAE") or diskID.startswith("ROOT"):
        return True
    
    log.info("[parseDisk][getDiskMapInfo][unhandled disk_id=%s]" % diskID)
    return False


def isCollectSucc(diskInfo):
    """
    @summary: 判断硬盘smart信息是否收集成功
    """
    if SMART_COLLECT_FAIED_FLAG in diskInfo:
        return False
    
    return True


def handleSmartInfo(inputPath, log, diskSnList, devObj):
    '''
    @summary: 处理smart信息压缩包文件存放目录中所有压缩包文件，获取总体的diskMapInfo集合
    @param inputPath: 收集到的smart信息压缩包文件存放目录
    '''
    diskMapInfo = {}
    gzFiles = os.listdir(inputPath)
    for gzFileName in gzFiles:
        gzFile = os.path.join(inputPath, gzFileName)
        if not gzFileName.endswith(".tgz"):
            log.info("[parseDisk][handleSmartInfo][unhandled gzfile=%s]" % gzFile)
            continue
        log.info("[parseDisk][handleSmartInfo][handled gzFile=%s]" % gzFile)
        getDiskMapInfo(gzFile, diskMapInfo, log, diskSnList, devObj)
    return diskMapInfo

def generageSmartFile(smartFilePath, collectContent, log):
    '''
    @summary: 生成smart*.txt
    @param smartFilePath: smart*.txt文件路径
    @param collectContent: 要写入的内容
    '''
    smartFile = None
    try:
        os_open_flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND # 读写 | 创建并打开
        smartFile = os.fdopen(os.open(smartFilePath, os_open_flags, OS_OPEN_MODES), 'a')
        smartFile.write(collectContent)
        smartFile.write("\n")
        smartFile.flush()
    except:
        logException(log)
        raise IOError
    finally:
        if smartFile is not None:
            smartFile.close()  

def generageSummaryFile(summaryFilePath, summaryInfo, log):
    '''
    @summary: 生成summary.ini
    @param summaryFilePath: summary.ini文件路径
    @param collectContent: 要写入的内容
    '''
    summaryFile = None
    try:
        os_open_flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC  # 写 | 创建并打开 | 截断
        summaryFile = os.fdopen(os.open(summaryFilePath, os_open_flags, OS_OPEN_MODES), 'w')
        summaryFile.write("total=%s" % summaryInfo["total"])
        summaryFile.write("\n")
        summaryFile.write("success=%s" % summaryInfo["success"])
        summaryFile.write("\n")
        currDate = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        summaryFile.write("date=%s" % currDate)
        summaryFile.write("\n")
        summaryFile.flush()
    except:
        logException(log)
        raise IOError
    finally:
        if summaryFile is not None:
            summaryFile.close()

def converSmartInfo(inputPath, outputPath, log, diskIdList, diskSnList, diskDhaInfo, devObj):
    '''
    @summary: 自动根据收集到的smart信息生成summary.ini和smart*.txt
    @param inputPath: smart硬盘信息所在目录
    @param outputPath: 生成summary.ini和smart*.txt所在目录
    '''
    log.info("[parseDisk][converSmartInfo][start]")
    log.info("[parseDisk][converSmartInfo][inputPath=%s]" % inputPath)
    log.info("[parseDisk][converSmartInfo][outputPath=%s]" % outputPath)
    
    successCnt = 0
    failureCnt = 0
    
    #获取总体的diskMapInfo集合
    diskMapInfo = handleSmartInfo(inputPath, log, diskSnList, devObj)
    diskIDs = diskMapInfo.keys()
    diskIDs.sort()
    diskDhaFormat = "\nStart Health Mark\nHealth Mark For DHA : %s\n"
    for diskID in diskIDs:
        collectFlag = diskMapInfo[diskID][0]
        log.info("[parseDisk][converSmartInfo][diskID=%s]" % diskID)
        log.info("[parseDisk][converSmartInfo][collectFlag=%s]" % collectFlag)
        
        #未接入系统的盘不做处理,show disk general能查到的盘为已接入系统的盘
        if diskID not in diskIdList:
            log.info("[parseDisk][converSmartInfo][the disk(%s) is not accessed]" % diskID)
            continue
        
        #只处理成功收集的信息
        if collectFlag != True:
            failureCnt += 1
            continue
        
        #将收集成功的硬盘信息写入文件
        smartFileName = "smart%s.txt" % (successCnt / 20)
        smartFilePath = os.path.join(outputPath, smartFileName)
        collectContent = diskMapInfo[diskID][1] + diskDhaFormat % diskDhaInfo.get(diskID)
        generageSmartFile(smartFilePath, collectContent, log)
        
        successCnt += 1
    
    #生成summary.ini
    summaryInfo = {"total": failureCnt + successCnt, "success": successCnt}
    summaryFileName = "summary.ini"
    summaryFilePath = os.path.join(outputPath, summaryFileName)
    generageSummaryFile(summaryFilePath, summaryInfo, log)
    
    log.info("[parseDisk][converSmartInfo][summaryInfo=%s]" % summaryInfo)
    log.info("[parseDisk][converSmartInfo][end]")
    return summaryInfo

def getDiskSnBySmartInfo(diskInfo):
    '''
    @summary:从SmartInfo中解析出硬盘SN号
    @param diskInfo: 从SmartInfo中得到的单块盘信息
    @return: diskSn
    '''
    diskSn = ""
    SNbeginIndex = diskInfo.find("Serial Number")
    SNEndIndex = diskInfo.find("\n", SNbeginIndex)
    diskSnInfo = diskInfo[SNbeginIndex:SNEndIndex]
    diskSnInfoSplit = diskSnInfo.split(":")
    
    #分割后Sn号存在则返回正确值，不存在则返回空值
    if len(diskSnInfoSplit) == 2:
        diskSn = diskSnInfoSplit[1].strip()
    return diskSn
