#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2020~ All rights reserved.
import shutil
import traceback

# noinspection PyUnresolvedReferences
from com.huawei.ism.tool.devicemanager.utils import FileHostUtils
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException
# noinspection PyUnresolvedReferences
from com.huawei.ism.tool.obase.connection import SftpTransporter

from cbb.business.collect.hostinfo import const
from cbb.business.collect.hostinfo.collect_host_info import (
    collect as collect_host_info,
)
from cbb.business.collect.hostinfo.collect_host_info_util import (
    HostInfoCollectException,
)
from cbb.frame.base.baseUtil import getString
from cbb.frame.base.funcUtils import fakeProgress
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from common.config import config
from common.util import log
from common.util import util
from support_query.modules.host_info import is_support_and_need_collect
from common.sshTools import SSHToolUtils


@fakeProgress(totalSecs=60 * 60, intervalSec=2, tool_name="infoCollect")
def collect(params_dict):
    """主机信息收集入口函数。

    :param params_dict: 上下文信息
    :return:
    """
    result_dict = dict()
    logger = params_dict.get("logger")
    try:
        if not is_support_and_need_collect(params_dict):
            return True, ""

        if contextUtil.is_run_in_svp():
            create_ctrl_connection(params_dict)

        result_dict = collect_host_info(params_dict)
        params_dict["result_dict"] = result_dict
        log.info(
            params_dict, "Host info collect result dict:{}".format(result_dict)
        )

        hostinfo_pkg_path_name = result_dict.get("host_pkg_path")
        local_save_path = util.getLocalInfoPathByType(
            params_dict, config.COLLECT_TYPE_HOST_INFO
        )

        shutil.move(hostinfo_pkg_path_name, local_save_path)

        should_collect_host_num = result_dict.get("should_collect_host_num")
        success_num = len(result_dict["success"])
        logger.info(
            "Should connect host num:{}, collect success num:{}".format(
                should_collect_host_num, success_num
            )
        )
        return True, ""
    except HostInfoCollectException as host_info_exception:
        log.error(
            params_dict,
            "Host info collect exception traceback:{}".format(traceback),
        )
        log.error(
            params_dict,
            "Host info collect exception error msg:{}".format(
                host_info_exception.log_msg
            ),
        )
        params_dict["py_detail"] = host_info_exception.res_msg
        return False, ""
    except (Exception, JException) as e:
        log.error(
            params_dict, "Host info collect generic exception:{}".format(e)
        )
        log.error(params_dict, "traceback:{}".format(traceback.format_exc()))
        params_dict["py_detail"] = getString(
            const.ERR_KEY_COLLECT_HOST_INFO_GENERIC_ERROR
        )
        return False, ""
    finally:
        try:
            params_dict["result_dict"] = result_dict
            FileHostUtils.dealCollectResult(result_dict)
        except (JException, Exception) as e:
            logger.error(
                "handle collect file for scene device "
                "writing exception:{}".format(e)
            )
            logger.error("trace back:{}".format(traceback.format_exc()))
        restore_connection(params_dict)


def create_ctrl_connection(params_dict):
    """
    遍历控制器IP，直到成功创建到控制器的连接
    :param params_dict:
    :return:
    """
    cli = params_dict.get("SSH")
    lang = params_dict.get("lang")
    logger = params_dict.get("logger")

    flag, controller_ip_list = cliUtil.get_controller_ip_list(cli, lang)
    logger.info("controller list {}".format(controller_ip_list))
    for controller_ip in controller_ip_list:
        if create_one_ctrl_connection(params_dict, controller_ip):
            return
    raise HostInfoCollectException(
            -1,
            "Error to connected to controller",
            getString(const.ERR_KEY_QUERY_ALL_ETH_PORTS_FAILED),
        )


def create_one_ctrl_connection(params_dict, controller_ip):
    """
    指定控制器IP，创建CLI和SFTP连接
    :param params_dict:
    :param controller_ip:
    :return:
    """
    ctrl_cli = None
    logger = params_dict.get("logger")
    try:
        logger.info("current controller ip {}".format(controller_ip))
        ssh_utils = SSHToolUtils(params_dict, logger)
        flag, new_cli = ssh_utils.newSshUsingCrntUsrCredential(controller_ip)
        sftp = SftpTransporter(new_cli)
        if flag and sftp:
            logger.info("connected controller ip {}".format(controller_ip))
            backup_connection(params_dict, new_cli, sftp)
            return True
        return False
    except (Exception, JException) as e:
        logger.error("create connection error.{}".format(e))
        if ctrl_cli:
            ctrl_cli.close()
        return False


def backup_connection(params_dict, cli, sftp):
    params_dict["ORIGIN_SSH"] = params_dict.get("SSH")
    params_dict["SSH"] = cli
    params_dict["ORIGIN_SFTP"] = params_dict.get("SFTP")
    params_dict["SFTP"] = sftp


def restore_connection(params_dict):
    if params_dict.get("ORIGIN_SSH") and params_dict.get("SSH"):
        params_dict["SSH"].close()
        params_dict["SSH"] = params_dict["ORIGIN_SSH"]
    if params_dict.get("ORIGIN_SFTP") and params_dict.get("SFTP"):
        params_dict["SFTP"].close()
        params_dict["SFTP"] = params_dict["ORIGIN_SFTP"]
