# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
import re
import time
import traceback

from config import config
from util import util
from util import log
from cbb.frame.util.tar_util import decompress_tar_all_file, check_tar_special_file

REG = '[\\w\\\\/]+\\d+(\\.log)?\\.tar\\.bz2'


def tar_contains_partial_flag_by_enclosure(dev_obj, local_dir, enclosure_file):
    tar_file_path = os.path.join(local_dir, enclosure_file)
    decompress_path = os.path.join(local_dir, enclosure_file).replace(".tgz", "")
    try:
        decompress_tar_all_file(tar_file_path, decompress_path)
        for inner_tar_file in os.listdir(decompress_path):
            if tar_enclosure_contains_partial_flag(dev_obj, decompress_path, inner_tar_file, tar_file_path):
                return True
        log.info(dev_obj, "{} does not need to collect enclosure information.".format(enclosure_file))
    except Exception as e:
        log.error(dev_obj, "failed open tar ball DAE files. {} {}".format(e, str(traceback.format_exc())))
    return False


def tar_enclosure_contains_partial_flag(dev_obj, decompress_path, inner_tar_file, tar_file_path):
    if "DAE" in inner_tar_file:
        log.info(dev_obj, "old DAE tar...")
        return old_tar_contains_partial_flag_by_enclosure(dev_obj, os.path.join(decompress_path, inner_tar_file))
    else:
        return check_tar_special_file(tar_file_path, check_tar_name_is_match)


def old_tar_contains_partial_flag_by_enclosure(dev_obj, tar_file_path):
    for enclosure_inner_tar_file in os.listdir(tar_file_path):
        if tar_contains_partial_flag(dev_obj, tar_file_path, enclosure_inner_tar_file):
            return True
    return False


def tar_contains_partial_flag(dev_obj, local_dir, file_name):
    tar_file_path = os.path.join(local_dir, file_name)
    decompress_path = os.path.join(local_dir, file_name).replace(".tgz", "")
    try:
        decompress_tar_all_file(tar_file_path, decompress_path)
        return check_tar_special_file(tar_file_path, check_tar_name_is_match)
    except Exception as e:
        log.error(dev_obj, "failed open tar ball files. {} {}".format(e, str(traceback.format_exc())))
        return False


def check_tar_name_is_match(tar):
    return re.compile(REG).match(str(tar.name))


def decompress_dha_tar(dev_obj, local_dir, new_file_name):
    decompress_path = os.path.join(local_dir, new_file_name).replace(".tar", "")
    try:
        decompress_tar_all_file(os.path.join(local_dir, new_file_name), decompress_path, "r")
        for dha_file in os.listdir(decompress_path):
            # 根据dha日志中的内容处理标志位。若不包含指定内容则表示所有日志已经收集完毕
            if "DAE" in dha_file:
                is_loop_required = tar_contains_partial_flag_by_enclosure(
                    dev_obj, decompress_path, dha_file)
            else:
                is_loop_required = tar_contains_partial_flag(dev_obj, decompress_path, dha_file)
            if is_loop_required:
                return True
    except Exception as e:
        log.error(dev_obj, "failed open tar ball DAE files : {} {}".format(e, str(traceback.format_exc())))
    finally:
        log.info(dev_obj, "DAE temp file path = {}".format(decompress_path))
        try:
            is_delete_success = util.cleanDir(decompress_path)
            if not is_delete_success:
                log.error(dev_obj, "del DAE temp file failed.")
        except Exception as e:
            log.error(dev_obj, "del DAE temp file error. {} {}".format(e, str(traceback.format_exc())))
    return False


def check_dha_result_catalog(dev_obj, down_load_list, loop_times):
    log.info(dev_obj, "start check dha result catalog")
    # 标记是否需要继续循环的标志位
    is_loop_required = False
    local_dir = util.getLocalInfoPathByType(dev_obj, config.COLLECT_TYPE_DHA)
    # 遍历结果目录，对下载到的包进行查看是否需要进行处理
    for file_name in os.listdir(local_dir):
        if file_name in down_load_list:
            continue
        # dha日志重命名
        is_rename_success, new_file_name = rename_local_dha_file_name(dev_obj, local_dir, file_name)
        if not is_rename_success or not new_file_name:
            return False, util.getMsg(dev_obj, get_dha_err_msg_key(
                "dha.local.alter.file.name.failure", loop_times)), is_loop_required
        down_load_list.append(new_file_name)
        is_loop_required = decompress_dha_tar(dev_obj, local_dir, new_file_name) or is_loop_required
    return True, "", is_loop_required


def rename_file_name(dev_obj, local_dir, origin_file_name):
    log.info(dev_obj, "rename file is: {}".format(origin_file_name))
    try:
        # 获取扩展名
        file_extend = origin_file_name.split(".")[-1]
        # 删除最后的后缀
        file_name = origin_file_name.replace(".{}".format(file_extend), "")
        # 添加时间参数
        new_file_name = "{}{}".format(file_name, str(time.time()).replace(".", ""))
        log.info(dev_obj, "altered file name to: {}".format(new_file_name))
        # 加后缀，获取完整的新文件名
        new_file_full_name = "{}.{}".format(new_file_name, file_extend)
        # 改名
        os.rename("{}{}".format(local_dir, origin_file_name), "{}{}".format(local_dir, new_file_full_name))
        log.info(dev_obj, "done alternation DHA log into new file name : {}".format(new_file_full_name))
        return new_file_full_name
    except Exception as e:
        log.error(dev_obj, "DHA collection error:rename file failed. {} {}".format(e, str(traceback.format_exc())))
        return ""


def rename_local_dha_file_name(dev_obj, local_dir, file_name):
    if not os.path.exists(os.path.join(local_dir, file_name)):
        return False, ""
    log.info(dev_obj, "local DHA output file dir: {}".format(local_dir))
    return True, rename_file_name(dev_obj, local_dir, file_name)


def get_dha_err_msg_key(origin_name, loop_times):
    # 已收集次数大于0，本次收集失败，错误信息打印部分成功
    return "partly.{}".format(origin_name) if loop_times > 0 else origin_name
