# -*- coding: UTF-8 -*-
import os
import traceback
import re

from common.util import *
from util import util
from util import log
from common.commonUtils import SpaceNotEnoughException
from cbb.frame.cli.cli_con_check_4_info_collect import CliUtil as CliCheckUtil
from common.cliFactory import cli
from common.config import config
from com.huawei.oss.cn.common.anonymize import FileAnonymizeUtil
from com.huawei.ism.tool.infocollect.service import DealFileInCopressPackage
from utils import Products
import com.huawei.ism.tool.obase.exception.ToolException as ToolException


'''
导出系统信息（系统日志，配置信息，告警信息），公共方法
'''
class exportInfo():

    @staticmethod
    def isNormalForRec(devObj, rec):
        '''
        @summary: 判断回显信息是否正常
        @param cliRet:  devObj=上下文对象,
                        rec=导出系统信息的回显
        @return: True or False
        '''
        if None == rec or "" == rec:
            return False

        #判断系统是否在正确模式下
        if cli.CLI_RET_END_FLAG not in rec:
            util.setPyDetailMsg(devObj, "system.not.in.right.mode", rec)
            return False

        #判断系统是否繁忙
        if cli.CLI_SYSTEM_BUSY_FLAG in rec:
            log.info(devObj, "Collect error, result is:" + rec)
            util.setPyDetailMsg(devObj, "system.busy")
            return False

        if cli.CLI_BASH_EXPORT_FLAG in rec or cli.CLI_MINISYSTEM_FLAG in rec:
            log.info(devObj, "Collect error, result is:" + rec)
            util.setPyDetailMsg(devObj, "system.not.in.right.mode")
            return False

        return True

    @staticmethod
    def isExecCmdSuccForRec(devObj, rec):
        '''
        @summary: 根据回显判断命令是否执行成功
        @param cliRet:  devObj=上下文对象,
                        rec=导出系统信息的回显
                        list=判断命令回显成功的回显列表
        @return 1: True or False,命令执行成功
        @return 2: True or False,收集的文件是否完整
        '''
        execSucc = False
        isAllSucc = False
        #首先判断命令执行是否正常
        isNormal = exportInfo.isNormalForRec(devObj, rec)
        if False == isNormal:
            return (execSucc, isAllSucc)

        for re in config.ISEXEC_CMD_SUCCESS_FLAG:
            if re in rec:
                execSucc = True
                break

        if execSucc:
            #收集成功的文件数量
            succCollectNum = 0
            cliRetList = rec.splitlines()
            cliRet4Log = ""
            for line in cliRetList:
                if config.COLLECT_INFO_RECROD_STANDARD in line:
                    succCollectNum += 1
                    cliRet4Log += "File Path : XXXX"
                else:
                    cliRet4Log += line
            #应该收集的文件数量
            allCollectNum = len(cliRetList) - 2
            if succCollectNum > 0 and succCollectNum < allCollectNum:
                log.info(devObj, "collect partfail because: %s" % cliRet4Log)
            else:
                isAllSucc = True

        return (execSucc, isAllSucc)

    @staticmethod
    def getRemotePathAndName(devObj, rec):
        '''
        @summary: 获取阵列端的文件名称和路径
        @param cliRet:  devObj=上下文对象
        @return: name:阵列端产生的文件名称
                 path:阵列端产生的文件的路径
        '''
        resultList = []
        cliRetList = rec.splitlines()
        for line in cliRetList:
            #判断回显是否正常
            if config.COLLECT_INFO_RECROD_STANDARD in line:
                tempDict = {}
                tempList = line.split(" : ")
                path = tempList[1]
                tempList1 = path.split("/")
                name = tempList1[-1]
                tempDict["path"] = path
                tempDict["name"] = name
                resultList.append(tempDict)

        return resultList

    @staticmethod
    def downloadFile(devObj, type, rec, isAnonymize = True):
        '''
        @summary: 下载收集到得文件信息到本地
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        rec=命令执行成功的回显
        @return: True or False
        '''
        try:
            sftp = util.getSftp(devObj)
            localDir = util.getLocalInfoPathByType(devObj, type)
            #下载信息到本地
            #获取阵列端存放信息的路径和文件名称
            listRemotePaths = exportInfo.getRemotePathAndName(devObj, rec)

            #下载到本地的目录信息
            loadLocalDirInfo = []
            localPath = ""
            for temp in listRemotePaths:
                localPath = localDir + temp["name"]
                file = File(localPath)
                sftp.getFile(temp["path"], file, None)
                tempDict = {"path":localDir, "name":temp["name"]}
                loadLocalDirInfo.append(tempDict)

            # dorado、新融合6.1.5及之后版本，二级存储1.2.0及之后版本，计算型存储，微存储，阵列做了脱敏操作，工具不再脱敏。
            if exportInfo.check_device_version(devObj):
                isAnonymize = False

            # 对配置信息文件、全部日志和近期日志进行脱敏
            # 由于新硬件存储存在IP框日志，增加判断是否进行脱敏的标志位
            if isAnonymize:
                if config.EXPORT_TYPE_RUNNING_DATA == type:
                    FileAnonymizeUtil.anonymizeWithEncoding(localDir, "utf-8")
                elif config.EXPORT_TYPE_ALLLOG == type or config.EXPORT_TYPE_LOG == type:
                    DealFileInCopressPackage.anonymizeCompressFile(localPath)
                CliCheckUtil.check_cli_connect(devObj)
            return (True, loadLocalDirInfo)
        except ToolException as e:
            log.info(devObj, "except e:{}".format(str(e)))
            err_key = "local free space is not enough"
            if err_key in str(e).lower():
                raise SpaceNotEnoughException(err_key)
            return False, None
        except:
            log.error(devObj, "down load %s fail" % str(type))
            log.info(devObj, "except trace back:" + str(traceback.format_exc()))
            return (False, None)

    @staticmethod
    def check_device_version(dev_obj):
        device_type = str(dev_obj.get("devNode").getDeviceType())
        if device_type in config.NON_ANONYMIZATION_DEV_TYPES:
            return True
        version = dev_obj.get("devNode").getProductVersion()
        if device_type in config.PARTIALLY_UNSENSITIZED_DEV_TYPES:
            return Products.compareVersion(version, "1.2.0") >= 0
        return Products.compareVersion(version, "6.1.5") >= 0

    @staticmethod
    def deleteRemoteFile(devObj, type, rec):
        '''
        @summary: 删除阵列端生成的文件
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        rec=CLI回显信息
        @return:Bool(True or False)
        '''
        #根据CLI回显信息判断当前环境下，是否允许删除
        tempList = config.DELETE_REMOTE_DATA_STANDARD
        deleteFlag = True
        for standard in tempList:
            if standard in rec:
                deleteFlag = False
                break

        if deleteFlag:
            strCmd = "delete file filetype=" + type
            log.info(devObj, "delete command is:" + strCmd)
            cli.executeCmdNoLogTimeout(devObj, strCmd)



    @staticmethod
    def decompressTgzFile(devObj, filePath, fileName):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tgz）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''

        tar_class = devObj.get("PYENGINE.PY_ZIP")
        dirName = fileName[0: fileName.find(".tgz")]

        tempFilePath = filePath + os.path.sep + fileName
        tempDirPath = filePath + os.path.sep + dirName

        if not tar_class.decompressZipFile(tempFilePath, tempDirPath):
            log.info(devObj, "decompressZipFile is err, use decompressTarGzFile")
            tar_class.decompressTarGzFile(tempFilePath, tempDirPath)

        return tempDirPath


    @staticmethod
    def decompressTarBz2File(devObj, filePath, fileName, f_type=".tar.bz2"):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tar.bz2）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (解压路径)
        '''
        tar_class = devObj.get("PYENGINE.PY_ZIP")
        dirName = fileName[0: fileName.find(f_type)]
        tempFilePath = filePath + os.path.sep + fileName
        tempDirPath = filePath + os.path.sep + dirName
        tar_class.decompressTarBz2File(tempFilePath, tempDirPath)
        return tempDirPath


    @staticmethod
    def getNodeCollectInfo(devObj, filePath, fileName, tail=".tar.bz2"):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tar.bz2）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''

        tempDirPath = exportInfo.decompressTarBz2File(devObj, filePath, fileName, tail)

        #筛选日志信息结果文件
        listFile = os.listdir(tempDirPath)
        result = None
        for fileName in listFile:
            if fileName in config.COLLECT_LOG_FILE_INFO:
                logPath = tempDirPath + os.path.sep + fileName
                result = util.readFile(logPath)

        return result


    @staticmethod
    def checkSystemLogByFile(devObj, info, item, nodeName):
        '''
        @summary: 解析节点,系统日志信息中收集结果的文件
        @param cliRet:  devObj=上下文对象,
                        info=文件内容，
                        item=错误信息编号
                        nodeName=控制器节点名称
        @return: (msg=当前节点错误信息, item=错误信息序号)
        '''
        msg = None
        if None == info:
            item +=1
            msg = util.getMsg(devObj, "controller.failed.information", (item, nodeName))
            return msg

        lines = info.splitlines()
        collectFailList = ""
        #如果收集结果文件（info_log.txt）中含有乱码则剔除含有乱码的项
        newLines = []
        for line in lines:
            try:
                line.decode("utf-8").encode("utf-8")
                newLines.append(line)
            except Exception:
                log.info(devObj, "decode and encode line error")
                pass

        lines = newLines
        for line in lines:
            if line.strip().startswith("Slot ID is"):
                break

            if not line.strip():
                continue

            try:
                #如果某项收集结果不是ok，则表示该项收集失败
                if not line.strip().lower().endswith(" ok"):
                    collectFailList += line.split()[0] + ', '
            except Exception as e:
                log.info(devObj, "collect item except." + str(e))
                continue

        if "" != collectFailList:
            item +=1
            msg = util.getMsg(devObj, "controller.failed.items", (item, nodeName, collectFailList[:-2]))

        return (msg, item)

    @staticmethod
    def check_ip_enc_by_file(dev_obj, info, item, node_name):
        '''
        @summary: 解析节点,系统日志信息中收集结果的文件
        @param cliRet:  devObj=上下文对象,
                        info=文件内容，
                        item=错误信息编号
                        nodeName=控制器节点名称
        @return: (msg=当前节点错误信息, item=错误信息序号)
        '''
        import re
        msg = None
        enc_name = node_name
        search_res = re.compile("(DAE\w+.\w+).tbz").search(node_name)
        if search_res:
            enc_name = search_res.group(1)
        log.info(dev_obj, "check {} ip enc is full, info:{}".format(
            enc_name, info))
        if not info:
            item += 1
            msg = util.getMsg(dev_obj, "ipenc.failed.information",
                              (item, enc_name))
            return msg, item

        collect_fail_list = []
        for line in info.splitlines():
            if "OK OVERSIZE" in line:
                continue
            columns = [column.strip() for column in line.split("  ") if
                       column.strip()]
            if not columns:
                continue
            if "OK" not in columns and len(columns) > 3:
                collect_fail_list.append(columns[1])
            elif "OK" not in columns and len(columns) == 3:
                collect_fail_list.append(columns[0])
        if collect_fail_list:
            item += 1
            msg = util.getMsg(dev_obj, "ipenc.failed.items",
                              (item, enc_name, ", ".join(collect_fail_list)))

        return msg, item

    @staticmethod
    def checkNodeSystemLogIsFull(devObj, filePath, fileName, item, nodeId=None):
        '''
        @summary: 获取当前收集节点，收集失败的项
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        #解压tgz压缩包，返回解压后的路径
        keepAliveThread = AsynProgressExecuteCmd(devObj, log)
        keepAliveThread.start()
        log.info(devObj, "check full fileName:{}, nodeId:{}".format(
            fileName, nodeId))
        decompressDir = exportInfo.decompressTgzFile(devObj, filePath, fileName)
        #获取解压后的tar.bz2的文件列表，一个tar.bz2代表一个节点信息
        nodeList = os.listdir(decompressDir)
        messsage = ""
        controllerCount = len(nodeList)
        for nodeInfo in nodeList:
            nodePath = decompressDir
            nodeName = nodeInfo
            log.info(devObj, "node name:{}".format(nodeName))
            # 判断:若不是tar.bz2、tbz、tar格式的压缩包，则不做处理
            if exportInfo.is_not_specified_compression_format(nodeName) or ".sha256sum" in nodeName:
                continue
            if nodeName.endswith(".tar"):
                result = exportInfo.get_node_result_info(devObj, nodePath, nodeName)
                log.info(devObj, "tail .tar, name:{},resultInfo:{}".format(nodeName, result))
                node_id = nodeId if nodeId is not None else nodeName
                msg, item = exportInfo.checkSystemLogByFile(devObj, result, item, node_id)
                if msg is not None:
                    messsage += msg
                continue
            #解压各个节点，获取节点中导出日志信息的结果信息
            tail = ".tar.bz2" if ".tar.bz2" in nodeName else ".tbz"
            nodeResultInfo = exportInfo.getNodeCollectInfo(
                devObj, nodePath, nodeName, tail)
            log.info(devObj, "tail {},name:{}, nodeResultInfo:{}".format(
                tail, nodeName, nodeResultInfo))
            if ".tbz" == tail:
                msg, item = exportInfo.check_ip_enc_by_file(
                    devObj, nodeResultInfo, item, nodeName)
            else:
                # 分析结果信息
                if nodeId is not None:
                    nodeName = nodeId
                (msg, item) = exportInfo.checkSystemLogByFile(
                    devObj, nodeResultInfo, item, nodeName)
            if None != msg:
                messsage += msg

        #清除解压后的临时目录
        util.cleanDir(decompressDir)
        keepAliveThread.setStopFlag(True)
        return (messsage, item, controllerCount)

    @staticmethod
    def is_not_specified_compression_format(node_name):
        return ".tar.bz2" not in node_name and ".tbz" not in node_name and not node_name.endswith(".tar")

    @staticmethod
    def get_node_result_info(dev_obj, file_path, file_name):
        temp_dir_file = exportInfo.decompress_tar_file(dev_obj, file_path, file_name)
        list_file = os.listdir(temp_dir_file)

        for temp_file in list_file:
            if temp_file in config.COLLECT_LOG_FILE_INFO:
                log_path = temp_dir_file + os.path.sep + temp_file
                return util.readFile(log_path)
        return None

    @staticmethod
    def decompress_tar_file(dev_obj, file_path, file_name):
        tar_class = dev_obj.get("PYENGINE.PY_ZIP")
        dir_name = file_name[0: file_name.find(".tar")]
        temp_file_path = file_path + os.path.sep + file_name
        temp_dir_path = file_path + os.path.sep + dir_name
        log.info(dev_obj, "start decompress tar file,  dir_name is {}".format(dir_name))
        tar_class.decompressTarFile(temp_file_path, temp_dir_path)
        return temp_dir_path

    @staticmethod
    def checkLogIsCollectFull(devObj, successCount, item):
        '''
        @summary: 下载系统config.txt文件，从文件中获取系统标配控制器个数
        @param cliRet:  devObj=上下文对象,
                        pathList=收集的系统日志的文件夹列表，
                        msgInfo=错误信息
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        #配置信息存放目录，若目录存在，则表示之前已经收集了配置信息
        runningDataType = config.EXPORT_TYPE_RUNNING_DATA
        runningDataDir = devObj[config.COLLECT_INFO_LOCAL_PATH] + os.path.sep
        runningDataDir = runningDataDir + config.COLLECT_INFO_DIR_NAME[runningDataType]
        deleteRunningDataDir = True

        #若存在则不删除本地配置信息
        if os.path.exists(runningDataDir):
            deleteRunningDataDir = False

        exportDataFalg = True
        tempCmd = "show file export_path file_type=%s" % runningDataType

        (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, tempCmd)
        if not isSuccess:
            item += 1
            msg = util.getMsg(devObj, "dev.conn.failure")
            return (msg, item)

        (execSucc, isAllSucc) = exportInfo.isExecCmdSuccForRec(devObj, rec)
        if not execSucc:
            exportDataFalg = False

        (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj, runningDataType, rec, True)
        if not loadSucc:
            exportDataFalg = False

        exportInfo.deleteRemoteFile(devObj, runningDataType, rec)
        if not exportDataFalg:
            item += 1
            msg = util.getMsg(devObj, "network.communication.abnormal", (item))
            return (msg, item)

        tempList = exportInfo.getRemotePathAndName(devObj, rec)
        fileName = tempList[0]["name"]
        ctrlCount = 0
        path = runningDataDir + os.path.sep + fileName
        collectConfig = util.readFile(path)
        if None == collectConfig:
            collectConfig = ""
        lines = collectConfig.splitlines()
        for line in lines:
            if "Number of total controllers" in line:
                ctrlCount = line.split(':')[-1].strip()
                break

        #删除收集到得文件
        if deleteRunningDataDir:
            util.cleanDir(runningDataDir)

        msg = ""
        if int(ctrlCount) != successCount:
            item +=1
            msg = util.getMsg(devObj, "controllers.some.failed", (item))
            log.info(devObj, "ctrlCount=" + str(ctrlCount) + ";collect success controller = " + str(successCount))

        return (msg, item)

    @staticmethod
    def exportLog(devObj, log_type, modules_ids="", controller_ids=""):
        '''
        #执行命令，导出系统日志信息
        '''
        #版本兼容，需要首先判断新命令是否存在，若不存在则执行老命令进行信息收集
        timeOut = config.EXPORT_INFO_TIMEOUT[log_type]
        existCmd = False
        successCount = 0
        #错误信息相关
        messages = ""
        item = 0
        if log_type in config.IS_ASYNCHRONOUS_CMD_HAS_IP_ENCLOSURE:
            #对支持异步方式收集信息的设备采用如下方式收集，适用于新方案的log,alllog,smart收集
            (isSuccess, existCmd) = exportInfo.canParallelExportInfoByCtrlId(
                devObj, log_type)
            if not isSuccess:
                return False
            if existCmd:
                return exportInfo.parallelExportInfoByCtrlId(devObj, log_type, modules_ids, controller_ids)

        if not existCmd:
            #使用旧的导出方式导出数据
            tempCmd = "show file export_path file_type=%s" % (log_type)
            log.info(devObj, "execute collect : %s" % str(tempCmd))
            (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, tempCmd, timeOut)
            if not isSuccess:
                util.setPyDetailMsg(devObj, "dev.conn.failure")
                return False
            util.refreshProcess(devObj, 50)
            #判断命令是否存在
            existCmd = cli.isExistCmdForRec(rec)
            if not existCmd:
                #命令不存在，表示环境版本过低，直接收集成功
                log.info(devObj, "commend not exist %s" % str(rec))
                return True
            util.refreshProcess(devObj, 52)
            (execSucc, isAllSucc) = exportInfo.isExecCmdSuccForRec(devObj, rec)
            if not execSucc:
                util.setPyDetailMsg(devObj, rec)
                log.info(devObj, "show file... execute failed, and delete file")
                exportInfo.deleteRemoteFile(devObj, log_type, rec)
                return False
            util.refreshProcess(devObj, 54)
            (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj,
                                                              log_type, rec,
                                                              True)
            if not loadSucc:
                util.setPyDetailMsg(devObj, "downLoad.file.failure")
                log.info(devObj, "download file failed, and delete file")
                exportInfo.deleteRemoteFile(devObj, log_type, rec)
                return False
            util.refreshProcess(devObj, 70)
            #删除阵列段临时文件
            log.info(devObj, "get file success, and delete file")
            exportInfo.deleteRemoteFile(devObj, log_type, rec)

            util.refreshProcess(devObj, 75)

            #若收集的是系统日志信息
            count = 0
            if log_type in config.IS_SYSTEM_INFO_FULL:
                #判断信息是否收集完全
                for tempDir in loadDirInfo:
                    fileName = tempDir["name"]
                    if ".tgz" in fileName:
                        (msg, item, count) = exportInfo.checkNodeSystemLogIsFull(devObj, tempDir["path"], fileName, item)
                        messages += msg
            util.refreshProcess(devObj, 78)
            #单独获取收集Smart信息的节点数量
            if log_type == config.COLLECT_TYPE_SMART:
                tempDir = loadDirInfo[0]
                tempDirPath = exportInfo.decompressTarBz2File(devObj, tempDir["path"], tempDir["name"])
                listFile = os.listdir(tempDirPath)
                count = len(listFile)

            successCount = count

        if log_type in config.IS_CONTROLLER_INFO_FULL:
            if successCount == 0:
                log.info(devObj, "collect fail successCount=0")
                return False
            (msg, item) = exportInfo.checkLogIsCollectFull(devObj, successCount, item)
            messages += msg
        util.refreshProcess(devObj, 86)
        if "" != messages:
            util.setCollectAllInfo(devObj, False)
            util.setPyDetailMsg(devObj, messages)

        return True

    @staticmethod
    def isRiskVersionFcCauseRisk(devObj, currentSoftVersion, currentHotpatchVersion):
        '''
        @summary: 工具规避在收集日志时触发的问题
        @param softwareVersionList: 当前阵列的版本所有控制器的版本列表
        @param hotPatchVersionList: 当前阵列的版本所有控制器的补丁版本列表
        retrun：False：表示是风险版本
                True：表示不是风险版本
        '''

        # 系统版本及热补丁风险版本
        riskVersionDict = {
            'V300R006C20': 'V300R006C20SPH015',
            'V300R006C20SPC100': 'V300R006C20SPH115',
            'V500R007C10': 'V500R007C10SPH015',
            'V500R007C30SPC100': 'V500R007C30SPH105',
            'V300R001C21SPC100': 'V300R001C21SPH112',
            'V500R007C10SPC100': 'V500R007C10SPH115',
            'V300R002C10SPC100': 'V300R002C10SPH101',
            'V300R006C50SPC100': 'V300R006C50SPH105',
            'V300R001C30SPC100': 'V300R001C30SPH106'
        }

        # 对版本型号及热补丁版本进行适配,如果系统版本不涉及则通过
        if currentSoftVersion not in riskVersionDict:
            return False

        needInstallHotpatchVersion = riskVersionDict.get(currentSoftVersion)
        # 如果热补丁版本匹配则通过
        if currentHotpatchVersion != '--' and currentHotpatchVersion >= needInstallHotpatchVersion:
            return False

        # 如果热补丁版本不匹配
        log.info(devObj, "hot patch version is lower than request")

        # 需判断启动器数量和控制器冗余
        cmd = "show initiator initiator_type=FC"
        __, cliRet = cli.executeCmdWithRetry(devObj, cmd)
        iniDictList = util.getHorizontalCliRet(cliRet)
        normalInitiatorCount = 0
        for iniDict in iniDictList:
            if iniDict.get("Running Status", '') == "Online":
                normalInitiatorCount += 1

        isAllCtrlNormal = True
        cmd = "show controller general |filterColumn include columnList=Health\sStatus,Running\sStatus,Controller"
        __, cliRet = cli.executeCmdWithRetry(devObj, cmd)
        controllerDictList = util.getVerticalCliRet(cliRet)
        for controllerDict in controllerDictList:
            if controllerDict.get("Health Status", '') != "Normal":
                isAllCtrlNormal = False
                break

            if controllerDict.get("Running Status", '') != "Online":
                isAllCtrlNormal = False
                break

        log.info(devObj, "hot patch version is lower than request, normal fc Initiator Count:%s, isAllCtrlNormal:%s"
                 % (normalInitiatorCount, isAllCtrlNormal))

        if normalInitiatorCount <= 200 and isAllCtrlNormal:
            return False

        # 未安装对应补丁，且启动器数量超过80个或控制器不正常，限制收集。
        util.setPyDetailMsg(devObj, "install.fc.driver.risk.patch", needInstallHotpatchVersion)
        return True

    @staticmethod
    def exportSysInfo(devObj, type, modules_ids="", controller_ids=""):
        '''
        @summary: 丛阵列端导出信息，日志收集、事件信息、配置信息收集的主入口
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
        @return:True or False
        '''
        try:
            #收集前检查
            #检查系统是否正常
            isSysNormal = util.checkSystemNormal(devObj)
            if not isSysNormal:
                return False
            #检查用户级别是否满足收集信息
            isPrivilege = util.checkUserPrivilege(devObj)
            if not isPrivilege:
                return False

            # 存在收集config的收集项，规避FC启动器缺陷，限制收集。
            if type in [config.EXPORT_TYPE_LOG, config.EXPORT_TYPE_ALLLOG, config.EXPORT_TYPE_RUNNING_DATA]:
                issucess, softwareVersionList, hotPatchVersionList = device.parse_upgradePackage(devObj)
                # 判断是否含异常场景
                if issucess != True:
                    return False

                # 获取系统版本
                __, currentSoftVersion = device.getCurrentVersion(devObj, softwareVersionList)

                # 获取热补丁版本
                __, currentHotpatchVersion = device.getCurrentVersion(devObj, hotPatchVersionList)

                currentSoftVersion = currentSoftVersion.strip()

                currentHotpatchVersion = currentHotpatchVersion.strip()

                # 规避FC启动器缺陷，限制收集。
                isRisk = exportInfo.isRiskVersionFcCauseRisk(devObj, currentSoftVersion, currentHotpatchVersion)
                # 存在风险，限制收集，否则继续收集
                if isRisk:
                    return False

            exportSucc = exportInfo.exportLog(devObj, type, modules_ids, controller_ids)

            return exportSucc

        except SpaceNotEnoughException:
            error_msg = util.getMsg(
                devObj, "failed.export.log.insufficient.space")
            util.setPyDetailMsg(devObj, error_msg)
            return False

        except Exception as ex:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            return False

    @staticmethod
    def export_meg_info(devObj, collect_type, modules_ids="", controller_ids=""):
        '''
        @summary: 丛阵列端导出信息，收集pool私有日志的主入口
        @param cliRet:  devObj=上下文对象
                        collect_type=收集的类型
        @return:True or False
        '''
        try:
            # 收集前检查
            # 检查系统是否正常
            if not util.checkSystemNormal(devObj):
                return False

            # 检查用户级别是否满足收集信息
            if not util.checkUserPrivilege(devObj):
                return False

            exportSucc = exportInfo.exportLog(devObj, collect_type, modules_ids, controller_ids)

            return exportSucc

        except Exception as exception:
            log.error(devObj, "except trace back:" + str(exception))
            return False

    @staticmethod
    def canParallelExportInfoByCtrlId(devObj, type):
        '''
        #判断设备是否支持并行方式收集-->设备支持此命令show file package_result file_type=%s
        @return 1:True or False,命令执行成功
        @return 2:True or False,是否支持此命令
        '''
        (isSuccess, exist_cmd) = (False, False)
        cmd = "show file package_result file_type=%s" % (type)
        count = 0
        while count < 4:
            (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, cmd)
            if "timed out" not in rec:
                exist_cmd = cli.isExistCmdForRec(rec)
                break
            exist_cmd = cli.isExistCmdForRec(rec)
            time.sleep(30)
            count += 1
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
        return (isSuccess, exist_cmd)

    @staticmethod
    def prepareCollect(devObj, type, modules_ids="", controller_ids=""):
        '''
        #下发CLI,通知OM开始收集，OM通知各节点做收集的前期准备
        '''
        is_success, ret = exportInfo.execute_command(controller_ids, devObj, modules_ids, type)
        log.info(devObj, "prepareCollect and the cliRet is {}".format(ret))
        if not is_success:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False, ret

        if config.CLI_EXECUTE_CMD_SUCCESS not in ret:
            util.setPyDetailMsg(devObj, "cli.excute.failure", ret)
            return False, ret

        return True, ret

    @staticmethod
    def execute_command(controller_ids, dev_obj, modules_ids, collect_type):
        is_success, controller_id = util.getController(dev_obj)
        cmd4collect = "show file export_path file_type={}".format(collect_type)
        if is_success:
            cmd4collect = "show file export_path file_type={} controller_id={}".format(collect_type, controller_id)
        if controller_ids:
            cmd4collect = "{} controller_id={}".format(cmd4collect, controller_ids)
        if modules_ids:
            cmd4collect = "{} enclosure_id={}".format(cmd4collect, modules_ids)
        if dev_obj.get("collectAllLogs") and collect_type == config.EXPORT_TYPE_ALLLOG:
            cmd4collect = "{} unlimit_size=yes".format(cmd4collect)
        is_success, ret = cli.executeCmd(dev_obj, cmd4collect)
        return is_success, ret

    @staticmethod
    def queryCollectStatus(devObj, type):
        '''
        #轮询收集进度
        @return 1:True or False,命令执行成功
        @return 2:收集进度{"总的进度" : "" , "单个节点的进度" : {"节点" : "进度"}..}
        @return 3:cli回显
        '''
        (isSuccess, collectStatus, cliRet) = (False, {"totalResult" : "" , "singleResultDict" : {}}, "")

        cmd4status = "show file package_result file_type=%s" % (type)
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4status)
        log.info(devObj, "queryCollectStatus and the cliRet is " + str(cliRet))
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (isSuccess, collectStatus, cliRet)
        if "Total Result" not in cliRet:
            if "Failed to query the export status" not in str(cliRet) \
                    or 'Run the "show file export_path" command' not in str(cliRet):
                 util.setPyDetailMsg(devObj, "cli.excute.failure", cliRet)
            return (isSuccess, collectStatus, cliRet)

        totalResult = ""
        NodeID = ""
        singleResult = ""

        dictList = cli.getCliTable2DictList(cliRet)
        for dict in dictList:
            totalResult = dict["Total Result"]
            if totalResult != "" and totalResult != "--":
                collectStatus["totalResult"] = totalResult

            singleResultDict = collectStatus["singleResultDict"]
            NodeID = dict["Controller ID"]
            singleResult = dict["Single Result"]
            singleResultDict.update({NodeID : singleResult})
            collectStatus["singleResultDict"] = singleResultDict

        return (True, collectStatus, cliRet)

    @staticmethod
    def notificationDownload(devObj, type, NodeID):
        '''
        #通知OM取文件
        @return 1:True or False,命令执行成功
        @return 2:CLI回显
        '''
        (isSuccess, cliRet) = (False, "")
        cmd4load = "show file notification file_type=%s controller_id=%s" % (type, NodeID)
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4load)
        log.info(devObj, "notificationDownload and the cliRet is " + str(cliRet))
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (isSuccess, cliRet)

        isSuccess = True
        return (isSuccess, cliRet)

    @staticmethod
    def deleteFile(devObj, type, NodeID):
        '''
        #删除阵列端数据
        '''
        cmd4delete = "delete file filetype=%s controller_id=%s" % (type, NodeID)
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4delete)
        log.info(devObj, "deleteFile and the cliRet is " + str(cliRet))
        if not isSuccess:
            log.info(devObj, "deleteFile failed because cli execute failed")

        return

    @staticmethod
    def collectAndDownload(devObj, file_type):
        '''
        #1，轮询收集进度；2，下载收集到的文件到本地；3，删除阵列端的文件
        @return 1:True or False,收集项成功or失败
        @return 2:本项收集项的所有失败记录
        '''
        (isSuccess, failMesage) = (False, "")

        cmd4statusRet = ""
        item = 0 #
        totalResult = "" #总的收集进度
        # 收集的超时时间，单位：分
        limitTime = config.PARALLEL_EXPORT_INFO_TIMEOUT[file_type]
        repeatTimes = limitTime * 2 #轮训次数

        shouldCollectNodeList = [] #应该收集的节点数量
        finishedNodes = [] #已经完成的节点，包括成功和失败
        successNoedList = [] #收集成功的节点
        hasErrMsgNodeList = [] #收集失败且已经设置错误信息的节点列表
        ftpGetFailedNodes = []
        #已经收集到的文件路径列表
        finishedFile = []

        #1，轮询收集进度
        currentProcess = 2
        percentNumber = 90.0 / 8
        while repeatTimes != 0:
            repeatTimes -= 1
            (isSuccess, collectStatus,
             cmd4statusRet) = exportInfo.queryCollectStatus(devObj,
                                                            file_type)
            if not isSuccess:
                return (isSuccess, failMesage)
            # 兼容没有IP框的场景，将包含的定错误信息的回显，当做收集结束，等同于Packaging消失(标志由产品提供)
            if "Failed to query the export status" in str(cmd4statusRet) \
                    and 'Run the "show file export_path" command' in str(cmd4statusRet):
                break

            totalResult = collectStatus["totalResult"]
            singleResultDict = collectStatus["singleResultDict"]

            for NodeID in singleResultDict.keys():
                singleResult = singleResultDict[NodeID]
                if shouldCollectNodeList.count(NodeID) == 0 and NodeID not in ["", "--"]:
                    shouldCollectNodeList.append(NodeID)
                
                #阵列内部包控制器收集失败    
                if singleResult in ["Busy", "Failed", "Out of Memory"] and finishedNodes.count(NodeID) == 0:
                    exportInfo.deleteFile(devObj, file_type, NodeID)
                    item += 1
                    errMsgKey = "get.node.file.faild.%s" % singleResult
                    errMsg = util.getMsg(devObj, errMsgKey, (item, NodeID, cmd4statusRet))
                    failMesage += errMsg
                    hasErrMsgNodeList.append(NodeID)
                    finishedNodes.append(NodeID)

                if "Successful" in singleResult:
                    finishedNodes.append(NodeID)
                    #通知OM取文件
                    (isSuccess, cliRet) = exportInfo.notificationDownload(
                        devObj, file_type, NodeID)
                    if not isSuccess:
                        return (isSuccess, failMesage)

                    if "File Path" not in cliRet:
                        if finishedNodes.count(NodeID) == 1:
                            item += 1
                            errMsg = util.getMsg(devObj, "cli.excute.failure.on.someContr", (item, NodeID, cliRet))
                            failMesage += errMsg
                            hasErrMsgNodeList.append(NodeID)
                            #未获取到相应的文件路径，向用户提示可能的原因和处理意见
                            ftpGetFailedNodes.append(NodeID)
                        else:
                            continue
                    else:
                        #判断此次回显中的文件，是否有已经下载过的，若存在，不在继续下载
                        isfinishedFileCount = False
                        filePaths = exportInfo.getRemotePathAndName(devObj, cliRet)
                        for temp in filePaths:
                            if finishedFile.count(temp["path"]) > 0:
                                isfinishedFileCount = True
                            else:
                                finishedFile.append(temp["path"])
                        if isfinishedFileCount:
                            continue

                        #将阵列端的数据下载到本地
                        isAnonymize = True
                        if finishedNodes.count(NodeID) > 1:
                            isAnonymize = False
                        (loadSucc, loadDirInfo) = exportInfo.downloadFile(
                            devObj, file_type, cliRet, isAnonymize)
                        #删除阵列端数据
                        exportInfo.deleteFile(devObj, file_type, NodeID)

                        if not loadSucc:
                            item += 1
                            if finishedNodes.count(NodeID) == 1:
                                errMsg = util.getMsg(devObj, "downLoad.file.failure.on.someContr", (item, NodeID))
                                failMesage += errMsg
                            else:
                                errMsg = util.getMsg(devObj, "downLoad.file.IPBox", (item, NodeID))
                                failMesage += errMsg
                            hasErrMsgNodeList.append(NodeID)
                            continue
                        successNoedList.append(NodeID)
                        log.info(devObj, "finishedNodes：{}".format(
                            finishedNodes))
                        if file_type not in config.IS_SYSTEM_INFO_FULL or \
                                (exportInfo.check_device_version(devObj) and singleResult == "Successful"):
                            continue
                        # 判断信息是否收集完全
                        for tempDir in loadDirInfo:
                            fileName = tempDir["name"]
                            if ".tgz" not in fileName:
                                continue
                            (errMsg, item, count) = exportInfo.\
                                checkNodeSystemLogIsFull(devObj,
                                                         tempDir["path"],
                                                         fileName, item,
                                                         NodeID)
                            failMesage += errMsg
                            hasErrMsgNodeList.append(NodeID)

            if "Packaging" not in totalResult:
                break  
            #已30秒为间隔时间进行轮询
            time.sleep(30)
            currentProcess = util.refreshProcessByStep(devObj, currentProcess, percentNumber)

        successNoedNum = len(successNoedList)
        shouldCollectNodeNum = len(shouldCollectNodeList)
        #收集失败 
        if successNoedNum == 0:
            isSuccess = False         
        #成功或部分成功 
        else:
            isSuccess = True

        #每个节点收集失败的原因
        if successNoedNum != shouldCollectNodeNum:
            for NodeID in shouldCollectNodeList:
                #如果是超时则设置轮询结果的CLI命令的回显为失败原因
                if successNoedList.count(NodeID) == 0 and hasErrMsgNodeList.count(NodeID) == 0:
                    item += 1
                    errMsg = util.getMsg(devObj, "cli.excute.failure.on.someContr", (item, NodeID, cmd4statusRet))
                    failMesage += errMsg

        if "Partly Exported" in totalResult and failMesage == "":
            item += 1
            errMsg = util.getMsg(devObj, "collect.controllers.some.failed", item)
            failMesage += errMsg

        if len(ftpGetFailedNodes) != 0:
            errMsg = util.getMsg(devObj, "sftp.get.file.faild", ",".join(ftpGetFailedNodes))
            failMesage += errMsg
        util.refreshProcess(devObj, 98)    
        return (isSuccess, failMesage)

    @staticmethod
    def parallelExportInfoByCtrlId(devObj, type, modules_ids="", controller_ids=""):
        '''
        #控制器之间以并行方式来收集信息，涉及log,alllog,smart
        '''
        #1, 下发CLI,通知OM开始收集;
        is_success = True
        ctrl_id_flag, ctrl_id_list = util.getCtrlIds(devObj)
        # 每5分钟收集一次，收集4次，如果收集失败，清理残留文件
        num = 4
        for i in range(num):
            is_success, ret = exportInfo.prepareCollect(devObj, type, modules_ids, controller_ids)
            if is_success:
                # 如果导出命令执行成功，清理可能存在错误信息
                util.initPyDetailMsg(devObj)
                break

            # 如果其他设备正在导出，直接返回失败。
            if exportInfo.is_export_file(ret):
                util.setPyDetailMsg(devObj, "system.is.exporting.files")
                return False

            # 收集，清理残留文件
            exportInfo.collect_and_delete_file(devObj, type, ctrl_id_list)

        if not is_success:
            return False

        #2, 轮询收集进度，一旦发现有收集成功的节点则下载收集到得文件到本地并删除阵列端的文件。
        (isSuccess, failMesage) = exportInfo.collectAndDownload(devObj, type)

        #设置错误消息，并将部分收集成功标志置为True
        if failMesage != "":
            util.setCollectAllInfo(devObj, False)
            util.setPyDetailMsg(devObj, failMesage)

        return isSuccess

    @staticmethod
    def is_export_file(ret):
        return "system is exporting files" in str(ret).lower()

    @staticmethod
    def collect_and_delete_file(dev_obj, type, ctrl_id_list):
        """
        如果收集失败，清理残留文件
        """
        for ctrl_id in ctrl_id_list:
            flag, cli_ret = exportInfo.notificationDownload(dev_obj, type, ctrl_id)
            if flag and "File Path" in cli_ret:
                exportInfo.deleteFile(dev_obj, type, ctrl_id)
        num = 10
        for i in range(num):
            cmd = "show system general"
            cli.executeCmd(dev_obj, cmd)
            time.sleep(30)

    @staticmethod
    def collect_item(dev_obj, collect_type):
        """
        1、无节点信息，走老逻辑
        2、ALL_LOG且只收集ip框场景，只下发modules_ids
        3、ALL_LOG其余场景需要下发modules_ids、controller_ids
        4、近期、SMART、关键日志、ftds日志只需要下发controller_ids
        :param dev_obj: dev对象
        :param collect_type: 收集类型
        :return: 收集结果
        """
        has_nodes, modules_ids, controller_ids = util.get_all_nodes_id(dev_obj)
        if not has_nodes:
            return exportInfo.exportSysInfo(dev_obj, collect_type)
        if dev_obj.get("collectSmartByQuicklyScene") and collect_type == config.EXPORT_TYPE_ALLLOG:
            return exportInfo.exportSysInfo(dev_obj, config.EXPORT_TYPE_ALLLOG, modules_ids=modules_ids)
        if collect_type == config.EXPORT_TYPE_ALLLOG:
            return exportInfo.exportSysInfo(dev_obj, config.EXPORT_TYPE_ALLLOG, modules_ids, controller_ids)
        if collect_type in config.CUSTOM_ONLY_COLLECT_CONTROLLER_ITEM:
            return exportInfo.exportSysInfo(dev_obj, collect_type, controller_ids=controller_ids)
        log.info(dev_obj, "the type: {} is error".format(collect_type))
        return False
