# -*- coding: UTF-8 -*-
import traceback
import tarfile
from util import util
from util import log
from util import cli
from util import File
from util import time
from util import device
from config import config
import os
import re
import shutil
from common.exportInfo import exportInfo
from common.util import add_permission
from com.huawei.ism.tool.obase.exception import ToolException
from frameone.util.str_util import StrUtils
from cbb.frame.util.tar_util import decompress_tar_all_file, \
    check_tar_special_file


class DhaInfoCollector():
    
    def __init__(self):
        return
    
    ##############
    #导出DHA日志执行入口
    ##############
    
    def exportDhaInfo(self, devObj):
        try:
            version = str(device.getDeviceVersion(devObj))
            log.info(devObj, " version:%s" % version)
            util.refreshProcess(devObj, 5)
            if not self.is_digital_ver(version) and (
                    version >= 'V500R007C61' or (
                    'V3' in version and version >= 'V300R006C61')):
                # 如果为R5R7C61及之后版本，用show file的形式进行采集，2020,02 25新增
                is_success, loopTimes, echoMsg = \
                    self.export_dha_with_show_file(devObj)
            else:
                # 检查系统状态
                isSysNormal = util.checkSystemNormal(devObj)
                if not isSysNormal:
                    return False
                # 获取执行参数
                commandArguments = self.fetchDhaCommandArgs(devObj)
                log.info(devObj, " now starting gethering DHA info data!")
                # 导出DHA日志
                is_success, loopTimes, echoMsg = \
                    self.processDhaCmd(devObj, commandArguments)
            # 2020年12月12日删除DHA无论如何都会收集成功的判断
            return is_success, echoMsg
        except Exception as exception:
            log.error(
                devObj, "collect dha log "
                        "except: %s" % str(exception))
            util.setPyDetailMsg(devObj, "failed.collect.information")
            return (False, "")

    def getRemoteDhaPathAndName(self, devObj, rec):
        '''
        @summary: 获取远端DHA日志存放路径
        @param rec:  DHA导出命令回显
        @return:resultList:list[{name:文件名, path:文件完整路径}]
        '''
        
        resultList = []
        cliRetList = rec.splitlines()
        for line in cliRetList:
            #获取回显中的Package Path字段
            if config.COLLECT_INFO_PACKAGE_EXPORT_PATH_HEADER in line:
                tempDict = {}
                path = line.split(":")[1].replace(" ","")
                log.info(devObj, "DHA path:%s" % path)
                tempDict.setdefault("path", path)
                tempDict.setdefault("name", path.split("/")[-1])
                resultList.append(tempDict)
                
        return resultList
    
    def fetchDhaCommandArgs(self, devObj):
        '''
        @summary: 获取命令执行的必要参数
        @param devObj:  devObj：上下文对象
        @return: DHA信息收集的必要参数字典
        '''
        
        status, userName = util.getUsername(devObj)
        ipAddr = util.getDevIp(devObj)
        
        # 工具加载SVP IP时，export命令中的IP需要使用特定的IP（此IP为linux vm，上面才有支持export功能的ftp服务）。
        if self.isConnectedSVP(devObj):
            ipAddr = config.SVP_LINUX_VM_IP if not devObj.get(
                config.READ_ONLY_DHA_IP) else devObj.get(
                config.READ_ONLY_DHA_IP)
        
        loginPasswd = util.getLoginPasswd(devObj)
        presetProtocol = config.PROTOCOL_SFTP
        port = util.getPort(devObj)

        dhaCmdArgs = {"userName": userName, "ipAddr": ipAddr,
                      "loginPasswd": loginPasswd,
                      "presetProtocol": presetProtocol, "port": port}
        
        return dhaCmdArgs
        
    def getRetMsgName(self, oriName, loopTimes):
        '''
        @summary: 根据循环次数返回错误消息对应的key 
        @param arguments:   默认错误消息的key
        @param arguments:   循环次数
        @return: 错误消息的key
        '''
        if loopTimes > 0:
            return "partly." + oriName
        else:
            return oriName
        
    def processDhaCmd(self, devObj, arguments):
        '''
        @summary: 收集DHA信息 
        @param arguments: 必要的参数字典
        '''
        
        try:
            timeOut = config.EXPORT_INFO_TIMEOUT[config.COLLECT_TYPE_DHA]
            loopTimes = 0
            cmd = "export event event_type=disk_log ip={0} user={1} " \
                  "password={2} path=/home/permitdir/ protocol={3} port={4}"
            cmd = cmd.format(arguments.get("ipAddr"), arguments.get(
                "userName"), arguments.get("loginPasswd"),
                arguments.get("presetProtocol"), arguments.get("port", "22"))
            
            per_pro_step = 70.0 / config.COLLECT_DHA_MAX_LOOP_TIMES
            
            while loopTimes < config.COLLECT_DHA_MAX_LOOP_TIMES :
                #由于参数中包含密码，所以使用无日志记录的执行方式
                (isSuccess, echos) = cli.executeCmdNoLogTimeout(devObj, cmd)
                
                if not isSuccess:
                    log.error(devObj,"error occurred while processing export dha logs")
                    return False, loopTimes, self.getRetMsgName("dha.process.failure", loopTimes)
                    
                #命令交互
                confirm = 0
                while confirm < 2:
                    confirm += 1
                    if "(y/n)" in echos:
                        
                        (isSuccess, echos) = cli.executeCmdWithTimout(devObj, "y", timeOut) 
                        if not isSuccess:
                            return False, loopTimes, self.getRetMsgName("dha.process.failure", loopTimes)
                    
                #根据回显解析日志存放路径  
                listRemotePaths = self.getRemoteDhaPathAndName(devObj, echos)
                
                if not listRemotePaths:
                    log.error(devObj, "  No output file path detected! "
                              + StrUtils.filter_sensitive_information_in_logs(echos))
                    return False, loopTimes, self.getRetMsgName("dha.remote.export.file.failure", loopTimes)
                
                #标记是否需要继续循环的标志位
                isLoopRequired = False   
                log.info(devObj, "  exported filePaths in device%s"%str(listRemotePaths))
                util.refresh_progress_by_add(devObj, per_pro_step)
                for listRemotePath in listRemotePaths:
                    filePath = listRemotePath.get("path")
                    fileName = listRemotePath.get("name")
                    
                    #下载日志文件
                    (isSuccess, result) = self.downloadSingleFile(devObj, filePath, fileName)
                    log.info(devObj, "  exported filePath to local-host:%s"%str(result))
                    
                    if isSuccess:
                        #下载成功后删除设备上的日志
                        self.deleteMiniSysFile(devObj, filePath)
                        
                        #将日志命名为时间相关的名称
                        (status, localDir, newFileFullName) = self.changeLocalDHAFileNameWithTimeArgs(devObj, fileName)
                        if not status:
                            return False, loopTimes, self.getRetMsgName("dha.local.alter.file.name.failure", loopTimes)
                        
                        #根据收回日志中的内容处理标志位。若不包含指定内容则表示所有日志已经收集完毕
                        if "DAE" in newFileFullName \
                                and util.is_X10(devObj):
                            isLoopRequired = self.tarContainsPartialFlagByDAE(
                                devObj, localDir, newFileFullName) \
                                or isLoopRequired
                        else:
                            isLoopRequired = self.tarballContainsPartialFlag(
                                devObj, localDir, newFileFullName) \
                                or isLoopRequired

                    else:
                        log.error(devObj,"DHA partition files not downLoaded correctly")
                        return isSuccess, loopTimes, self.getRetMsgName(result, loopTimes)
                        
                #如果不需要继续收集，直接返回
                if not isLoopRequired:
                    log.info(devObj, " The file has been fully exported, quit...")
                    return True, loopTimes, ""
                
                #需要继续收集则记录循环次数
                loopTimes += 1
                log.info(devObj, " The file has been partly exported, looping...")
            
            #达到收集次数上限
            log.info(devObj, " >>> the DHA file looping reached to max loop times, quit...")
            util.refreshProcess(devObj, 97)
            return True, loopTimes, "dha.export.reach.to.loop.limit.info"
        except:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            return False,"dha.process.failure"

    def downloadSingleFile(self, devObj, filePath, fileName):
        '''
        @summary: 下载文件
        @param 
        @return: status: boolean, localFilePath: {"path" : 本地目录, "name" : 文件名}
        '''
        
        try:
            sftp = util.getSftp(devObj)
            
            #根据收集类型确定本地保存路径
            localDir = util.getLocalInfoPathByType(devObj, config.COLLECT_TYPE_DHA)
            localPath = localDir + fileName
            localFile = File(localPath)
            
            #获取filePath指定的文件到file指定的路径
            sftp.getFile(filePath, localFile, None)
            localFilePath = {"path" : localDir, "name" : fileName}
            
            return (True, localFilePath)
        except:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            log.error(devObj, "down load %s fail" % str(filePath))
            return False,"sftp.get.file.faild"
    
    def deleteMiniSysFile(self, devObj, filePath):
        '''
        @summary: 日志成功收集后，应删除设备上的日志文件  
        @param filePath:  日志文件绝对路径
        @return: status (boolean)
        '''
        
        try:
            log.info(devObj, "start to remove remote file using SFTP client")
            sftpAdmin= devObj.get("SFTP")
            status = sftpAdmin.deleteFile(filePath)
            log.info(devObj, "done deleting files")
            
            return status,""
        except:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            log.error(devObj, "failed processing SFTP remove opt ")
            return False, "dha.sftp.remote.delete.file.failure"

    def changeLocalDHAFileNameWithTimeArgs(self, devObj, oriFileName):
        '''
        @summary: 每次收集回来的部分是同名文件，需要重命名为时间相关的文件名
        @param oriFileName:  包含文件名的日志文件绝对路径
        @return: status (boolean), 本地路径 , 新文件名
        '''
        
        localDir = util.getLocalInfoPathByType(devObj, config.COLLECT_TYPE_DHA)
        
        #每个日志包都应重命名，本地应不存在相同名字的文件
        if not os.path.exists(os.path.join(localDir,oriFileName)):
            return (False, localDir, "")
        
        log.info(devObj,"  local DHA output file dir:%s" % localDir)
        
        #重命名为时间相关文件名
        newFileFullName = self.changeFileNameWithTimeStamp(devObj, localDir, oriFileName)
        
        return (True, localDir, newFileFullName)

    @staticmethod
    def _check_tar_name_is_match(tar):
        formatter = re.compile('[\\w\\\\/]+\\d+(\\.log)?\\.tar\\.bz2')
        return formatter.match(str(tar.name))

    def tarballContainsPartialFlag(self, devObj, localDir, newFileFullName):
        '''
        @summary: 判断收集到的结果中是否包含DHA日志 
        @param localDir: 本地路径
        @param newFileFullName:  重命名后的文件名
        @return: boolean flag (boolean)
        '''
        try:
            tar_file = os.path.join(localDir, newFileFullName)
            exPath = os.path.join(localDir, newFileFullName).replace(".tgz", "")
            # 解压全部文件
            decompress_tar_all_file(tar_file, exPath)
            # 判断是否存在指定规则的文件
            flag = check_tar_special_file(tar_file, self._check_tar_name_is_match)
            
            #将无用文件删除后重新写入压缩
            tarFile = \
                tarfile.open(os.path.join(localDir, newFileFullName), 'w:gz')

            if util.is_X10(devObj):
                tarFile.add(exPath, arcname="download")
            else:
                tarFile.add(exPath + os.path.sep + "download", arcname="download")

            #删除解压的临时文件
            log.info(devObj, "temp file path = %s" % exPath)
            if os.path.exists(exPath):
                add_permission(exPath)
                shutil.rmtree(exPath, True)
                log.info(devObj, "del temp file success")
                
            log.info(devObj, " found DHA logfile?: no" )
            tarFile.close()
            
            return flag
        except:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            log.error(devObj, "failed open tar ball files : the file might be empty")
            return False

    def tarContainsPartialFlagByDAE(
            self, devObj, localDir, newFileFullName):
        '''
        @summary: 判断收集到的结果中是否包含DHA日志
        @param localDir: 本地路径
        @param newFileFullName:  重命名后的文件名
        @return: boolean flag (boolean)
        '''
        try:
            flag = False
            tarFile = tarfile.open(
                os.path.join(localDir, newFileFullName), mode="r:gz")
            exPath = \
                os.path.join(localDir, newFileFullName).replace(".tgz", "")
            tarFile.extractall(exPath)
            for innerTarFile in os.listdir(exPath):
                if ".tgz" in innerTarFile:
                    flag = self.tarballContainsPartialFlag(
                        devObj, exPath, innerTarFile) or flag
            tarFile = \
                tarfile.open(os.path.join(localDir, newFileFullName), 'w:gz')
            tarFile.add(exPath, arcname="DAE")

            return flag
        except Exception, e:
            log.error(devObj,
                      "except trace back:" + unicode(e))
            log.error(devObj, "except trace back:" +
                      str(traceback.format_exc()))
            log.error(devObj, "failed open tar ball DAE files : "
                              "the file might be empty")
        finally:
            try:
                # 删除解压的临时文件
                log.info(devObj, "DAE temp file path = %s" % exPath)
                if os.path.exists(exPath):
                    add_permission(exPath)
                    shutil.rmtree(exPath, True)
                    log.info(devObj, "del DAE temp file success")
                log.info(devObj, " found DAE DHA logfile?: no")
                tarFile.close()
            except Exception, e:
                log.error(devObj,
                          "except trace back:" + unicode(e))

        return False
    
    def changeFileNameWithTimeStamp(self, devObj, localDir, oriFileName):
        '''
        @summary: 将文件重命名为时间相关的文件名
        @param localDir: 本地路径
        @return: oriFileName: 源文件名
        '''
        
        try:
            #获取扩展名
            fileExtend = oriFileName.split(".")[-1]
            
            #删除最后一个后缀
            fileName = oriFileName.replace("." + fileExtend,"")
            log.info(devObj, "  current file name:%s"%str(fileName)) 
            
            #加上时间参数，并去除时间参数中包含的"."
            newFileName = fileName + str(time.time()).replace(".","")
            log.info(devObj, "  altered file name to:%s"%str(newFileName)) 
            
            #将后缀加回，完成新文件名获取
            newFileFullName = newFileName + "." + fileExtend
            
            #改名
            os.rename(localDir + oriFileName, localDir + newFileFullName)
            log.info(devObj, "  done alternation DHA log into new file name :"+ newFileFullName)
            
            return newFileFullName
        except:
            log.error(devObj, "except trace back:" + str(traceback.format_exc()))
            log.error(devObj, "DHA collection error:rename file failed")
            return ""
        
        
    def isConnectedSVP(self, devObj):
        """
        @summary: 判断工具连接的是否为SVP:若是18000设备，且工具加载的IP不是阵列的管理端口IP，则默认其为SVP的IP。
        """
        devType = str(device.getDeviceType(devObj)).strip()
        
        # 非18000设备没有SVP
        flag, high_end_flag = util.is18000Series(devObj)
        log.info(devObj, "is 18000 series:%s." % high_end_flag)
        if not high_end_flag:
            return False
        # 通过svp作代理连接的控制器
        if devObj.get("devNode").canEnterDiagnose():
            log.info(devObj, 'can enter diagnose')
            return False
        # 工具加载的IP
        connectedIp = util.getDevIp(devObj)
        
        cmd = "show port general logic_type=Management_Port"
        isSucc, echos = cli.executeCmd(devObj, cmd)
        if not isSucc:
            log.info(devObj, "get management port failed when get dha info.")
            # 无法判断时默认其为SVP
            return True
        
        # SVP ip不是阵列的管理端口
        if connectedIp in echos:
            log.info(devObj, "The ip tool connected is management ip, so it is not svp ip.")
            log.info(devObj, "The device is 18000, and tool connected array.")
            return False
        
        log.info(devObj, "The device is 18000, and tool connected svp.")
        return True

    def export_dha_with_show_file(self, devObj):
        """
        V5R7C61及之后信息收集
        :param devObj:
        :return:
        """
        loop_times = 0
        down_load_list = []
        try:
            process_step = 86.0 / config.COLLECT_DHA_MAX_LOOP_TIMES
            while loop_times < config.COLLECT_DHA_MAX_LOOP_TIMES:
                log.info(devObj, "dha collect with show file")
                is_succ = exportInfo.export_meg_info(
                    devObj,
                    config.COLLECT_TYPE_DHA_LOG,
                    process_step=process_step
                )
                is_exist_msg = util.isExistDetailMsg(devObj)
                if is_succ is False and is_exist_msg is False:
                        util.setPyDetailMsg(devObj,
                                            "failed.collect.information")
                if is_succ is False:
                    return is_succ, loop_times, ""
                local_dir = \
                    util.getLocalInfoPathByType(devObj,
                                                config.COLLECT_TYPE_DHA)
                flag, meg, is_loop_required = \
                    self.check_result_catalog(devObj, local_dir,
                                              down_load_list, loop_times)
                if not flag:
                    return flag, loop_times, meg
                # 如果不需要继续收集，直接返回
                if not is_loop_required:
                    log.info(devObj, " The file has been fully exported, quit...")
                    return True, loop_times, ""
                loop_times += 1
                log.info(devObj, " The file has been partly exported, looping...")
            # 达到收集次数上限
            log.info(devObj, " >>> the DHA file looping reached to max loop times, quit...")
            util.refreshProcess(devObj, 97)
            return True, loop_times, "dha.export.reach.to.loop.limit.info"
        except (Exception, ToolException) as exception:
            log.error(
                devObj, "collect dha log "
                        "except: %s" % str(exception))
            log.error(devObj,
                      "except trace back:" + str(traceback.format_exc()))
            return (False, loop_times, "dha.process.failure")

    def is_digital_ver(self, version):
        """
        判断是否为数字版本号
        :param version: 版本号
        :return: True：数字版本号，False：非数字版本号
        """
        if re.compile(r"\d+\.\d+.*").match(version):
            return True
        return False

    def check_result_catalog(self, dev_obj, local_dir, down_load_list,
                             loop_times):
        """
         遍历结果目录，对下载到的包进行查看是否需要进行处理
        :param dev_obj: dev对象
        :param local_dir:  本地目录
        :param down_load_list:  下载目录
        :param loop_times: 循环次数
        :return: 是否失败，循环次数， 错误信息
        """

        # 标记是否需要继续循环的标志位
        is_loop_required = False
        # 遍历结果目录，对下载到的包进行查看是否需要进行处理
        for file_name in os.listdir(local_dir):
            if file_name in down_load_list:
                continue
            (status, local_dir,
             new_file_full_name) = \
                self.changeLocalDHAFileNameWithTimeArgs(
                    dev_obj, file_name)
            down_load_list.append(new_file_full_name)
            if not status:
                return False, self.getRetMsgName(
                    "dha.local.alter.file.name.failure", loop_times), is_loop_required
                # 根据收回日志中的内容处理标志位。若不包含指定内容则表示所有日志已经收集完毕
            if "DAE" in new_file_full_name \
                    and util.is_X10(dev_obj):
                is_loop_required = self.tarContainsPartialFlagByDAE(
                    dev_obj, local_dir, new_file_full_name) \
                                   or is_loop_required
            else:
                is_loop_required = self.tarballContainsPartialFlag(
                    dev_obj, local_dir, new_file_full_name) \
                                   or is_loop_required
        return True, "", is_loop_required
