# -*- coding: UTF-8 -*-
from java.io import File
from com.huawei.ism.tool.obase.exception import ToolException
from com.huawei.ism.tool.obase.connection import SshConnection
import os
import re
import sys
import resource
import time
from util import util

class CommonUtils():
    
    '''
    @summary: python common utils : this is a class , NEW one before using it! 
    # 
    '''
    
    def __init__(self, devObject, logger):
        '''
        @summary: initial parameters, devObj & log instance is required!
        '''
        
        self.devObj = devObject
        self.log = logger
    
    def  initPyDetailMsg(self):
        '''
        @summary: set py_detail to vacant
        '''
        self.devObj["py_detail"] = ""
        
        
    def checkSystemNormal(self):
        '''
        @summary: check whether this device is in admin normal mode
        @return: True or False, "error type"
        '''
        cmd = "show system general" 
        isCmdSuccess, cmdEcho = self.execCliWithTimeout(cmd)
        if isCmdSuccess and re.search("System Name", cmdEcho, re.I):
            if not "upgrade:/>" in cmdEcho:
                return True, ""
            else :
                return False, "system.isUpgrading"
            
        return False, "dev.conn.failure"
    
    def getEnginNum(self):
        '''
        @summary: get device engine counts
        @return: status(boolean), count(int)
        '''
        cmd = 'show enclosure |filterRow column=Logic\sType predict=equal_to value=Engine'
        isCmdSuccess, cmdEcho = self.execCliWithTimeout(cmd)
        if isCmdSuccess:
            engNum = 0
            for line in cmdEcho.splitlines()[1:] :
                fields = line.strip().split()
                if line and len(fields) > 2 and re.search("ENGINE", fields[1].strip().upper(), re.I):
                    engNum += 1
            return True, engNum
        else:
            self.log.error(self.devObj, "execute show enclosure failed.")
            return False, 0
            
    def isLocalDirEmpty(self, localDir):
        '''
        @summary: judge whether the input dir locally is empty
        @param param:  localdir local directory
        @return: boolean 
        '''
        contents = os.listdir(localDir)
        if not contents:
            return True
        else:
            emptyCnt = 0
            for f in contents:
                fn = os.path.join(localDir, f)
                if os.path.isfile(fn):
                    return False
                else:
                    if util.isEmptyDir(fn):
                        emptyCnt += 1
                    else:
                        return False
            return len(contents) == emptyCnt
    
    
    def rmRemoteFiles(self, remoteDir):
        '''
        @summary: delete remote dirtctory using SFTP and SSH
        @param remoteDir: remote directory ,force remove ;
        '''
        sftp = self.devObj.get("SFTP")
        try:
            sftp.deleteDir(remoteDir)
        except ToolException, te:
            self.log.error("Delete directory by SFTP exception:" + unicode(te))
            try:
                cmd = 'rm -rf ' + remoteDir
                isExeSucc, cmdEcho = self.execCliWithTimeout(cmd)
                for _ in range(5) :
                    if isExeSucc and 'y/n' in cmdEcho:
                        isExeSucc, cmdEcho = self.execCliWithTimeout("y")
            except Exception, e:
                self.log.error('Delete directory by CLI exception, directory:' + unicode(remoteDir) + unicode(e))
        
        
                 
    def  execCliWithTimeout(self, cmd, timeout = 3*60):
        '''
        @summary: proceed CMD with default SSH connection timeout
        @param cmd:str timeout default 180s
        @return: isSuccess boolean, command echo  str
        '''
        ssh = self.devObj.get("SSH")
        retries = 0
        while retries < 3:
            try:
                cmdEcho = ssh.execCmdWithTimout(cmd, timeout)
                return True, cmdEcho
            except:
                self.log.error( "execute cmd failed, retrying ...")
                retries += 1
                time.sleep(5*retries)
                continue
        
        return False, ''

    def execCliTimeoutNoLog(self, cmd, timeout = 3*60):
        '''
        @summary: proceed CMD with  timeout ,no log debug echos
        @param cmd:str timeout default 180s
        @return: issuccess boolean, command echo  str
        '''
        ssh = self.devObj.get("SSH")
        retries = 0
        while retries < 3:
            try:
                cmdEcho = ssh.execCmdNoLogTimout(cmd, timeout)
                return True, cmdEcho
            except:
                self.log.error( "execute cmd failed, retrying ...")
                retries += 1
                time.sleep(5*retries)
                continue
        
        return False, ''

    def execCliTimeoutGivenSsh(self, ssh, cmd, timeout = 3*60):
        '''
        @summary: execute command with given ssh connection
        '''

        if (ssh is None) or (not isinstance(ssh, SshConnection)):
            raise Exception, "failed.build.ssh.connection.on.ctrlr"
        retries = 0
        while retries < 3:
            try:
                cmdEcho = ssh.execCmdWithTimout(cmd, timeout)
                return True, cmdEcho
            except:
                retries += 1
                self.log.error( "execute cmd on given ssh failed, retrying ...")
                time.sleep(5*retries)
                continue
        
        return False, ''
        
        
    def execCliTimeoutGivenSshNoLog(self, ssh, cmd, timeout = 3*60):
        '''
        @summary: execute command with given ssh connection while no log shall be printed
        '''
        if (ssh is None) or (not isinstance(ssh, SshConnection)):
            raise Exception, "failed.build.ssh.connection.on.ctrlr"
        retries = 0
        while retries < 3:
            try:
                cmdEcho = ssh.execCmdNoLogTimout(cmd, timeout)
                return True, cmdEcho
            except:
                retries += 1
                self.log.error( "execute cmd on given ssh failed, retrying ...")
                time.sleep(5*retries)
                continue
        
        return False, ''
        
    def enterMiniSystemModOnHost(self, ssh, ip ):
        '''
        @summary: enter mini-system on pointed SSH connection
        @raise exception 
        '''
        try:
            if self.enterDeveloperModeOnHost(ssh, ip):
                isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, "minisystem")
                if  cmdEcho.strip().endswith('minisystem>'):
                    self.log.info( "minisystem successfully entered.")
                    return True
                else:
                    self.developerMode2CliModeOnHost(ssh, ip)
                    self.log.error( "failed entering minisystem mode.")
                    raise Exception , 'failed.enter.pointed.ctlr.minisystem' + "||" + str(ip)
            
            else:
                raise  Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
                
        except Exception , e:
            raise Exception , e

    def tryExitMiniSystemOnHost(self, ssh , ip):
        '''
        @summary:  try to exit mini-system mode , no exception shall be raised
        '''
        try:
            self.log.info("checking whether system is in mini-system mode...")
            isInMiniSystem = self.isInMiniSystemModeOnHost(ssh)
            if not isInMiniSystem:
                return 
            
            isCmdSuccess, cmdEcho =  self.execCliTimeoutGivenSsh(ssh, "exit")
            if not isCmdSuccess:
                self.setPyDetailMsg("exit.minisystem.failed.on.ctlr", str(ip))
                return
            
            retries = 3
            while retries > 0:
                if isCmdSuccess and  "(y/n)" in cmdEcho:
                    isCmdSuccess, cmdEcho =  self.execCliTimeoutGivenSsh(ssh, "y")    
                retries -= 1
                
            self.developerMode2CliModeOnHost(ssh, ip)
            return 
        except Exception, e:
            self.log.info("try quit mini-system mode but failed.%s"%str(e))
    
    def isInMiniSystemModeOnHost(self, ssh ):
        '''
        @summary : check whether now in minisystem mode 
        @return: true if in mini-system mode
        '''
        cmd = "ls"
        isCmdSuccess, cmdEcho =  self.execCliTimeoutGivenSsh(ssh, cmd)
        if not isCmdSuccess:
            self.log.error("cannot figure out whether system is now in mini-system mode!")
            return False
        if "minisystem>" in cmdEcho.strip().splitlines()[-1]:
            self.log.info("system is now in mini-system mode")
            return True
        return False

    def isInDeveloperModeOnHost(self, ssh):
        '''
        @summary: check whether the system is in developer mode 
        @return: True if so else False
        '''             
        cmd = "show system general"
        isCmdSuccess, cmdEcho =self.execCliTimeoutGivenSsh(ssh, cmd)
        if not isCmdSuccess:
            self.log.error("cannot proceed , checking whether in developer mode failed ")
            return False
        
        if "developer:/>" in cmdEcho.lower():
            return True
        
        return False
        
        
    def enterDiagnoseModeOnHost(self, ssh, ip):
        '''
        @summary: enter diagnose mode on specified host from admin-mode
        @raise exception 
        '''
        isSuccess = self.enterDeveloperModeOnHost(ssh, ip)
        if isSuccess and self.isInDeveloperModeOnHost(ssh):
            cmd = "debug"
            isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
            if isSuccess and cmdEcho.strip().endswith("diagnose>"):
                return True
            else:
                self.log.error("failed in entering into diagnose mode on ctrlr!")
                raise Exception, "failed.enter.pointed.ctlr.diagnose" + "||" + ip
            
        else:
            self.log.error("exception  occurred while entering into diagnose mode on ctrlr!")
            raise Exception , "failed.enter.pointed.ctlr.diagnose" + "||" + ip
    
    def tryExitDiagnoseModeOnHost(self, ssh, ip):
        '''
        @summary:  try to exit diagnose mode , no exception will be raised
        '''
        cmd = "ls"
        isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
        if cmdEcho and "diagnose>" in cmdEcho.lower():
            cmd = "exit"
            isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
            retries = 3
            while retries >= 0:
                retries -= 1
                if isSuccess and "(y/n)" in cmdEcho.lower() :
                    cmd = "y"
                    isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
                if isSuccess and "developer:/>" in cmdEcho.lower():
                    break
            if self.isInDeveloperModeOnHost(ssh):
                self.developerMode2CliModeOnHost(ssh, ip)
                
                return True
            else:
                self.addPyDetailMsg("exit.diagnose.minisystem.failed")
                return False
        else:
            self.addPyDetailMsg("exit.diagnose.minisystem.failed")
            return False
            
    def developerMode2CliModeOnHost(self, ssh , ip):
        '''
        @summary: developer模式下退回到cli模式
        @param cliRet: devObj = 上下文对象
        '''        
        cmd = "show system general"
        isSuccess, cliRet =  self.execCliTimeoutGivenSsh(ssh, cmd)
        if not isSuccess:
            self.setPyDetailMsg("failed.build.ssh.connection.on.ctrlr", str(ip))
            return

        if ("Password" in cliRet) or ("developer:/>" in cliRet):
                    
            isSuccess, cliRet =  self.execCliTimeoutGivenSsh(ssh, "exit")
            if not isSuccess:
                self.setPyDetailMsg("failed.build.ssh.connection.on.ctrlr", str(ip))
                return
            
            cliRet =  self.execCliTimeoutGivenSsh(ssh, "exit")
            index = 0
            while ("admin:/>" not in cliRet):
                index = index + 1
                (isSuccess, cliRet) =  self.execCliTimeoutGivenSsh(ssh, "exit")
                if not isSuccess:
                    self.setPyDetailMsg("failed.build.ssh.connection.on.ctrlr", str(ip))
                    return
                
                if "Are you sure to exit?(y/n):" in cliRet:
                    (isSuccess, cliRet) =  self.execCliTimeoutGivenSsh(ssh, "n")
                    if not isSuccess:
                        self.setPyDetailMsg("failed.build.ssh.connection.on.ctrlr", str(ip))
                        return
                    
                if index > 5:
                    break


    def enterDeveloperModeOnHost(self, ssh, ip):
        '''
        @summary: enter developer mode using set SSH connection
        @raise exception 
        '''
        cmd = "change user_mode current_mode user_mode=developer" 
        try:
            isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
            
            if re.search("developer:", cmdEcho, re.I):
                return True
            for times in range(3):
                if re.search("(y/n)", cmdEcho, re.I):
                    isSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, "y")
                    
            if re.search("developer:/>", cmdEcho, re.I):
                return True
            # after all things done , if we are still not in developer mode 
            #and the echos is not requiring for password,throw exception
            if not re.search("Password:", cmdEcho, re.I):
                raise Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
            
            developerPwd = self.devObj.get("developer", "")
            if None == developerPwd or "" == developerPwd :
                raise  Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
            
            isSuccess, cmdEcho =  self.execCliTimeoutGivenSshNoLog(ssh, developerPwd)            
            del developerPwd
            if not re.search("developer:", cmdEcho, re.I):
                raise  Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
            
            else:
                return True
        except:
            raise  Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
        
                   
    def setCollectAllInfo(self, flag):
        '''
        @summary: set collectAllInfo as flag 
        '''
        self.devObj["collectAllInfo"] = flag
        
    def getMsg(self, msg, args= ""):
        
        if  None == msg or "" == msg:
            return ""
        lang = self.devObj.get("lang")
        if resource.MESSAGES_DICT.has_key(msg):
            msgDict = resource.MESSAGES_DICT.get(msg)
            msg = msgDict[lang]
        if "" != args:
            msg = msg % args
        
        return msg
            
    def addPyDetailMsg(self, msg, args = ""):
        '''
        @summary: append  py_detail object
        '''
        msg = self.getMsg(msg, args)
        oldMsg = self.devObj.get("py_detail", "")
        self.devObj["py_detail"] = (oldMsg + os.linesep + msg) if oldMsg else msg
        self.log.info("py_detail:"+self.devObj["py_detail"])
        
    def setPyDetailMsg(self, msg, args = ""):
        
        msg = self.getMsg(msg, args)
        
        self.devObj["py_detail"] = msg  
        
    
    def ping(self, ipV4):
        '''
        @summary: start ping process
        @param ipV4: IP address in V4 mode
        @return: isReachable : boolean
        '''        
        isParamValid = re.match("((2[0-4]\d|25[0-5]|[01]?\d\d?)\.){3}(2[0-4]\d|25[0-5]|[01]?\d\d?)", ipV4)
        if not isParamValid : 
            return False
        strCmd = "ping -c 5 " + ipV4
        isSuccess, cmdEcho = self.execCliWithTimeout(strCmd)
        if not isSuccess:
            return False
        
        if re.search("ttl", cmdEcho, re.I):
            return True
        
        return False
        
    def getControllerIpList(self):
        '''
        @summary: get controller ip list using SSH connection in devObj
        @return: status, controllerIpLists([] if qry failed)
        '''
        controllerIpLists = []
        cmd = "show upgrade package"
        isQrySuccess, echos = self.execCliWithTimeout(cmd)
        if not isQrySuccess or not echos :
            return False, None

        for line in echos.splitlines():
            
            if "Controller" in line :
                ctlrParams = re.split("[ ]+", line.strip())
                if len(ctlrParams) >3 and ctlrParams[2] not in controllerIpLists:
                    controllerIpLists.append(ctlrParams[2])
                
            elif "HotPatch" in line :
                break
        if not controllerIpLists:
            return False, None
        
        return True, controllerIpLists
    
    def getEnginCntrIpMap(self):
        '''
        @summary: get controller ip Map using SSH connection in devObj
        @return: status, controllerIpLists([] if qry failed)
        '''
        controllerIpMap = {}
        cmd = "show upgrade package"
        isQrySuccess, echos = self.execCliWithTimeout(cmd)
        if not isQrySuccess or not echos :
            return False, None
        echos = echos.lower()
        titlePoi = echos.find("software version")
        cutPoiStart = (titlePoi + len("software version")) if titlePoi > 0 else 0
        cutPoiEnd = echos.find("hotpatch version")
        echos = echos[cutPoiStart:cutPoiEnd]
        dictList = self.getHorizontalCliRet(echos)
        for lineDist in dictList:
            name = lineDist.get("name", "")
            ctrIp = lineDist.get("ip", "")
            if name and ctrIp:
                enginNum = ""
                for charAt in str(name).strip():
                    if charAt.isdigit():
                        enginNum += str(charAt)
                enginNum = str(enginNum).strip()
                curEngineCtrIps = controllerIpMap.get(enginNum, '')
                if not curEngineCtrIps:
                    curEngineCtrIps = []
                curEngineCtrIps.append(ctrIp)
                controllerIpMap[enginNum] = curEngineCtrIps
                self.log.info("current engine's controller ip list: %s" % str(controllerIpMap[enginNum]))
        if not controllerIpMap:
            return False, None

        return True, controllerIpMap
    def getControllerIpMap(self):
        '''
        @summary: get controller and ip map using SSH connection in devObj
        @return: status, controllerIpMap(None if qry failed)
        '''
        controllerIpMap = {}
        cmd = "show upgrade package"
        isQrySuccess, echos = self.execCliWithTimeout(cmd)
        if not isQrySuccess or not echos :
            return False, controllerIpMap
        for line in echos.splitlines():
            if "Controller" in line :
                ctlrParams = re.split("[ ]+", line.strip())
                if len(ctlrParams) > 3:
                    controllerIpMap[ctlrParams[1]] = ctlrParams[2]
            elif "HotPatch" in line :
                break
        if not controllerIpMap:
            return False, controllerIpMap
        return True, controllerIpMap
    def getCtrlIDByIP(self, ip):
        '''
        @summary: get controller and ip map using SSH connection in devObj
        @return: status, controllerIpMap(None if qry failed)
        '''
        flag, ctrlIPMap = self.getControllerIpMap()
        if not (flag and ctrlIPMap):
            return False, ""
        defaultCtrlID = ""
        for ctrlId, ctrlIp in ctrlIPMap.items():
            defaultCtrlID = ctrlId
            if ctrlIp == ip:
                return True, ctrlId
        return True, defaultCtrlID
    def getLoginIp(self, devObj):
        '''
        @summary: get ip in devObj
        @return: status, controllerIpMap(None if qry failed)
        '''
        devNode = self.devObj.get("devNode")
        return devNode.getIp()
    
    def getHorizontalCliRet(self, cliRet):
        '''
        @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
        @param cliRet: cli回显
        @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合, 处理不正常时，返回空集合
        '''
        self.log.info("parsing command echo ...")
        try:
            headline = ""
            i = 0
            cliRetList = cliRet.encode("utf8").splitlines()
            for line in cliRetList:
                reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
                match_headline = reg_headline.search(line)
                if match_headline:
                    headline = match_headline.group()
                    break
                i += 1
            
            if headline == "" or i == 0 or i >= len(cliRetList) - 1:
                self.log.info("mismatch echo format, return ")
                return []
            
            title = cliRetList[i - 1]
            self.log.info("title:" + str(title))
            field_words = cliRetList[(i + 1):]
            self.log.info("field_words:" + str(field_words))
            reg_split = re.compile("\s*-+\s*")
            tuple_idxs = []
            start_pos = 0
            end_pos = 0
            while (start_pos <= len(headline)):
                match = reg_split.search(headline, start_pos)
                if match:
                    end_pos = match.end()
                    tuple_idxs.append((start_pos, end_pos))
                    start_pos = end_pos
                else:
                    break
                
            keys = []
            for item in tuple_idxs:
                key = title[item[0]:item[1]].strip()
                if keys.count(key):
                    key += "_" + str(str(keys).count(key + "_") + 1)
                keys.append(key.decode("utf8"))
            
            requiredLineLen = tuple_idxs[-1][0]
            dictList = []
            for line in field_words:
                if ":/>" in line:
                    break
                
                if re.search("^-+(\s+-+)*\s*$", line):
                    continue
                
                if len(line.strip()) == 0:
                    continue
                
                if len(line) <= requiredLineLen:
                    continue
                
                vals = []
                for item in tuple_idxs:
                    vals.append(line[item[0]:item[1]].strip().decode("utf8"))
                dictList.append(dict(zip(keys, vals)))
            self.log.info("done parsing CLI echo...")
            return dictList
        except Exception, e:
            self.log.error("exception occurred :%s"%str(e))
            return []

    
        
    def getLocalInfoPathByType(self, subDir):
        '''
        @summary: get local infocollect checkpoint absolute dir 
        @param subdir, the infocollect sub-function dir
        @return: localDir that is absolute 
        '''
        
        localDir = self.devObj["collect_info_local_path"]
        self.log.info("collect_info_local_path:%s" % self.devObj["collect_info_local_path"])
        if not os.path.exists(localDir):
            os.makedirs(localDir)
     
        localDir = os.path.join(localDir, subDir)
        self.log.info("localDir%s" % localDir)
        if not os.path.exists(localDir):
            os.makedirs(localDir)  
        
        return localDir

    def getDeviceVersion(self):
        '''
         @summary: device version , eg  V200R002C11
        ''' 
        devNode = self.devObj.get("devNode")
        if None == devNode:
            return ""
        
        version = devNode.getProductVersion()
        return version
           
           
     
    def getFreeMemInMinisystem(self):
        '''
        @summary: get free memory info under  minisystem mode
        @raise exception 
        '''
        try:
            cmd = "free -m"
            isCmdSucc, cmdEcho = self.execCliWithTimeout(cmd)
            if not isCmdSucc:
                raise Exception, "query.free.memory.of.current.ctrl.failed"
            for memLine in cmdEcho.splitlines():
                if 'Mem:' in memLine:
                    return int(memLine.split()[3].strip())
        except:
            raise Exception, "query.free.memory.of.current.ctrl.failed"


    def checkPointedSystemAvailable(self, ssh, ip):
        '''
        @summary: check whether the pointed system is available
        @raise exception 
        @return: status,""
        '''
        cmd = "show system general"
        isCmdSuccess, cmdEcho = self.execCliTimeoutGivenSsh(ssh, cmd)
        if isCmdSuccess and re.search("System Name", cmdEcho, re.I):
            if not "upgrade:/>" in cmdEcho:
                return True, ""
            else :
                raise Exception, "system.isUpgrading"
            
        raise Exception, "failed.build.ssh.connection.on.ctrlr" + "||" + str(ip)
    
        
class Log():
    '''
    @summary: log class
    '''
    
    def __init__(self, devObject):    
        self.devObj = devObject

    def getCallerStack(self, MAX_CALLER_LEVEL = 5):
        '''
        @summary: get caller funtion name and line from system context
        '''
        funcBack = sys._getframe().f_back

        callerInfo = ""    
        for index in range(MAX_CALLER_LEVEL):
        
            if hasattr(funcBack, "f_code") and hasattr(funcBack, "f_lineno"):
                funcName = funcBack.f_code.co_name
                lineNumber = funcBack.f_lineno
                callerInfo = " [" + str(funcName) + ":" +  str(lineNumber) + "]" + callerInfo
            else:
                break
            if hasattr(funcBack, "f_back"):
                funcBack = funcBack.f_back
            else:
                break
        
        return callerInfo
            
    def info(self, info):
        '''
        @summary: logging info class echos using java logger environment
        '''
        logInfo = info + self.getCallerStack()
        if "logger" in self.devObj:
            self.devObj.get("logger").info('[ToolLog]:' + logInfo)
        else:
            raise Exception("[failed]: logger does not exist! info=" + logInfo)
        
    def warn(self, info):
        '''
        @summary: logging warn class echos using java logger environment
        '''
        logInfo = info + self.getCallerStack()
        if "logger" in self.devObj:
            self.devObj.get("logger").warn('[ToolLog]:' + logInfo)
        else:
            raise Exception("[failed]: logger does not exist! info=" + logInfo)

    def error(self, info):
        '''
        @summary: logging error class echos using java logger environment
        '''
        logInfo = info + self.getCallerStack()
        if "logger" in self.devObj:
            self.devObj.get("logger").error('[ToolLog]:' + logInfo)
        else:
            raise Exception("[failed]: logger does not exist! info=" + logInfo)

class MsgInfo():
    def __init__(self, logger, commonUtils, disklogConfig):
        self.util = commonUtils
        self.log = logger
        self.disklogConf = disklogConfig
        
    def writeMsgLine(self, msgLine, msgFile=None):
        if not msgLine:
            return
        if msgFile is None:
            msgFile = self.disklogConf.DISK_LOG_COLLECT_RESULT_FILE_NAME
        disklogResultFile = os.path.join(self.util.getLocalInfoPathByType("Disklog"), msgFile)
        
        try:
            f = open(disklogResultFile, 'a+')
            f.write(msgLine + '\n')
        except Exception, e:
            self.log.error('Write disk log result file exception:' + unicode(e))
        else:
            self.log.info('Write disk log result file success.')
        finally:
            try:
                f.close()
            except Exception, e:
                self.log.error('Close disk log result file exception:' + unicode(e))
                
    def writeMsgLines(self, msgLines, msgFile=None):
        if not msgLines:
            return
        
        if msgFile is None:
            msgFile = self.disklogConf.DISK_LOG_COLLECT_RESULT_FILE_NAME
        disklogResultFile = (self.util.getLocalInfoPathByType("Disklog"), msgFile)
        
        try:
            f = open(disklogResultFile, 'a+')
            for msgLine in msgLines:
                f.write(msgLine + '\n')
        except Exception, e:
            self.log.error( 'Write disk log result file exception:' + unicode(e))
        else:
            self.log.info( 'Write disk log result file success.')
        finally:
            try:
                f.close()
            except Exception, e:
                self.log.error( 'Close disk log result file exception:' + unicode(e))


class SpaceNotEnoughException(Exception):
    def __init__(self, msg):
        self.msg = msg
