﻿# -*- coding: UTF-8 -*-
from java.io import File
from common.util import util, log
from common.cliFactory import cli
from common.config import config
import traceback
import time
import os


def collect(devObj):
    """
    @summary: 确保客户业务连续性，需要有故障快速定界手段，快速获取故障定界信息，支撑三板斧手段的使用。
            因此，除了收集日志之外，需要新增一种故障信息快速获取，快速决策恢复方案的机制，即三板斧快速恢复。
            当接到客户紧急问题需要处理的诉求后，首先5分钟内需要采集三板斧定界信息，即通过一键诊断命令，将系统中硬件、业务、内部资源、内部任务、数据损坏这5类关键信息采集出来，
            生成诊断信息文件，并完成下载到本地服务器。
    """
    util.refreshProcess(devObj, 1)
    util.initPyDetailMsg(devObj)
    
    # 检查系统状态
    isNormal = util.checkSystemNormal(devObj)
    util.refreshProcess(devObj, 3)
    if not isNormal:
        return False
    
    # 检查用户权限：管理员/超级管理员
    isMeet = util.checkUserPrivilege(devObj)
    util.refreshProcess(devObj, 5)
    if not isMeet:
        return False
    
    # 通知导出系统诊断文件
    isSucc = notifyArray(devObj)
    util.refreshProcess(devObj, 10)
    if not isSucc:
        return False
    
    # 查看诊断文件的导出状态。
    isSucc = checkExportingStatus(devObj)
    util.refreshProcess(devObj, 85)
    if not isSucc:
        return False
    
    # 查看导出的诊断文件在阵列上的存放路径。
    filePath = getFilePath(devObj)
    util.refreshProcess(devObj, 90)
    if not isSucc:
        return False
    
    # 下载阵列上的诊断文件到本地
    isSucc = downLoadFile(devObj, filePath)
    util.refreshProcess(devObj, 99)
    
    # 删除阵列上的临时诊断文件
    deleteFile(devObj)
    
    util.refreshProcess(devObj, 100)        
    return isSucc
    

def notifyArray(devObj):
    """
    @summary: Send a notification of exporting the system diagnostic file and return a message about whether the export is successful.
    """
    cmd = "show file export_path file_type=diagnostic_file"
    timeout = config.PARALLEL_EXPORT_INFO_TIMEOUT.get(config.COLLECT_TYPE_DIAGNOSTIC)
    isSucc, rec = cli.executeCmdWithTimout(devObj, cmd, timeout * 60)
    if not isSucc:
        util.setPyDetailMsg(devObj, "dev.conn.failure")
        return False
    
    if config.CLI_EXECUTE_CMD_SUCCESS not in rec: 
        util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
        return False
    
    return True

def checkExportingStatus(devObj):
    """
    @summary: Check the diagnostic file exporting status 
    """
    cmd = "show file package_result file_type=diagnostic_file"
    process = 10
    timeout = config.PARALLEL_EXPORT_INFO_TIMEOUT.get(config.COLLECT_TYPE_DIAGNOSTIC)
    checkTimes = timeout * 60 / 30
    for times in range(0, checkTimes):
        isSucc, rec = cli.executeCmdWithTimout(devObj, cmd, 30)
        
        if process < 85:
            process += 10
        util.refreshProcess(devObj, process)
        
        if not isSucc:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        statusDictList = cli.getCliTable2DictList(rec)
        if not statusDictList:
            util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
            return False
    
        status = statusDictList[0].get("Total Result")
        if status == "Packaging":
            time.sleep(30)
            continue
        
        if status == "Successful":
            return True
        else:
            util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
            return False
    
    # 收集超时（5min）   
    log.info(devObj, "Get diagnostic file failed, time over!")
    util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
    return False

def getFilePath(devObj):
    """
    @summary: Check the path of the exported diagnostic file in the storage system 
    """
    cmd = "show file notification file_type=diagnostic_file"
    timeout = config.PARALLEL_EXPORT_INFO_TIMEOUT.get(config.COLLECT_TYPE_DIAGNOSTIC)
    isSucc, rec = cli.executeCmdWithTimout(devObj, cmd, timeout * 60)
    if not isSucc:
        util.setPyDetailMsg(devObj, "dev.conn.failure")
        return ""
    
    if config.COLLECT_INFO_RECROD_STANDARD not in rec: 
        util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
        return ""
    
    for line in rec.splitlines():
        if config.COLLECT_INFO_RECROD_STANDARD in line:
            filePath = line.split(":")[1].strip()
            return filePath
    
    return ""


def downLoadFile(devObj, filePath):
    """
    @summary: 下载阵列诊断文件到本地
    """
    sftp = util.getSftp(devObj)
    type = config.COLLECT_TYPE_DIAGNOSTIC
    localDir = util.getLocalInfoPathByType(devObj, type)
    fileName = os.path.basename(filePath)
    localPath = localDir + fileName
    try:
        log.info(devObj, "diagnostic file path is %s." % filePath)
        log.info(devObj, "diagnostic file is putted %s." % localPath)
        localFile = File(localPath)
        sftp.getFile(filePath, localFile, None)
        return True
    except:
        util.setPyDetailMsg(devObj, "downLoad.file.failure")
        log.error(devObj, "download diagnostic file failed.")
        log.error(devObj, "except trace back:" + str(traceback.format_exc()))
        return False
    
def deleteFile(devObj):
    """
    @summary: 删除阵列上的临时诊断文件
    """
    cmd = "delete file filetype=diagnostic_file"
    isSucc, rec = cli.executeCmdWithTimout(devObj, cmd, 30)
    if not isSucc:
        log.warn(devObj, "delete diagnostic file failed.")