# -*- coding: UTF-8 -*-
import traceback
from common.util import device
from common.util import util
from common.util import log
from common.cliFactory import cli
from common.config import config
import os.path
import tempfile
import time
from java.io import File
from com.huawei.ism.tool.obase.exception import ToolException

'''
导出系统信息（系统日志），公共方法
'''
class exportInfo():
    
    @staticmethod
    def isNormalForRec(devObj, rec):
        '''
        @summary: 判断回显信息是否正常
        @param cliRet:  devObj=上下文对象,
                        rec=导出系统信息的回显
        @return: True or False
        '''
        if cli.CLI_NOT_EXIST_FLAG in rec:
            util.setPyDetailMsg(devObj, rec)
            return False
        
        #判断系统是否在小系统模式下
        if cli.CLI_RET_END_FLAG not in rec:
            util.setPyDetailMsg(devObj, rec)
            return False
        
        #判断系统是否繁忙
        if cli.CLI_SYSTEM_BUSY_FLAG in rec:
            log.info(devObj, "Collect error, result is:" + rec)
            util.setPyDetailMsg(devObj, "system.busy")
            return False
        
        return True
    
    @staticmethod
    def isExecCmdSuccForRec(devObj, rec):
        '''
        @summary: 根据回显判断命令是否执行成功
        @param cliRet:  devObj=上下文对象,
                        rec=导出系统信息的回显
                        list=判断命令回显成功的回显列表
        @return 1: True or False,命令执行成功
        @return 2: True or False,收集的文件是否完整
        '''
        execSucc = False
        isAllSucc = False
        #首先判断命令执行是否正常
        isNormal = exportInfo.isNormalForRec(devObj, rec)
        if False == isNormal:
            return (execSucc, isAllSucc)
        
        for re in config.ISEXEC_CMD_SUCCESS_FLAG:
            if re in rec:
                execSucc = True
                break
                
        return (execSucc, isAllSucc)
    
    @staticmethod
    def getRemotePathAndName(devObj, fileList):
        '''
        @summary: 获取阵列端的文件名称和路径
        @param cliRet:  devObj=上下文对象
        @return: name:阵列端产生的文件名称
                 path:阵列端产生的文件的路径
        '''
        resultList = []
        
        for fileName in fileList:
            tempDict = {}
            tempDict["path"] = "/home/permitdir/log_controller_%s.tar.bz2" % (fileName)
            tempDict["name"] = "log_controller_%s.tar.bz2" % (fileName)
            resultList.append(tempDict)
        
        return resultList
    
    @staticmethod
    def isTarFileExist(devObj, fileDict):
        '''
        @summary: 判断要下载的文件是否存在
        @param cliRet:  devObj=上下文对象
        @return: fileName:阵列端产生的文件名称
        '''
        tempCmd = fileDict["path"].split("/log")[0]
        tempFile = fileDict["name"]
        tempCmd = "cd " + tempCmd
        
        (isSuccessCd, _) = cli.executeCmdNoLogTimeout(devObj, tempCmd, 60)
        if not isSuccessCd:
            return False
        (isSuccessLs, recLs) = cli.executeCmdNoLogTimeout(devObj, "ls", 60)
        if not isSuccessLs:
            return False
        count = 0
        while True:
            count = count + 1
            recLsList = []
            if count > 10:
                return False
            (isSuccessLs, recLs) = cli.executeCmdNoLogTimeout(devObj, "ls", 60)
            if not isSuccessLs:
                return False
            
            recLsList = recLs.split()
            for temp in recLsList:
                if tempFile == temp:
                    log.info(devObj, "fileName %s is in path %s" % (tempFile, recLs))
                    return True
            else:
                time.sleep(6)
                
        return False
        
    @staticmethod
    def downloadFile(devObj, type, fileList):
        '''
        @summary: 下载收集到得文件信息到本地
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        fileList=下载文件 List
        @return: True or False
        '''
        try:
            sftp = util.getSftp(devObj)
            localDir = util.getLocalInfoPathByType(devObj, type)
            #下载信息到本地
            #获取阵列端存放信息的路径和文件名称
            listRemotePaths = exportInfo.getRemotePathAndName(devObj, fileList)
            
            #下载到本地的目录信息
            loadLocalDirInfo = []
            for temp in listRemotePaths:
                localPath = localDir + temp["name"].replace(':', '_')#supports IPv6
                file = File(localPath)
                log.info(devObj, "file = " + temp["path"])
                
                #判断文件是否存在
                isExistFlag = exportInfo.isTarFileExist(devObj, temp)
                if not isExistFlag:
                    log.error(devObj, "the file is not exist.fileName = %s" % str(temp["path"]))
                    return (False, None)
                sftp.getFile(temp["path"], file, None)
                tempDict = {"path":localDir, "name":temp["name"]}
                loadLocalDirInfo.append(tempDict)
            
            return (True, loadLocalDirInfo)
        except:
            log.error(devObj, "down load %s fail" % str(type))
            log.info(devObj, "except trace back:" + str(traceback.format_exc()))
            return (False, None)
    
    @staticmethod
    def deleteRemoteFile(devObj, type, rec):
        '''
        @summary: 删除阵列端生成的文件
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        rec=CLI回显信息
        @return:Bool(True or False)
        '''
        #根据CLI回显信息判断当前环境下，是否允许删除
                  
        tempList = config.DELETE_REMOTE_DATA_STANDARD
        deleteFlag = True
        for standard in tempList:
            if standard in rec:
                deleteFlag = False
                break
            
        if deleteFlag:
            strCmd = "delete file filetype=" + type
            log.info(devObj, "delete command is:" + strCmd)
            cli.executeCmdNoLogTimeout(devObj, strCmd)
    
    @staticmethod
    def deleteRemoteFilefromSFTP(devObj, path, fileName):
        '''
        @summary: 删除阵列端生成的文件
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        rec=CLI回显信息
        @return:Bool(True or False)
        '''
        deleteFilePathAndName = path + "/log_controller_%s.tar.bz2" % (fileName)
        
        fileDict = {}
        fileDict["path"] = deleteFilePathAndName
        fileDict["name"] = "log_controller_%s.tar.bz2" % (fileName)
        
        #判断文件是否存在
        isExistFlag = exportInfo.isTarFileExist(devObj, fileDict)
        if not isExistFlag:
            log.error(devObj, "the file is not exist")
            return (False, None)
        
        try:
            util.getSftp(devObj).deleteFile(deleteFilePathAndName)
        except Exception:
            pass
 
    @staticmethod
    def deleteRemoteFilefromLinux(devObj, type, oppositeEquip):
        '''
        @summary: 删除阵列端生成的文件
        @param cliRet:  devObj=上下文对象
                        type=收集的类型
                        rec=CLI回显信息
        @return:Bool(True or False)
        '''
        #根据CLI回显信息判断当前环境下，是否允许删除
        timeOut = 60
        tempCmd = "rm /home/permitdir/log_controller_%s.tar.bz2" % (oppositeEquip)
        log.info(devObj, "execute collect : %s" % str(tempCmd))            
        (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, tempCmd, timeOut)
        if not isSuccess:
            return False
        
        count = 0
        while True:
            if "(y/n)" in rec and count < 3:
                count = count + 1
                (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, "y", timeOut)
                log.info(devObj, "rec = %s" % str(rec))            
                
            else:
                break
            
        if cli.CLI_RET_END_FLAG not in rec:
            log.info(devObj, "delete file fail")
            return False
        
        return True
    
    @staticmethod
    def decompressTgzFile(devObj, filePath, fileName):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tgz）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        
        tar_class = devObj.get("PYENGINE.PY_ZIP")
        dirName = fileName[0: fileName.find(".tgz")]
        
        tempFilePath = filePath + os.path.sep + fileName
        tempDirPath = filePath + os.path.sep + dirName
        
        tar_class.decompressZipFile(tempFilePath, tempDirPath)
        
        return tempDirPath
    
    
    @staticmethod
    def decompressTarBz2File(devObj, filePath, fileName):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tar.bz2）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (解压路径)
        '''
        tar_class = devObj.get("PYENGINE.PY_ZIP")
        dirName = fileName[0: fileName.find(".tar.bz2")]
        tempFilePath = filePath + os.path.sep + fileName
        tempDirPath = filePath + os.path.sep + dirName
        tar_class.decompressTarBz2File(tempFilePath, tempDirPath)
        return tempDirPath
        
    
    @staticmethod
    def getNodeCollectInfo(devObj, filePath, fileName):
        '''
        @summary: 解压文件(只支持解压系统日志包（.tar.bz2）)
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        
        tempDirPath = exportInfo.decompressTarBz2File(devObj, filePath, fileName)
        
        #筛选日志信息结果文件
        listFile = os.listdir(tempDirPath)
        result = None
        for fileName in listFile:
            if fileName == config.COLLECT_LOG_FILE_INFO:
                logPath = tempDirPath + os.path.sep + fileName
                result = util.readFile(logPath)
        
        return result
    
    
    @staticmethod
    def checkSystemLogByFile(devObj, info, item, nodeName):
        '''
        @summary: 解析节点,系统日志信息中收集结果的文件
        @param cliRet:  devObj=上下文对象,
                        info=文件内容，
                        item=错误信息编号
                        nodeName=控制器节点名称
        @return: (msg=当前节点错误信息, item=错误信息序号)
        '''
        msg = None
        if None == info:
            item +=1
            msg = util.getMsg(devObj, "controller.failed.information", (item, nodeName))
            return msg
            
        lines = info.splitlines()
        collectFailList = ""
        #如果收集结果文件（info_log.txt）中含有乱码则剔除含有乱码的项
        newLines = []
        for line in lines:
            try:
                line.decode("utf-8").encode("utf-8")
                newLines.append(line)
            except Exception:
                pass
        
        lines = newLines  
        for line in lines:
            if line.strip().startswith("Slot ID is"):
                break
            #如果某项收集结果不是ok，则表示该项收集失败
            if not line.strip().lower().endswith(" ok"):
                collectFailList += line.split()[0] + ', ' 
                
        if "" != collectFailList:
            item +=1
            msg = util.getMsg(devObj, "controller.failed.items", (item, nodeName, collectFailList[:-2]))
            
        return (msg, item)
    
    @staticmethod
    def checkNodeSystemLogIsFull(devObj, filePath, fileName, item, nodeId=None):
        '''
        @summary: 获取当前收集节点，收集失败的项
        @param cliRet:  devObj=上下文对象,
                        filePath=需要检查的节点的路径，
                        fileName=检查的文件名称
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        #解压tgz压缩包，返回解压后的路径
        decompressDir = exportInfo.decompressTgzFile(devObj, filePath, fileName)
        #获取解压后的tar.bz2的文件列表，一个tar.bz2代表一个节点信息
        nodeList = os.listdir(decompressDir)
        messsage = ""
        controllerCount = len(nodeList)
        for nodeInfo in nodeList:
            nodePath = decompressDir
            nodeName = nodeInfo
            
            #判断，若不是tar.bz2的压缩包，则不做处理
            if ".tar.bz2" not in nodeName or ".sha256sum" in nodeName:
                continue
            #解压各个节点，获取节点中导出日志信息的结果信息
            nodeResultInfo = exportInfo.getNodeCollectInfo(devObj, nodePath, nodeName)
            #分析结果信息
            if nodeId != None:
                nodeName = nodeId
            (msg, item) = exportInfo.checkSystemLogByFile(devObj, nodeResultInfo, item, nodeName)
            if None != msg:
                messsage += msg
        
        #清除解压后的临时目录
        util.cleanDir(decompressDir)
        return (messsage, item, controllerCount)
    
    
    @staticmethod
    def checkLogIsCollectFull(devObj, successCount, item):
        '''
        @summary: 下载系统config.txt文件，从文件中获取系统标配控制器个数
        @param cliRet:  devObj=上下文对象,
                        pathList=收集的系统日志的文件夹列表，
                        msgInfo=错误信息
        @return: (msgInfo=错误信息, item=错误信息序号)
        '''
        #配置信息存放目录，若目录存在，则表示之前已经收集了配置信息
        runningDataType = config.EXPORT_TYPE_RUNNING_DATA
        runningDataDir = devObj[config.COLLECT_INFO_LOCAL_PATH] + os.path.sep
        runningDataDir = runningDataDir + config.COLLECT_INFO_DIR_NAME[runningDataType]
        deleteRunningDataDir = True
        
        #若存在则不删除本地配置信息
        if os.path.exists(runningDataDir):
            deleteRunningDataDir = False
        
        exportDataFalg = True
        tempCmd = "show file export_path file_type=%s" % runningDataType

        (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, tempCmd)
        if not isSuccess:
            item += 1
            msg = util.getMsg(devObj, "dev.conn.failure")
            return (msg, item)
        
        (execSucc, isAllSucc) = exportInfo.isExecCmdSuccForRec(devObj, rec)
        if not execSucc:
            exportDataFalg = False
        
        (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj, runningDataType, rec)
        if not loadSucc:
            exportDataFalg = False
        
        exportInfo.deleteRemoteFile(devObj, runningDataType, rec)
        if not exportDataFalg:
            item += 1
            msg = util.getMsg(devObj, "network.communication.abnormal", (item))
            return (msg, item)
        
        tempList = exportInfo.getRemotePathAndName(devObj, rec)
        fileName = tempList[0]["name"]
        ctrlCount = 0
        path = runningDataDir + os.path.sep + fileName
        collectConfig = util.readFile(path)
        if None == collectConfig:
            collectConfig = ""
        lines = collectConfig.splitlines()
        for line in lines:
            if "Number of total controllers" in line:
                ctrlCount = line.split(':')[-1].strip()
                break
            
        #删除收集到得文件
        if deleteRunningDataDir:
            util.cleanDir(runningDataDir)
            
        msg = ""
        if int(ctrlCount) != successCount:
            item +=1
            msg = util.getMsg(devObj, "controllers.some.failed", (item))
            log.info(devObj, "ctrlCount=" + str(ctrlCount) + ";collect success controller = " + str(successCount))
        
        return (msg, item)

    @staticmethod
    def moveFileFromTmpToHome(devObj, curFile, moveToFile):
        
        timeOut = 2*30
        
        tempCmd = "mv %s %s" % (curFile, moveToFile)
        log.info(devObj, "execute collect : %s" % str(tempCmd))
        (isSuccess, rec) = cli.executeCmdWithTimout(devObj, tempCmd, timeOut)
        if not isSuccess:
            return (False, rec)
        
        log.info(devObj, "move %s to %s" % (curFile, moveToFile))
        return (True ,"")
    @staticmethod
    def exportLogFromCurEquip(devObj, type, curDevIp):   
         
        timeOut = config.EXPORT_INFO_TIMEOUT[type]
    
        #a.获取本节点日志
        
        tempCmd = "os_backup_info.sh -f log_controller_%s.tar.bz2" % (curDevIp)
        log.info(devObj, "execute collect : %s" % str(tempCmd))
        (isSuccess, rec) = cli.executeCmdWithTimout(devObj, tempCmd, timeOut)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        execSucc = exportInfo.isNormalForRec(devObj, rec)
        if not execSucc:
            log.info(devObj, "os_backup_info.sh -d... execute failed, and delete file")
            util.setPyDetailMsg(devObj, rec)
            exportInfo.deleteRemoteFilefromSFTP(devObj, "/tmp", curDevIp)
            return False
        
        #b.判断文件 是否存在
        curFile = r"/tmp/log_controller_%s.tar.bz2" % (curDevIp)
        moveToFile = r"/home/permitdir"
        fileDict = {}
        fileDict["path"] = curFile
        fileDict["name"] = curFile.split("/")[-1]
        
        isExistFlag = exportInfo.isTarFileExist(devObj, fileDict)
        if not isExistFlag:
            util.setPyDetailMsg(devObj, "dev.log.big.memory.insufficient")
            log.error(devObj, "the file is not exist")
            return False
        
        #c.将文件从tmp移到home
        (isSuccess,rec) = exportInfo.moveFileFromTmpToHome(devObj, curFile, moveToFile)
        if not execSucc:
            util.setPyDetailMsg(devObj, rec)
            log.info(devObj, "mv execute failed, and delete file")
            return False
        
        #d.下载文件
        fileList = []
        fileList.append(curDevIp)
        (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj, type, fileList)
        if not loadSucc:
            util.setPyDetailMsg(devObj, "downLoad.file.failure")
            log.info(devObj, "download file failed, and delete file")
            exportInfo.deleteRemoteFilefromSFTP(devObj, "/home/permitdir", curDevIp)
            return False
        
        #e.删除阵列端临时文件
        log.info(devObj, "get file success, and delete file") 
        exportInfo.deleteRemoteFilefromSFTP(devObj, "/home/permitdir", curDevIp)
        return True
    
    @staticmethod
    def copyFileFromOppositeDevToCurDev(devObj, type, curDevIp, oppositeEquip):  
        
        #1、TV2有SCP命令的版本
        timeOut = config.EXPORT_INFO_TIMEOUT[type]
        usrPwd = device.getDevicePassword(devObj)
        (isSuccess, name) = util.getUsername(devObj)
        
        tempCmd = "scp /home/permitdir/log_controller_%s.tar.bz2 %s@%s:/home/permitdir/log_controller_%s.tar.bz2" % (oppositeEquip,name,curDevIp,oppositeEquip)
        log.info(devObj, "execute collect : %s" % str(tempCmd))            
        (isSuccess, scpRec) = cli.executeCmdWithTimout(devObj, tempCmd, timeOut)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            device.exitHeartbeatToPeer(devObj)
            return False
        
        if "(yes/no)" in scpRec:
            (isSuccess, scpRec) = cli.executeCmdWithTimout(devObj, "yes", 60)
        
        if "password:" in scpRec:
            (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, usrPwd, 60)
            if not isSuccess:
                util.setPyDetailMsg(devObj, rec)
                log.info(devObj, "execute sshtoremote failed")
                device.exitHeartbeatToPeer(devObj)
                return False
            
            #如果密码错误连续尝试三次，退出到小系统
            if "password:" in rec:
                for i in range(2):
                    (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, usrPwd, 60)
                util.setPyDetailMsg(devObj, rec)
                log.info(devObj, "The password is error!")
                device.exitHeartbeatToPeer(devObj)
                return False
            
            if cli.TOOLKIT_SEND_CMD_TIME_OUT == rec:
                util.setPyDetailMsg(devObj, rec)
                log.info(devObj, "execute cmd timeout.") 
                device.exitHeartbeatToPeer(devObj) 
                return False
            
            if cli.CLI_RET_END_FLAG not in rec: #对端正常不进行日志收集。
                util.setPyDetailMsg(devObj, rec)
                device.exitHeartbeatToPeer(devObj)
                return False
            
        log.info(devObj, "scpRec = : %s" % str(scpRec))  
        
        #退出心跳
        device.exitHeartbeatToPeer(devObj)
        #2、只有getremotefile命令的版本，V3R5C00\V3R3C20
        if cli.CLI_NOT_EXIST_FLAG in scpRec:
            
            
            count = 0
            heartbeatIP = "127.127.127.1"
            while True:
                heartbeatIP = "127.127.127.1" + str(count)
                
                count = count + 1
                if count > 4:
                    break
                
                tempCmd = "getremotefile %s /home/permitdir/log_controller_%s.tar.bz2 /home/permitdir" % (heartbeatIP,oppositeEquip)
                log.info(devObj, "execute collect : %s" % str(tempCmd))            
                (isSuccess, rec) = cli.executeCmdWithTimout(devObj, tempCmd, timeOut)
                log.info(devObj, "getremotefile rec =  %s" % str(rec))  
                if "getremotefile failed" in rec:
                    log.info(devObj, rec)
                    continue
                else:
                    break
                
        log.info(devObj, "copy file from opposite equip to curDev successfully!")      
        return True
            
    @staticmethod
    def exportLogFromOppositeEquip(devObj, type, curDevIp):  
        
        timeOut = config.EXPORT_INFO_TIMEOUT[type]
        usrPwd = device.getDevicePassword(devObj)
        (isSuccess, name) = util.getUsername(devObj)
        
        oppositeEquip = device.getPeerEth2IP(devObj) #对端设备命名
        
        #1、执行心跳sshtoremote，收对端日志.
        execSucc = device.heartbeatToPeer(devObj, usrPwd)
        if not execSucc:
            log.info(devObj, "Heart beat to peer failed!")
            util.setPyDetailMsg(devObj, "dev.conn.failure.and.so.on")
            return False
        log.info(devObj, "Heart beat to peer successfully!")
        
        #2、获取对端sys日志 
        tempCmd = "os_backup_info.sh -f log_controller_%s.tar.bz2" % (oppositeEquip)
        log.info(devObj, "execute collect : %s" % str(tempCmd))            
        (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, tempCmd, timeOut)
        log.info(devObj, "after sshtoremote, to collect log! rec = " + rec)
        if not isSuccess:
            util.setPyDetailMsg(devObj, rec)
            device.exitHeartbeatToPeer(devObj)
            return False
        
        execSucc = exportInfo.isNormalForRec(devObj, rec)
        if not execSucc:
            util.setPyDetailMsg(devObj, rec)
            log.info(devObj, "after sshtoremote, os_backup_info.sh -d... execute failed, and delete file")
            exportInfo.deleteRemoteFilefromLinux(devObj, type, oppositeEquip)
            device.exitHeartbeatToPeer(devObj)
            return False
        
        curFile = r"/tmp/log_controller_%s.tar.bz2" % (oppositeEquip)
        moveToFile = r"/home/permitdir"
        
        #3、判断文件是否存在
        fileDict = {}
        fileDict["path"] = curFile
        fileDict["name"] = curFile.split("/")[-1]
        
        isExistFlag = exportInfo.isTarFileExist(devObj, fileDict)
        if not isExistFlag:
            util.setPyDetailMsg(devObj, "dev.log.big.memory.insufficient")
            log.error(devObj, "the file is not exist")
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #4、将文件从tmp移到home
        (isSuccess,rec) = exportInfo.moveFileFromTmpToHome(devObj, curFile, moveToFile)
        if not execSucc:
            util.setPyDetailMsg(devObj, rec)
            log.info(devObj, "mv execute failed, and delete file")
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #5、判断文件 是否存在
        curFile = r"/home/permitdir/log_controller_%s.tar.bz2" % (oppositeEquip)
        fileDict = {}
        fileDict["path"] = curFile
        fileDict["name"] = curFile.split("/")[-1]
        
        isExistFlag = exportInfo.isTarFileExist(devObj, fileDict)
        if not isExistFlag:
            util.setPyDetailMsg(devObj, "dev.log.big.memory.insufficient")
            log.error(devObj, "the file is not exist")
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #6、将对端日志拷贝到本端
        isCopySucc = exportInfo.copyFileFromOppositeDevToCurDev(devObj, type, curDevIp, oppositeEquip) 
        if not isCopySucc:
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #7、判断本端文件是否存在
        fileDict = {}
        fileDict["path"] = "/home/permitdir/log_controller_%s.tar.bz2" % (oppositeEquip)
        fileDict["name"] = curFile.split("/")[-1]
        
        isExistFlag = exportInfo.isTarFileExist(devObj, fileDict)
        if not isExistFlag:
            util.setPyDetailMsg(devObj, "dev.log.big.memory.insufficient")
            log.error(devObj, "the file is not exist")
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #8、下载文件
        fileList = []
        fileList.append(oppositeEquip)
        (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj, type, fileList)
        if not loadSucc:
            util.setPyDetailMsg(devObj, "downLoad.file.failure")
            log.info(devObj, "download file of opposite equip failed, and delete file")
            exportInfo.deleteRemoteFilefromSFTP(devObj, "/home/permitdir", oppositeEquip)
            device.exitHeartbeatToPeer(devObj)
            return False
        
        #9、删除阵列段临时文件
        log.info(devObj, "get file success, and delete file") 
        exportInfo.deleteRemoteFilefromSFTP(devObj, "/home/permitdir",oppositeEquip)
        
        #10、执行心跳sshtoremote，删除对端日志
        execSucc = device.heartbeatToPeer(devObj, usrPwd)
        if not execSucc:
            log.info(devObj, "Heart beat to peer failed!,delete the file in opposite equip failed")
        log.info(devObj, "Heart beat to peer successfully!")
        
        #11、删除阵列临时文件
        log.info(devObj, "copy file form opposite equip to this equip succ, delete the file in opposite equip") 
        exportInfo.deleteRemoteFilefromLinux(devObj, type, oppositeEquip)
        
        #12、退出心跳
        device.exitHeartbeatToPeer(devObj)
        
        return True
    
    @staticmethod
    def exportLogFromDevIpForTv1(devObj, collectType, curDevIp, nodeNum):
        '''
        #执行命令，导出系统日志信息
        '''
        log.info(devObj, "Device node number is:" + unicode(nodeNum))
        
        localSaveRootDir = util.getLocalInfoPathByType(devObj, collectType)
        localThisCtrlLogDir = localSaveRootDir + curDevIp.replace(':', '_')
        
        devTmpDirName = os.path.basename(tempfile.mkdtemp(prefix='Tmp'))
        
        isThisCtrlLogOk = exportInfo.getThisCtrlLogs(devObj, 'log', localThisCtrlLogDir)
        isThisCtrlCofferLogOk = exportInfo.getThisCtrlLogs(devObj, 'coffer_log', localThisCtrlLogDir)
        isThisCtrlAllLogOk = isThisCtrlLogOk and isThisCtrlCofferLogOk
        isThisCtrlLogResult = isThisCtrlLogOk or isThisCtrlCofferLogOk
        
        if nodeNum == 1:
            util.setCollectAllInfo(devObj, isThisCtrlAllLogOk)
            return isThisCtrlLogResult
        
        #Double controller
        localPeerCtrlLogDir = localSaveRootDir + device.getPeerEth2IP(devObj).replace(':', '_')#support IPV6
        peerHeartBeatIp = device.getPeerBond0IP(devObj)

        if not peerHeartBeatIp:
            log.error(devObj, "Getting peer heart IP failed." )
            util.addPyDetailMsg(devObj, "dev.conn.failure.and.so.on")
            #状态：只要有一端收集成功log收集项状态即为成功(GUI上的对号）
            #返回状态：所有节点的log收集成功时表示全部成功，否则表示部分成功或者失败。
            util.setCollectAllInfo(devObj, False)
            return isThisCtrlLogResult
        
        #Get peer log and coffer_log to local directory.
        isPeerLogOk = exportInfo.getPeerLogs(devObj, 'log', peerHeartBeatIp, localPeerCtrlLogDir, devTmpDirName)
        isPeerCofferLogOk = exportInfo.getPeerLogs(devObj, 'coffer_log', peerHeartBeatIp, localPeerCtrlLogDir, devTmpDirName)
        
        isPeerCtrlAllLogOk = isPeerLogOk and isPeerCofferLogOk             
        isPeerCtrlLogResult = isPeerLogOk or isPeerCofferLogOk
        
        isDoubleCtrlAllLogOk = isThisCtrlAllLogOk and isPeerCtrlAllLogOk
        isDoubleCtrlLogResult = isThisCtrlLogResult or isPeerCtrlLogResult

        util.setCollectAllInfo(devObj, isDoubleCtrlAllLogOk)
        return isDoubleCtrlLogResult

    @staticmethod
    def getThisCtrlLogs(devObj, logName, localThisCtrlLogDir):
        sftp = util.getSftp(devObj)
        try:
            sftp.getDirRecurse("/home/permitdir/" + logName, File(localThisCtrlLogDir + os.sep + logName))
        except ToolException, te:
            log.error(devObj, 'Caught ToolException when down loading [/home/permitdir/' + logName + ']:' + unicode(te))
            util.addPyDetailMsg(devObj, "download.local.%s.failed" % logName)
            return False
        except Exception, e:
            log.error(devObj, 'Caught Exception when down loading [/home/permitdir/' + logName + ']:' + unicode(e))
            util.addPyDetailMsg(devObj, "download.local.%s.failed" % logName)
            return False
        else:
            log.info(devObj, 'Collect current controller /home/permitdir/' + logName + ' success.')
            return True
                    
    @staticmethod
    def getPeerLogs(devObj, logName, peerHeartBeatIp, localPeerCtrlLogDir, devTmpDirName):
        sftp = util.getSftp(devObj)
        usrPwd = device.getDevicePassword(devObj) 
        #Copy peer coffer_log dir to /home/permitdir/tmpPeerCofferLog
        tmpLogDir = '/home/permitdir/' + devTmpDirName + logName
        cpPeerLogCmd = 'scp -r admin@' + peerHeartBeatIp + ':/home/permitdir/' + logName + ' ' + tmpLogDir
        (isSuccess, cpCofferLogRet) = cli.executeCmdWithTimout(devObj, cpPeerLogCmd, 60)
        if not isSuccess:
            log.info(devObj, "Copy peer log dir to current controller failed when executing copy command." )
            return False
        else:
            if 'password:' in cpCofferLogRet:
                (isInpuPwdSuc, _) = cli.executeCmdNoLogTimeout(devObj, usrPwd, 10*60)
                if not isInpuPwdSuc:
                    log.error(devObj, "Copy peer log dir to current controller failed when input password." )
                    return False
                else:
                    try:
                        sftp.getDirRecurse(tmpLogDir, File(localPeerCtrlLogDir + os.sep + logName))
                    except ToolException, te:
                        log.error(devObj, "Caught ToolException when down loading [" + tmpLogDir + "] exception:" + unicode(te))
                        util.addPyDetailMsg(devObj, "download.peer.%s.failed" % logName)
                        return False
                    except Exception, e:
                        log.error(devObj, "Caught Exception when down loading [" + tmpLogDir + "] exception:" + unicode(e))
                        util.addPyDetailMsg(devObj, "download.peer.%s.failed" % logName)
                        return False
                    else:
                        log.info(devObj, "Down loading  [" + tmpLogDir + "] success." )
                        return True
                    finally:
                        #delete peerTmpLogDir
                        rmTmpLogDirCmd = 'rm -rf ' + tmpLogDir
                        cli.executeCmdWithTimout(devObj, rmTmpLogDirCmd, 60)
            else:
                log.error(devObj, 'Copy peer ' + logName + ' dir failed.' ) 
                return False
                
    @staticmethod
    def exportLogFromDevIp(devObj, type, curDevIp, soltIdList):
        '''
        #执行命令，导出系统日志信息
        '''
        nodeCfg = device.getDeviceNodeCfg(devObj)
        
        log.info(devObj, "Deveice node number is:" + unicode(nodeCfg))
        
        #1、获取本端日志 文件 
        getSuccCur = exportInfo.exportLogFromCurEquip(devObj, type, curDevIp)
        
        if nodeCfg == 1:  #单控
            collectFlag = getSuccCur
            isAllSucc = getSuccCur
            log.info(devObj, "the device is single controller.")
            util.setCollectAllInfo(devObj, isAllSucc)
            return collectFlag
        elif nodeCfg == 2: #双控
            #获取对端日志 文件 
            getSuccOpp = exportInfo.exportLogFromOppositeEquip(devObj, type, curDevIp)
            log.info(devObj, "the device is double controller.")
            collectFlag = getSuccCur or getSuccOpp   #状态：对号与错号
            isAllSucc = getSuccCur and getSuccOpp #状态：成功，部分成功，失败
            util.setCollectAllInfo(devObj, isAllSucc)
            return collectFlag
        elif nodeCfg > 2: #多控
            #获取对端日志 文件 
            getSuccOpp = exportInfo.exportLogFromOppositeEquip(devObj, type, curDevIp)
            log.info(devObj, "the device is more than double controller.")
            collectFlag = getSuccCur or getSuccOpp
            isAllSucc = False
            if getSuccOpp and getSuccCur:
                util.setPyDetailMsg(devObj, "cur.only.support.double.controller")
            util.setCollectAllInfo(devObj, isAllSucc)
            return collectFlag
        else:#获取该字段失败
            #获取对端日志 文件 
            getSuccOpp = exportInfo.exportLogFromOppositeEquip(devObj, type, curDevIp)
            log.info(devObj, "get the number of controller information fail")
            collectFlag = getSuccCur or getSuccOpp
            isAllSucc = False
            if getSuccOpp and getSuccCur:
                util.setPyDetailMsg(devObj, "get.controller.info.fail")
            util.setCollectAllInfo(devObj, isAllSucc)
            return collectFlag
            
    @staticmethod
    def exportSysInfo(devObj, type):
        '''
        @summary: 丛阵列端导出信息，日志收集、事件信息、配置信息收集的主入口
        @param cliRet:  devObj=上下文对象
        @return:True or False
        '''
        try:
            
            #检查是否能下发showsysstatus命令，获取 ctrlIdList。
            ctrlIdList = util.soltIdStrToList(devObj)
            curDevIp = device.getDeviceIP(devObj)
            
            #1、获取节点失败，直接sshtoremote心跳，当前IP作为日志包后缀
            exportSucc = exportInfo.exportLogFromDevIp(devObj, type, curDevIp, ctrlIdList)
            return exportSucc
        
        except Exception, ex:
            log.info(devObj, "except trace back:" + str(traceback.format_exc()))
            return False
            
    @staticmethod
    def exportSysInfoForTv1(devObj, collectType):
        '''
        @summary: 丛阵列端导出信息，日志收集、事件信息、配置信息收集的主入口
        @param cliRet:  devObj=上下文对象
        @return:True or False
        '''
        try:
            #检查是否能下发showsysstatus命令，获取 ctrlIdList。
            nodeNum = device.getDeviceNodeCfg(devObj)
            curDevIp = device.getDeviceIP(devObj)
            
            log.info(devObj, "Node number is:" + unicode(nodeNum))
            log.info(devObj, "Current device IP is:" + unicode(curDevIp))
            
            #1、获取节点失败，直接sshtoremote心跳，当前IP作为日志包后缀
            exportSucc = exportInfo.exportLogFromDevIpForTv1(devObj, collectType, curDevIp, nodeNum)
            return exportSucc
        
        except Exception, ex:
            log.info(devObj, "except trace back:" + str(traceback.format_exc()))
            return False
        
    @staticmethod
    def canParallelExportInfoByCtrlId(devObj, type):
        '''
        #判断设备是否支持并行方式收集-->设备支持此命令show file package_result file_type=%s
        @return 1:True or False,命令执行成功
        @return 2:True or False,是否支持此命令
        '''
        (isSuccess, exist_cmd) = (False, False)
        cmd = "show file package_result file_type=%s" % (type)
        count = 0
        while count < 4:
            (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, cmd)
            if "timed out" not in rec:
                exist_cmd = cli.isExistCmdForRec(rec)
                break
            exist_cmd = cli.isExistCmdForRec(rec)
            time.sleep(30)
            count += 1
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
        return (isSuccess, exist_cmd)

    @staticmethod
    def prepareCollect(devObj, type):
        '''
        #下发CLI,通知OM开始收集，OM通知各节点做收集的前期准备
        '''
        (issuccess,controllerId) = util.getController(devObj)
        
        cmd4collect = "show file export_path file_type=%s" % (type)
        
        if issuccess:
            cmd4collect = "show file export_path file_type=%s controller_id=%s" % (type,controllerId)
        (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, cmd4collect)    
            
        log.info(devObj, "prepareCollect and the cliRet is " + str(rec))
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        if config.CLI_EXECUTE_CMD_SUCCESS not in rec: 
            util.setPyDetailMsg(devObj, "cli.excute.failure", rec)
            return False
        
        return True
    
    @staticmethod
    def queryCollectStatus(devObj, type):
        '''
        #轮询收集进度
        @return 1:True or False,命令执行成功
        @return 2:收集进度{"总的进度" : "" , "单个节点的进度" : {"节点" : "进度"}..}
        @return 3:cli回显
        '''
        (isSuccess, collectStatus, cliRet) = (False, {"totalResult" : "" , "singleResultDict" : {}}, "")
        
        cmd4status = "show file package_result file_type=%s" % (type)
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4status)
        log.info(devObj, "queryCollectStatus and the cliRet is " + str(cliRet))
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (isSuccess, collectStatus, cliRet)
        if "Total Result" not in cliRet:
            util.setPyDetailMsg(devObj, "cli.excute.failure", cliRet)
            return (isSuccess, collectStatus, cliRet)
        
        totalResult = ""
        NodeID = ""
        singleResult = ""
        
        dictList = cli.getCliTable2DictList(cliRet)
        for dict in dictList:
            totalResult = dict["Total Result"]
            if totalResult != "" and totalResult != "--":
                collectStatus["totalResult"] = totalResult
            
            singleResultDict = collectStatus["singleResultDict"]
            NodeID = dict["Controller ID"]
            singleResult = dict["Single Result"]
            singleResultDict.update({NodeID : singleResult})
            collectStatus["singleResultDict"] = singleResultDict
        
        return (True, collectStatus, cliRet)
    
    @staticmethod
    def notificationDownload(devObj, type, NodeID):
        '''
        #通知OM取文件
        @return 1:True or False,命令执行成功
        @return 2:CLI回显
        '''
        (isSuccess, cliRet) = (False, "")
        cmd4load = "show file notification file_type=%s controller_id=%s" % (type, NodeID)   
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4load)
        log.info(devObj, "notificationDownload and the cliRet is " + str(cliRet))
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (isSuccess, cliRet)
        
        isSuccess = True
        return (isSuccess, cliRet)
        
    @staticmethod
    def deleteFile(devObj, type, NodeID):
        '''
        #删除阵列端数据
        '''
        cmd4delete = "delete file filetype=%s controller_id=%s" % (type, NodeID)
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd4delete)
        log.info(devObj, "deleteFile and the cliRet is " + str(cliRet))
        if not isSuccess:
            log.info(devObj, "deleteFile failed because cli execute failed")
            
        return
       
    @staticmethod
    def collectAndDownload(devObj, type):
        '''
        #1，轮询收集进度；2，下载收集到的文件到本地；3，删除阵列端的文件
        @return 1:True or False,收集项成功or失败
        @return 2:本项收集项的所有失败记录
        '''
        (isSuccess, failMesage) = (False, "")
        
        cmd4statusRet = ""
        item = 0 #
        totalResult = "" #总的收集进度
        limitTime = config.PARALLEL_EXPORT_INFO_TIMEOUT[type] #收集的超时时间，单位：分
        repeatTimes = limitTime * 2 #轮训次数
        
        shouldCollectNodeList = [] #应该收集的节点数量
        finishedNodes = [] #已经完成的节点，包括成功和失败
        successNoedList = [] #收集成功的节点
        hasErrMsgNodeList = [] #收集失败且已经设置错误信息的节点列表
        ftpGetFailedNodes = []
        #1，轮询收集进度
        while repeatTimes != 0:
            repeatTimes -= 1

            (isSuccess, collectStatus, cmd4statusRet) =exportInfo.queryCollectStatus(devObj, type)
            if not isSuccess:
                return (isSuccess, failMesage)
            
            totalResult = collectStatus["totalResult"]
            singleResultDict = collectStatus["singleResultDict"]
            
            for key in singleResultDict.keys():
                NodeID = key
                singleResult = singleResultDict[key]
                if shouldCollectNodeList.count(NodeID) == 0 and NodeID not in ["", "--"]:
                    shouldCollectNodeList.append(NodeID)
                
                #阵列内部包控制器收集失败    
                if singleResult in ["Busy", "Failed", "Out of Memory"] and finishedNodes.count(NodeID) == 0:
                    item += 1
                    errMsgKey = "get.node.file.faild.%s" % singleResult
                    errMsg = util.getMsg(devObj, errMsgKey, (item, NodeID, cmd4statusRet))
                    failMesage += errMsg
                    hasErrMsgNodeList.append(NodeID)
                    finishedNodes.append(NodeID)
                    
                if "Successful" in singleResult and finishedNodes.count(NodeID) == 0:
                    finishedNodes.append(NodeID)
                    #通知OM取文件
                    (isSuccess, cliRet) = exportInfo.notificationDownload(devObj, type, NodeID)
                    if not isSuccess:
                        return (isSuccess, failMesage)
                    
                    if "File Path" not in cliRet:
                        item += 1
                        errMsg = util.getMsg(devObj, "cli.excute.failure.on.someContr", (item, NodeID, cliRet))
                        failMesage += errMsg
                        hasErrMsgNodeList.append(NodeID)
                        #未获取到相应的文件路径，向用户提示可能的原因和处理意见
                        ftpGetFailedNodes.append(NodeID)  
                    else:
                        #将阵列端的数据下载到本地
                        (loadSucc, loadDirInfo) = exportInfo.downloadFile(devObj, type, cliRet)
                        #删除阵列端数据
                        exportInfo.deleteFile(devObj, type, NodeID)
                        
                        if not loadSucc:
                            item += 1
                            errMsg = util.getMsg(devObj, "downLoad.file.failure.on.someContr", (item, NodeID))
                            failMesage += errMsg
                            hasErrMsgNodeList.append(NodeID)
                        else:
                            successNoedList.append(NodeID)
                            if type in config.IS_SYSTEM_INFO_FULL:
                                #判断信息是否收集完全
                                for tempDir in loadDirInfo:
                                    fileName = tempDir["name"]
                                    if ".tgz" in fileName:
                                        (errMsg, item, count) = exportInfo.checkNodeSystemLogIsFull(devObj, tempDir["path"], fileName, item, NodeID)
                                        failMesage += errMsg
                                        hasErrMsgNodeList.append(NodeID)
                        
            if "Packaging" not in totalResult:
                break  
            #已30秒为间隔时间进行轮询
            time.sleep(30)
        
        successNoedNum = len(successNoedList)
        shouldCollectNodeNum = len(shouldCollectNodeList)
        #收集失败 
        if successNoedNum == 0:
            isSuccess = False         
        #成功或部分成功 
        else:
            isSuccess = True
        
        #每个节点收集失败的原因
        if successNoedNum != shouldCollectNodeNum:
            for NodeID in shouldCollectNodeList:
                #如果是超时则设置轮询结果的CLI命令的回显为失败原因
                if successNoedList.count(NodeID) == 0 and hasErrMsgNodeList.count(NodeID) == 0:
                    item += 1
                    errMsg = util.getMsg(devObj, "cli.excute.failure.on.someContr", (item, NodeID, cmd4statusRet))
                    failMesage += errMsg
                    
        if "Partly Exported" in totalResult and failMesage == "":
            item += 1
            errMsg = util.getMsg(devObj, "collect.controllers.some.failed", item)
            failMesage += errMsg
                    
        if len(ftpGetFailedNodes) != 0:
            errMsg = util.getMsg(devObj, "sftp.get.file.faild", ",".join(ftpGetFailedNodes))
            failMesage += errMsg
            
        return (isSuccess, failMesage)
    
    @staticmethod
    def parallelExportInfoByCtrlId(devObj, type):
        '''
        #控制器之间以并行方式来收集信息，涉及log,alllog,smart
        '''
        #1, 下发CLI,通知OM开始收集;
        isSuccess = exportInfo.prepareCollect(devObj, type)
        if not isSuccess:
            return False
        
        #2, 轮询收集进度，一旦发现有收集成功的节点则下载收集到得文件到本地并删除阵列端的文件。
        (isSuccess, failMesage) = exportInfo.collectAndDownload(devObj, type)
            
        #设置错误消息，并将部分收集成功标志置为True
        if failMesage != "":
            util.setCollectAllInfo(devObj, False)
            util.setPyDetailMsg(devObj, failMesage)
        
        return isSuccess
