# -*- coding: UTF-8 -*-
from java.io import File
import sys
import resource
import os
import re
import traceback
import shutil
import codecs
from cliFactory import cli
from config import config
import time

from com.huawei.ism.tool.obase.exception import ToolException


class systemMode():
    @staticmethod
    def reconnect_ssh(cli, logger):
        try:
            cli.reConnect()
            logger.logInfo(
                "it is afresh to connect to device by ssh gallery.")
            return True
        except (ToolException, Exception):
            logger.logError(str(traceback.format_exc()))
            return False

    @staticmethod
    def enterDeveloperMode(devObj):
        '''
        @summary: 进入developer模式
        @param cliRet: devObj=上下文对象
        @return: True=成功，False=失败
        '''
        #发送命令
        cmd = "change user_mode current_mode user_mode=developer"
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        
        if re.search("developer:", cliRet, re.IGNORECASE):
            return True
        
        if None == re.search("Password:", cliRet, re.IGNORECASE):
            log.info(devObj, "enterDeveloperMode fail")
            return False
        
        developerPwd = util.getDevDeveloperPwd(devObj)
        #判断developer密码是否为空
        if None == developerPwd or "" == developerPwd :
            del developerPwd
            return False
        
        cliRet = ""
        (isSuccess, cliRet) =  cli.executeCmdNoLogTimeout(devObj, developerPwd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        del developerPwd
        if None == re.search("developer:", cliRet, re.IGNORECASE):
            log.info(devObj, "enterDeveloperMode fail")
            return False
        
        return True
    
    @staticmethod
    def developerMode2CliMode(devObj):
        '''
        @summary: developer模式下退回到cli模式
        @param cliRet: devObj=上下文对象
        '''        
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "show system general")
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return
        
        #密码输入错误时及正常情况从developer模式下退出
        if ("Password" in cliRet) or ("developer:/>" in cliRet):
                    
            (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "exit")
            if not isSuccess:
                util.setPyDetailMsg(devObj, "dev.conn.failure")
                return
            
            cliRet = cli.executeCmdNoLogTimeout(devObj, "exit")
            log.info(devObj, "Sequence[5] change to cli model")
            index = 0
            while ("admin:/>" not in cliRet):
                index = index + 1
                (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "exit")
                if not isSuccess:
                    util.setPyDetailMsg(devObj, "dev.conn.failure")
                    return
                if "Are you sure to exit?(y/n):" in cliRet:
                    (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "n")
                    if not isSuccess:
                        util.setPyDetailMsg(devObj, "dev.conn.failure")
                        return
                if index > cli.CLI_ENTER_DEVELOPER_MODEL_MAX_RETRYS:
                    break
    
    @staticmethod
    def isInDeveloperMode(devObj):
        '''
        @summary: 判断当前是否在developer模式下
        @param cliRet: cli回显
        @return: 
            True: 当前在developer模式下
            False: 当前不在developer模式下
        '''
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "show system general")
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        if re.search(cli.CLI_DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
            return True
        return False
    
class device():
    @staticmethod
    def getDeviceSn(devObj):
        '''
                    获取设备SN号
        '''
        devNode = devObj.get("devNode")
        
        if None == devNode:
            return ""
        
        sn = devNode.getDeviceSerialNumber()
        
        return sn
    
    @staticmethod
    def getDeviceType(devObj):
        '''
                    获取设备具体型号,如S5600T
        '''
        devNode = devObj.get("devNode")
        
        if None == devNode:
            return ""
        
        type = devNode.getDeviceType()
        
        return type
    
    
    @staticmethod
    def getDeviceVersion(devObj):
        '''
                    获取设备版本号,如：V200R002C11
        '''
        
        devNode = devObj.get("devNode")
        
        if None == devNode:
            return ""
        
        version = devNode.getProductVersion()
        
        return version
    
    @staticmethod
    def getDevicePassword(devObj):
        '''
                    获取设备密码，心跳连接
        '''
        
        password = devObj.get("password")
        
        if None == password:
            return ""
        
        return password
    
    @staticmethod
    def getDeviceSoltIdStr(devObj):
        '''
                    获取设备soltIdStr
        '''
        soltIdStr = devObj.get("soltIdStr")
        
        if None == soltIdStr:
            return ""
        
        return soltIdStr
    
    @staticmethod
    def getDeviceNodeCfg(devObj):
        '''
                    获取设备NodeCfg
        '''
        nodeCfg = devObj.get("nodeCfg")
        if None == nodeCfg or "" == nodeCfg:
            return 0
        
        return int(nodeCfg)
    
    @staticmethod
    def getDeviceIP(devObj):
        '''
                    获取当前控制器IP，生成日志包后缀
        '''
        
        devIp = devObj.get("devIp")
        
        if None == devIp:
            return "this"
        
        return devIp
        
    @staticmethod
    def getEth2IP(devObj):
        '''
                    通过解析ifconfig命令获取控制器IP
        '''
        qryEth2Cmd = "ifconfig eth2"
        (isCmdExeSucc, eth2Ret) = cli.executeCmdWithTimout(devObj, qryEth2Cmd, 60)
        if not isCmdExeSucc:
            log.info(devObj, "Execute ifconfig eth2 failed.")
            return ""
        
        for line in eth2Ret.splitlines():
            if line.strip().startswith("inet addr"):
                return line.split(':')[1].split()[0].strip()
        
        return ""
    
    @staticmethod
    def getPeerEth2IP(devObj):
        '''
                    功能说明: 获取对端eht2 网口IP,如果获取失败：返回“thePeer".
        '''
        usrPwd = device.getDevicePassword(devObj)
        execSucc = device.heartbeatToPeer(devObj, usrPwd)
        if not execSucc or execSucc == 'Normal':
            log.info(devObj, "Heart beat to peer failed or peer CTRL is not in minisystem when getting peer eth2 IP.")
            return 'thePeer'
        
        thePeerIp = device.getEth2IP(devObj)
        device.exitHeartbeatToPeer(devObj)
        return  thePeerIp if thePeerIp else 'thePeer'
    
    @staticmethod
    def getBond0IP(devObj):
        '''
                    通过解析ifconfig命令获取控制器心跳IP,，如果获取失败，返回空字符串。
        '''
        qryBond0Cmd = "ifconfig bond0"
        (isCmdExeSucc, bond0Ret) = cli.executeCmdWithTimout(devObj, qryBond0Cmd, 60)
        if not isCmdExeSucc:
            log.error(devObj, "Execute ifconfig bond0 failed.")
            return ''
        
        for line in bond0Ret.splitlines():
            if line.strip().startswith("inet addr"):
                return line.split(':')[1].split()[0].strip()
            
        return ''
    
    @staticmethod
    def getPeerBond0IP(devObj):
        '''
                    通过解析ifconfig命令获取控制器心跳IP
        '''
        thisBond0Ip = device.getBond0IP(devObj)
        if not thisBond0Ip:
            return ''
        elif thisBond0Ip == '127.127.127.10':
            return '127.127.127.11'
        elif thisBond0Ip == '127.127.127.11':
            return '127.127.127.10'
        else:
            return ''
    
    @staticmethod
    def getDeviceSsh(devObj):
        '''
                    获取设备版本号,如：V200R002C11
        '''
        ssh = devObj.get("SSH")
        
        return ssh

    @staticmethod
    def heartbeatToPeer(devObj, usrPwd):
        '''
                    函数名称: heartbeatToPeer
                    功能说明: 从本端minisystem模式心跳到对端，此方法须在minisystem模式下调用
                    输入参数: devObj, usrPwd
                    输出参数: 无
                    返 回 值: True or False    
        '''
        try:
            (isSuccess, rec) = cli.executeCmdWithTimout(devObj, 'sshtoremote', 60)
            if not isSuccess:
                log.info(devObj, "execute sshtoremote failed.")
                return False
            
            log.info(devObj, "XXXX:" + rec)
             
            if "System is not ready" in rec:
                log.info(devObj, "execute sshtoremote failed")
                return False
            
            if "password:" in rec:
                (isSuccess, rec) = cli.executeCmdNoLogTimeout(devObj, usrPwd, 60)
                if not isSuccess:
                    log.info(devObj, "sshtoremote failed when inputting user password.")
                    return False
                
                if cli.TOOLKIT_SEND_CMD_TIME_OUT == rec:
                    log.info(devObj, "[heartbeatToPeer] execute cmd timeout.")
                    return False
                
                if  rec.endswith(":/>"):    #对端正常不进行日志 收集,并且退出对端。
                    (isSuccessExit, recExit) = cli.executeCmdWithTimout(devObj, "exit", 60)
                    if not isSuccessExit:
                        log.info(devObj, recExit)
                        return False
                    
                    (isSuccessY, recY) = cli.executeCmdWithTimout(devObj, "y", 60)
                    if not isSuccessY:
                        log.info(devObj, recY)
                        return False
                    
                    if "Storage:~ #" in recY:  #兼容debug包
                        (isSuccessExit, recExit) = cli.executeCmdWithTimout(devObj, "exit", 60)
                        if not isSuccessExit:
                            log.info(devObj, recExit)
                            return False
                    
                    log.info(devObj, "[heartbeatToPeer] the Opposite Equip is right need not to collect")
                    return "Normal"
                
                if rec.strip().endswith(cli.CLI_RET_END_FLAG): 
                    return True
                
            log.info(devObj, "[heartbeatToPeer] heart to peer failed.")
            return False
        except:
            log.info(devObj, "[heartbeatToPeer] catch trace backxxx:" + unicode(traceback.print_exc()))
            return False

    @staticmethod
    def exitHeartbeatToPeer(devObj):
        '''
                    函数名称: exitHeartbeatToPeer
                    功能说明: 退出心跳
                    输入参数: devObj
                    输出参数: 无
                    返 回 值: 
        ''' 
        (isSuccessExit, rec) = cli.executeCmdWithTimout(devObj, "exit", 60)
        if not isSuccessExit:
            return False
        
        if "y/n" in rec:
            (isSuccess, rec) = cli.executeCmdWithTimout(devObj, "y", 60)
            if not isSuccess:
                return False
            
        if "Storage:~ #" in rec:  #兼容debug包
            (isSuccessExit, _) = cli.executeCmdWithTimout(devObj, "exit", 60)
            if not isSuccessExit:
                return False
            
        log.info(devObj, "[exitHeartbeatToPeer] Exit heart beat successfully.")
        return True
            
'''
信息收集，基本的工具方法
'''
class util():
    
    cliInfoFalg = False
    
    @staticmethod
    def isGetSN(devObj):
        '''
                    函数名称: isGetSN
                    功能说明: 检验是否成功获取SN号
                    输入参数: devObj
                    输出参数: True, False
        '''
        SerialNumber = device.getDeviceSn(devObj)
        if "" == SerialNumber:
            return False
        return True
    
    @staticmethod
    def soltIdStrToList(devObj):
        '''
                    函数名称: soltIdStrToList
                    功能说明: 将soldIdStr转化为List
                    输入参数: devObj
                    输出参数: True, False
        '''
        
        soltIdStr = device.getDeviceSoltIdStr(devObj)
        if "" == soltIdStr:
            return []
        else:
            soltIdList = soltIdStr.strip().split(",")[:-1]
            return soltIdList
    
    @staticmethod
    def isSystemUpgrading(devObj):
        '''
                    函数名称: isSystemUpgrading
                    功能说明: 检验当前设备是否处于升级中
                    输入参数: devObj
                    输出参数: True, False
        '''
        flag = False
        cmd = "show system general"        
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return True
        
        cliList = cliRet.splitlines()
        #只关注最后一行回显
        if config.SYSTEM_RUNNING_STATUS_UPGRADING in cliList[-1]:
            util.setPyDetailMsg(devObj, 'system.isUpgrading')
            flag = True
        return flag
    
    @staticmethod
    def checkSystemNormal(devObj):
        '''
                    函数名称: checkSystemNormal
                    功能说明: 检验当前设备是否normal
                    输入参数: devObj
                    输出参数: True, False
        '''
        flag = False
        cmd = "show system general"        
        (isSuccess, cliRet) =  cli.executeCmdNoLogTimeout(devObj, cmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        if re.search("System Name", cliRet, re.IGNORECASE):
            flag = True
        else:
            util.setPyDetailMsg(devObj, 'system.abnormal')
            
        return flag
    
    @staticmethod
    def getUserNameCli(devObj):
        '''
        @summary: 用CLI的方式，获取当前的用户名（当上下文对象中不存在用户名时，用到）
        @param cliRet: devObj=上下文对象
        @return: userName
        '''
        userName = ""        
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout("show system general")
        if not isSuccess:
            return (isSuccess, userName)
        
        #输入CLI回显是否有效
        if None == cliRet or "" == cliRet: 
            return (isSuccess, userName)
        
        #执行命令获取当前登录的用户名      
        lineList = cliRet.splitlines()
        #用户名一定包含在最后一行中
        if lineList[-1].find(cli.CLI_RET_END_FLAG):
            userName = lineList[-1].replace(" ","").replace(cli.CLI_RET_END_FLAG,"")
        return (isSuccess, userName)
    
    @staticmethod
    def getUsername(devObj):
        '''
        @summary: 获取上下文中的用户名
        @param cliRet: devObj=上下文对象
        @return: name
        '''
        isSuccess = True
        name = devObj.get("username")
        
        if None == name or "" == name: 
            (isSuccess, name) = util.getUserNameCli(devObj)
            
        return (isSuccess, name)
        
    @staticmethod
    def getUserPrivilege(devObj):
        '''
        @summary: 获取用户权限级别
        @param cliRet: devObj=上下文对象
        @return: userLevel
        '''
        userLevel = ""
        #首先获取用户名
        (isSuccess, userName) = util.getUsername(devObj)
        
        if not isSuccess:
            (isSuccess, userLevel)
        
        if None == userName or "" == userName:
            return (isSuccess, userLevel)
        
        log.info(devObj, "Get login name is:"+str(userName))
        
        #根据用户名，查询用户权限        
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, "show user user_name=" + userName)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (isSuccess, userLevel)
        
        rec = cli.getCliTable2DictList(cliRet)
        if len(rec) == 0:
            return (isSuccess, userLevel)
        
        #获取用户权限的行信息
        level = rec[0]
        if "Level" in level:
            userLevel = level["Level"]
        
        return (isSuccess, userLevel)
    
    
    @staticmethod
    def checkUserPrivilege(devObj, type=None):
        '''
        @summary: 检查用户权限，用户权限为["Super_admin" 或  "Admin"],则返回True，否则返回False
        @param cliRet: devObj=上下文对象
        @return: userLevel
        '''
        result = False
        (isSuccess, userLevel) = util.getUserPrivilege(devObj)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return result
        
        privileges = ["Super_admin", "admin"]
            
        if userLevel in privileges:
            result = True
        else:
            util.setPyDetailMsg(devObj, "user.level.low")
        
        return result
    
    
    @staticmethod
    def isChinese(devObj):
        '''
        @summary: 判断当前语言类型
        @param lang: 语言lang
        @return: 
            True: 中文
            False: 非中文
        '''
        lang = devObj.get("lang")
        if "zh" == lang:
            return True
        return False
    
    @staticmethod
    def getMsg(devObj, msg, args = ""):
        '''
        @summary: 获取资源信息
        @param cliRet: devObj=上下文对象, msg=资源脚本（resource.py）中对应的Key，args=占位符对应的值
        @return: msg
        '''
        if  None == msg or "" == msg:
            return ""
        
        lang = devObj.get("lang")
        if resource.MESSAGES_DICT.has_key(msg):
            msgDict = resource.MESSAGES_DICT.get(msg)
            msg = msgDict[lang]
        
        #若占位符的值不为”“，则天冲高占位符
        if "" != args:
            msg = msg % args
            
        return msg
    
    @staticmethod
    def setCollectAllInfo(devObj, flag):
        '''
        @summary: 设置是否部分收集成功
        @param flag: flag = True 该项收集成功，flag=False 该项部分收集成功
        '''
        devObj["collectAllInfo"] = flag
    
    @staticmethod
    def isExistDetailMsg(devObj):
        '''
        @summary: 判断错误信息是否已经设置
        @param devObj: 上下文对象
        @return: True or False
        '''
        msg = devObj["py_detail"]
        if None == msg or "" == msg:
            return False
        
        return True
    
    @staticmethod
    def initPyDetailMsg(devObj):
        '''
        @summary: 清除之前设置的错误信息
        @return: 无
        '''
        devObj["py_detail"] = ""
    
    @staticmethod
    def setPyDetailMsg(devObj, msg, args = ""):
        '''
        @summary: 消息国际化
        @param lang: 语言lang
        @param msg: 消息
        @param args: 消息参数
        @return: 经过国际化处理后的消息
        '''
        msg = util.getMsg(devObj, msg, args)
        
        devObj["py_detail"] = msg
    @staticmethod
    def addPyDetailMsg(devObj, msg, args = ""):
        '''
        @summary: 消息国际化
        @param lang: 语言lang
        @param msg: 消息
        @param args: 消息参数
        @return: 经过国际化处理后的消息
        '''
        msg = util.getMsg(devObj, msg, args)
        
        devObj["py_detail"] = devObj["py_detail"] + msg
        
    @staticmethod
    def getDevIp(devObj):
        '''
        @summary: 根据上下文，获取当前连接的IP
        @param cliRet: devObj=上下文对象
        @return: ip
        '''
        ip = devObj.get("devIp")
        return ip
    
    @staticmethod
    def getSftp(devObj):
        '''
        @summary: 根据上下文，获取sftp对象
        @param cliRet: devObj=上下文对象
        @return: sftp
        '''
        sftp = devObj.get("SFTP")
        return sftp
    
    @staticmethod
    def readFile(filePath):
        try:
            f = None
            f = open(filePath)
            info = f.read()
            return info
        except:
            return None
        finally:
            if None != f:
                f.close()
    
    @staticmethod
    def cleanDir(dir):
        '''
        @summary: 根据路径，删除路径
        @param cliRet: devObj=上下文对象，dir=路径
        @return: True or False
        '''
        if False == os.path.exists(dir):
            return True
        
        if False == os.path.isdir(dir):
            return False
        
        shutil.rmtree(dir, True)
        return True
    
    @staticmethod
    def writeFile(filePath, info, isAddFlag):
        try:
            file = None
            openType = "w"
            if True == isAddFlag:
                openType = "a"
            
            file = codecs.open(filePath, openType, "utf-8")
            file.write(info)
            return True
        except:
            return False
        finally:
            if None != file:
                file.close()
    
    @staticmethod
    def getLocalDir(devObj):
        '''
        @summary: 获取用户选择的路径
        @param cliRet:  devObj=上下文对象
        @return: localDir:用户选择的路径
        '''
        localDir = devObj.get("collectRetDir")
        return localDir
    
    @staticmethod
    def getLocalInfoPathByType(devObj, type):
        '''
        @summary: 获取信息存放的路径
        @param cliRet: devObj=上下文对象
        @return: localDir
        '''
        localDir = devObj[config.COLLECT_INFO_LOCAL_PATH] + os.path.sep
        localDir = localDir + config.COLLECT_INFO_DIR_NAME[type] + os.path.sep
        if not os.path.exists(localDir):
            os.mkdir(localDir)
        
        return localDir
    
    @staticmethod
    def getCurrentDate():
        '''
        @summary: 获取本地当前时间
        @param cliRet: devObj=上下文对象
        @return: localDir
        '''
        result = str(time.strftime('%Y%m%d%H%M%S',time.localtime(time.time())))
        return result
    
    @staticmethod
    def getDevDeveloperPwd(devObj):
        '''
        @summary: 获取设备developer密码
        @param cliRet: devObj=上下文对象
        @return: pwd
        '''
        pwd = devObj.get("developer")
        return pwd
    
    @staticmethod
    def getCtrlIds(devObj):
        '''
        @summary: 获取当前环境下，控制器的ID列表
        @param cliRet: devObj=上下文对象
        @return: (Bool,list)，如：(True,[0A,0B])
        '''
        ctrlIds = []
        strCmd = "show controller general"
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, strCmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (False, ctrlIds)
        
        list_cliRet = cliRet.splitlines()
        #根据CLI回显信息获取当前环境中，主控的ID
        for field in list_cliRet:
            
            field = field.replace(" ", "")
            if field.startswith("Controller"):
                list_ctrl = field.split(":")
                ctrlIds.append(list_ctrl[1])
        
        #判断是否获取到控制器的ID列表
        if len(ctrlIds) == 0:
            log.error(devObj, "show controller general:recrod " + str(cliRet))
            util.setPyDetailMsg(devObj, "controller.information.cannot.be.obtained")
            return (False, ctrlIds)
        
        return (True, ctrlIds)

    @staticmethod 
    def getController(devObj):
        if devObj.get("SettingInfo") == None or devObj.get("SettingInfo").get("modeList") == None:
            return (False,0)
        if (devObj.get("SettingInfo").get("modeList")).size() == 0:
            return (False,0)
        else:
            modeList = devObj.get("SettingInfo").get("modeList")            
            return (True, ",".join([modeList.get(i) for i in range(0, modeList.size())]))
   
    @staticmethod
    def changeCliMoreEnabled(devObj):
        '''
        @summary: 关闭回显more功能
        @return: True 命令执行成功
                False 命令执行失败
        '''
        flag = False
        cmd = "change cli more_enabled=no"        
        (isSuccess, cliRet) = cli.executeCmd(devObj, cmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return False
        
        if config.CLI_EXECUTE_CMD_SUCCESS in cliRet:
            flag = True
            
        return flag
    
    @staticmethod
    def getDiskNum(devObj):
        '''
        @summary: 获取系统中硬盘的数量
        @return: True|False 成功或者失败
                                        硬盘数量
        '''
        
        cmd = "show disk general|filterColumn include columnList=ID"
        (isSuccess, cliRet) = cli.executeCmdNoLogTimeout(devObj, cmd)
        if not isSuccess:
            util.setPyDetailMsg(devObj, "dev.conn.failure")
            return (False, 0)
        
        dictList = cli.getCliTable2DictList(cliRet)
        return (isSuccess, len(dictList))
        
class log():
    
    """
            功能：日志记录函数的子函数，用于获取调用函数和调用行号
            参数： MAX_CALLER_LEVEL：最大调用关系层数
            返回值：调用函数信息
    """
    @staticmethod
    def getCallerInfo(MAX_CALLER_LEVEL=5, skipLastLevel=True):
        #从堆栈中获取调用函数和行号
        
        #初始化参数
        funcBack = sys._getframe().f_back
        if True == skipLastLevel: #忽略最近的调用关系
            funcBack = funcBack.f_back
            MAX_CALLER_LEVEL -= 1
        
        #生成函数调用关系
        callerInfo = ""    
        for index in range(0, MAX_CALLER_LEVEL):
        
            #获取该级调用函数和行号
            if hasattr(funcBack, "f_code") and hasattr(funcBack, "f_lineno"):
                funcName = funcBack.f_code.co_name
                lineNumber = funcBack.f_lineno
                callerInfo = " [" + str(funcName) + ":" +  str(lineNumber) + "]" + callerInfo
            else:
                break
            
            #刷新Back函数
            if hasattr(funcBack, "f_back"):
                funcBack = funcBack.f_back
            else:
                break
        
        #返回函数调用关系
        return callerInfo
    
    @staticmethod
    def debug(devObj, info):
        """
                    功能：记录调试信息info到工具日志中
                    参数：devObj=工具上下文；info=要记录的信息
                    返回值：True=成功；False=失败
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in devObj:
            devObj.get("logger").debug('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)
    
    @staticmethod
    def error(devObj, info):
        """
                    功能：记录错误信息info到工具日志中
                    参数：devObj=工具上下文；info=要记录的信息
                    返回值：True=成功；False=失败
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in devObj:
            devObj.get("logger").error('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)
        
    @staticmethod
    def info(devObj, info):
        logInfo = info + log.getCallerInfo()
        if "logger" in devObj:
            devObj.get("logger").info('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)
        
    @staticmethod
    def warn(devObj, info):
        logInfo = info + log.getCallerInfo()
        if "logger" in devObj:
            devObj.get("logger").warn('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)

