# -*- coding: UTF-8 -*-

import os
import re 
import shutil
import time
import traceback
import stat
import platform

from Properties import*

CODE_ZH_FILE_NAME = r"..\res\resource_zh.properties" if platform.system() == "Windows" \
    else r"../res/resource_zh.properties"
CODE_EN_FILE_NAME = r"..\res\resource_en.properties" if platform.system() == "Windows" \
    else r"../res/resource_en.properties"
#解压时不做递归全解压，只将子文件原样移动的目录（目录之间以“|”分隔）
BLACKS = "|alarm_log|cli_command_info|dha_information|disksmartinfo|electronical_label|running_data|smart_information|other|"
SILENT_START_MODE = "1"
OS_OPEN_FLAGS = os.O_WRONLY | os.O_CREAT | os.O_TRUNC  # 写 | 创建并打开 | 截断
OS_OPEN_MODES = stat.S_IWUSR | stat.S_IRUSR  # 拥有者具有读写权限


class fileHandler():
    
    def __init__(self, dataDict):
        
        self.file = dataDict.get("filePath") #待切割文件路径
        self.SaveDir = dataDict.get("fileSavePath")#切割后文件保存路径
        self.startTime = dataDict.get("startTime")#切割开始时间
        self.endTime = dataDict.get("endTime")#切割结束时间
        self.FileCompress = dataDict.get("collectManagement")#文件压缩解压接口
        self.uiObserver = dataDict.get("uiObserver")#进度条刷新接口
        self.logger = dataDict.get("Logger")#日志打印接口
        self.lang = dataDict.get("lang")#国际化接口
        self.startMode = dataDict.get("startMode")#获取启动模式
        
        #获取当前时间
        curTime = time.strftime('%Y%m%d%H%M%S',time.localtime(time.time()))
        self.timeStamp = curTime
        
        #文件名称
        self.srcFileName = os.path.basename(self.file)
        srcFileBaseName, extension = os.path.splitext(self.srcFileName)
        self.cutFileName = srcFileBaseName + "_cut_" + curTime + extension
        
        #系统日志基础路径
        self.logDirName = os.path.sep + "Messages" + os.path.sep
    
    def logCutProcess(self):
        """
                    文件裁剪功能入口:
                    （1）预处理
                    （2）日志切割
                    （3）切割后处理
        """
        try:
            #步骤1：调用预处理接口
            flag, errMsg = self.preProcess()
            if not flag:
                return(flag, errMsg)
            
            #步骤2：日志切割
            flag, errMsg = self.fileMoveProcess()
            if not flag:
                return(flag, errMsg)
            
            #步骤3：切割后处理
            flag, errMsg = self.postProcess()
            if not flag:
                return(flag, errMsg)
            return (True, self.getResCode(["fileCutSucc", self.cutFileName]))
        except:
            self.logger.error("Execute Process Failed: %s" % str(traceback.format_exc()))
            return (False, self.getResCode(["fileCutError", str(traceback.format_exc())]))
        finally:
            self.clean_temp_dir()
            self.uiObserver.setProgress(100)

    def clean_temp_dir(self):
        """
        清理临时目录，当文件太大时，第一次删除因windows内部处理问题导致清理会失败，
        文件残留，需要重试。
        :return:
        """
        time_out = 60
        self.add_permission(self.tempDir)
        while time_out > 0:
            try:
                if not os.path.exists(self.tempDir):
                    self.logger.info("already delete success! {}".format(
                        self.tempDir))
                    break
                shutil.rmtree(self.tempDir, ignore_errors=False)
                self.logger.info("delete success! {}".format(self.tempDir))
                break
            except Exception:
                self.logger.error("clean_temp_dir Failed: {}".format(
                    str(traceback.format_exc())))
            time_out -= 1
            time.sleep(5)

    def add_permission(self, input_path):
        """
        递归增加目标目录下所有权限
        :param input_path:
        :return:
        """
        if not os.path.exists(input_path):
            return

        if os.path.isdir(input_path):
            file_dir_list = os.listdir(input_path)
            # 开始解压缩迭代及写入硬盘信息
            for file_dir in file_dir_list:
                file_path = os.path.join(input_path, file_dir)
                self.add_permission(file_path)
        else:
            try:
                os.chmod(input_path, stat.S_IWRITE)
            except Exception:
                self.logger.error(
                    "add permission failed: {}".format(str(
                        traceback.format_exc())))

    def preProcess(self):
        """
                    预处理功能入口:
                    （1）创建临时文件夹
                    （2）拷贝源文件到文件保存路径
                    （3）拷贝后文件改名temp.7z
                    （4）解压重命名后文件
                    （5）设置切割后文件夹
        """

        #初始化中，进度设定为1%
        self.uiObserver.setProgress(1)
        try:
            #（1）创建临时文件夹
            tempDirName = "temp_" + self.timeStamp
            self.tempDir = os.path.join(self.SaveDir, tempDirName)
            try:
                os.makedirs(self.tempDir)
            except:
                self.logger.error("Execute preProcess Failed: %s" % str(traceback.format_exc()))
                return (False, self.getResCode(["createDirError", self.tempDir]))
            self.uiObserver.setProgress(5)
    
            try:
                #（2）拷贝源文件到文件保存路径
                if self.srcFileName not in os.listdir(self.tempDir):
                    shutil.copy2(self.file, self.tempDir)
            except:
                self.logger.error("Execute preProcess Failed: %s" % str(traceback.format_exc()))
                return (False, self.getResCode(["copyMoveFileError", self.file]))
            self.uiObserver.setProgress(15)
    
            #（3）拷贝后文件改名temp.7z：为了避免名字过长导致路径超过操作系统限制，文件名称缩减
            try:
                fileShortName = os.path.join(self.tempDir, "temp.7z")
                os.rename(os.path.join(self.tempDir, self.srcFileName), fileShortName)
                self.file = fileShortName
            except:
                self.logger.error("Execute preProcess Failed: %s" % str(traceback.format_exc()))
                return (False, self.getResCode(["fileRenameError", self.srcFileName]))
            self.uiObserver.setProgress(20)
            
            #（4）解压重命名后文件
            self.baseDir = self._decompressFile()
            self.uiObserver.setProgress(40)

            #（5）设置切割后文件夹（暂时不创建，防止无文件需要移动）
            self.setCutDir()
            return (True, "")
        except:
            self.logger.error("Execute preProcess Failed: %s" % str(traceback.format_exc()))
            return (False, self.getResCode(["preProcessError", str(traceback.format_exc())]))

    def fileMoveProcess(self):
        """
                    子文件移动流程:
                    （1）遍历解压后的所有文件
                    （2）非Messages文件直接移动到新文件夹；满足时间要求的Messages文件也移动到新文件夹
        """
        try: 
            #路径有效性判断
            if not os.path.exists(self.baseDir) or \
                not os.path.isdir(self.baseDir):
                return (False, self.getResCode(["fileOrDirNotExist", self.baseDir]))
            
            self.uiObserver.setProgress(45)
            #遍历所有文件及文件夹
            self.moveFileCnt = 0
            for subDir, dirs, files in os.walk(self.baseDir):
                for fileTmp in files:
                    fileFullPath = os.path.join(subDir, fileTmp)
                    if not self.isFileNeedCut(fileFullPath):
                        continue            
                    self.moveFileCnt += 1
                    
                    #递归移动文件 :文件移动过程中缓慢更新进度条
                    self.moveObserverUpdate(45, 80)
                    moveFlag = self.move_file_to_cut_dir(fileFullPath, subDir)
                    #如果是静默模式调用日志拆分如果无法拆分则忽略。
                    self.logger.info("Get startMode is [%s]" % self.startMode)
                    if not moveFlag and self.startMode != SILENT_START_MODE:
                        self.logger.error("fileMoveProcess: file %s cut Error!" % fileFullPath)
                        return (False, self.getResCode(["fileMoveProcessfailed"]))
            self.uiObserver.setProgress(80)
            #创建拆分基本信息文件
            self.createCutInfoFile()
            return (True, "")
        except:
            self.logger.error("Execute fileMoveProcess Failed: %s" % str(traceback.format_exc()))
            return (False, self.getResCode(["fileMoveProcessError", str(traceback.format_exc())]))

    def postProcess(self):
        """
                    分割后处理流程:
                    （1）cut文件夹压缩，重新命名
                    （2）移动文件到设定的保存路径中
        """
        try:        
            if not os.path.exists(self.cutDir):
                return (True, "")

            #压缩cut文件夹，重命名后移动到保存路径
            CutFile = self.FileCompress.compressPackage(os.path.join(self.cutDir))
            newFile = os.path.join(os.path.dirname(CutFile), self.cutFileName)
            os.rename(CutFile, newFile)
            shutil.move(newFile, self.SaveDir)
            self.uiObserver.setProgress(90)
            return (True, "")
        except:
            self.logger.error("Execute postProcess Failed: %s" % str(traceback.format_exc()))
            return (False, self.getResCode(["postProcessError", str(traceback.format_exc())]))
        finally:
            #删除临时文件夹
            shutil.rmtree(self.tempDir, ignore_errors=True)
            self.uiObserver.setProgress(100)

    def moveObserverUpdate(self, start, end):
        """
                    文件移动过程中的进度条刷新（每移动10个文件刷新一次，最多不能超过截止进度）
        """
        newProcess = start + self.moveFileCnt /10
        if newProcess >= end:
            newProcess = end
        self.uiObserver.setProgress(newProcess)
    
    def getResCode(self, params):
        """
                    界面显示信息获取（可带参数）
        """
        curPath = os.path.split(os.path.realpath(__file__))[0]
        #根据语言环境解析文件
        if self.lang == 'zh':
            fileName = curPath + os.sep + CODE_ZH_FILE_NAME
        else:
            fileName = curPath + os.sep + CODE_EN_FILE_NAME        

        p = Properties(fileName)
        p.setParamPartStr("#???")#设置参数分隔符
        msgDesc = p.getProperties(params)
        return msgDesc if msgDesc else "Unknown Error"

    def _decompressFile(self):
        """
                    解压文件
        """
        deCompressCmds = ['OceanspaceS5000T', '']
        return self.FileCompress.deCompressPackage(self.file, BLACKS, deCompressCmds) 
    
    def setCutDir(self):
        """
                    设置cut文件夹名称
        """
        #获取文件夹名称
        cutTmpDirName = "cut_" + self.timeStamp
        #创建新文件夹（切割后文件夹）
        self.cutDir = os.path.join(os.path.dirname(self.baseDir), cutTmpDirName)   

    def contructTime(self, timeStamp):
        """功能：构造规范性的时间戳
                      传入格式为：20160620133322
                      输出格式为：2011-09-28 10:00:00
        """
        if len(timeStamp) != 14:
            return ""
        newTimeStamp = timeStamp[0:4] + "-" + timeStamp[4:6] + \
                "-" + timeStamp[6:8] + " " + timeStamp[8:10] + ":" + timeStamp[10:12] + ":" + timeStamp[12:14]
        return newTimeStamp
    
    def getfileNameTimeStamp(self, fileName):
        """
                     功能：获取文件名称中的时间戳
        """

        patternList = ["_(\d{14})", "_(\d{8}_\d{6})", "_(\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2})"]
        timeStamp = ""
        for pattern in patternList:
            findRes = re.findall(pattern, fileName)
            if findRes:
                timeStamp = findRes[-1].replace("-", "").replace("_", "")
                break

        #获取规范化的时间戳
        timeStamp = self.contructTime(timeStamp)
        return timeStamp

    def _isFileInTime(self, subFile):
        """
                    判断文件的是否在规定范围内
        """
        #优先使用文件名称中的时间戳
        fileTimeStamp = self.getfileNameTimeStamp(os.path.basename(subFile))
        if not fileTimeStamp:
            fileStat = os.stat(subFile)
            #文件名称中时间获取失败，使用文件的修改时间
            modifyTime = fileStat.st_mtime
            timeArray = time.localtime(modifyTime)
            fileTimeStamp = time.strftime("%Y-%m-%d %H:%M:%S", timeArray)
        
        #文件在选择时间段内
        if fileTimeStamp and self.startTime <= fileTimeStamp <= self.endTime:
            return True
        return False

    def createCutInfoFile(self):
        """
                    创建拆分基本信息文件
        """ 
        #cut文件夹不存在，不需要创建
        if not os.path.exists(self.cutDir):
            return
        #cut文件夹存在（有文件满足拆分要求，已经创建cut文件夹）
        cutFile = os.path.join(self.cutDir, "cutInformation.txt")
        cutFileHandle = os.fdopen(os.open(cutFile, OS_OPEN_FLAGS, OS_OPEN_MODES), 'w')
        cutFileHandle.write("sourcefile=%s\n"% self.srcFileName) 
        cutFileHandle.write("startTime=%s\n"% self.startTime)
        cutFileHandle.write("endTime=%s\n"% self.endTime)
        cutFileHandle.close()
        return

    def _isMessageFile(self, subFile):
        """
                    判断文件是否为日志文件
        """ 
        filePath = os.path.dirname(subFile)
        #去掉基础路径
        filePath = filePath.replace(self.baseDir, "") + os.path.sep
        if self.logDirName in filePath:
            return True
        return False

    def isFileNeedCut(self, subFile):
        """
                    判断文件是否应该被切割
        """ 
        #满足如下条件不需要切割：message文件，并且不在规定时间内
        if self._isMessageFile(subFile) and not self._isFileInTime(subFile):
            return False
        return True

    def moveFile2CutDir(self, subFile, subDir):
        """
                    移动文件到切割后文件夹
        """ 
        try:
            #拷贝后文件所在新目录
            newFullDir = subDir.replace(self.baseDir, self.cutDir)
            #递归创建新目录
            if not os.path.exists(newFullDir):
                os.makedirs(newFullDir)
            #移动文件到新目录
            shutil.move(os.path.join(subDir, subFile), newFullDir)
        except:
            self.logger.error("moveFile2CutDir: file %s move Error! Details:" % str(traceback.format_exc()))
            return False
        return True

    def move_file_to_cut_dir(self, sub_file, sub_dir):
        """
        移动文件到切割后文件夹
        :param sub_file: 文件
        :param sub_dir: 切割后文件夹
        :return:
        """
        retries_num = 60
        while retries_num > 0:
            try:
                # 拷贝后文件所在新目录
                new_full_dir = sub_dir.replace(self.baseDir, self.cutDir)
                # 递归创建新目录
                if not os.path.exists(new_full_dir):
                    os.makedirs(new_full_dir)
                # 移动文件到新目录
                shutil.move(os.path.join(sub_dir, sub_file), new_full_dir)
                return True
            except Exception:
                self.logger.error("moveFile2CutDir: file %s move Error! Details:" % str(traceback.format_exc()))
            retries_num -= 1
            time.sleep(5)
        self.logger.error("Failed to move file to cut dir after 60 retries.")
        return False

"""流程入口"""
def execute(dataDict):
    fileHand = fileHandler(dataDict)
    return fileHand.logCutProcess()
