#!/usr/bin/python
# -*- coding: utf-8 -*-
import json
import os
import shutil

from check_item.check_util.sftp import SftpClient
from baseline import rpc_opcode
from config import path as cfg_path
from config.rpc import RpcRet
from infra.debug.log import swm_logger as log
from infra.rpc.rpc_client import RpcClient
from plat.host.host import IpInfo
from plat.host.host_mgr import HostMgr


class ExtendParamMgr(object):
    _LOG_PREFIX = "ExtendParamMgr"
    _MAX_HOST_TO_GET_FILE = 8

    @classmethod
    def get_extend_param(cls, read_param):
        if not os.path.exists(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE):
            cls.get_file_from_other_node(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE)
            if not os.path.exists(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE):
                log.info("%s: No extend param tmp file(%s)",
                         cls._LOG_PREFIX, cfg_path.CHECK_EXTEND_PARAM_TMP_FILE)
                return True, ""
        try:
            with open(cfg_path.CHECK_EXTEND_PARAM_TMP_FILE, 'r') as f:
                read_str = str(f.read()).replace("'", '"')
                if not read_str:
                    log.info("%s: No extend param in tmp file(%s) for %s",
                             cls._LOG_PREFIX,
                             cfg_path.CHECK_EXTEND_PARAM_TMP_FILE, read_param)
                    return True, ""
                log.info("%s: Read str(%s).", cls._LOG_PREFIX, read_str)
                extend_param = json.loads(read_str)
                out_param = \
                    str(extend_param.get(read_param, ''))
                log.info("%s: Key param read from extend param file is '%s'.",
                         cls._LOG_PREFIX, out_param)
                return True, out_param
        except Exception:
            log.exception("%s: Exeception.", cls._LOG_PREFIX)
            return False, ""

    @classmethod
    def get_file_from_other_node(cls, file_path):
        try:
            host_list = HostMgr.get_ctl_host_list()
            # Obtain files only from the previous eight controllers
            # to prevent long time when no file is available.
            if len(host_list) > cls._MAX_HOST_TO_GET_FILE:
                host_list = host_list[:cls._MAX_HOST_TO_GET_FILE]
            local_tmp_file = file_path + "_tmp"
            for host in host_list:
                host_ips = [ip.addr for ip in host.host_ip if
                            ip.state == IpInfo.STATE_VALID]
                sftp = SftpClient(host_ips[0])
                if sftp.get_file(local_tmp_file, file_path):
                    log.info("%s: Get file(%s) from(%s) successfully.",
                             cls._LOG_PREFIX, file_path, host_ips[0])
                    shutil.move(local_tmp_file, file_path)
                    break
                log.error("%s: Get file(%s) failed.",
                          cls._LOG_PREFIX, file_path)
        except Exception:
            log.exception(
                "%s: An exception occurred when obtaining the file(%s).",
                cls._LOG_PREFIX, file_path)
        return

    @classmethod
    def _get_file(cls, management_ip, file_path):
        host_id_list = HostMgr.get_ctrl_host_id_list()
        # Obtain files only from the previous eight controllers
        # to prevent long time when no file is available.
        local_tmp_file = file_path + "_tmp"
        host_ip_list = []
        for host_id in host_id_list:
            host_ip_list = HostMgr.get_node_ip(host_id)
            rpc_ret, mgt_ip = RpcClient. \
                start_rpc_call(host_ip_list, rpc_opcode.RPC_OPCODE_GET_MGMT_IP)
            if rpc_ret == RpcRet.OK and mgt_ip and mgt_ip == management_ip:
                log.info("%s: Management ip(%s) have found, host id is(%s)"
                         ", begin to get file",
                         cls._LOG_PREFIX, management_ip, host_id)
                break

        for host_ip in host_ip_list:
            sftp = SftpClient(host_ip)
            if sftp.get_file(local_tmp_file, file_path):
                log.info("%s: Get file(%s) from(%s) successfully.",
                         cls._LOG_PREFIX, file_path, host_ip)
                shutil.move(local_tmp_file, file_path)
                break

    @classmethod
    def get_file_from_management_ip(cls, management_ip, file_path):
        if not management_ip:
            log.error("%s: Management ip is empty, can not get file "
                      "from other.", cls._LOG_PREFIX)
            return False
        try:
            cls._get_file(management_ip, file_path)

            log.error("%s: Get file(%s) failed.",
                      cls._LOG_PREFIX, file_path)
            return True
        except Exception as e:
            log.exception(
                "%s: An exception occurred when obtaining the file(%s): %s.",
                cls._LOG_PREFIX, file_path, e)
        return False
