#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
# @brief   : 2018/3/1 15:23
# @File    : shell.py
# @Software: Software management
"""

__version__ = '1.1.0.1'
__date__ = '2018/3/1'

import functools
import os
import platform
import subprocess
import sys
import time

from infra.debug.log import swm_logger as log


__all__ = ['call_system_cmd', 'remount_disk']

PYTHON_2 = '2.7' in sys.version
MSWIN = (platform.system() == "Windows")
PATH_FIX = '.' if MSWIN else ''
IMAGE_DISK = PATH_FIX + '/startup_disk/image'


def decode_str(value):
    if not PYTHON_2 and isinstance(value, bytes):
        return value.decode(errors='ignore')
    return value


# pylint: disable=W0703
def call_system_cmd(cmd, input_str=''):
    """

    :param cmd:
    :param input_str:
    :return:
    """
    ret_code = -1
    output = ""
    try:
        process = subprocess.Popen(cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE, shell=True)
        if input_str:
            process.stdin.write(input_str)
            process.stdin.flush()
        output = decode_str(process.stdout.read())
        ret_code = process.wait()
        if ret_code:
            log.warning("Exec cmd(%s) result abnormal, retCode(%d), result(%s).", cmd, ret_code, output)
        else:
            log.info("Exec cmd(%s) successfully.", cmd)
        return ret_code, output
    except Exception:
        log.error("Exec cmd(%s) except, retCode(%d), result(%s).", cmd, ret_code, output)
        return ret_code, output


def remount_disk(disk, mode='rw'):
    """

    :param disk:
    :param mode:
    :return:
    """
    if not os.path.ismount(disk):
        log.error("The disk(%s) is not mounted.", disk)
        return False
    ret_code, output = call_system_cmd("mount -o remount,%s %s" % (mode, disk))
    if ret_code:
        log.error("Remount disk(%s) to mode(%s) failed(ret: %d, output: %s).", disk, mode, ret_code, output)
        return False
    log.debug("Remount disk(%s) to mode(%s) successfully.", disk, mode)
    return True


def remount_image_disk(rw):
    """

    :param rw:
    :return:
    """
    mode = 'rw,async' if rw else 'ro,sync'
    for dummy in range(10):
        ret = remount_disk(IMAGE_DISK, mode)
        if ret:
            return True
        time.sleep(1)
    log.error("SHELL: Remount image disk(mode: %s) for 10 times failed.", mode)
    return False


def remount_image(run_func):
    """

    :param run_func:
    :return:
    """

    @functools.wraps(run_func)
    def wrapper(*args, **kwargs):
        """

        :param self:
        :param args:
        :param kwargs:
        :return:
        """
        if not remount_image_disk(True):
            log.error("Rmount image disk for(%s) failed.", run_func)
            return False
        result = run_func(*args, **kwargs)
        remount_image_disk(False)
        return result

    return wrapper
