#!/usr/bin/python
# -*- coding: utf-8 -*-
import sys
import yaml
import os
import getopt
import logging


def get_patch_conf(conf_file):
    if not os.path.exists(conf_file):
        return dict()
    fd = open(conf_file)
    patch_conf = yaml.safe_load(fd)
    fd.close()
    return patch_conf


def get_patch_version(conf_file):
    patch_conf = get_patch_conf(conf_file)
    if not patch_conf:
        return None, None
    ver_conf = patch_conf.get("PatchConf", None)
    if not ver_conf:
        return None, None
    return ver_conf.get('base_version', None), ver_conf.get('patch_version', None)


def install_patch_file_check(patch_dir, new_conf, old_conf):
    # 1.检查欧拉补丁是增量的
    new_euler_conf = new_conf.get('EulerPatch', [])
    old_euler_conf = old_conf.get('EulerPatch', [])

    for old_patch in old_euler_conf:
        if old_patch.get('patch_file') not in [x.get('patch_file') for x in new_euler_conf]:
            return 'consistence_error', 'patch %s not inherited' % old_patch.get('patch_file')

    # 2.检查Dopra补丁是增量的:
    new_dopra_conf = new_conf.get('DopraPatch', dict())
    old_dopra_conf = old_conf.get('DopraPatch', dict())
    for old_process, old_patches in list(old_dopra_conf.items()):
        if old_process not in list(new_dopra_conf.keys()):
            return 'consistence_error', 'process %s not inherited' % old_process
        new_patch_list = [x.get('patch_file') for x in new_dopra_conf[old_process] ]
        for old_patch in old_patches:
            if old_patch.get('patch_file') not in new_patch_list:
                return 'consistence_error', 'process %s patch %s not inherited'\
                       % (old_process, old_patch.get('patch_file'))

    # 3.检查欧拉补丁存在
    for patch in new_euler_conf:
        if not os.path.exists(os.path.join(patch_dir, patch.get('patch_file'))):
            return 'existence_error', 'patch %s not exist' % patch.get('patch_file')

    # 4.检查Dopra补丁存在
    for process, patches in list(new_dopra_conf.items()):
        for patch in patches:
            if not os.path.exists(os.path.join(patch_dir, process, patch.get('patch_file'))):
                return 'existence_error', 'process %s patch %s not exist'\
                       % (process, patch.get('patch_file'))

    return 'pass', 'patch file check passed'


def filter_num(string, type_int=True):
    res_list = list(filter(str.isdigit, string))
    res = "".join(res_list)
    return int(res) if type_int else res


def install_check():
    version = os.readlink('/startup_disk/image/pkg_cur').split('/')[0]
    check_patch_dir = os.path.join('/startup_disk/image', version, 'hotpatch/patch_temp')
    check_patch_conf_file = os.path.join(check_patch_dir, 'patch.yml')
    if not os.path.exists(check_patch_conf_file):
        return 'error', 'patch not exist'
    check_patch_conf = get_patch_conf(check_patch_conf_file)
    if not check_patch_conf:
        return 'error', 'patch conf parse failed'

    check_version, check_sph = get_patch_version(check_patch_conf_file)
    # 1.检查与基线版本是否一致
    if not check_version or str(check_version) != str(version):
        return 'error_base_ver', 'base version consistence check failed'

    old_patch_dir = os.path.join('/startup_disk/image', version, 'hotpatch/patch_cur')
    old_patch_conf_file = os.path.join(old_patch_dir, 'patch.yml')
    old_version, old_sph = get_patch_version(old_patch_conf_file)
    old_patch_conf = get_patch_conf(old_patch_conf_file)

    # 2.检查补丁版本是否收编
    if old_sph and filter_num(old_sph) == filter_num(check_sph):
        return 'error_patch_ver', 'sph version same'

    # 3.检查补丁文件正确
    result, output = install_patch_file_check(check_patch_dir, check_patch_conf, old_patch_conf)
    if result != 'pass':
        return result, output
    return 'pass', 'patch check passed'


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "ih", ["help", "install"])
        for o, a in opts:
            if o in ['-i', '--install']:
                ret, output = install_check()
                print(ret)
                print(output)
                return 0
    except Exception as e:
        print("exception")
        print("exception")
        print(e)
        logging.exception(e)
        return 1
    return 0


if __name__ == '__main__':
    sys.exit(main())
