#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import getopt
import os
import sys
import re
import json

from check_util.cli_executor import CliExecutor
from comm_check_func import get_upg_sys_version, show_verbose, get_patch_version
from comm_check_func import log as logging

CUR_MANIFEST_PATH = '/OSM/conf/manifest.yml'
IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'

VERSION_V6 = "6.1.6"
VERSION_V6_NO = "7600513292"

CHECK_EXTEND_PARAM_PATH = "/tmp/check_extend_param"


def compare_version(version1, version2):
    versions1 = version1.split(".")
    versions2 = version2.split(".")
    for i, item in enumerate(versions1):
        if item == versions2[i]:
            continue
        elif item > versions2[i]:
            return 1
        else:
            return -1
    return 0


def is_exit_roce_port():
    res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file("show logical_port general")
    if not res:
        return False
    with open(cli_exec_tmp_file) as lines:
        lines_array = lines.readlines()
        for line in lines_array:
            if 'NVMe over RoCE' in line:
                return True
    return False


def get_target_version():
    with open(CHECK_EXTEND_PARAM_PATH, 'r') as fp:
        read_str = str(fp.read()).replace("'", '"')
        if not read_str:
            return 0
        target_version = json.loads(read_str).get("targetVersion")
        target_digit_ver = target_version.split(".SPC")[0]
        return target_digit_ver.split("RC")[0]


def check_upgrade_version():
    if not is_exit_roce_port():
        show_verbose('True', '', '', '')
        return 0
    target_version = get_target_version()
    
    if not target_version:
        logging.warning("CHECK_NOF_UPGRADE_VERSION: get target version failed.")
        show_verbose('False', '', '', '')
        return 1

    logging.info("CHECK_NOF_UPGRADE_VERSION: upgrade target_version is %s", target_version)
    cmp = compare_version(target_version, VERSION_V6)
    if cmp >= 0:
        show_verbose('True', '', '', '')
        return 0
    else:
        show_verbose('False', '', '', '')
        return 0


def check_software_and_patch_version():
    if not is_exit_roce_port():
        show_verbose('True', '', '', '')
        return 0
    upgrade_patch_cfg = {
        '7600506206': 'SPH52',  # 6.1.2版本
        '7600509200': 'SPH22',  # 6.1.3版本
        '7600511219': 'SPH20'   # 6.1.5版本
    }

    res, upg_sys_ver = get_upg_sys_version()
    if not res:
        logging.warning("CHECK_NOF_UPGRADE_VERSION: get system version failed.")
        show_verbose('False', '', '', '')
        return 1

    upg_patch_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), 'hotpatch', 'patch_cur', PATCH_MANIFEST)
    upg_patch_ver = get_patch_version(upg_patch_path)

    if not upg_patch_ver:
        patch_digit_ver = 0
    else:
        patch_digit_ver = int(re.split('(\\d+)', upg_patch_ver)[1])
    logging.info("CHECK_NOF_UPGRADE_VERSION: upg_sys_ver(%s), upg_patch_ver(%s)", upg_sys_ver, upg_patch_ver)

    if upg_sys_ver >= VERSION_V6_NO:
        show_verbose('True', '', '', '')
        return 0
    elif upg_sys_ver in upgrade_patch_cfg:
        require_patch_digit_ver = int(re.split('(\\d+)', upgrade_patch_cfg[upg_sys_ver])[1])
        if patch_digit_ver >= require_patch_digit_ver:
            show_verbose('True', '', '', '')
            return 0
        else:
            show_verbose('False', '', '', '')
            return 0
    else:
        show_verbose('False', '', '', '')
        return 0


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, _ = getopt.getopt(argv[1:], "ce", ["check_nof_software_and_patch_version", "check_nof_upgrade_version"])
        for opt, _ in opts:
            if opt in ['-c', '--check_nof_software_and_patch_version']:
                return check_software_and_patch_version()
            if opt in ['-e', '--check_nof_upgrade_version']:
                return check_upgrade_version()

    except Exception as e:
        logging.exception("CHECK_NOF_UPGRADE_VERSION: upgrade version failed %s", e)
        show_verbose('False', '', '', '')
        return 1
    return 0


if __name__ == '__main__':
    sys.exit(main())
