#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import getopt
import os
import sys
import re
import json
import subprocess
import shlex

from check_util.cli_executor import CliExecutor
from comm_check_func import get_upg_sys_version, show_verbose, get_patch_version, get_cur_sys_version
from comm_check_func import log as logging

# 当前版本信息
CUR_MANIFEST_PATH = '/OSM/conf/manifest.yml'
# 升级目标版本信息
UPG_MANIFEST_PATH = '/startup_disk/image/pkg_upd/manifest.yml'
IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'
CHECK_EXTEND_PARAM_PATH = "/tmp/check_extend_param"

PRODUCT_MODEL_OCEAN_PROTECT = "OceanProtect"
PRODUCT_MODEL_OCEAN_DISK = "OceanDisk"
PRODUCT_MODEL_OCEAN_STOR_2910 = "OceanStor 2910"

SPH_30 = 'SPH30'
SPH_7 = 'SPH7'
TIMEOUT = 30
HOTPATCH = "hotpatch"


def diagnose(command):
    cmd = 'timeout -s 9 10 diagsh --attach=*_12 --cmd="%s"' % command
    process = subprocess.Popen(shlex.split(cmd), stdout=subprocess.PIPE)
    out = process.communicate(timeout=TIMEOUT)
    return out


def is_ctrl_2p():
    res, ctrl_tmp = CliExecutor.exec_show_cmd_to_file("show controller general")
    if not res:
        return False

    with open(ctrl_tmp) as lines:
        lines = lines.readlines()
        for line in lines:
            match = re.findall(r"Description=([\s\S]+2P)", line)
            if not match:
                continue
            logging.info("CHECK_ZERO_SLOT_CATD_TYPE:controller = (%s)", line)
            return True
    return False


def is_open_mem_policy():
    mem_info = str(diagnose("dpumm_shm show ramnumainfo"))
    logging.info("CHECK_ZERO_SLOT_CATD_TYPE:mem_info = (%s)", mem_info)
    if "More reliability numa info" in mem_info:
        return True
    return False


def is_roce_card_in_slot_zero():
    res, interface = CliExecutor.exec_show_cmd_to_file("show interface_module")
    if not res:
        logging.info("CHECK_ZERO_SLOT_CATD_TYPE: show interface_module failed")
        return False
    with open(interface) as lines:
        lines = lines.readlines()
        for line in lines:
            if "IOM.H0" in line and "FE" in line and "ROCE" in line:
                logging.info("CHECK_ZERO_SLOT_CATD_TYPE:card_info = (%s)", line)
                return True
    return False


# 获取升级前工勘的目标版本与目标补丁版本
def get_target_version():
    with open(CHECK_EXTEND_PARAM_PATH, 'r') as fp:
        read_str = str(fp.read()).replace("'", '"')
        if not read_str:
            return 0
        target_version = json.loads(read_str).get("targetVersion")
        target_digit_ver = target_version.split(".SPC")[0]
        return target_digit_ver.split("RC")[0]


def check_patch_version(target_version, patch_ver):
    upgrade_patch_cfg = {
        '7600511219': 'SPH30',  # 6.1.5版本
        '7600513292': 'SPH7',  # 6.1.6版本
        '7600516210': 'SPH2'  # 6.1.7版本
    }

    if target_version not in upgrade_patch_cfg:
        return True

    require_patch_ver = int(re.split('(\\d+)', upgrade_patch_cfg.get(target_version))[1])
    logging.info("CHECK_ZERO_SLOT_CATD_TYPE: check_patch_version version(%s)(%s)(%s)",
                 patch_ver, require_patch_ver, patch_ver < require_patch_ver)
    if patch_ver < require_patch_ver:
        return False
    return True


def filter_version(target_version):
    pass_version_dict = {
        "6.1.5": "SPH30",
        "6.1.6": "SPH7",
        "6.1.7": "SPH2",
    }

    upgrade_patch_cfg = {
        '7600511219': 'SPH30',  # 6.1.5版本
        '7600513292': 'SPH7',  # 6.1.6版本
        '7600516210': 'SPH2'  # 6.1.7版本
    }

    # 过滤不涉及的版本
    return target_version not in upgrade_patch_cfg and target_version not in pass_version_dict


def check_software_and_patch_version(cmd):
    # 获取补丁版本
    target_version, target_spc_version, target_apollo_version = get_cur_sys_version()
    logging.info("CHECK_ZERO_SLOT_CATD_TYPE: get_cur_sys_version = (%s)、(%s)、(%s)", target_version, target_spc_version,
                 target_apollo_version)

    # 获得该系统版本的补丁版本目录
    patch_path = os.path.join(IMAGE_DISK, str(target_version), HOTPATCH, "patch_cur", PATCH_MANIFEST)
    # 获得该系统版本的目标安装补丁版本目录
    temp_patch_path = os.path.join(IMAGE_DISK, str(target_version), HOTPATCH, "patch_temp", PATCH_MANIFEST)
    # 获得补丁版本
    cur_patch_ver = get_patch_version(patch_path)
    target_patch_ver = get_patch_version(temp_patch_path)
    logging.info("CHECK_ZERO_SLOT_CATD_TYPE: patch_path(%s), temp_patch_path(%s), patch_ver(%s)(%s)",
                 patch_path, temp_patch_path, cur_patch_ver, target_patch_ver)

    # 过滤当前版本
    if filter_version(target_version):
        return True

    # 快速升级评估(smartkit上是升级工勘)
    if cmd == "evaluate_fast_upgrade":
        target_version = get_target_version()
        logging.warning("CHECK_ZERO_SLOT_CATD_TYPE: evaluate fast upgrade target_version(%s).", target_version)
        if not target_version:
            logging.warning("CHECK_ZERO_SLOT_CATD_TYPE: get target version failed.")
            return False
        return filter_version(target_version)

    # 快速升级前检查
    if cmd == "fast_upgrade_pre_check" or cmd == "fast_upgrade_pre_check_cli":
        res, upg_sys_ver = get_upg_sys_version()
        logging.info("CHECK_ZERO_SLOT_CATD_TYPE: upg_sys_ver = (%s)", upg_sys_ver)
        if not res:
            logging.warning("CHECK_ZERO_SLOT_CATD_TYPE: get system version failed.")
            return False
        # 过滤目标版本
        if filter_version(upg_sys_ver):
            return True
        # 获得补丁版本
        upg_patch_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), HOTPATCH, "patch_cur", PATCH_MANIFEST)
        upg_patch_ver = get_patch_version(upg_patch_path)
        if not upg_patch_ver:
            patch_digit_ver = 0
        else:
            patch_digit_ver = int(re.split('(\\d+)', upg_patch_ver)[1])
        logging.info(
            "CHECK_ZERO_SLOT_CATD_TYPE: upg_sys_ver(%s), upg_patch_path(%s), upg_patch_ver(%s), patch_digit_ver(%s)",
            upg_sys_ver, upg_patch_path, upg_patch_ver, patch_digit_ver)

        return check_patch_version(upg_sys_ver, patch_digit_ver)

    # 补丁安装检查
    if cmd == "hotpatch_install_pre_check":
        if not target_patch_ver:
            patch_digit_ver = 0
        else:
            patch_digit_ver = int(re.split('(\\d+)', target_patch_ver)[1])
        return check_patch_version(target_version, patch_digit_ver)

    # 补丁评估 hotpatch_evaluate
    if cmd == "hotpatch_evaluate":
        # 补丁评估获取不到补丁版本直接拦截提示
        return False

    return True


def filter_product_model():
    models = {
        'OceanStor 18500K', 'OceanStor 18510', 'OceanStor 18810', 'OceanStor 6810',
        'OceanStor Dorado 18800K V6', 'OceanStor Dorado 18000 V6', 'OceanStor Dorado 18500 V6',
        'OceanStor Dorado 18800 V6', 'OceanStor Dorado 8000 V6', 'OceanStor Dorado 6800 V6'
    }

    res, interface = CliExecutor.exec_show_cmd_to_file("show system general")
    if not res:
        logging.info("CHECK_ZERO_SLOT_CATD_TYPE: show system general failed")
        return False
    with open(interface) as lines:
        lines = lines.readlines()
        for line in lines:
            if "Product Model" in line:
                logging.info("CHECK_ZERO_SLOT_CATD_TYPE:product_model(%s)", line)
                product_model = line.split(":")[1].strip()
                logging.info("CHECK_ZERO_SLOT_CATD_TYPE:product_model(%s)(%d)", product_model, product_model in models)
                return product_model in models
    return False


def check_zero_slot_card():
    return is_ctrl_2p() and is_open_mem_policy() and is_roce_card_in_slot_zero()


def show_true():
    show_verbose("True", "", "", "")


def show_false():
    show_verbose("False", "", "", "")


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        if not filter_product_model():
            show_true()
            return 0

        res = check_zero_slot_card()
        logging.info("CHECK_ZERO_SLOT_CATD_TYPE: pre-check condition res(%s)", res)
        if not res:
            show_true()
            return 0

        opts, _ = getopt.getopt(argv[1:], "cethp",
                                ["fast_upgrade_pre_check_cli", "evaluate_fast_upgrade", "fast_upgrade_pre_check",
                                 "hotpatch_evaluate", "hotpatch_install_pre_check"])
        ret = False
        for opt, _ in opts:
            if opt in ['-t', '--fast_upgrade_pre_check']:
                ret = check_software_and_patch_version("fast_upgrade_pre_check")  # EntireFastUpgrade smartkit升级前检查
            if opt in ['-c', '--fast_upgrade_pre_check_cli']:
                ret = check_software_and_patch_version("fast_upgrade_pre_check_cli")  # FastUpgrade
            if opt in ['-e', '--evaluate_fast_upgrade']:
                ret = check_software_and_patch_version("evaluate_fast_upgrade")  # FastUpgradeEvaluate 升级评估
            if opt in ['-h', '--hotpatch_evaluate']:
                ret = check_software_and_patch_version("hotpatch_evaluate")  # HotpatchEvaluate
            if opt in ['-p', '--hotpatch_install_pre_check']:
                ret = check_software_and_patch_version("hotpatch_install_pre_check")  # HotpatchPre-check

        if ret:
            show_true()
        else:
            show_false()
    except Exception as e:
        logging.exception("CHECK_ZERO_SLOT_CATD_TYPE: %s", e)
        show_false()
    return 0


if __name__ == '__main__':
    sys.exit(main())
