#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
import re
import sys

from config import path
from infra.debug.log import swm_logger as log
from infra.util import pkg
from service.version.ver_mgr import VerMgr

BSP_FILE = "/proc/osp/bsp"
MEMFLOW_ACTIVED_FLAG = "/OSM/coffer_runningdata/container/memflow_activated"


def _is_cur_version_activated(src_ver):
    if not os.path.exists(MEMFLOW_ACTIVED_FLAG):
        log.info("DOWNGRADE_CHECK: container is not memflow activated.")
        return False
    with open(MEMFLOW_ACTIVED_FLAG, 'r') as flag_file:
        try:
            active_version = flag_file.read().strip()
        except Exception as e:
            log.error("DOWNGRADE_CHECK: flag file is err %s.", e)
            return False
        if str(src_ver).strip() == str(active_version).strip():
            log.info("DOWNGRADE_CHECK: container is activated in cur version")
            return True
    log.info("DOWNGRADE_CHECK: container is activated in past version")
    return False


def _container_enabled():
    with open(BSP_FILE, "rb") as f:
        lines = f.readlines()
    for line in lines:
        try:
            line = line.decode("utf-8")
        except Exception as e:
            log.error("DOWNGRADE_CHECK: line is not utf-8, error %s", e)
            continue
        pos = line.find("conta:")
        if pos == -1:
            continue
        info = line[pos:len(line)]
        tmp = re.split(":|;", info)
        level = tmp[1].strip()
        if level != "0":
            log.info("DOWNGRADE_CHECK: container is enabled.")
            return True
    log.info("DOWNGRADE_CHECK: container is not enabled.")
    return False


def downgrade_check(src_ver, target_ver):
    """
    通过调用diagnose或者CLI命令方式，实现降级检查逻辑，如果需要通过执行shell命令
    检查，建议直接按本脚本逻辑通过shell脚本实现。
    此脚本根据实际检查情况，打印检查结果、错误码和错误信息，
    升级模块解析打印回显，将检查结果、错误码和错误信息上报到升级工具
    检查结果说明：
    True或False
    错误码规则说明：
    十六进制，前四位为模块id，后四位为模块错误码，模块ID参考模块PID，
    错误码由模块自行分配管理，如：0x01B50001表示SWM模块（PID十进制为437）的001
    号错误码。
    错误信息说明：
    表示检查不通过的详细信息，用于匹配升级工具资源文件中的关键字，
    比如资源文件描述“硬盘利用率检查不通过，当前最大利用率为%s（检查阈值50%），
    检查不通过的硬盘为(%s)。”，说明资源文件需要匹配两个关键字，
    这种情况错误信息打印格式为：当前最大利用率;硬盘id列表，如90;DAE000.0,DAE000.1,
    关键字之间使用英文分号“;”分隔，参数内部如果需要返回列表使用英文逗号“,”分隔
    :param src_ver: 当前版本，如7600501032
    :param target_ver:降级的目标版本，如7600500099
    :return:
    """

    # 如果是走免重启激活容器的环境，则需要将容器去激活后才允许降级
    if _container_enabled() and _is_cur_version_activated(src_ver):
        print("False")
        return 1
    print("True")
    return 0


def main():
    """
    检查主函数，一般不需要修改
    :return:
    """
    try:
        # 当前版本，如果采用shell脚本实现，需要自行解析
        # /startup_disk/image/pkg_cur/manifest.yml中SYS下的version字段值
        src_ver = VerMgr.get_src_version()[0]
        # 降级/升级目标版本，如果采用shell脚本实现，需要自行解析
        # /startup_disk/image/pkg_upd/manifest.yml中SYS下的version字段值，
        # 此配置文件只有升级开始并完成软件包上传后才有
        target_ver = pkg.get_pkg_version(path.TARGET_CONFIG_PATH)[0]
        log.info("DOWNGRADE_CHECK: System Version: %s target version: %s.",
                 src_ver, target_ver)
        if not src_ver or not target_ver:
            log.error("DOWNGRADE_CHECK: The src or target version is empty.")
            print("False")
            print("1077949019")
            print("src version or target version is not exist")
            return 1
        downgrade_check(src_ver, target_ver)
        return 0
    except Exception as e:
        print("False")
        print("1077949019")
        print(e)
        log.exception(e)
        return 1


if __name__ == '__main__':
    sys.exit(main())