#!/usr/bin/python
# -*- coding: utf-8 -*-
import getopt
import re
import sys

from config import db as db_cfg
from config import path
from config.upd import UpgStatus, UpgType
from infra.db.swm_db import DbService as DbSrv
from infra.debug.log import swm_logger as log
from infra.util import pkg
from service.version.ver_mgr import VerMgr

ERR_CODE_UPG_TYPE_NOT_MATCH = 0x01B50001

ERR_CODE_UPG_PATH_NOT_MATCH = 0x01B50002

ERR_CODE_NO_UPG_PATH = 0x01B50003

ERR_CODE_UPG_PATH_NOT_SUPPORT = 0x01B50004

NOT_ALLOWED_DOWNLOAD_VERSION = ['7600301133', '7600302133']

D_3P_2P_MIN_VERSION = 7600710230

PYTHON_2 = '2.7' in sys.version

ROLL_APOLLO_UPG = 8
OFFLINE_APOLLO_UPG = 9


def load_record_info_from_db(exclude_apollo_upg=True):
    """

    :return:
    """
    if exclude_apollo_upg:
        sql = "select upg_type, pre_ver, post_ver from {0} where sn=" \
              "(select max(sn) from {0} where state={1} and upg_type not in ({2}, {3}))". \
            format(db_cfg.UPGRADE_RECORD_TBL, int(UpgStatus.UPD_SUCCESS), ROLL_APOLLO_UPG, OFFLINE_APOLLO_UPG)
    else:
        sql = ("select upg_type, pre_ver, post_ver from {0} where sn=(select max(sn) from {0} where state={1})".
               format(db_cfg.UPGRADE_RECORD_TBL, int(UpgStatus.UPD_SUCCESS)))
    ret = DbSrv.exec_sql(sql)
    if ret is None:
        log.error("Load progress failed.")
        return None
    if not ret:
        log.info("No progress info in db.")
        return None
    return [ret[0][0], ret[0][1], ret[0][2]]


def get_pangea_model():
    """
    获取盘古型号
    """
    try:
        with open('/OSM/script/proc_osp_bsp.info', 'r') as fd:
            bsp_info = fd.read()
        groups = re.search(r'Product Name is (\S+)', bsp_info).groups()
        if not groups:
            return ""
        board_type = groups[0]
        if PYTHON_2:
            board_type = board_type.decode(errors='ignore')
        return board_type
    except Exception:
        log.exception("DOWNGRADE_CHECK: Get pangea model failed.")
        return ""


def downgrade_check(src_ver, target_ver):
    """

    :param src_ver: 原版本
    :param target_ver: 目标版本
    :return:
    """
    upg_record = load_record_info_from_db()
    if not upg_record:
        log.warning("DOWNGRADE_CHECK: There is no any upgrade path in the db.")
        print("False")
        print(ERR_CODE_NO_UPG_PATH)
        print("")
        return False

    # 只能原路径降级
    if (upg_record[1] != target_ver) or (upg_record[2] != src_ver):
        log.warning("DOWNGRADE_CHECK: upgrade path is not original path, "
                    "his_src(%s), cur_target(%s), his_target(%s), cur_src(%s).",
                    upg_record[1], target_ver, upg_record[2], src_ver)
        print("False")
        print(ERR_CODE_UPG_PATH_NOT_MATCH)
        print("{0};{1}".format(target_ver, upg_record[1]))
        return True

    # 新硬件支持的最小版本
    pangea_model = get_pangea_model()
    if pangea_model in ['PANGEA_V6_HI1630_2P_2U2C_BOARD100G'] and int(target_ver) < D_3P_2P_MIN_VERSION:
        print("False")
        print(ERR_CODE_UPG_PATH_NOT_SUPPORT)
        print("")
        return True

    print("True")
    print("")
    print("")
    return True


def downgrade_version_check(src_ver, target_ver, fast_upg):
    if target_ver < src_ver and target_ver in \
            NOT_ALLOWED_DOWNLOAD_VERSION:
        log.warning("DOWNGRADE_CHECK: Can not downgrade version: %s.",
                    target_ver)
        print("False")
        print(ERR_CODE_UPG_PATH_NOT_SUPPORT)
        print("{0};{1}".format(src_ver, target_ver))
        return 1

    ret = check_apollo(fast_upg)
    if not ret:
        return 1
    if not downgrade_check(src_ver, target_ver):
        return 1
    return 0


def check_apollo(fast_upg):
    """
    快速升级+最新一次是内核升级：不允许快速降级
    :param fast_upg:
    :return: 检查是否成功，如果成功后续查询升级记录是否排除内核升级
    """
    if not fast_upg:
        log.info("DOWNGRADE_CHECK: Not fast upgrade no need check apollo.")
        return True
    upg_record = load_record_info_from_db(exclude_apollo_upg=False)
    if not upg_record:
        print("False")
        print(ERR_CODE_NO_UPG_PATH)
        print("")
        return False
    upg_type = upg_record[0]
    #
    if int(upg_type) in [ROLL_APOLLO_UPG, OFFLINE_APOLLO_UPG]:
        log.error("DOWNGRADE_CHECK: Exists apollo upgrade, can not downgrade.")
        print("False")
        print(ERR_CODE_UPG_PATH_NOT_SUPPORT)
        print("")
        return False
    return True


def main(argv=None):
    """
    检查主函数，一般不需要修改
    :return:
    """
    if argv is None:
        argv = sys.argv
    try:
        src_ver = VerMgr.get_src_version()[0]
        target_ver = pkg.get_pkg_version(path.TARGET_CONFIG_PATH)[0]
        log.info("DOWNGRADE_CHECK: Src Version: %s target version: %s.",
                 src_ver, target_ver)
        opts, args = getopt.getopt(argv[1:], "rf", ["fast", "roll"])
        if not src_ver or not target_ver:
            log.error("DOWNGRADE_CHECK: The src or target version is empty.")
            print("False")
            print("1077949019")
            print("src version or target version is not exist")
            return 1
        for option, _ in opts:
            if option in ['-f', '--fast', '-r', '--roll']:
                fast_upg = option in ['-f', '--fast']
                return downgrade_version_check(src_ver, target_ver, fast_upg)
        downgrade_check(src_ver, target_ver)
        return 0
    except Exception as e:
        print("False")
        print("1077949019")
        print(e)
        log.exception(e)
        return 1


if __name__ == '__main__':
    sys.exit(main())
