#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import getopt
import logging
import os
import sys

import yaml

from comm_check_func import (check_current_patch_version, check_sys_and_patch_product, check_upgrade_patch_product,
                             check_upgrade_patch_version, clean_patch_backup)


class PatchType(object):
    COLD_PATCH = 'ColdPatch'
    SCRIPTONLY_PATCH = 'ScriptOnly'
    KERNEL_PATCH = 'KernelPatch'
    COMPRESSION_PATCH = 'CompressionPatch'


def get_patch_conf(conf_file):
    if not os.path.exists(conf_file):
        return dict()
    fd = open(conf_file)
    patch_conf = yaml.safe_load(fd)
    fd.close()
    return patch_conf


def get_patch_version(conf_file):
    patch_conf = get_patch_conf(conf_file)
    if not patch_conf:
        return None, None
    ver_conf = patch_conf.get("PatchConf", None)
    if not ver_conf:
        return None, None
    return ver_conf.get('base_version', None), ver_conf.get('patch_version', None)


def install_patch_file_check(patch_dir, new_conf, old_conf, check_sph=None):
    # 1.检查欧拉补丁是增量的
    new_euler_conf = new_conf.get('EulerPatch', [])
    old_euler_conf = old_conf.get('EulerPatch', [])
    new_patch_file_list = [x.get('patch_file', None) for x in new_euler_conf]
    new_script_list = [x.get('script', None) for x in new_euler_conf]

    for old_patch in old_euler_conf:
        if old_patch.get('patch_file') not in new_patch_file_list or \
                old_patch.get('script', None) not in new_script_list:
            return 'consistence_error', 'patch %s not inherited' % old_patch.get('patch_file')

    # 2.检查Dopra补丁是增量的:
    new_dopra_conf = new_conf.get('DopraPatch', dict())
    old_dopra_conf = old_conf.get('DopraPatch', dict())
    for old_process, old_patches in list(old_dopra_conf.items()):
        if old_process not in list(new_dopra_conf.keys()):
            return 'consistence_error', 'process %s not inherited' % old_process
        new_patch_list = [x.get('patch_file') for x in new_dopra_conf[old_process]]
        for old_patch in old_patches:
            if old_patch.get('patch_file') not in new_patch_list:
                return 'consistence_error', 'process %s patch %s not inherited' \
                       % (old_process, old_patch.get('patch_file'))

    # 3.检查欧拉补丁存在
    for patch in new_euler_conf:
        if not os.path.exists(os.path.join(patch_dir, patch.get('patch_file'))):
            return 'existence_error', 'patch %s not exist' % patch.get('patch_file')

    # 4.检查Dopra补丁存在
    for process, patches in list(new_dopra_conf.items()):
        for patch in patches:
            if not os.path.exists(os.path.join(patch_dir, process, patch.get('patch_file'))):
                return 'existence_error', 'process %s patch %s not exist' \
                       % (process, patch.get('patch_file'))

    # 5.检查replace_version是否正确配置和CompressionPatch压缩类补丁配置正确
    error_type, error_info = is_compression_patch_vaild(new_euler_conf, check_sph)
    if error_type:
        return error_type, error_info
    return 'pass', 'patch file check passed'


def is_compression_patch_vaild(new_euler_conf, check_sph):
    valid_tail_list = [".zip", ".tar", ".tgz", ".tar.gz"]
    for patch in new_euler_conf:
        replace_version = patch.get("replace_version")
        # 补丁收编版本不能大于当前sph版本
        if replace_version and check_sph < replace_version:
            return 'existence_error', 'patch %s version is invalid' % patch.get('replace_version')
        if patch.get('patch_type') == PatchType.COMPRESSION_PATCH:
            # 补丁尾缀是否有效
            is_file_name_valid = tail_valid(valid_tail_list, patch)
            if not is_file_name_valid:
                return 'existence_error', 'patch_file name %s is invalid' % patch.get('patch_file')
    return '', ''


def filter_num(string, type_int=True):
    res_list = list(filter(str.isdigit, string))
    res = "".join(res_list)
    return int(res) if type_int else res


def tail_valid(valid_tail_list, patch):
    # patch_file补丁名只能以 [.zip, .tar, .tgz, .tar.gz]结尾。
    if [valid for valid in valid_tail_list if patch.get("patch_file", "").endswith(valid)]:
        return True
    return False


def install_check():
    version = os.readlink('/startup_disk/image/pkg_cur').split('/')[0]
    check_patch_dir = os.path.join('/startup_disk/image', version, 'hotpatch/patch_temp')
    check_patch_conf_file = os.path.join(check_patch_dir, 'patch.yml')
    if not os.path.exists(check_patch_conf_file):
        return 'error', 'patch not exist'
    check_patch_conf = get_patch_conf(check_patch_conf_file)
    if not check_patch_conf:
        return 'error', 'patch conf parse failed'

    check_version, check_sph = get_patch_version(check_patch_conf_file)
    # 1.检查与基线版本是否一致
    if not check_version or str(check_version) != str(version):
        return 'error_base_ver', 'base version consistence check failed'

    # 2.检查与阵列产品型号是否一致
    if not check_sys_and_patch_product('/OSM/conf/manifest.yml', check_patch_conf_file):
        return 'error_product_type', 'product type consistence check failed'

    old_patch_dir = os.path.join('/startup_disk/image', version, 'hotpatch/patch_cur')
    old_patch_conf_file = os.path.join(old_patch_dir, 'patch.yml')
    old_version, old_sph = get_patch_version(old_patch_conf_file)
    old_patch_conf = get_patch_conf(old_patch_conf_file)

    # 3.检查补丁版本是否收编
    if old_sph and filter_num(old_sph) == filter_num(check_sph):
        return 'error_patch_ver', 'sph version same'

    # 4.检查补丁文件正确
    result, output = install_patch_file_check(check_patch_dir, check_patch_conf, old_patch_conf, check_sph)
    if result != 'pass':
        return result, output
    return 'pass', 'patch check passed'


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "icdhpb", ["help", "install", "upgrade_check_current_patch",
                                                        "upgrade_check_upgrade_patch", "check_upgrade_patch_product",
                                                        "clear_patch_backup"])
        for opt, _ in opts:
            if opt in ['-i', '--install']:
                ret, output = install_check()
                print(ret)
                print(output)
                return 0
            if opt in ['-c', '--upgrade_check_current_patch']:
                return check_current_patch_version()
            if opt in ['-d', '--upgrade_check_upgrade_patch']:
                return check_upgrade_patch_version()
            if opt in ['-p', '--check_upgrade_patch_product']:
                return check_upgrade_patch_product()
            if opt in ['-b', '--clear_patch_backup']:
                return clean_patch_backup()
    except Exception as e:
        print("exception")
        print("exception")
        print("exception")
        logging.exception(e)
        return 1
    return 0


if __name__ == '__main__':
    sys.exit(main())
