#!/usr/bin/python
# -*- coding: utf-8 -*-
import logging
import os
import pwd
import shlex
import subprocess
import sys
import re
import yaml


# 功能：判定mkv fpi rc 开关状态是否打开，打开情况下则升级检查不通过，因为有可能导致介质错误
# 实现：阵列上，进入CLI模式获取diskDomain的id列表，遍历ddId列表，获取fpi rc开关的状态，若为打开状态则检查不通过。

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

RESULT_CLI_FILE_PATH = "tempCheckCLIFpiResult"
RESULT_DIA_FILE_PATH = "tempCheckDiaFpiResult"

# 直接执行diagnose命令的字符串格式：
# {0}表示attach的进程id,比如app_data为12，
# {1}表示需要执行的diagnose命令，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
DIAGNOSE_CHECK_CMD = "diagsh --attach=*_{0} --cmd='{1}' > '{2}'"

# 直接执行CLI命令的字符串格式：
# {0}表示需要执行的CLI命令，
# {1}表示执行的CLI命令的用户名，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
USER_CMD = "cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
CLI_CHECK_CMD = "echo -e '{0}' | /ISM/cli/ismcli -u {1}  > '{2}'"

CUR_MANIFEST_PATH = '/OSM/conf/manifest.yml'
UPG_MANIFEST_PATH = '/startup_disk/image/pkg_upd/manifest.yml'
IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'

RETURN_TRUE = 0
RETURN_FALSE = -1
RETURN_CONTINUE = 1


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def get_user_name():
    try:
        pwd.getpwnam('admin')
        return 'admin'
    except Exception as exp:
        logging.warning("not find admin: %s", exp)
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


# 获取diskDomainId列表，字符串处理函数
def get_disk_domain_id_from_txt():
    list_of_diskdomain_id = []

    with open(RESULT_CLI_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()

        # 获取id
        for line in lines_array:
            if ("-" in line) or ("Health Status" in line) or ("#" in line):
                continue
            if ("." in line) and ("B" in line):
                dd_id = re.findall(r"\d+.?\d*", line)[0]
                # 去除空格等非数字
                dd_id = ''.join([c for c in dd_id if c in '0123456789'])
                list_of_diskdomain_id.append(dd_id)

    return list_of_diskdomain_id


# 从CLI中获取diskDomainId列表
def mkv_check_fpi_rc_stats_get_cli_ddid():
    cli_cmd_str = "show disk_domain general"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_CLI_FILE_PATH)
    result = os.system(cli_cmd_str)

    list_of_disk_domains = get_disk_domain_id_from_txt()

    return list_of_disk_domains


# 获取rc字符串中对应开关是否打开
def find_first_str(given, targets):
    for tar in targets:
        index = given.find(tar, 10, 45)
        if index != -1:
            if tar == '0x0':
                return False
            else:
                return True
    return True


def is_fpi_rc_stats_on():
    targets = ['0x0', '0x1', '0xff']

    # 从文本中获取fpi rc开关对应的位置，并判定是否打开
    with open(RESULT_DIA_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if "rc" not in line:
                continue
            fpi_rc_stats = find_first_str(line, targets)
            if fpi_rc_stats is False:
                return False

    return True


# 从dia中获取rc开关状态
def mkv_check_fpi_rc_stats_dia(diag_cmd_str):
    diag_cmd_str = DIAGNOSE_CHECK_CMD.format(12, diag_cmd_str, RESULT_DIA_FILE_PATH)
    result = os.system(diag_cmd_str)
    return is_fpi_rc_stats_on()


def clear_temp_file():
    shell_cmd_str = "rm -f " + RESULT_CLI_FILE_PATH
    os.system(shell_cmd_str)
    shell_cmd_str = "rm -f " + RESULT_DIA_FILE_PATH
    os.system(shell_cmd_str)
    return


# 获取升级的目标版本，找不到路径时返回空字符，只在升级时才会有此配置信息
def get_upg_sys_version_new():
    if not os.path.exists(UPG_MANIFEST_PATH):
        logging.warning("sys file didnt exist.")
        return "", ""
    with open(UPG_MANIFEST_PATH) as file_handle:
        cfg_yml = yaml.safe_load(file_handle)
        try:
            target_version = str(cfg_yml.get("SYS")["Version"])
            target_spc_version = str(cfg_yml.get("SYS")["SpcVersion"])
            return target_version, target_spc_version
        except Exception:
            logging.warning("CHECK_FUNC: Failed to get version.")
            return "", ""


def get_hotpatch_version():
    """
    :return:
    """
    version = get_upg_sys_version_new()[0]
    cur_patch_path = os.path.join(IMAGE_DISK, str(version),
                                  'hotpatch/patch_cur/', PATCH_MANIFEST)
    cur_sph = get_patch_version(cur_patch_path)
    #获取不到补丁号，则默认补丁为0
    if cur_sph == '':
        cur_sph = 0
        return cur_sph
    sph = int(re.findall(r"\d+\.?\d*", cur_sph)[0])
    return sph


def get_patch_version(conf_file):
    """
    :param conf_file:补丁配置文件
    :return:
    """
    if not os.path.exists(conf_file):
        logging.warning("patch conf file didnt exist.")
        return ""
    file_handle = open(conf_file)
    patch_conf = yaml.safe_load(file_handle)
    file_handle.close()
    ver_conf = patch_conf.get("PatchConf", None)
    if not ver_conf:
        logging.warning("Patch Conf file didnt exist.")
        return ""
    return ver_conf.get('patch_version', "")


# 判断此版本是否需要检查
def is_need_check_in_upg_version():
    upg_spc_version = get_upg_sys_version_new()[1]
    upg_sph = get_hotpatch_version()
    logging.info("mkv upgrade check version, dst_spc(%s), patch(%s).", upg_spc_version, upg_sph)
    if "6.1.3" in upg_spc_version:
        if upg_sph < 5:
            return RETURN_CONTINUE
    elif "6.1.0" in upg_spc_version:
        if upg_sph < 15:
            return RETURN_CONTINUE
    elif "6.0.1" in upg_spc_version:
        if upg_sph < 25:
            return RETURN_CONTINUE
    elif "6.0.0" in upg_spc_version:
        return RETURN_CONTINUE
    elif "" == upg_spc_version:
        # 获取不到目标版本号，直接返回False
        return RETURN_FALSE
    return RETURN_TRUE


def mkv_check_fpi_rc_stats():
    # 首先检查版本
    check_version_result = is_need_check_in_upg_version()
    if check_version_result == RETURN_TRUE:
        print("True")
        return True
    elif check_version_result == RETURN_FALSE:
        print("False")
        return False

    list_of_diskdomain_id = mkv_check_fpi_rc_stats_get_cli_ddid()
    # 没有diskDomain的情况下不会有开关，检查通过
    if len(list_of_diskdomain_id) == 0:
        print("True")
        return True

    # 遍历diskDomain，查看fpi rc开关的状态
    for dd_id in list_of_diskdomain_id:
        diag_cmd_str = "pmgr showPoolTaskSwitch -d " + dd_id
        stats = mkv_check_fpi_rc_stats_dia(diag_cmd_str)
        if stats is False:
            clear_temp_file()
            print("False")
            return False

    clear_temp_file()
    print("True")
    return True


def main():
    try:
        mkv_check_fpi_rc_stats()
        return 0

    except Exception as exp:
        logging.exception("mkv_check_fpi_rc_stats: %s", exp)
        print("False")
        return 1


if __name__ == '__main__':
    sys.exit(main())
