#!/usr/bin/python
# -*- coding: utf-8 -*-
import logging
import os
import pwd
import sys
import re
import subprocess
import shlex
import yaml

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

RESULT_FILE_PATH = "tempResult"

# 直接执行CLI命令的字符串格式：
# {0}表示需要执行的CLI命令，
# {1}表示执行的CLI命令的用户名，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
USER_CMD = "cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
CLI_CHECK_CMD = "echo -e '{0}' | /ISM/cli/ismcli -u {1}  > '{2}'"
DEV_CHECK_CMD = "sudo /ISM/cli/start.sh -u {0} -c 'change user_mode current_mode " \
                "user_mode=developer;y;y;{1}' > '{2}'"


# 直接执行diagnose命令的字符串格式：
# {0}表示attach的进程id,比如app_data为12，
# {1}表示需要执行的diagnose命令，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
DIAGNOSE_CHECK_CMD = "diagsh --attach=*_{0} --cmd='{1}' > '{2}'"

# 最大循环次数
MAX_LOOP_CNT = 1000

# 涉及的大版本
PRODUCT_SOFTWARE_VERSION_MAP = {
    '6.1.0': '6.1.0',
    '6.1.2': '6.1.2',
    '6.1.3': '6.1.3',
    '6.1.5': '6.1.5',
}

# 涉及大版本对应的补丁版本
PRODUCT_HOTPATCH_VERSION_MAP = {
    '6.1.0': 33,
    '6.1.2': 37,
    '6.1.3': 27,
    '6.1.5': 17,
}


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def get_user_name():
    try:
        pwd.getpwnam('admin')
        return 'admin'
    except Exception as exp:
        logging.warning("not find admin: %s", exp)
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


# 判断版本是否在属于检查范围内
def check_version(product_model, software_version, hot_patch_version):
    if not ("Dorado" in product_model) and software_version == '6.1.5':
        return True

    if software_version not in PRODUCT_SOFTWARE_VERSION_MAP:
        return True

    if int(hot_patch_version) < PRODUCT_HOTPATCH_VERSION_MAP.get(software_version):
        return True

    return False


def get_ddp_read_io_collision_switch():
    ddp_read_io_collision_switch = ''
    dia_cmd_str = "ddp set setDdpReadIoCollisionSwitch status"
    dia_cmd_str = DIAGNOSE_CHECK_CMD.format(12, dia_cmd_str, RESULT_FILE_PATH)
    result = os.system(dia_cmd_str)
    if result:
        logging.warning("not find the diagnose cmd.")
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return ddp_read_io_collision_switch
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if 'Current ddp collision switch status' in line:
                ddp_read_io_collision_switch = line.split()[-1].split('.')[0]
    return ddp_read_io_collision_switch


def set_ddp_read_collision_on():
    ddp_read_switch_open_success = False
    dia_cmd_str = "ddp set setDdpReadIoCollisionSwitch on"
    dia_cmd_str = DIAGNOSE_CHECK_CMD.format(12, dia_cmd_str, RESULT_FILE_PATH)
    result = os.system(dia_cmd_str)
    if result:
        logging.warning("not find the diagnose cmd.")
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return ddp_read_switch_open_success
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if 'success' in line:
                ddp_read_switch_open_success = True
    return ddp_read_switch_open_success


def get_system_general_info():
    product_model = ''
    product_version = ''
    software_version = ''
    hotpatch_version = ''
    cli_cmd_str = "show system general"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_FILE_PATH)
    result = os.system(cli_cmd_str)
    if result:
        logging.warning("not find the controller.")
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return product_model, software_version, hotpatch_version

    # 获取cli回显的每一行的信息
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for _, line in enumerate(lines_array):
            if 'Product Model' == line.split(':')[0].strip():
                product_model = line.split(':')[-1].strip()
            if 'Product Version' == line.split(':')[0].strip():
                product_version = line.split(':')[-1].strip()
    if len(product_version.split('.SPH')) <= 1:
        software_version = product_version
        hotpatch_version = "0"
    else:
        software_version = product_version.split('.SPH')[0]
        hotpatch_version = product_version.split('.SPH')[-1]
    # 回显格式: OceanStor 6810 6.1.3 27
    return product_model, software_version, hotpatch_version


def check_version_is_meet_upgrade_require():
    product_model, software_version, hotpatch_version = get_system_general_info()  # 获取产品型号，软件版本，补丁版本
    res = check_version(product_model, software_version, hotpatch_version)  # 检查版本
    logging.info("(mkv_check_version_is_meet_upgrade_require)check_version %s:%s:%s",
                 product_model, software_version, hotpatch_version)
    if res:
        logging.info("(mkv_check_version_is_meet_upgrade_require)"
                     "the upgrade version not meets the requirements, the check is passed.")
        return True
    # 查看读相似指纹开关状态
    ddp_read_io_collision_switch = get_ddp_read_io_collision_switch()
    if ddp_read_io_collision_switch == '' or ddp_read_io_collision_switch == '1':
        logging.info("(mkv_check_version_is_meet_upgrade_require)"
                     "the upgrade version meets the requirements and the switch is open, the check is passed.")
        return True
    # 打开读相似指纹开关（每控执行）
    ddp_read_switch_open_success = set_ddp_read_collision_on()
    if not ddp_read_switch_open_success or get_ddp_read_io_collision_switch() == '0':
        logging.error("(mkv_check_version_is_meet_upgrade_require)"
                      "open switch fail, ddp_read_switch_open_success: %d, switch: %d",
                      ddp_read_switch_open_success, get_ddp_read_io_collision_switch())
        return False
    return True


def main():
    try:
        res = check_version_is_meet_upgrade_require()
        if res:
            print("True")
        else:
            print("False")
        shell_cmd_str = "rm " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return 0

    except Exception as exp:
        logging.exception("mkv_check_similar_fp_switch: %s", exp)
        return 1


if __name__ == '__main__':
    sys.exit(main())
