#!/bin/sh
set +x

snasLogCommFile="/opt/huawei/snas/bin/snas_log_comm.sh"
. ${snasLogCommFile}

LOG_FILE="/var/log/scriptlog/dealLog.log"
FS_Back_Path="/mnt/fs/system/log"
Swift_Back_Path="/mnt/s3fs/log"
S3_Client_Path="/opt/obs/scripts/osc/s3client"
S3_Client_File="s3client.sh"
S3FS_Bucket_Name=".Ya89Soshla6S4Uma_s3fs"
S3_Object_Prefix=""
Collect_Product_Type="DFS"
ParserFile="/opt/huawei/snas/script/ConfigParser.sh"
S3_CollectServiceType=0
DFS_CollectServiceType=0
Swift_CollectServiceType=0

source ${ParserFile}

#默认0表示9000 DFS场景以及9000 swift场景，1表示uds场景、9000 S3场景
CollectServiceType=0

#===================================================
# 函数名称 ：showErr
# 函数功能 ：打印日志
# 参数     ：日志内容
#===================================================
function showErr
{
    local errFile="$1"
    local errLog="$2"
    echo -e "[$(date "+%F %T")] ${errLog}" >> "${errFile}"
}

######################################################################
#   函数名称   : getTimeZoneDiffSeconds
#   函数功能   : 将UTC时间转换为对应时区
#   参数       : 时间
######################################################################
function getTimeZoneDiffSeconds()
{
    timeSeconds=$1
    curTimeZone="$(date +%z)"
    curTimeZoneSign="${curTimeZone:0:1}"
    curTimeZoneHour="${curTimeZone:1:2}"
    curTimeZoneMin="${curTimeZone:3:2}"
    diffSeconds=$(echo "${curTimeZoneHour} * 3600 + ${curTimeZoneMin} * 60" | bc)
    if [ "X${curTimeZoneSign}" == "X+" ];then
        timeSeconds=$(($timeSeconds - $diffSeconds))
    else
        timeSeconds=$(($timeSeconds + $diffSeconds))
    fi
    echo "${timeSeconds}"
}

######################################################################
#   函数名称   : get_seconds
#   函数功能   : 将时间转换为秒
#   参数       : 时间
######################################################################
function get_seconds
{
    local time=$1
    local hour=""
    local year=""
    local secondsTime=""
    year=$(echo ${time} | awk -F'_' '{print $1}')
    hour=$(echo ${time} | awk -F'_' '{print $2}')
    secondsTime=$(date -d "${year} ${hour}" +%s)
    if [ $? -ne 0 ]; then
        showErr "${LOG_FILE}" "[$FUNCNAME] Fail! Time para is ${year} ${hour}"
        secondsTime=$(date -u -d "${year} ${hour}" +%s)
        secondsTime=$(getTimeZoneDiffSeconds "${secondsTime}")
    fi
    echo ${secondsTime}
}

######################################################################
#   函数名称   : get_seconds_after_one_hour
#   函数功能   : 获取前一小时秒
#   参数       : 日期
######################################################################
function get_seconds_after_one_hour
{
    local time=$1
    echo $((${time} + 60*60))
}

######################################################################
#   函数名称   : get_string_time
#   函数功能   : 转换时间格式为2017-01-21_10
#   参数       : 日期
######################################################################
function get_string_time
{
    local time=$1
    echo $(date -d "@${time}" "+%Y-%m-%d_%H")
}

######################################################################
#   函数名称   : show
#   函数功能   : 以默认色（白色）显示错误
#   参数       : 打印信息
######################################################################
function show
{
    echo -e "${1}"
    return 0
}

######################################################################
#   函数名称   : usage
#   函数功能   : 打印usage帮助
#   参数       : 无
######################################################################
function usage
{
    show "Usage:"
    show "      $(basename \"$0\")"
    show "      --option(must)      : The type of operation which should be "get" or "list" or "upload" or "delete" or "batchlist" or "batchget" or "batchall" or "listall"."
    show "                get       : Get one log."
    show "                batchget  : Get log by date."
    show "                list      : List one log."
    show "                batchlist : List all log."
    show "                batchall  : List all log with size."
    show "                upload    : Upload one log to bucket."
    show "                delete    : Delete one log from bucket"
    show "      --fileName(option)  : The name of log."
    show "      --startTime(option) : The starting time which use to look for log(time eg:2017-02-13_01)."
    show "      --endTime(option)   : The terminal time which use to look for log(time eg:2017-02-13_01)."
    show "      --nodeName(option)  : The name of node which use to look or upload for log."
    show "      --desDir(option)    : The target path which use to store log(DIR eg:/home/dir)."
    show "      --srcDir(option)    : The target path which use for uoload log(DIR eg:/home/dir)."
    show ""
    show "      e.g. sh snas_log_deal.sh --option=get --nodeName=OMS01 --fileName=snas.log.2017-02-13_01.tar.gz --desDir=/home/permitdir"
    show "      e.g. sh snas_log_deal.sh --option=batchget --startTime=2017-01-01_20 --endTime=2017-01-24_18 --nodeName=OMS01 --desDir=/home/permitdir"
    show "      e.g. sh snas_log_deal.sh --option=list --nodeName=OMS01 --fileName=snas.log.2017-02-13_01.tar.gz"
    show "      e.g. sh snas_log_deal.sh --option=batchlist --startTime=2017-01-01_20 --endTime=2017-01-24_18 --nodeName=OMS01"
    show "      e.g. sh snas_log_deal.sh --option=batchall --nodeName=OMS01"
    show "      e.g. sh snas_log_deal.sh --option=listall --nodeName=OMS01"
    show "      e.g. sh snas_log_deal.sh --option=upload --nodeName=OMS01 --fileName=snas.log.2017-02-13_01.tar.gz --srcDir=/home/permitdir"
    show "      e.g. sh snas_log_deal.sh --option=delete --nodeName=OMS01 --fileName=snas.log.2017-02-13_01.tar.gz"
}

######################################################################
#   函数名称   : checkDateFormat
#   函数功能   : 检查时间参数格式
#   参数       :
######################################################################
function checkDateFormat
{
    local time=$1
    echo "${time}" |grep -qPo "[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{2}"
    if [ $? -ne 0 ];then
        return 1
    fi
    return 0
}

######################################################################
#   函数名称   : checkDateFormat
#   函数功能   : 检查时间参数，与当前时间比较以及比较开始时间和结束时间
#   参数       :
######################################################################
function checkDate
{
    local startTime=$1
    local endTine=$2
    local currentTime=$(date "+%Y-%m-%d_%H")
    if [ -z "${startTime}" ] || [ -z "${endTine}" ];then
        showErr "${LOG_FILE}" "the parameter of startTime or endTime is empty."
        return 1
    fi
    if [ "$(get_seconds "${startTime}")" -gt "$(get_seconds "${endTine}")" ];then
        showErr "${LOG_FILE}" "the parameter of startTime is more than endTime."
        return 1
    fi

    if [ "$(get_seconds "${startTime}")" -gt "$(get_seconds "${currentTime}")" ];then
        showErr "${LOG_FILE}" "the parameter of startTime is more than currentTime."
        return 1
    fi

    if [ "$(get_seconds "${endTine}")" -gt "$(get_seconds "${currentTime}")" ];then
        g_logEndTime=${currentTime}
        return 0
    fi
    return 0
}

#===================================================
# 函数名称：product_type
# 函数功能：判断场景
# 参数：
#===================================================
function product_type
{
    local NodeServiceType=""
    local value=$(cat "/opt/huawei/snas/etc/snas.ini"|grep productType|awk -F= '{print $2}')
    #UDS环境处理
    if [ "X80" == "X${value}" ];then
        S3_CollectServiceType=1
        Collect_Product_Type="FusionStorage Object"
        showErr "${LOG_FILE}" "${Collect_Product_Type}: CollectServiceType is FusionStorage Object"
    else
    #9000环境处理
        NodeServiceType=$(GetNodeServiceType)
        if [ $? -ne 0 ];then
            showErr "${LOG_FILE}" "Get Node Service Type [$NodeServiceType] failed"
            exit 1
        fi
        if [ "X${NodeServiceType}" == "XS3Node" ];then
            S3_CollectServiceType=1
            Collect_Product_Type="S3"
            showErr "${LOG_FILE}" "${Collect_Product_Type}: CollectServiceType is s3"
        elif [ "X${NodeServiceType}" == "XSwiftNode" ];then
            Swift_CollectServiceType=1
            Collect_Product_Type="SWIFT"
            showErr "${LOG_FILE}" "${Collect_Product_Type}: CollectServiceType is swift"
        elif [ "${NodeServiceType}" == "FSNode" ];then
            DFS_CollectServiceType=1
            Collect_Product_Type="DFS"
            showErr "${LOG_FILE}" "${Collect_Product_Type}: CollectServiceType is dfs"
        fi
    fi
}

#===================================================
# 函数名称：list_obj
# 函数功能：封装ls单个文件操作
# 参数：
#===================================================
function list_obj
{
    local logListInfo=""
    logListInfo=$(ls ${FS_Back_Path}/${g_logNodeName}/${g_fileName} 2>&1)
    if [ $? -ne 0 ];then
       showErr "${LOG_FILE}" "${Collect_Product_Type}: [list_obj]List the detail log information(${S3_Object_Prefix}/${g_fileName}) failed.\n$logListInfo"
       return 1
    fi

    if [ -z "${logListInfo}" ];then
        showErr "${LOG_FILE}" "${Collect_Product_Type}: The ${S3_Object_Prefix}/${g_fileName} not exit."
        return 2
    else
        echo ${logListInfo}
        return 0
    fi
}

#===================================================
# 函数名称：batch_list_obj
# 函数功能：按时间批量获取文件操作
# 参数：
#===================================================
function batch_list_obj
{
    local tmpListFile="/tmp/log_list_deal.$$"
    local logListInfo=""
    local fileTarget=""
    local startTimeSec=$(get_seconds "${g_logStartTime}")
    local endTimeSec=$(get_seconds "${g_logEndTime}")
    local fileTime=""
    local fileTimeSec=""
    local currTimeSec=${startTimeSec}
    local currTime=${g_logStartTime}
    local currDay=$(echo ${g_logStartTime} | awk -F'_' '{print $1}')
    local currDayFlag=${currDay}
    local firstGetBuket=0   #标记桶被查询是否为第一次
    local line=""

    if [ -f ${tmpListFile} ];then
        rm -f ${tmpListFile}
    fi
    touch ${tmpListFile}
    for fileTarget in ${File_Target_List}
    do
        currTimeSec=${startTimeSec}
        currTime=${g_logStartTime}
        firstGetBuket=0
        currDayFlag=$(echo ${g_logStartTime} | awk -F'_' '{print $1}')
        currDay=$(echo ${g_logStartTime} | awk -F'_' '{print $1}')
        while [ ${currTimeSec} -le ${endTimeSec} ]
        do
            ls ${FS_Back_Path}/${g_logNodeName}/"${fileTarget}.${currTime}.tar.gz" 2>/dev/null

            currTimeSec=$(get_seconds_after_one_hour "${currTimeSec}")
            currTime=$(get_string_time "$currTimeSec")
            currDay=$(echo ${currTime} | awk -F'_' '{print $1}')
            firstGetBuket=1
        done
    done

    rm -f ${tmpListFile}
}

#===================================================
# 函数名称：batch_list_all_obj
# 函数功能：按节点批量获取所有文件操作
# 参数：
#===================================================
function batch_list_all_obj
{
    local tmpListFile="/tmp/log_bash_list_deal.$$"
    local line=""

    if [ -f ${tmpListFile} ];then
        rm -f ${tmpListFile}
    fi
    touch ${tmpListFile}

    #FS_Back_Path后面不能跟"/"，特殊处理，显示的日志不带路径
    ls ${FS_Back_Path}/${g_logNodeName}* >${tmpListFile} 2>/dev/null

    while read line
    do
        #跳过空行
        if [ "X" == "X${line}" ];then
            continue
        fi
		echo ${line} | grep "/mnt/" >/dev/null 2>&1
		if [ $? -ne 0 ];then
			echo "${line}" 2>/dev/null
		fi
    done <${tmpListFile}

    rm -f ${tmpListFile} >/dev/null 2>&1
}

#===================================================
# 函数名: batch_list_all_detail_obj
# 函数功能：按节点批量获取所有文件名和文件大小操作,信息收集专用增加
# 参数： $1 结果输出文件
#===================================================
function batch_all_list_detail_obj
{
    local tmpListFile="/tmp/batch_all_list_detail_obj.$$"
    local line=""
    local TimeStyle="--time-style=\"+%m/%d/%Y %H:%I\""

	#FS_Back_Path后面不能跟"/"，特殊处理，显示的日志不带路径
	ls -l ${FS_Back_Path}/${g_logNodeName} > ${tmpListFile} 2>/dev/null
	if [ $? -ne 0 ];then
		showErr "${LOG_FILE}" "${Collect_Product_Type}: [batch_list_all_detail_obj]List the detail log information(${S3_Object_Prefix}) failed."
		return 1
	fi

    while read line
    do
        #跳过空行
        if [ "X" == "X${line}" ];then
            continue
        fi
		echo ${line} | grep "/mnt/" >/dev/null 2>&1
		if [ $? -ne 0 ];then
			echo ${line} | awk '{print $5" "$NF}' 2>/dev/null
		fi

    done <${tmpListFile}

    rm -f ${tmpListFile} >/dev/null 2>&1
    return 0
}

#===================================================
# 函数名称：get_obj
# 函数功能：封装cp单个文件操作,从s3fs桶中获取文件
# 参数：
#===================================================
function get_obj
{
    local logGetInfo=""
    if [ -L "${g_desDir}" ]; then
        showErr "${LOG_FILE}" "There are security risks, exit -1"
        rm -f "${g_desDir}"
    fi

    logGetInfo=$(cp ${FS_Back_Path}/${g_logNodeName}/${g_fileName} ${g_desDir} 2>&1)

    if [ $? -ne 0 ];then
       showErr "${LOG_FILE}" "${Collect_Product_Type}: [get_obj]Get the detail log information(${S3_Object_Prefix}/${g_fileName}) failed.\n$logGetInfo"
       return 1
    fi

    return 0
}

#===================================================
# 函数名称：batch_get_obj
# 函数功能：按时间批量获取文件操作
# 参数：
#===================================================
function batch_get_obj
{
    local logGetInfo=""
    local fileTarget=""
    local startTimeSec=$(get_seconds "${g_logStartTime}")
    local endTimeSec=$(get_seconds "${g_logEndTime}")
    local currTimeSec=${startTimeSec}
    local currTime=${g_logStartTime}
    local currDay=$(echo ${g_logStartTime} | awk -F'_' '{print $1}')
    local tmpListFile="/tmp/tmp_log_bash_get_deal.$$"
    local ListFile="/tmp/log_bash_get_deal.$$"
    local tmpFile=""
    local line=""

    if [ -f ${tmpListFile} ];then
        rm -f ${tmpListFile}
    fi
    if [ -f ${ListFile} ];then
        rm -f ${ListFile}
    fi
    touch ${tmpListFile}
    touch ${ListFile}

	for fileTarget in ${File_Target_List}
	do
		endTimeSec=$(get_seconds "${g_logEndTime}")
		currTimeSec=${startTimeSec}
		while [ ${currTimeSec} -le ${endTimeSec} ]
		do
			if [ ${Swift_CollectServiceType} -eq 1 ];then
				if [ ${fileTarget} == "audit.log" ];then
					cp ${Swift_Back_Path}/${g_logNodeName}/audit/"${fileTarget}.${currTime}.tar.gz" ${g_desDir} 2>/dev/null
				else
					cp ${Swift_Back_Path}/${g_logNodeName}/archive_${currDay}/"${fileTarget}.${currTime}.tar.gz" ${g_desDir} 2>/dev/null
				fi
			else
				cp ${FS_Back_Path}/${g_logNodeName}/"${fileTarget}.${currTime}.tar.gz" ${g_desDir} 2>/dev/null
			fi
			currTimeSec=$(get_seconds_after_one_hour "${currTimeSec}")
			currTime=$(get_string_time "$currTimeSec")
			currDay=$(echo ${currTime} | awk -F'_' '{print $1}')
		done
	done

    rm -f ${tmpListFile}
    rm -f ${ListFile}
}

#===================================================
# 函数名称：upload_obj
# 函数功能：封装cp单个文件操作,从本地上传文件到s3fs桶中
# 参数：
#===================================================
function upload_obj
{
    local logUploadInfo=""
    local time=""

	if [ -L "${FS_Back_Path}/${g_logNodeName}/${g_fileName}" ]; then
		showErr "${LOG_FILE}" "There are security risks, exit -1"
		rm -f "${FS_Back_Path}/${g_logNodeName}/${g_fileName}"
	fi
	logUploadInfo=$(cp ${g_srcDir}/${g_fileName} ${FS_Back_Path}/${g_logNodeName}/${g_fileName}  2>&1)

    if [ $? -ne 0 ];then
       showErr "${LOG_FILE}" "${Collect_Product_Type}: Upload the detail log information(${S3_Object_Prefix}/${g_fileName}) failed.\n$logUploadInfo"
       return 1
    fi

    return 0
}

#===================================================
# 函数名称：delect_obj
# 函数功能：封装rm单个文件操作
# 参数：
#===================================================
function delect_obj
{
    local logDelInfo=""

	if [ -L "${FS_Back_Path}/${g_logNodeName}/${g_fileName}" ]; then
		showErr "${LOG_FILE}" "There are security risks, exit -1"
		rm -f "${LOG_FILE}" "There are security risks, exit -1"
	fi
	logDelInfo=$(rm -rf ${FS_Back_Path}/${g_logNodeName}/${g_fileName} 2>&1)

    if [ $? -ne 0 ];then
       showErr "${LOG_FILE}" "${Collect_Product_Type}: Delete the log(${S3_Object_Prefix}/${g_fileName}) failed.\n$logDelInfo"
       return 1
    fi

    return 0
}

######################################################################
#   函数名称   : logOperation
#   函数功能   : 日志操作
#   参数       : 脚本参数
######################################################################
function logOperation
{
    case "${g_operOption}" in
        list)
            list_obj
        ;;
        get)
            get_obj
        ;;
        batchget)
            batch_get_obj
        ;;
        batchlist)
            batch_list_obj
        ;;
        listall)
            batch_list_all_obj
        ;;
        upload)
            upload_obj
        ;;
        delete)
            delect_obj
        ;;
        batchall)
            batch_all_list_detail_obj
        ;;

    esac
}

######################################################################
#   函数名称   : check_Para
#   函数功能   : 入参检查
#   参数       :
######################################################################
function check_Para
{
    #操作类型检查
    if [ -z "${g_operOption}" ];then
        showErr "${LOG_FILE}" "the parameter of option is empty."
        return 1
    fi
    #节点名称检查，节点名称必须存在
    if [ -z "${g_logNodeName}" ];then
        showErr "${LOG_FILE}" "the parameter of nodeName is empty."
        return 1
    fi
    #get操作参数检查
    if [ "Xget" == "X${g_operOption}" ];then
        if [ -z "${g_fileName}" ];then
            showErr "${LOG_FILE}" "the parameter of fileName is empty."
            return 1
        fi
        if [ -z "${g_desDir}" ];then
            showErr "${LOG_FILE}" "the parameter of desDir is empty."
            return 1
        fi
    fi
    #batchget操作参数检查
    if [ "Xbatchget" == "X${g_operOption}" ];then
        checkDate "${g_logStartTime}" "${g_logEndTime}"
        if [ $? -ne 0 ];then
            return 1
        fi
        if [ -z "${g_desDir}" ];then
            showErr "${LOG_FILE}" "the parameter of desDir is empty."
            return 1
        fi
    fi
    #upload操作检查
    if [ "Xupload" == "X${g_operOption}" ];then
        if [ -z "${g_fileName}" ];then
            showErr "${LOG_FILE}" "the parameter of fileName is empty."
            return 1
        fi
        if [ -z "${g_srcDir}" ];then
            showErr "${LOG_FILE}" "the parameter of srcDir is empty."
            return 1
        fi
    fi
    #list操作检查
    if [ "Xlist" == "X${g_operOption}" ];then
        if [ -z "${g_fileName}" ];then
            showErr "${LOG_FILE}" "the parameter of fileName is empty."
            return 1
        fi
    fi
    #batchlist操作检查
    if [ "Xbatchlist" == "X${g_operOption}" ];then
        checkDate "${g_logStartTime}" "${g_logEndTime}"
        if [ $? -ne 0 ];then
            return 1
        fi
    fi
    #delete操作检查
    if [ "Xdelete" == "X${g_operOption}" ];then
        if [ -z "${g_fileName}" ];then
            showErr "${LOG_FILE}" "the parameter of fileName is empty."
            return 1
        fi
    fi
    return 0
}

######################################################################
#   FUNCTION   : getPara
#   DESCRIPTION: 获取参数
#   INPUT      : 脚本参数
######################################################################
getPara()
{
    local g_operOptionS

    umask 022

    g_operOptionS=$(getopt -o h --long option:,fileName:,startTime:,endTime:,nodeName:,srcDir:,desDir:  -n "$0" -- "$@")
    if [ $? -ne 0 ];then
        showErr "${LOG_FILE}" "Invalid parameter!";
        return 1
    fi

    if [ $# -lt 2 ];then
        showErr "${LOG_FILE}" "Not enough parameter!";
        return 1
    fi

    while [ -n "$1" ];
    do
        arg=$1
        echo "$arg" | grep -qE "\-\-.*=.*"
        if [ $? -ne 0 ];then
            showErr "${LOG_FILE}" "Invalid parameter format($arg).";
            return 1
        fi
        opts=$(echo $arg | awk -F= '{print $1}')
        case "$opts" in
        --option)
            g_operOption=$(echo $arg | awk -F= '{print $2}')
            echo "$g_operOption" | grep -qE "^get$|^batchget$|^list$|^batchlist$|^upload$|^delete$|^listall$|^batchall$"
            if [ $? -eq 0 ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_operOption). The option should be "get" or "list" or "upload" or "delete" or "batchlist" or "batchget" or "listall" or "batchall"";
                return 1
            fi
            ;;
        --startTime)
            g_logStartTime=$(echo $arg | awk -F= '{print $2}')
            checkDateFormat "$g_logStartTime"
            if [ $? -eq 0 ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_logStartTime). The start time should be yyyy-mm-dd_hh";
                return 1
            fi
            ;;
        --endTime)
            g_logEndTime=$(echo $arg | awk -F= '{print $2}')
            checkDateFormat "$g_logEndTime"
            if [ $? -eq 0 ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_logEndTime). The end time should be yyyy-mm-dd_hh";
                return 1
            fi
            ;;
        --nodeName)
            g_logNodeName=$(echo $arg | awk -F= '{print $2}')
            if [ -n "$g_logNodeName" ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($nodeName). The node name must be exit";
                return 1
            fi
            ;;
        --fileName)
            g_fileName=$(echo "$arg" | awk -F= '{print $2}')
            if [ -n "$g_fileName" ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_operOption). The file name must be exit";
                return 1
            fi
            ;;
        --desDir)
            g_desDir=$(echo "$arg" | awk -F= '{print $2}')
            if [ -d "$g_desDir" ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_operOption). The option should be valid path.";
                return 1
            fi
            ;;
        --srcDir)
            g_srcDir=$(echo "$arg" | awk -F= '{print $2}')
            if [ -d "$g_srcDir" ];then
                shift 1
                continue
            else
                showErr "${LOG_FILE}" "Invalid parameter($g_operOption). The option should be valid path.";
                return 1
            fi
            ;;
            *)
            showErr "${LOG_FILE}" "Invalid parameter($arg).";
            return 1
            ;;
        esac
    done

    check_Para
    if [ $? -ne 0 ];then
        return 1
    fi
    return 0
}

#==========================================================主处理流程=============================================================
#参数定义
g_operOption=""    #操作类型
g_logStartTime=""  #起始时间
g_logEndTime=""    #结束时间
g_logNodeName=""   #节点名字
g_fileName=""      #日志名称
g_desDir=""        #日志下载路径
g_srcDir=""        #日志上传的本地路径
g_swiftBackDir=""  #swift节点存放日志目录
#本脚本日志超过100MB，则覆盖文件内容
JudgeLogFileSize "${LOG_FILE}" 100
#设置场景
product_type
#获取参数
getPara "$@" || { usage; exit 1; }

S3_Object_Prefix=${g_logNodeName}
#处理传入参数带"/"，只获取第一个域
echo ${g_logNodeName} | grep "/" >/dev/null 2>&1
if [ $? -eq 0 ];then
    g_logNodeName=$(dirname ${g_logNodeName})
fi

#操作日志
logOperation

