# -*- coding: UTF-8 -*-
import os
import traceback
import time
import re
from cbb.frame.rest.restUtil import Tlv2Rest
from cbb.frame.rest import restData
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.frame.base.baseUtil import UpgradeRes
from cbb.frame.base import constants
from cbb.frame.base.exception import UnCheckException
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tool.obase.utils import GZipAssistant
from com.huawei.ism.tool.framework.platform.util import ApplicationContext
from java.lang import Exception as JException
from utils import Products

SFTP_CAN_NOT_CONNECT_CODE = "1073949187"

def excute(dataDict):
    lang = contextUtil.getLang(dataDict)
    logger = contextUtil.getLogger(dataDict)
    try:
        return VersionUpdate(dataDict).execute()
    except UnCheckException as e:
        logger.error("updatecheck uncheckException:%s" % traceback.format_exc())
        return False, e.errorMsg
    except:
        logger.error("updatecheck exception:%s" % traceback.format_exc())
        return False, baseUtil.getPyResource(lang, "updateCheck.common.error")

class VersionUpdate():
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.lang = contextUtil.getLang(dataDict)
        self.dev = contextUtil.getDevObj(self.dataDict)
        self.logger = dataDict.get("logger")
        self.dev_type = contextUtil.get_dev_type(self.dataDict)
        self.upgrade_res = UpgradeRes(self.dev_type)
        self.dev_version = contextUtil.getCurVersion(dataDict)
        self.checkType = self.dataDict.get("checkType")

    def execute(self):
        self.logger.info("enter versionUpdate.dev_type:{}".format(self.dev_type))
        packagePath = self._getPackagePath()
        self.logger.info("upgrade check component path=%s" % packagePath)
        if not os.path.exists(packagePath):
            self.logger.info("package not exist.")
            return True, ""

        if not self.need_update_inner_check():
            return True, ""

        restService = RestService(self.dataDict)
        package_version = self.getPackageVersion(packagePath)
        self.logger.info("packageVersion={}".format(package_version))
        if not package_version:
            return False, baseUtil.getPyResource(
                self.lang, "updateCheck.getPackageVersion.error")

        self.logger.info("begin to query upload path")
        # 查询上传路径
        uploadPath = restService.queryUploadPath(packagePath)

        # 开始SFTP上传
        self.logger.info("begin to upload component package.")
        self.upload_component_package(packagePath, uploadPath)

        # 通知上传完成
        restService.notifyFinish(contextUtil.getIp(self.dataDict), uploadPath)

        # 查询分发、安装进度
        status = None
        startTime = time.clock()
        while constants.OM_MSG_OP_UPD_LST_SYS_PROGRESS.UPD_DLD_SUCCESS != status:
            if time.clock() - startTime > constants.DISTRIBUTE_PKG_TIMEOUT:
                return False, baseUtil.getPyResource(self.lang, "updateCheck.query.timeout")

            recs = restService.queryProgress()
            if recs:
                status = Tlv2Rest.getRecordValue(recs[0], restData.Upgrade.UpdLstSysProgress.CMO_UPD_SYS_STEP_TASK_STATUS)
                self.logger.info("current status=%s" % status)
                if status == constants.OM_MSG_OP_UPD_LST_SYS_PROGRESS.UPD_DLD_FAIL:
                    return False, baseUtil.getPyResource(self.lang, "updateCheck.query.failed")
            baseUtil.safeSleep(5)

        self.logger.info("update inner check version succeed.")
        return True, ""

    def upload_component_package(self, packagePath, uploadPath):
        if self.checkType == "Inspection":
            try:
                sftp = self.dataDict.get("SFTP")
                self.logger.info("sftp:{}".format(type(sftp)))
                sftp.putFile(packagePath, uploadPath, None)
            except (Exception, JException) as e:
                self.logger.error("sftp exception.{}".format(e))
                raise UnCheckException(contextUtil.getMsg(self.lang, SFTP_CAN_NOT_CONNECT_CODE))
        else:
            sftp = contextUtil.getSftp(self.dataDict)
            sftp.upload(packagePath, uploadPath)

    def need_update_inner_check(self):
        base_dev_node = contextUtil.get_base_dev(self.dataDict)
        dev_hotpatch_version = base_dev_node.getHotPatchVersion()
        self.logger.info(
            "current hot patch version is:%s" % dev_hotpatch_version)
        if self.dev_version \
                and Products.compareVersion(self.dev_version,
                                            "V500R007C60SPC100 Kunpeng") == 0:
            if Products.compareVersion(dev_hotpatch_version,
                                       "V500R007C60SPH105") < 0:
                return False
            return True
        # V500R007C60SPC200及之前的版本不支持更新
        if self.dev_version and not Products.isDigitalVer(self.dev_version) \
                and Products.compareVersion(self.dev_version,
                                            "V500R007C60SPC200 Kunpeng") <= 0:
            self.logger.info("current version not need update inner check.")
            return False
        return True

    def _getPackagePath(self):
        if baseUtil.is_dorado_series_devs(self.dev_type) or "_container" in self.dev_type:
            packagePath = "\packages\upgrade_check_component_dorado.tar.gz"
        else:
            packagePath = "\packages\upgrade_check_component.tar.gz"
        return ApplicationContext.getInstance().getWorkPath() + packagePath

    def getSuggestion(self, recs):
        suggestion = ""
        for rec in recs:
            itemState = Tlv2Rest.getRecordValue(rec, restData.Upgrade.UpdListDetaiInfo.ITEM_STATE)
            errorKey = Tlv2Rest.getRecordValue(rec, restData.Upgrade.UpdListDetaiInfo.ERROY_KEY)
            self.logger.info("itemState=%s, errorKey=%s" % (itemState, errorKey))
            if itemState == "3":
                errorMsg = self.upgrade_res.get_res(errorKey)
                if not errorMsg or "--" == errorMsg:
                    errorMsg = errorKey
                suggestion += errorMsg
        return suggestion

    # 查询检查包的版本
    def getPackageVersion(self, pkgpath):
        packegeConfList = ['manifest.yml']
        versionNum = ''
        bf = None
        try:
            ga = GZipAssistant(pkgpath)
            fileMap = ga.getFileBufferMap(packegeConfList)
            if not fileMap:
                return versionNum  # 第二个参数返回0表示用户上传的是错误文件包

            fileName = ga.getFileNameFromMap(fileMap)
            if fileName != 'manifest.yml':
                return versionNum
            bf = fileMap.get(fileName)
            line = bf.readLine()
            while line != None:
                if bool(re.search('Version', line, re.IGNORECASE)):
                    if "#" in line:
                        line = line.split("#")[0]
                    versionNum = line.split(':')[1].strip()
                    return versionNum
                line = bf.readLine()
        except IOError:
            pass
        finally:
            # 关闭文件流
            GZipAssistant.closeBufferedReader(bf)
        return versionNum

class RestService():
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.dev = dataDict.get("dev")
        self.lang = contextUtil.getLang(dataDict)
        self.logger = dataDict.get("logger")
        self.queryFailedCnt = 0

    # 8599109754
    def queryUploadPath(self, packagePath):
        try:
            self.logger.info("begin to query upload path.")
            rest = contextUtil.getRest(self.dataDict)
            upgradePackageSize = os.path.getsize(packagePath)
            param0 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_TYPE,
                      UnsignedInt32(restData.Enum.UploadPackageType.CHEKCITEM_PACKAGE))
            param2 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_SIZE, UnsignedInt32(upgradePackageSize))
            params = [param0, param2]
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.OM_MSG_OP_GET_PACKAGE_UPLOADPATH, params)
            rec = recs[0]
            uploadPath = Tlv2Rest.getRecordValue(rec, restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_DIR_FILE)
            return uploadPath
        except Exception as e:
            self.logger.error("query check progress error.")
            raise UnCheckException(baseUtil.getPyResource(self.lang, "updateCheck.queryUploadpath.error", e.args[1]))

    # 8599117947
    def notifyFinish(self, ip, uploadPath):
        try:
            self.logger.info("begin to notify upload finish.")
            rest = contextUtil.getRest(self.dataDict)
            devPkgPath = ip + ":" + uploadPath
            # 上传包位置ip+
            pkgPathParam = (restData.Upgrade.NotifyPackagePath.CMO_NOTIFY_PACKAGE_PATH, str(devPkgPath))
            params = [pkgPathParam]
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.OM_MSG_OP_NOTIFY_PACKAGE_PATH, params)
            return recs
        except:
            self.logger.error("query check progress error.")
            raise UnCheckException(baseUtil.getPyResource(self.lang, "updateCheck.notifyFinish.error"))

    # 8599109744
    def queryProgress(self):
        try:
            self.logger.info("begin to query progress.")
            rest = contextUtil.getRest(self.dataDict)
            msgParam0 = (restData.Upgrade.LstSysProgress.CMO_UPD_FLOW_ID,
                         restData.Enum.UpgQueryFlowEnum.UPDATE_CHECKITEM)
            params = [msgParam0]
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.OM_MSG_OP_UPD_LST_SYS_PROGRESS, params)
            return recs
        except:
            self.logger.error("query check progress error.")
            return []

    # 8599150995
    def queryDetails(self):
        ##查询检查详细信息
        try:
            self.logger.info("begin to query detail")
            rest = contextUtil.getRest(self.dataDict)
            msgParam0 = (restData.Upgrade.LstSysProgress.CMO_UPD_FLOW_ID,
                         restData.Enum.UpgQueryFlowEnum.UPDATE_CHECKITEM)
            params = [msgParam0]
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.OM_MSG_OP_UPD_LIST_DETAILINFO, params)
            return recs
        except:
            self.logger.error("query check details error.")
            return []