# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

import re
import time
import traceback

from com.huawei.ism.tool.obase.exception import ToolException

from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.base.exception import UnCheckedException
from cbb.frame.cli.execute_on_all_controllers import (
    ExecuteOnAllControllers,
    ResultType,
    FuncResult,
)
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlContext
from cbb.frame.ruleConfig import common

MESSAGE_PROCESS_MAX_NUM = 26
MESSAGE_PROCESS_REBOOT_TIME = 5
# 风险版本白名单，只有下列版本。
RISK_VERSION = ["V300R003C20SPC200",
                "V300R006C00SPC100",
                "V300R006C10SPC100",
                "V500R007C00SPC100"]


class MessageConnectionNumberCheck:
    """
    消息连接数检查
    """
    def __init__(self, data_dict, dev_node=None):
        self.data_dict = data_dict
        self.logger = contextUtil.getLogger(data_dict)
        self.lang = contextUtil.getLang(data_dict)
        self.product_version = str(dev_node.getProductVersion()) if dev_node else str(data_dict.get("dev").getProductVersion())
        self.origin_info = []
        self.err_msg_list = []

    def execute(self):
        if not self.is_risk_version():
            self.logger.info("{} is not a risk version.".format(
                self.product_version))
            return True, '', "Risk Version:{}".format(RISK_VERSION)

        exe_context = ExeOnAllCtrlContext(self.data_dict)
        self.check_msg_process(exe_context)
        ret = "\n".join(self.origin_info)
        if exe_context.err_msg:
            suggest_msg = common.getMsg(self.lang,
                                        "contact.technical.support")
            exe_context.append_err_msg(suggest_msg)
            return False, exe_context.err_msg, ret

        return True, '', ret

    @ExecuteOnAllControllers
    def check_msg_process(self, exec_context):
        current_ctrl_id = exec_context.cur_ctrl_id
        self.cli = exec_context.dev_info.cli
        lang = exec_context.lang
        self.logger.info("check ctrl {} start.".format(current_ctrl_id))
        self.origin_info.append("\nCTRL NODE ID:{}".format(current_ctrl_id))
        try:
            if self.is_msg_server_normal():
                self.logger.info("msg server is normal.node id {}".format(
                    current_ctrl_id))
                return FuncResult(ResultType.SUCCESS, "", "")

            self.logger.info("msg server is not normal.")
            if not self.is_daemon_process_normal():
                self.logger.info("daemon process is not normal.")
                err_msg = common.getMsg(
                    self.lang,
                    "msg.conn.check.daemon.process.check.not.pass",
                    current_ctrl_id,
                )
                exec_context.append_err_msg(err_msg)
                return FuncResult(ResultType.FAILED, "", "")
            self.logger.info("daemon process is normal.")
            if self.repair_msg_server(current_ctrl_id):
                self.logger.info("repair msg server success.")
                return FuncResult(ResultType.SUCCESS, "", "")

            self.logger.info("repair msg server fail.")
            err_msg = common.getMsg(
                self.lang, "msg.conn.check.repair.not.success",
                current_ctrl_id,
            )
            exec_context.append_err_msg(err_msg)
            return FuncResult(ResultType.FAILED, "", "")
        except UnCheckedException as e:
            exec_context.append_err_msg(e.errorMsg)
            return FuncResult(ResultType.FAILED, "", "")
        except (ToolException, Exception):
            # 如果执行命令过程发生异常，则此次查询忽略
            exec_context.logger.error(str(traceback.format_exc()))
            return FuncResult(ResultType.FAILED, "", "")
        finally:
            cliUtil.enterCliModeFromSomeModel(self.cli, lang)

    def is_risk_version(self):
        return self.product_version in RISK_VERSION

    def is_msg_server_normal(self):
        """
        进程个数不超过26个是正常，否则不正常。
        pis：不处理异常，查询不到就通过。小于26就通过。
        :return: True: 正常， False: 不正常。
        """
        cmd = "omm show_status msg_server"
        flag, ret, msg = cliUtil.excuteCmdInDebugModel(
            self.cli, cmd, True, self.lang)
        self.origin_info.append(ret)
        return len(re.compile(
            "Client\s*\d+,\s*Socket\s*=\s*\d+\s*",
            re.IGNORECASE).findall(ret)) < MESSAGE_PROCESS_MAX_NUM

    def is_daemon_process_normal(self):
        """
        daemon进程是否在位
        :return: True: 正常， False: 不正常。
        """
        cmd = "ps -C daemon"
        flag, ret, msg = cliUtil.excute_cmd_in_minisystem_with_end_sign(
            self.cli, cmd, self.lang, [cliUtil.MINISYSTEM_MODEL_FLAG])
        self.origin_info.append(ret)
        return bool(re.compile("\d+.*\sdaemon").findall(ret))

    def query_msg_server_id(self, ctrl_id):
        """
        查询 msg 进程
        如果存在，确认只有一个
        :return: True: 正常， False: 不正常。
        """
        cmd = "ps -C msg_server"
        flag, ret, msg = cliUtil.excute_cmd_in_minisystem_with_end_sign(
            self.cli, cmd, self.lang, [cliUtil.MINISYSTEM_MODEL_FLAG])
        self.origin_info.append(ret)
        msg_process_id = re.compile("(\d+).*\smsg_server").findall(ret)
        if not msg_process_id:
            err_key = "msg.conn.check.msg.server.not.exist"
            err_msg = common.getMsg(self.lang, err_key, ctrl_id)
            raise UnCheckedException(err_msg)

        return msg_process_id[0]

    def repair_msg_server(self, ctrl_id):
        """
        修复 msg 进程
        :return: True: 成功， False: 不成功。
        """
        before_process_id = self.query_msg_server_id(ctrl_id)
        cmd = "kill -9 {}".format(before_process_id)
        flag, ret, msg = cliUtil.excuteCmdInMinisystemModel(
            self.cli, cmd, self.lang)
        self.origin_info.append(ret)
        # 睡眠5秒
        time.sleep(MESSAGE_PROCESS_REBOOT_TIME)

        after_process_id = self.query_msg_server_id(ctrl_id)
        return before_process_id != after_process_id
