# coding=UTF-8

#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re
import copy
import traceback
import codecs

from com.huawei.ism.tool.obase.exception import ToolException
import com.huawei.ism.tool.obase.utils.DeviceTypeUtil as DeviceTypeUtil
import com.huawei.ism.tool.protocol.factory.ConnectorFactory as connFactory

from cbb.frame.context.contextUtil import createCliConnection
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils

from cbb.business.collect.deviceInfo import parse_cmd_config
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.cli.execute_on_all_controllers import (
    ExecuteOnAllControllers,
    ExeOnAllCtrlContext,
    ResultType,
    FuncResult
)
from cbb.frame.util.common import wrapAllException


class MultiCmdExecuter():
    """
    批量命令采集器（可适用于多命令模式）
    """

    def __init__(self, context, cmd_conf_file, result_file):
        self.context = context
        # 命令配置文件模板：参照同级目录下的cmd_list_template.xml
        self.cmd_conf_file = cmd_conf_file
        self.result_file = result_file

        # 命令采集过程中的关键信息缓存：保存ID等信息，用于其他命令做参数输入
        # 格式：{"lun_id": ["0","1","2"]}
        self.shared_key_msg = {}
        self.lang = context["lang"]
        self.logger = context["logger"]
        self.debug_password = context["debug"]
        self.dev_node = context["devNode"]
        self.is_hyper_device = bool(self.dev_node.getRemoteSNs())
        self.ssh = context["SSH"]
        self.new_ssh = context["SSH"]
        context["connectorFactory"] = connFactory
        self.minisystem_cmd_conf_list = []
        self.debug_cmd_conf_list = []
        self.welcome_cmd = "show login message"
        # 待补充扩展
        self.function_map = {"Horizontal": cliUtil.getHorizontalCliRet,
                             "Vertical": cliUtil.getVerticalCliRet}

    def execute(self):
        """
        命令采集器功能入口
        :return: None
        """
        try:
            collect_cmd_list = parse_cmd_config.get_cmd_list(
                self.cmd_conf_file)
            for model_conf in collect_cmd_list:
                cmd_model = model_conf.get("model")
                cmd_conf_list = model_conf.get("commands")
                if cmd_model == "CLI":
                    self.exe_cmd_list_in_cli_model(cmd_conf_list)

                elif cmd_model == "Developer":
                    self.exe_cmd_list_in_developer_model(cmd_conf_list)

                elif cmd_model == "Minisystem":
                    self.minisystem_cmd_conf_list = cmd_conf_list

                elif cmd_model == "Diagnose":
                    self.debug_cmd_conf_list = cmd_conf_list

                elif cmd_model == "rest":
                    self.exe_cmd_list_in_rest_model(cmd_conf_list)

            # 创建用于遍历所有控制器的上下文环境（用于采集minisystem或者diagnose命令）
            if DeviceTypeUtil.hasSVPModule(self.dev_node) \
                    and not contextUtil.is_run_in_svp() and not self.dev_node.canEnterDiagnose():
                self.logger.info("not need collect debug cmd out in svp.")
                return
            if self.debug_cmd_conf_list or self.minisystem_cmd_conf_list:
                tmp_context = self.create_context()
                self.new_ssh = self.get_cli_high_end(
                    self.ssh, self.dev_node, tmp_context
                )
                all_nodes_context = self.create_all_node_exec_context()
                welcome_msg = self.ssh.getLoginMessage()
                all_nodes_context.clear_cli_ret()
                all_nodes_context.append_cli_ret(welcome_msg)
                self.logger.info("cli_ret:{}".format(
                    all_nodes_context.cli_ret))
                # minisystem或者debug命令：可能需要采集所有控制器节点，统一处理
                self.exe_cmd_list_for_minisystem_debug_model(all_nodes_context)
        except Exception as e:
            self.logger.error(
                "[MultiCmdExecuter] exception: {}".format(str(e)))
            self.logger.error("[MultiCmdExecuter] traceback:: {}".format(
                str(traceback.format_exc())))
        except ToolException as e:
            self.logger.error(
                "[MultiCmdExecuter] exception: {}".format(str(e)))
            self.logger.error("[MultiCmdExecuter] traceback:: {}".format(
                str(traceback.format_exc())))

            # 连接等异常，需要重连SSH
            cliUtil.reConnectionCli(self.ssh, self.logger)
        finally:
            if self.new_ssh is not self.ssh:
                self.close_high_cli(self.new_ssh, self.dev_node)
            cliUtil.enterCliModeFromSomeModel(self.ssh, self.lang)
        return

    def get_cli_high_end(self, ssh, dev_node, context):
        if DeviceTypeUtil.hasSVPModule(dev_node):
            flag, ctrl_ips = cliUtil.getCtrlIps(ssh, self.logger, self.lang)
            for ip_tuple in ctrl_ips:
                cli = createCliConnection(context, ip_tuple[1])
                if not cli:
                    continue
                return cli

        return ssh

    def close_high_cli(self, ssh, dev_node):
        """
        关闭ssh连接
        :param dev_node: dev_node
        :param ssh: 新建的ssh连接
        :return:
        """
        if DeviceTypeUtil.hasSVPModule(dev_node):
            cliUtil.closeCliConnection(ssh)

    def create_context(self):
        new_context = contextUtil.getContext(self.context)
        new_context["logger"] = self.logger
        new_context["lang"] = self.lang
        dev_node = self.context.get("devNode")
        new_dev_node = EntityUtils.toNewDev(dev_node)
        new_context['dev'] = new_dev_node
        new_context['devInfo'] = new_dev_node
        new_context['debug'] = self.debug_password
        return new_context

    def create_all_node_exec_context(self):
        """
        创建用于遍历所有控制器的上下文环境（用于采集minisystem或者diagnose命令）
        :return: exe_context
        """
        new_context = self.create_context()
        new_context["ssh"] = self.new_ssh
        dev_node = self.context.get("devNode")
        exe_context = ExeOnAllCtrlContext(
            new_context,
            self.new_ssh,
            self.logger,
            str(dev_node.getDeviceType()),
            str(dev_node.getProductVersion())
        )

        # 保存实例，在收集时使用
        exe_context.cmd_exe_instance = self
        return exe_context

    def common_exe_single_cmd_and_save(self, ssh, cmd, output_conf_list):
        """
        单条命令执行入口（公共入口）
        :param cmd: 命令
        :param output_conf_list: 命令回文解析关键字配置列表
        :return: None
        """
        # 步骤1：执行命令
        cmd_result = ssh.execCmd(cmd)

        # 步骤2：解析回文中的关键信息并保存，方便后续命令取用
        self._parse_and_save_cmd_result(cmd, cmd_result, output_conf_list)
        return

    def common_create_cmd_list_and_exe(self, ssh, cmd_conf_list):
        """
        创建命令列表并批量执行，保存（所有命令模式均可使用）
        :param cmd_conf_list: 命令配置列表
        :return: None
        """

        for cmd_ins in cmd_conf_list:
            if not self.is_hyper_device and cmd_ins.hyper_cmd:
                self.logger.info("not hyper dev, not collect cmd:{}".format(
                    cmd_ins.base_cmd))
                continue
            output_conf_list = cmd_ins.output_conf_list
            # 拼装命令
            final_cmd_list = self.create_cmd_list_by_params(cmd_ins)
            for final_cmd in final_cmd_list:
                if re.search("#.*#", final_cmd):
                    continue
                self.common_exe_single_cmd_and_save(
                    ssh, final_cmd, output_conf_list,
                )
        return

    def common_create_rest_cmd_list_and_exe(self, rest_conn, base_uri,
                                            cmd_conf_list):
        """
        创建命令列表并批量执行，保存（所有命令模式均可使用）
        :param cmd_conf_list: 命令配置列表
        :return: None
        """

        for cmd_ins in cmd_conf_list:
            if not self.is_hyper_device and cmd_ins.hyper_cmd:
                self.logger.info("not hyper dev, not collect cmd:{}".format(
                    cmd_ins.base_cmd))
                continue
            output_conf_list = cmd_ins.output_conf_list
            # 拼装命令
            final_cmd_list = self.create_cmd_list_by_params(cmd_ins)
            for final_cmd in final_cmd_list:
                if re.search("#.*#", final_cmd):
                    continue

                # 步骤1：执行命令
                response_info = rest_conn.execGet("{}{}".format(
                    base_uri, final_cmd)
                )
                cmd_result = response_info.getContent()
                # 步骤2：解析回文中的关键信息并保存，方便后续命令取用
                self._parse_and_save_cmd_result(final_cmd, cmd_result,
                                                output_conf_list)
        return

    @wrapAllException
    def exe_cmd_list_in_rest_model(self, cmd_conf_list):
        """
        采集Rest命令信息
        :param cmd_conf_list: 命令配置列表
        :return:
        """
        if self.context.get("concurrent") is True:
            self.exe_cmd_list_in_new_rest_model(cmd_conf_list)
            return
        java_map = contextUtil.getContext(self.context)
        rest_conn_wrapper = contextUtil.getRest(java_map)
        base_uri = rest_conn_wrapper.getBaseUri()
        rest_connection = rest_conn_wrapper.getRest()
        self.common_create_rest_cmd_list_and_exe(
            rest_connection, base_uri, cmd_conf_list
        )
        rest_connection.closeSession()

    @wrapAllException
    def exe_cmd_list_in_new_rest_model(self, cmd_conf_list):
        self.logger.info("start creat a new rest_conn to collect!")
        dev_node = self.context.get("devNode")
        ip = dev_node.getIp()
        rest_conn = self.context.get("restMamager").getRestConnectionThrowLoginError(dev_node)
        uri = r"https://{}:8088/deviceManager/rest/{}/" \
            .format("[{}]".format(ip) if ":" in str(ip) else str(ip),
                    dev_node.getDeviceSerialNumber())
        self.common_create_rest_cmd_list_and_exe(rest_conn, uri, cmd_conf_list)
        self.context.get("restMamager").releaseConn(dev_node)

    def exe_cmd_list_in_cli_model(self, cmd_conf_list):
        """
        采集CLI命令信息
        :param cmd_conf_list: 命令配置列表
        :return:
        """
        try:
            cliUtil.enterCliModeFromSomeModel(self.ssh, self.lang)
            self.common_create_cmd_list_and_exe(self.ssh, cmd_conf_list)
        except (Exception, ToolException):
            self.ssh.reConnect()
            self.logger.error("[MultiCmdExecuter] traceback:: {}".format(
                str(traceback.format_exc())))

    def exe_cmd_list_in_developer_model(self, cmd_conf_list):
        """
        批量执行developer命令
        :param cmd_conf_list: 命令配置列表
        :return: None
        """
        try:
            # 步骤1：切换到developer模式
            flag, _, _ = cliUtil.enterDeveloperMode(self.ssh, self.lang,
                                                    self.debug_password)
            if flag is not True:
                self.logger.error("Change to developer model failed!")
                # 退出到Cli模式
                cliUtil.developerMode2CliMode(self.ssh)
                return

            # 步骤2：执行所有命令
            self.common_create_cmd_list_and_exe(self.ssh, cmd_conf_list)

            # 步骤3：退出到Cli模式
            cliUtil.developerMode2CliMode(self.ssh)
            return
        except (Exception, ToolException):
            cliUtil.reConnectionCli(self.ssh, self.logger)
            self.logger.error("[MultiCmdExecuter] traceback:: {}".format(
                str(traceback.format_exc())))

    @staticmethod
    @ExecuteOnAllControllers
    def exe_cmd_list_for_minisystem_debug_model(exe_context):
        """
        采集当前存储设备所有控制器的命令：minisystem和debug命令
        :param exe_context: 特定上下文环境
        :return: FuncResult: 执行结果
        """
        # 需要使用exe_context的cli链接，兼容高端SVP场景
        ssh = exe_context.dev_info.cli
        exe_context.cmd_exe_instance.ssh = ssh
        # 获取采集器实例
        executer_ins = exe_context.cmd_exe_instance
        logger = executer_ins.logger
        if not isinstance(executer_ins, MultiCmdExecuter):
            logger.error("Instance in context is not a MultiCmdExecuter!")
            return FuncResult(ResultType.FAILED, '', '', False)
        local_node_id = MultiCmdExecuter.prepare_collect(
            executer_ins, exe_context)
        # 获取minisystem和debug采集命令配置列表
        mini_cmd_conf_list = executer_ins.minisystem_cmd_conf_list
        debug_cmd_conf_list = executer_ins.debug_cmd_conf_list
        if not (mini_cmd_conf_list or debug_cmd_conf_list):
            return FuncResult(ResultType.FAILED, '', '', False)
        # 获取结果文件
        result_file = executer_ins.result_file
        # 步骤3：采集debug信息
        ret = ssh.execCmd("sys showcls")
        if debug_cmd_conf_list and cliUtil.isInDebugMode(ret):
            MultiCmdExecuter.execute_in_debug_mode(
                exe_context, executer_ins, debug_cmd_conf_list)
        # 步骤4：minisystem模式，采集命令
        if mini_cmd_conf_list:
            MultiCmdExecuter.execute_in_mini_system_mode(
                executer_ins, exe_context, mini_cmd_conf_list
            )
        MultiCmdExecuter.teardown_debug_model(
            executer_ins, local_node_id, result_file, exe_context
        )
        return FuncResult(ResultType.SUCCESS, '', '', True)

    @staticmethod
    def execute_in_mini_system_mode(
            executer_ins, exe_context, mini_cmd_conf_list
    ):
        ssh = exe_context.dev_info.cli
        lang = executer_ins.lang
        logger = executer_ins.logger
        ret = ssh.execCmd("sys showcls")
        if cliUtil.isInDebugMode(ret):
            flag, ret = cliUtil.enterMinisystemFromDebugMode(ssh, lang)
            # 如果是故障模式，进来就是minisystem
            if flag is not True:
                logger.error("Change to minisystem model failed!")
                cliUtil.enterCliModeFromSomeModel(ssh, lang)
                return FuncResult(ResultType.FAILED, '', '', False)
        if not cliUtil.isInMinisystemMode(ret):
            return FuncResult(ResultType.FAILED, '', '', False)
        # 执行命令采集
        executer_ins.common_create_cmd_list_and_exe(
            ssh, mini_cmd_conf_list
        )

    @staticmethod
    def prepare_collect(executer_ins, exe_context):
        result_file = executer_ins.result_file
        ssh = exe_context.dev_info.cli
        lang = executer_ins.lang
        logger = executer_ins.logger
        debug_pwd = executer_ins.debug_password
        local_node_id = '0'
        # 步骤1：切换到debug模式
        flag, cli_ret, err_msg = cliUtil.enterDebugMode(ssh, lang, debug_pwd)
        if flag is not True and not cliUtil.isInMinisystemMode(cli_ret):
            logger.error("Change to debug model failed!")
            cliUtil.enterCliModeFromSomeModel(ssh, lang)
            return local_node_id

        cmd = "sys showcls"
        if cliUtil.isInMinisystemMode(cli_ret):
            cmd = "showsysstatus"
        cli_ret = ssh.execCmd(cmd)
        local_node_id = executer_ins._get_local_node_id(cli_ret)
        cmd_result_list = list()
        cmd_result_list.append(
            "#############################################\n")
        cmd_result_list.append("[Controller Start] node_id:{}\n".format(
            local_node_id))
        cmd_result_list.append(executer_ins._wrap_cmd_result(cmd, cli_ret))
        executer_ins._write_cmd_result("".join(cmd_result_list), result_file)
        welcom_msg = executer_ins._wrap_cmd_result(
            executer_ins.welcome_cmd, exe_context.cli_ret)
        executer_ins._write_cmd_result(welcom_msg, result_file)
        logger.info("welcom_msg:{}".format(welcom_msg))
        return local_node_id

    @staticmethod
    def execute_in_debug_mode(exe_context, executer_ins, debug_cmd_conf_list):
        ssh = exe_context.dev_info.cli
        logger = executer_ins.logger
        try:
            # 步骤1：切换到debug模式
            logger.info("all_nodes_context.cli_ret:{}".format(
                exe_context.cli_ret))
            # 步骤2：采集debug信息
            if debug_cmd_conf_list:
                executer_ins.common_create_cmd_list_and_exe(
                    ssh, debug_cmd_conf_list
                )
        except (Exception, ToolException):
            cliUtil.reConnectionCli(ssh, logger)
            logger.error("[MultiCmdExecuter] traceback:: {}".format(
                str(traceback.format_exc())))

    @staticmethod
    def teardown_debug_model(executer_ins, local_node_id, result_file,
                             exe_context):
        ssh = exe_context.dev_info.cli
        lang = executer_ins.lang
        logger = executer_ins.logger
        # 增加控制器间信息间隔符，并写入文件
        cmd_result = "#############################################\n"
        cmd_result += "[Controller End] node_id:%s\n" % local_node_id
        executer_ins._write_cmd_result(cmd_result, result_file)

        # 切换回CLI模式
        cliUtil.enterCliModeFromSomeModel(ssh, lang)
        logger.info("clear welcom_msg:{}".format(exe_context.cli_ret))
        exe_context.clear_cli_ret()

    @staticmethod
    def _get_local_node_id(sys_showcls_cmd_result):
        """
        通过已执行好的sys showcls命令回文解析本节点node_id
        :param sys_showcls_cmd_result: 命令回文
        :return: node_id： 当前控制器的ID
        """

        node_id = ""
        vertical_ret = cliUtil.getVerticalCliRet(sys_showcls_cmd_result)
        for vertical_info in vertical_ret:
            node_id = vertical_info.get("local node id", "")

        return node_id

    @staticmethod
    def _write_cmd_result(cmd_result, result_file):
        """
        通用的写文件函数
        :param cmd_result: 命令回文
        :param result_file: 需写入的文件
        :return: None
        """
        with codecs.open(result_file, "a", encoding='utf-8') as file_obj:
            file_obj.write(cmd_result)

    @staticmethod
    def _wrap_cmd_result(cmd, cmd_result):
        """
        命令执行结果，增加识别符，方便离线导入分析功能解析
        :param cmd: 命令
        :param cmd_result: 命令执行结果
        :return: cmd_result_wrapped： 修饰后的执行结果
        """
        cmd_result_wrapped = "#############################################\n"
        cmd_result_wrapped += "StartCmd:%s\n" % cmd
        cmd_result_wrapped += cmd_result + "\n"
        cmd_result_wrapped += "EndCmd:%s\n" % cmd
        return cmd_result_wrapped

    def _parse_and_save_cmd_result(self, cmd, cmd_result, output_conf_list):
        """
        解析命令执行结果并保存
        :param cmd: 命令
        :param cmd_result: 命令执行结果
        :param output_conf_list: 命令回文输出关键配置信心
        :return: None
        """

        # 保存命令结果
        final_cmd_result = self._wrap_cmd_result(cmd, cmd_result)
        self._write_cmd_result(final_cmd_result, self.result_file)

        # 解析保存关键信息，用作后续命令输入
        for out_conf in output_conf_list:
            key_name = out_conf.get("name", "")
            key_value = out_conf.get("key_value", "")
            parse_func_desc = out_conf.get("parse_func")
            save_key_list = []

            parse_exe_func = self.get_parse_func(parse_func_desc)
            if parse_exe_func is None:
                self.logger.error("Parse Function {} config invalid!".format(
                    str(parse_func_desc)))
                return

            # 执行解析函数
            obj_info_list = parse_exe_func(cmd_result)
            # 多参数场景，入参和出参都是用,分隔。
            if "," in key_value and "," in key_name:
                key_name_list = key_name.split(",")
                key_value_list = key_value.split(",")
                for obj_info in obj_info_list:
                    value_list = [obj_info.get(t) for t in key_value_list]
                    save_key_list.append(dict(zip(key_name_list, value_list)))
            else:
                for obj_info in obj_info_list:
                    save_key_list.append(obj_info.get(key_value, "Error"))

            # 关键信息存在重复，说明配置文件中存在重复的param配置，需要更正
            if key_name in self.shared_key_msg:
                self.logger.error(
                    "Key {} duplicate configured!".format(str(key_name)))
                return
            # 保存关键信息，便于后续其他命令使用
            self.shared_key_msg[key_name] = save_key_list
        return

    def get_parse_func(self, func_desc):
        """
        获取对应的回文解析函数
        :param func_desc: 函数配置关键字
        :return: func：解析函数
        """

        # 待补充扩展
        return self.function_map.get(func_desc)

    def create_cmd_list_by_params(self, cmd_instance):
        """
        通过原始命令和输入参数信息，创建最终命令执行列表
        :param cmd_instance: 命令实例
        :return: final_cmd_list：最终执行的命令列表
        """

        base_cmd = cmd_instance.base_cmd
        input_conf_list = cmd_instance.input_conf_list
        self.logger.info("base_cmd:{},\ninput_conf_list:{}".format(
            base_cmd, input_conf_list)
        )
        final_cmd_list = [base_cmd]
        # 没有入参，直接执行命令
        if not input_conf_list:
            return final_cmd_list

        for input_param_conf in input_conf_list:
            input_param_list = self.get_input_param_list(input_param_conf)
            # 获取列表失败，报错
            if not input_param_list:
                self.logger.error(
                    "[collect_cmd] key message {} is invalid!".format(
                        str(input_param_conf)))
                return final_cmd_list

            value_key = input_param_conf.get("name")
            # 复制一份当前命令列表
            cmd_stack = copy.deepcopy(final_cmd_list)
            while cmd_stack:
                cmd_tmp = cmd_stack.pop()
                # 一条命令中带多个参数的，需要使用‘逗号’分隔。
                value_key_list = value_key.split(",")
                self.deal_multi_param(value_key_list, input_param_list,
                                      cmd_tmp, final_cmd_list)
        self.logger.info(
            "Config cmd [{}]: Final cmd list is {}".format(str(base_cmd), str(
                final_cmd_list)))
        return final_cmd_list

    def deal_multi_param(self, value_key_list, input_param_list,
                         cmd_tmp, final_cmd_list):
        """
        组装命令，将命令中参数全部替换为实际数据。
        :param value_key_list:
        :param input_param_list:
        :param cmd_tmp:
        :param final_cmd_list:
        :return:
        """
        for param_value_info in input_param_list:
            new_cmd = cmd_tmp
            for tmp_value_key in value_key_list:
                match_msg = "#%s#" % tmp_value_key
                if match_msg not in cmd_tmp:
                    continue
                if type(param_value_info) == dict:
                    param_value = param_value_info.get(tmp_value_key)
                else:
                    param_value = param_value_info
                new_cmd = new_cmd.replace(match_msg, param_value)
            if new_cmd:
                final_cmd_list.append(new_cmd)
        final_cmd_list.remove(cmd_tmp)

    def get_input_param_list(self, input_param_conf):
        """
        解析命令配置，获取参数输入列表
        :param input_param_conf: 命令输入参数的配置信心
        :return: input_param_list：实际输入参数列表
        """
        limit_range = input_param_conf.get("limit_count")
        input_param_list = input_param_conf.get("params")
        if not input_param_list:
            # 未配置，从共享信息里面获取
            input_param_list = self.shared_key_msg.get(
                input_param_conf.get("name"), [])
        if limit_range:
            return input_param_list[limit_range[0]: limit_range[1]]
        return input_param_list
