# -*- coding: UTF-8 -*-

import traceback

from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common import FuncFactory

from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest.restData import RestCfg as restData
from cbb.frame.rest import restData as tlvRestData
from cbb.frame.rest import restUtil
from cbb.frame.rest.restUtil import CommonRest
from cbb.frame.tlv import tlvData

PORT_CMD = {
    tlvData.OM_OBJ_E["ETH_PORT"]:
        "show port general physical_type=ETH logic_type=Host_Port",
    tlvData.OM_OBJ_E["FC_PORT"]:
        "show port general physical_type=FC logic_type=Host_Port",
}

PORT_CLI_TYPE = {
    tlvData.OM_OBJ_E["ETH_PORT"]: "ETH",
    tlvData.OM_OBJ_E["FC_PORT"]: "FC",
}

PORT_CLI_TYPE_LIST = ["ETH", "FC", "RoCE"]

# 计算型存储只支持ETH端口
PORT_CLI_TYPE_LIST_FOR_COMPUTE = ["ETH"]


class CheckItem:
    def __init__(self, cli, rest, context):
        self.context = context
        self.logger = baseUtil.getLogger(context.get("logger"), __file__)
        self.lang = contextUtil.getLang(context)

        self.cli = cli
        self.rest = rest

        # controller, front interface, front port
        self.fru_obj = contextUtil.getItem(context, "FRU_OBJ")
        self.fru_loc = contextUtil.getItem(context, "FRU_LOCATION")
        self.fru_id = baseUtil.cntrlIdFormat(
            contextUtil.getItem(context, "FRU_ID"))
        self.product_model = contextUtil.getProductModel(context)

        self.result = True
        self.err_msg = ""
        self.suggestion = ""
        self.errHosts = set()

        self.all_ports = {}
        self.checked_ports = {}
        self.other_ports = {}
        self.port_for_host = {}
        self.checked_host_info = {}
        self.total_ports = {}
        self.host_for_port_group = {}

    def check(self):
        """检查主流程

        :return:
        """
        try:
            if not self.checkRunningStatus():
                self.logger.logInfo("The fru is not Link Up or Online.")
                return

            # get all front ports that the running status is LinkUp
            self.all_ports = self.getAllPorts()
            self.logger.logInfo("allPorts:%s" % self.all_ports)
            self.logger.logInfo("totalPorts:%s" % self.total_ports)

            # get front ports for checking
            self.checked_ports = self.getCheckedPorts()
            self.logger.logInfo("checkedPorts:%s" % self.checked_ports)

            # get front ports exclude checked ports
            self.other_ports = self.getOtherPorts()
            self.logger.logInfo("otherPorts:%s" % self.other_ports)

            # get hosts for checking
            self.port_for_host = {}
            self.checked_host_info = {}
            self.getCheckedHost()
            self.logger.logInfo("checkedHostInfo:%s" % self.checked_host_info)

            if not self.checked_host_info:
                self.logger.logInfo(
                    "donot have host on the fru(%s)" % self.fru_loc)
                return

            # get host of port group
            self.getPortGroup4Host()
            self.logger.logInfo("host4PortGroup:%s" % self.host_for_port_group)

            # check host link
            self.checkHost()
        except Exception as ex:
            self.result = False
            self.logger.logException(ex)
        return

    def checkRunningStatus(self):
        """检查端口运行状态

        :return: "Link Up"和"Online"时返回True；其他返回False
        """
        if self.fru_obj == tlvData.OM_OBJ_E["INTF_MODULE"]:
            return True

        running_status = "Link Up"
        if self.fru_obj == tlvData.OM_OBJ_E["CONTROLLER"]:
            cmd = "show controller general controller=%s" % self.fru_id
            running_status = self.getRunningStatus(cmd)
        elif self.fru_obj in [
            tlvData.OM_OBJ_E["ETH_PORT"],
            tlvData.OM_OBJ_E["FC_PORT"],
            tlvData.OM_OBJ_E["FCoE_PORT"]
        ]:
            cmd = "show port general port_id=%s" % self.fru_loc
            running_status = self.getRunningStatus(cmd)
        elif self.fru_obj in [tlvData.OM_OBJ_E["IB_PORT"]]:
            cmd = "show ib_port general ib_port_id=%s" % self.fru_loc
            running_status = self.getRunningStatus(cmd)
        self.logger.logInfo(
            "get ruuning status of the fru:%s" % running_status)
        if running_status in ["Link Up", "Online"]:
            return True
        return False

    def getRunningStatus(self, cmd):
        """执行命令获取运行状态

        :param cmd: cli命令
        :return: 运行状态
        """
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        if not flag:
            self.result = False
            raise Exception("get running status error")

        info_list = cliUtil.getVerticalCliRet(cli_ret)
        for info in info_list:
            if "Running Status" in info:
                return info.get("Running Status")

        raise Exception("There is not Running Status in he fru info")

    def getAllPorts(self):
        """获取所有端口信息

        :return:
        """

        self.total_ports = {}
        ports = {}
        if baseUtil.is_support_NVMe_intf(self.context):
            port_type_list = PORT_CLI_TYPE_LIST
        else:
            port_type_list = PORT_CLI_TYPE.values()
        # 计算型存储只支持ETH端口
        if baseUtil.is_computing_dev(contextUtil.getProductModel(self.context)):
            port_type_list = PORT_CLI_TYPE_LIST_FOR_COMPUTE
        for pt in port_type_list:
            cmd = "show port general physical_type=%s" \
                  " logic_type=Host_Port" % pt
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.lang)

            if not cliUtil.hasCliExecPrivilege(cli_ret):
                continue

            if cliUtil.queryResultWithNoRecord(cli_ret):
                continue

            if not flag:
                # 获取端口信息失败
                msg, sug = common.getMsg(
                    self.lang, "check.cli.host.red.link.ports.fail")
                self.err_msg = msg
                self.suggestion = sug
                self.result = False
                return {}

            port_list = cliUtil.getHorizontalCliRet(cli_ret)
            for p in port_list:

                if not p.get("ID", None):
                    continue
                # RoCE类型端口保存为ETH端口，因为从rest接口查询时，RoCE是属于ETH类型的
                if pt == "RoCE":
                    pt = "ETH"
                self.total_ports[p.get("ID")] = pt
                if p.get("Running Status", None) != "Link Up":
                    continue
                ports[p.get("ID")] = pt

        return ports

    def isOnCtrl(self, loc, ctrId, eth_port):
        """判断归属控制器，针对以太网口

        :param loc:
        :param ctrId:
        :param eth_port:
        :return:
        """
        eng = "CTE%s" % ctrId[0] if "CTE" in loc else "ENG%s" % ctrId[0]
        prefix = "%s.%s" % (eng, ctrId[-1])
        if prefix in loc:
            return True

        if not eth_port:
            return False
        own_ctrl = restUtil.Tlv2Rest.getRecordValue(
            eth_port,
            tlvRestData.Hardware.EthPort.OWNING_CONTROLLER)
        return own_ctrl and own_ctrl == ctrId

    def getPortsByCtrl(self, ctrId):
        """获取控制器上的所有端口

        :param ctrId: 控制器ID
        :return:
        """
        ports = {}
        eth_ports = FuncFactory.getFruListInfo(
            self.context,
            tlvRestData.Enum.ObjEnum.ETH_PORT)
        for (pId, pType) in self.all_ports.items():
            condition = restUtil.Tlv2Rest.getCondition(
                tlvRestData.PublicAttributes.LOCATION,
                tlvRestData.Enum.ConditionTypeEnum.EQ,
                pId)
            conditions = restUtil.Tlv2Rest.getConditionList(condition)
            eth_port = restUtil.Tlv2Rest.filter(eth_ports, conditions)
            check_eth_port = eth_port[0] if eth_port else None
            ok = self.isOnCtrl(pId, ctrId, check_eth_port)
            if ok:
                ports[pId] = pType

        return ports

    def getPortsByIntf(self, inftLoc):
        """获取接口卡上的所有端口

        :param inftLoc: 接口卡location
        :return:
        """
        ports = {}
        format_loc = "%s." % inftLoc
        for (pId, pType) in self.all_ports.items():
            if format_loc in pId:
                ports[pId] = pType
        return ports

    def getCheckedPorts(self):
        """获取所有要检查的端口

        :return:
        """
        self.logger.logInfo("get port of the fru(%s)" % self.fru_obj)
        if self.fru_loc in self.total_ports:
            return {self.fru_loc: self.total_ports.get(self.fru_loc)}
        elif self.fru_obj == tlvData.OM_OBJ_E["CONTROLLER"]:
            self.logger.logInfo("get port of the ctrl(%s)" % self.fru_id)
            return self.getPortsByCtrl(self.fru_id)
        elif self.fru_obj == tlvData.OM_OBJ_E["INTF_MODULE"]:
            self.logger.logInfo("get port of the intf(%s)" % self.fru_loc)
            return self.getPortsByIntf(self.fru_loc)
        elif self.fru_obj in [
            tlvData.OM_OBJ_E["ETH_PORT"],
            tlvData.OM_OBJ_E["FC_PORT"]
        ]:
            self.logger.logInfo("get port(%s) info" % self.fru_loc)
            return {self.fru_loc: self.all_ports.get(self.fru_loc)}
        return {}

    def getCliRet4Port(self, cmd, isLog, isHorizontalRet=True):
        """执行命令获取返回值

        :param cmd: cli命令
        :param isLog: 是否记录日志
        :return:
        """
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, isLog, self.lang)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return []

        if not cliUtil.hasCliExecPrivilege(cli_ret):
            return []

        if flag is not True:
            msg, sug = common.getMsg(
                self.lang, "check.cli.host.red.link.ports.fail")
            self.err_msg = msg
            self.suggestion = sug
            self.result = False
            raise Exception("get port info error.")

        if isHorizontalRet:
            info_list = cliUtil.getHorizontalCliRet(cli_ret)
        else:
            info_list = cliUtil.getVerticalCliRet(cli_ret)
        if not info_list:
            msg, sug = common.getMsg(
                self.lang, "check.cli.host.red.link.ports.fail")
            self.err_msg = msg
            self.suggestion = sug
            self.result = False
            return Exception("parse port info error.")
        return info_list

    def getOtherPorts(self):
        """获取其他端口

        :return:
        """
        other_posts = {}
        for x in self.all_ports:
            if x not in self.checked_ports:
                other_posts[x] = self.all_ports.get(x)
        return other_posts

    def isValidHost(self, hostId):
        """判断主机是否合法

        :param hostId: 主机ID
        :return:
        """
        record = CommonRest.getHostRecord(self.rest, hostId)
        if CommonRest.getRecordValue(record, restData.Host.ISADD2HOSTGROUP) != "true":
            return False
        return True

    def getHostIdbyPort(self, portId, portType, isCheckLun=True):
        """根据端口ID获取主机信息

        :param portId: 端口ID
        :param portType: 端口类型
        :param isCheckLun: lun标志
        :return:
        """

        if portId in self.port_for_host:
            return self.port_for_host.get(portId)

        hosts = set()

        cmd = "show port initiator port_type=%s" \
              " port_id=%s" % (portType, portId)
        info_list = self.getCliRet4Port(cmd, True)
        # 如果是ETH端口，一般命令查询无结果且支持NVMe接口卡时，尝试查询该端口的NVMe启动器
        if portType == "ETH" \
                and not info_list \
                and baseUtil.is_support_NVMe_intf(self.context):
            cmd = "show port nvme_over_roce_initiator port_id={}".format(
                portId)
            info_list = self.getCliRet4Port(cmd, True, isHorizontalRet=False)

        for info in info_list:
            host = info.get("Host ID", "--")
            if not host.isdigit():
                continue

            if isCheckLun:
                if not baseUtil.isDoradoV6Dev(self.product_model) and \
                        not self.isValidHost(host):
                    self.logger.logInfo("host is invalid.")
                    continue
                if baseUtil.is_micro_dev(self.product_model):
                    count = CommonRest.get_namespace_count(self.rest, host)
                else:
                    count = CommonRest.getHostLunAndSnapCount(self.rest, host)
                if 0 >= count:
                    self.logger.logInfo(
                        "host has no luns or snapshots:%s" % count)
                    continue

            hosts.add(host)

        self.port_for_host[portId] = hosts
        return hosts

    def getPortGroup4Host(self):
        """获取主机的端口分组

        :return:
        """
        self.host_for_port_group = {}
        mapping_views = CommonRest.getMappingViews(self.rest)
        for mapping_view in mapping_views:
            mapping_view_id = CommonRest.getRecordValue(
                mapping_view, restData.PublicAttributes.ID)
            port_groups = CommonRest.getPortgroupByMapping(
                self.rest, mapping_view_id)
            self.logger.logInfo("getPortGroup4Host mappingviewId:%s portgroups:%s" % (
                mapping_view_id, port_groups))
            if not port_groups:
                continue
            host_groups = CommonRest.getHostgroupByMapping(
                self.rest, mapping_view_id)
            if not host_groups:
                continue
            port_group_id = CommonRest.getRecordValue(
                port_groups[0], restData.PublicAttributes.ID)
            ports = CommonRest.getPortByPortGroup(
                self.rest, port_group_id, contextUtil.getProductModel(self.context))
            tmp_ports = {}
            for port in ports:
                loc = CommonRest.getRecordValue(
                    port, restData.PublicAttributes.LOCATION)
                port_type = PORT_CLI_TYPE.get(
                    CommonRest.getRecordValue(
                        port, restData.PublicAttributes.TYPE), "")
                tmp_ports[loc] = port_type
            self.get_host_port_group_ports(host_groups, tmp_ports)
        return

    def get_host_port_group_ports(self, host_groups, tmp_ports):
        for host_group in host_groups:
            host_group_id = str(CommonRest.getRecordValue(host_group, restData.PublicAttributes.ID))
            host_group_host_list = CommonRest.getHostByHostGroup(self.rest, host_group_id)
            for host in host_group_host_list:
                host_id = str(CommonRest.getRecordValue(host, restData.PublicAttributes.ID))
                if host_id not in self.host_for_port_group:
                    self.host_for_port_group[host_id] = []
                self.host_for_port_group.get(host_id).append(tmp_ports)

    def getCheckedHost(self):
        """获取待检查的主机

        :return:
        """
        # {hostid:[porttype]}
        self.checked_host_info = {}
        for pId in self.checked_ports:
            p_type = self.checked_ports.get(pId)
            hosts = self.getHostIdbyPort(pId, p_type)
            for host in hosts:
                if host not in self.checked_host_info:
                    self.checked_host_info[host] = set()
                self.checked_host_info.get(host).add((pId, p_type))
        return

    def checkOnePortGroup(self, hostId, group):
        """检查一组端口

        :param hostId: 主机ID
        :param group: 分组
        :return:
        """
        # 检查端口有无对应主机, 存在冗余返回True
        flag = False
        for portId in group:
            portType = group.get(portId)
            if portId not in self.other_ports:
                continue

            hosts = self.getHostIdbyPort(portId, portType, isCheckLun=False)
            if hostId in hosts:
                return True

        # 主机（ID:%s）不存在冗余路径。
        self.result = False
        self.errHosts.add(hostId)
        return

    def isSamePortType(self, type1, type2):
        """判断端口类型是否相同

        :param type1: 端口类型1
        :param type2: 端口类型2
        :return:
        """
        if type1 == type2:
            return True

        if type1 in ["FC", "FCoE"] and type2 in ["FC", "FCoE"]:
            return True

        return False

    def checkOneHost(self, hostId):
        """检查一个主机

        :param hostId: 主机ID
        :return:
        """

        port_groups = []
        if hostId in self.host_for_port_group:
            port_groups = self.host_for_port_group.get(hostId)

        for p_id, p_type in self.checked_host_info.get(hostId):
            if port_groups:
                for group in port_groups:
                    if p_id not in group:
                        continue
                    self.checkOnePortGroup(hostId, group)
            else:
                is_red = False
                for port_id in self.other_ports:
                    port_type = self.other_ports.get(port_id)
                    if not self.isSamePortType(p_type, port_type):
                        continue
                    hosts = self.getHostIdbyPort(
                        port_id, port_type,
                        isCheckLun=baseUtil.isDoradoV6Dev(self.product_model))
                    self.logger.logInfo("red host is: %s" % str(hosts))
                    if hostId in hosts:
                        is_red = True
                        break

                if not is_red:
                    self.result = False
                    self.errHosts.add(hostId)
        return

    def checkHost(self):
        """检查所有主机

        :return:
        """
        for host in self.checked_host_info:
            self.checkOneHost(host)
        return

    def getResult(self):
        """获取检查结果

        :return:
        """

        if self.errHosts:
            err_host_string = ','.join(self.errHosts)
            self.err_msg, self.suggestion = common.getMsg(
                self.lang,
                "check.cli.host.red.link.host.red.not.pass",
                suggestionArgs=err_host_string)
            return self.result, self.err_msg, self.suggestion

        if not self.result:
            if not self.err_msg or not self.suggestion:
                self.err_msg, self.suggestion = common.getMsg(
                    self.lang, "execute.error")
            return self.result, self.err_msg, self.suggestion

        return self.result, self.err_msg, self.suggestion
