# -*- coding: UTF-8 -*-
import re

from cbb.business.operate.fru.common import BaseFactory
from cbb.frame.base.exception import UnCheckException
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.business.operate.fru.common import common

RISK_VERSION = {
    "V500R007C60 Kunpeng": "",
    "V500R007C60SPC100 Kunpeng": "V500R007C60SPH101",
    "V500R007C60SPC200 Kunpeng": "",
    "V500R007C60SPC300 Kunpeng": "V500R007C60SPH302",
}


class Check_failover_event_ip_error:
    def __init__(self, context, location_id, is_controller=False):
        self.context = context
        self.cli = contextUtil.getCli(context)
        self.lang = contextUtil.getLang(context)
        self.location_id = location_id
        self.logger = baseUtil.getLogger(context.get("logger"), __file__)
        self.dialog_util = context.get("dialogUtil")
        self.is_controller = is_controller
        self.pro_model = ""
        """
        H0 - H6        归属A         
        H7 - H13       归属C
        L0 - L6        归属B
        L7 - L13       归属D"""
        self.high_card_dict = {
            "A": ["H0", "H1", "H2", "H3", "H4", "H5", "H6"],
            "C": ["H7", "H8", "H9", "H10", "H11", "H12", "H13"],
            "B": ["L0", "L1", "L2", "L3", "L4", "L5", "L6"],
            "D": ["L7", "L8", "L9", "L10", "L11", "L12", "L12"],
        }

    def execute(self):
        """

        :return:
        """
        flag, err_meg, suggest = self.check_risk()
        if flag is not True:
            BaseFactory.result.setResultFailByDesc(
                self.context, err_meg[0], suggest[0]
            )
            return

        BaseFactory.result.setResultPass(self.context)

    def check_risk(self):
        """
        漂移事件中IP地址检查
        software_failover_event_ip_error
        :param location_id: 接口卡，控制器，ID
        :return:
        """
        try:
            (
                flag,
                product_model,
                version,
                err_msg,
            ) = cliUtil.getProductModelAndVersion(self.cli, self.lang)
            if flag is not True:
                raise UnCheckException(err_msg)

            (
                flag,
                product_version,
                hot_patch_version,
            ) = cliUtil.getSystemVersion(self.cli, self.lang)
            if "Kunpeng" in version:
                product_version += " Kunpeng"
            if flag is not True:
                raise UnCheckException(
                    common.getMsg(self.lang, "system.status.abnormal")
                )
            self.pro_model = product_model
            self.logger.logInfo(
                "ver is:{} patch is:{}, location_id is {}".format(
                    product_version, hot_patch_version, self.location_id
                )
            )

            if not self.check_risk_version(product_version, hot_patch_version):
                return True, "", ""

            flag, ctrl_ids_list, err_msg = cliUtil.getControllerIdList(
                self.cli, self.lang
            )
            if flag is not True:
                raise UnCheckException(err_msg)

            flag, cli_ret, e_msg = cliUtil.enterDeveloperMode(
                self.cli, self.lang
            )

            if flag is not True:
                raise UnCheckException(e_msg)

            if self.is_all_ctrl_nfs_s_lock_zero(self.cli, ctrl_ids_list):
                return True, "", ""

            if self.check_all_logic_port_number(self.cli, self.lang) is True:
                return True, "", ""
            return (
                False,
                common.getMsg(self.lang, "failover_event_ip_error.suggestion"),
                self.build_err_meg(product_version),
            )
        except UnCheckException as e:
            return (
                False,
                (e.errorMsg, ""),
                common.getMsg(
                    self.lang, "failover_event_ip_error.suggestion.other"
                ),
            )
        except Exception as e:
            self.logger.logException(e)
            err_msg = common.getMsg(self.lang, "query.result.abnormal")
            return (
                False,
                err_msg,
                common.getMsg(
                    self.lang, "failover_event_ip_error.suggestion.other"
                ),
            )
        finally:
            flag, ret, err_msg = cliUtil.enterCliModeFromSomeModel(
                self.cli, self.lang
            )
            self.logger.logInfo("enter cli mode ret is : {}".format(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not flag:
                cliUtil.reConnectionCli(self.cli, self.logger)

    def is_all_ctrl_nfs_s_lock_zero(self, cli, ctrl_ids_list):
        """
        检查所有控制器s_lock是否为都为0
        :param cli:
        :param ctrl_ids_list:
        :param all_cli_ret_list:
        :return: True: 都为0
                  False: 存在不为0的
        """
        cmd = "show nfs nfsv3_nlm_stats controller={}"
        for ctrl_id in ctrl_ids_list:
            cmd_str = cmd.format(ctrl_id)
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                cli, cmd_str, True, self.lang
            )
            if flag is not True:
                raise UnCheckException(err_msg)

            reg = re.compile(r"s_lock:\s*(\d+)")
            reg_res = reg.findall(cli_ret)
            for s_lock in reg_res:
                if s_lock != "0":
                    return False
        return True

    def check_all_logic_port_number(self, cli, lang):
        """
        若步骤6中逻辑端口的数量大于1个，则检查结果为不通过，否则检查通过
        :param cli:
        :return: True: 无风险
                  False: 有风险
        """
        cmd = "show logical_port general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            cli, cmd, True, lang
        )
        if flag is not True:
            raise UnCheckException(err_msg)

        record_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(record_list) <= 1:
            return True

        for logic_port in record_list:
            cur_port_id = logic_port.get("Current Port ID", "")
            home_port_id = logic_port.get("Home Port ID", "")
            if self.check_risk_logical_port(cur_port_id, home_port_id) is True:
                return False
        return True

    def check_risk_version(self, p_version, p_patch):
        """
        检查是否是风险版本
        若系统软件版本为V500R007C60SPC300且安装V500R007C60SPH302
        或更高版本补丁，则检查通过
        :param p_version: 版本
        :param p_patch: 补丁
        :return: False: 非风险版本，True: 风险版本
        """
        if p_version in RISK_VERSION:
            if (
                RISK_VERSION.get(p_version)
                and p_patch >= RISK_VERSION.get(p_version)
                and not baseUtil.is_special_patch(p_patch)
            ):
                return False
            return True
        return False

    def build_err_meg(self, version):
        """
        构建错误消息
        :param version:
        :return:
        """
        if "V500R007C60SPC100" in version:
            meg_key = "failover_event_ip_error.suggestion.V5R7C60SPC100"
        elif "V500R007C60SPC300" in version:
            meg_key = "failover_event_ip_error.suggestion.V5R7C60SPC300"
        else:
            meg_key = "failover_event_ip_error.suggestion.other"

        return common.getMsg(self.lang, meg_key)

    def check_risk_logical_port(self, cur_port_id, home_port_id):
        """
        确认是否有风险
        :param cur_port_id:
        :param home_port_id:
        :return:  True 有风险 False 无风险
        """
        if self.location_id in cur_port_id or self.location_id in home_port_id:
            return True
        if not (
            baseUtil.is_v5_high_end_devs(self.pro_model)
            and self.is_controller is True
        ):
            return False
        high_card_list = self.location_id.split(".")
        if len(high_card_list) != 2:
            return False
        high_controller_list = self.high_card_dict.get(high_card_list[1])
        cur_port_id_list = cur_port_id.split(".")
        home_port_id_list = home_port_id.split(".")
        for high_controller in high_controller_list:
            if (
                high_controller in cur_port_id_list
                and high_card_list[0] in cur_port_id_list
            ):
                return True
            if (
                high_controller in home_port_id_list
                and high_card_list[0] in home_port_id_list
            ):
                return True
        return False
