# -*- coding: UTF-8 -*-
import re

from cbb.business.operate.fru.common import BaseFactory
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.base.exception import UnCheckException
from cbb.business.operate.fru.common import common

RISK_VERSION = re.compile("6.0.1.*")
SAFETY_HOT_PATCH = 6

# LUN的卷ID检查,风险版本范围
RISK_NIC_BOND_MODE_MODEL = (
    "OceanStor Dorado 18800K V6",
    "OceanStor Dorado 18000 V6",
    "OceanStor Dorado 18500 V6",
    "OceanStor Dorado 18800 V6",
    "OceanStor Dorado 8000 V6",
    "OceanStor Dorado 6800 V6",
)


class CheckNicBondMode:
    """
    高端环境绑定端口检查
    """

    def __init__(
        self, context, port_list, is_ctr_enc=False, is_front_end_intf=False
    ):
        self.context = context
        self.cli = contextUtil.getCli(context)
        self.lang = contextUtil.getLang(context)
        self.logger = baseUtil.getLogger(context.get("logger"), __file__)
        self.port_list = port_list
        self.is_ctr_enc = is_ctr_enc
        self.is_front_end_intf = is_front_end_intf
        self.all_ret_list = []
        self.product_model = ""
        self.software_version = ""
        self.hot_patch_version = ""

    def execute(self):
        # 查询到的版本不在名单内检查项直接通过
        if not self.is_risk_version():
            return
        flag = self.risk_judgment()
        er_ms, sug_ms = common.getMsg(
            self.lang,
            "nic.bond.mode.sph6.error_message",
            suggestionArgs=self.software_version,
        )
        if flag is not True:
            BaseFactory.result.setResultFailByDesc(self.context, er_ms, sug_ms)
            return
        BaseFactory.result.setResultPass(self.context)

    def risk_judgment(self):
        """
        执行show bond_port获取绑定信息,如果是更换控制框那么有绑定信息就有风险
        如果是更换前端接口卡，那么需要更换的接口卡绑定有端口才不通过
        :return:
        """
        cmd = "show bond_port"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)
        member_list = cliUtil.getHorizontalCliRet(ret)
        if self.is_ctr_enc and member_list:
            return False
        if self.is_front_end_intf and member_list:
            bond_list_info = self.get_bond_port_info(member_list)
            for port_name in bond_list_info:
                if self.port_list[0] == port_name[0:len(self.port_list[0])]:
                    return False
        return True

    @staticmethod
    def get_bond_port_info(member_list):
        """
        获取每个绑定口的"name"和"Port ID List"；
        :return:
        """
        bond_list_info = []
        for line in member_list:
            bond_port_list_str = line.get("Port ID List", "")
            if bond_port_list_str == "--":
                continue
            bond_port_list = bond_port_list_str.split(",")
            if not bond_port_list:
                continue
            bond_list_info.extend(bond_port_list)
        return bond_list_info

    def is_risk_version(self):
        (
            flag,
            self.product_model,
            _,
            msg,
        ) = cliUtil.getProductModelAndVersion(self.cli, self.lang)
        self.all_ret_list.append(msg)
        if flag is not True:
            raise UnCheckException(msg)
        if self.product_model not in RISK_NIC_BOND_MODE_MODEL:
            return False
        (
            flag,
            self.software_version,
            self.hot_patch_version,
            ret,
        ) = cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.all_ret_list.append(ret)
        if self.software_version == "6.0.1":
            pattern = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
            match_hot_path = pattern.search(self.hot_patch_version)
            if not match_hot_path:
                return True
            if int(match_hot_path.group(1)) < SAFETY_HOT_PATCH:
                return True
        return False
