# -*- coding: UTF-8 -*-
import math
import os
import shutil
import time
import traceback

from java.lang import Exception as JException

from cbb.business.operate.inspection import common, config
from cbb.frame.adapter import connectionFactory
from cbb.frame.base import constants
from cbb.frame.base.exception import ErrorCode, ErrorCodeSet, TlvCmdException
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.util.GZipHandler import GZipHandler


def uploadUpgradePkg(dataDict):
    """校验阵列状态是否支持升级，将补丁包上传到整列端，并获取阵列端处理结果

    :param dataDict: 上下文字典
    :return:
    """
    logger = dataDict.get("logger")
    lang = dataDict.get("language")
    resource = dataDict.get("resource")
    connectionAdapter = connectionFactory.getAdapter(dataDict)

    if checkUploadState(dataDict) == constants.OM_MSG_OP_UPD_LST_SYS_PROGRESS.UPD_DLDING:
        return False, resource.getString("upgrade.notsupport")
    # noinspection PyBroadException
    try:

        # 将硬盘固件包打包成补丁包形式
        if contextUtil.isSupportParallel(dataDict):
            dev = contextUtil.getDevObj(dataDict)
            upgradePkgFile = dev.getUpgradeSetInfo().getUpgradePackageFile()
            hotPatchTgzPkgName = upgradePkgFile.getAbsolutePath()
        else:
            hotPatchTgzPkgName = compressPkg(dataDict)
            if '' == hotPatchTgzPkgName:
                return False, resource.getString("upload.compress.failed")

        # 上传补丁包
        upload(dataDict, hotPatchTgzPkgName)
        try:
            time.sleep(5)
        except KeyboardInterrupt:
            logger.error("Interrupted")

        # 轮询获取阵列端处理状态
        status = None
        DEVEICE_DLD_QUERY_TIME = 0  # 轮询计数器
        while constants.OM_MSG_OP_UPD_LST_SYS_PROGRESS.UPD_DLD_SUCCESS != status:
            DEVEICE_DLD_QUERY_TIME = DEVEICE_DLD_QUERY_TIME + 1
            if DEVEICE_DLD_QUERY_TIME > constants.UPLOAD_PACKAGE_POLL_TIMES:
                return False, resource.getString("upload.failed")
            # 查询处理状态
            status = connectionAdapter.queryExcuteResult()
            if status == constants.OM_MSG_OP_UPD_LST_SYS_PROGRESS.UPD_DLD_FAIL:
                errMsg = connectionAdapter.queryExcuteDetail()
                return False, errMsg
            try:
                time.sleep(constants.UPLOAD_PACKAGE_POLL_INTERVAL)
            except KeyboardInterrupt:
                logger.error("Interrupted")
    except TlvCmdException, e:
        logger.error("notify upgrade package upload over error!" + str(traceback.format_exc()))
        return False, e.getMsg(lang)
    except Exception:
        logger.error("notify upgrade package upload over error!" + str(traceback.format_exc()))
        return False, resource.getString("upload.failed")

    return True, ""


def compressPkg(dataDict):
    """将硬盘升级包打包成补丁包的形式

    :param dataDict: 上下文字典
    :return:
    """
    dev = dataDict.get("dev")
    logger = dataDict.get("logger")
    homeDir = dataDict.get("home")

    devSN = contextUtil.getDevSN(dataDict)
    upgradePkgFile = dev.getUpgradeSetInfo().getUpgradePackageFile()
    upgradePkgPath = upgradePkgFile.getAbsolutePath()
    patchConfPath = homeDir + os.sep + config.PATCH_CONF_FILE
    try:
        # 创建打包临时目录
        compressDir = os.path.join(homeDir, config.FW_TO_PACKAGE_DIR_TEMP)
        if not os.path.exists(compressDir):
            os.mkdir(compressDir)
        # 复制硬盘固件包到临时目录并重命名
        diskPkgFile = os.path.join(compressDir, config.FW_PKG_NAME)
        shutil.copyfile(upgradePkgPath, diskPkgFile)
        # 压缩固件包为补丁包形式
        fileList = [diskPkgFile, patchConfPath]
        GZipHandler.compressToTgz(fileList, compressDir, devSN)
        destFilePath = os.path.join(compressDir, devSN + config.FW_COMPRESS_EXT)
        # 删除临时目录下的固件包
        os.remove(diskPkgFile)

        logger.info('compress pkg success !')
        return destFilePath
    except (Exception, JException), exception:
        logger.error('compress pkg exception:%s' % unicode(exception))
        return ''


def upload(dataDict, hotPatchTgzPkgName):
    """上传压缩后的固件包

    :param dataDict:上下文字典
    :param hotPatchTgzPkgName: 包路径名
    :return:
    """
    logger = dataDict.get("logger")

    if not os.path.isfile(hotPatchTgzPkgName):
        logger.error('patchPath is follow:%s' % hotPatchTgzPkgName)
        raise TlvCmdException(ErrorCode(ErrorCodeSet.FILE_PATH_INVALID))

    # 校验补丁包大小是否安全
    patchSize = os.stat(hotPatchTgzPkgName).st_size

    if contextUtil.isSupportParallel(dataDict):
        pkgType = constants.TLV_PACKAGE_TYPE.PARALLEL_PKG
    else:
        pkgType = constants.TLV_PACKAGE_TYPE.HOT_PATCH_PKG

    connectionAdapter = connectionFactory.getAdapter(dataDict)
    connectionAdapter.uploadDiskFw(hotPatchTgzPkgName, pkgType, patchSize)


def checkUploadState(dataDict):
    """查询阵列状态

    :param dataDict: 上下文字典
    :return:
    """
    logger = dataDict.get("logger")
    connectionAdapter = connectionFactory.getAdapter(dataDict)
    status = connectionAdapter.queryExcuteResult()
    logger.info("upload statusIndex is :" + str(status))
    return status


def uploadWithSftp(dataDict):
    """使用是sftp上传固件包，不需要对硬盘固件包做特殊处理，支持上传到阵列指定路径

    :param dataDict: 上下文字典
    :return:
    """
    protocalContext = dataDict.get("protocalContext")
    dev = dataDict.get("dev")
    sftp = protocalContext.get("SFTP")
    logger = dataDict.get("logger")
    lang = dataDict.get("language")

    try:
        devSN = contextUtil.getDevSN(dataDict)
        upgradePkgPath = dev.getUpgradeSetInfo().getUpgradePackageFile()
        absolutePath = upgradePkgPath.getAbsolutePath()
        _, ext = common.getFileNameAndExt(absolutePath)
        serverPath = config.SERVER_UPGRADE_PKG_DIR + devSN + ext
        logger.info("upgradePkgPath :%s " % serverPath)

        # 判断系统空闲空间是否满足要求
        checkFlag, errMsg = checkMemEnough(dataDict, absolutePath)
        if not checkFlag:
            return False, errMsg

        user = dev.getDevNode().getLoginUser()
        devIp = dev.getDevNode().getIp()
        sftp.putFile(devIp, absolutePath, serverPath, user, None)
        dataDict['path'] = serverPath
        logger.info('upload file success for dev(%s)!' % devIp)

        return True, errMsg
    except (Exception, JException), exception:
        logger.error('upload file occured except:' + unicode(exception))
        errMsg = common.getMsg(lang, "upload.pkg.failed")
        return False, errMsg


def getPkgSize(filePath):
    """获取文件大小，单位MB

    :param filePath: 文件路径
    :return:
    """
    fsize = os.path.getsize(filePath)
    # 转换为MB
    fsize = fsize / float(1024 * 1024)
    return fsize


def getSystemFreeMem(dataDict):
    """系统空闲空间大小，单位MB

    :param dataDict: 上下文字典
    :return:
    """
    lang = dataDict.get("language")
    logger = dataDict.get("logger")
    cli = contextUtil.getCli(dataDict)
    cmd = "free -m"
    freeMem = 0
    execRets = cliUtil.excuteCmdInMinisystemModel(cli, cmd, lang)
    cliRet = execRets[1]
    if not execRets[0]:
        logger.info("[uploadUtil] get free mem error:%s" % str(execRets[2]))
        return False, freeMem, common.getMsg(lang, "cannot.get.system.info")
    flag, freeMem = cliUtil.getFreeMemInCliRet(cliRet)
    if not flag:
        logger.info("[uploadUtil] format free mem error")
        return False, freeMem, common.getMsg(lang, "cannot.get.system.info")
    logger.info("[uploadUtil]free mem:%s ." % str(freeMem))
    return True, freeMem, ""


def checkMemEnough(dataDict, absolutePath):
    """查询阵列空闲内存

    :param dataDict: 上下文字典
    :param absolutePath: 绝对路径
    :return:
    """
    logger = dataDict.get("logger")
    lang = dataDict.get("language")

    pkgSize = getPkgSize(absolutePath)
    logger.info("disk fw package size :%s " % pkgSize)
    flag, freeMem, errMsg = getSystemFreeMem(dataDict)
    if not flag:
        return False, errMsg

    freeMem = float(freeMem)
    # 空闲空间 > 包大小 + 80MB
    if pkgSize + config.MEM_SIZE > freeMem:
        errMsg = common.getMsg(lang, "upload.pkg.no.free.mem", math.ceil(pkgSize + 80))
        return False, errMsg
    return True, ""
