# coding:utf-8
from cbb.business.operate.fru.common import BaseFactory
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil

import com.huawei.ism.tool.obase.exception.ToolException as ToolException
import time
import json
import traceback


RESULT_PASS = "pass"
RESULT_NOT_PASS = "not_pass"
RESULT_NOT_CHECK = "not check"
# 组件类型
OBJ_TYPE_LIST = [4, 6, 17, 21]


def check_compatibility_response(records, location):
    """检查是否兼容，compatibilityVersion为1，则兼容

    :param records: 回显
    :param location: location
    :return: “pass”: 兼容，“not_pass”：不兼容，“not_check”：未查到
    """
    for record in records.get("data", []):
        if record.get("objName", '') == location:
            return RESULT_PASS if record.get("compatibilityVersion") == '1' \
                else RESULT_NOT_PASS
    return RESULT_NOT_CHECK


def is_right_response(record):
    """校验响应数据是否正确

    :param record:
    :return:
    """
    err_info = restUtil.CommonRest.getErrInfo(record)
    if not err_info:
        return False
    err_code = restUtil.CommonRest.getRecordValue(
        err_info, restData.ErrorInfo.CODE
    )
    # 不支持该命令, 直接返回通过
    if str(err_code) == '-1':
        return True
    # 如果命令执行失败
    if str(err_code) != '0':
        return False
    return True


class InterfaceCardCompatibility(object):
    def __init__(self, context):
        self.context = context

    def is_compatibility(self, check_param):
        """
        校验接口卡是否兼容，入口;
        负责选择轮询还是不轮询的方式进行查询校验
        :param check_param: 校验所需参数,CompatibilityCheckParam
        """
        if check_param.is_only_check_once:
            return self.is_interface_compatibility_sas_and_high(
                check_param.location)
        else:
            return self.is_interface_compatibility_polling(check_param)

    def is_interface_compatibility_check_once(self, location, obj_type):
        """检查接口卡是否兼容,不轮询

        :param location: location
        :param obj_type: 组件类型
        :return "pass": 检查通过, "not_pass"：检查不通过, "not_check"：未查到结果
        """
        # 1:控制框;2:bios内核;3:管理板;4:sas卡;5:sas硬盘框;6:高端1822卡;
        # 7:高端1822前端卡;8:高端1822共享前端卡;9:高端1822共享eth卡;
        # 10:高端1822后端卡;11:高端1822交换卡;12:管理板;13:mcu;14:电源;15:bbu;
        # 16:io 组件;17:高端1822前端容器卡
        try:
            param_dict = {"objType": obj_type}
            params_str = restUtil.CommonRest.getParamsJsonStr(
                param_dict, False)
            cmd = "get_compatibility/check_compatibility"
            rest = contextUtil.getRest(self.context)
            base_uri = rest.getBaseUri()
            rest_connection = rest.getRest()
            response_info = rest_connection.execGet(
                "{}{}".format(base_uri, cmd), params_str
            )
            data_string = response_info.getContent()
            records = json.loads(data_string)
            if not is_right_response(records):
                BaseFactory.log.info(self.context, "error response")
                return RESULT_NOT_PASS
            return check_compatibility_response(records, location)
        except (Exception, ToolException):
            BaseFactory.log.error(self.context, traceback.format_exc())
            return RESULT_NOT_PASS

    def is_interface_compatibility_sas_and_high(self, location):
        """检查sas卡或1822高端口是否兼容

        :param location: location
        :return 是否兼容
        """
        # 1、卡件查不到，则查Mcu是否通过，Mcu的枚举是13
        # 2、卡件能查到，则按原逻辑（卡件是否通过）
        for obj_type in OBJ_TYPE_LIST:
            check_result = self.is_interface_compatibility_check_once(
                location, obj_type)
            if check_result == RESULT_NOT_CHECK:
                continue
            return check_result == RESULT_PASS
        return self.is_interface_compatibility_check_once(location, 13) == \
            RESULT_PASS

    def is_interface_compatibility_polling(self, check_param):
        """轮询，是否兼容

        :param check_param: 调用参数CompatibilityCheckParam
        :return: True 兼容， False 不兼容
        """
        time_out = check_param.time_out
        time_begin = time.time()
        while time.time() - time_begin < time_out:
            if self.is_interface_compatibility_sas_and_high(
                    check_param.location):
                return True
            BaseFactory.log.info(self.context,
                                 "compatibility check fail, more.")
        BaseFactory.log.info(self.context, "compatibility check timeout.")
        return False


class CompatibilityCheckParam(object):
    """兼容性接口调用所需参数
    is_only_check_once: 表示是否只调用一次该接口
    location: 表示fru的location
    time_out: 轮询调用接口的超时时间
    """
    def __init__(self, is_only_check_once, location, time_out):
        self.is_only_check_once = is_only_check_once
        self.location = location
        self.time_out = time_out
