# -*- coding: UTF-8 -*-
import traceback
import functools

from cbb.frame.base import baseUtil
from cbb.frame.base.exception import CliCmdException, ErrorCode, ErrorCodeSet
from cbb.frame.cli import cliUtil
from cbb.frame.cli.cliFactory import CliConnection
from cbb.frame.context import contextUtil
from java.lang import Exception as JException


# 捕获异常-->重试（闪断）-->重连（连接失效）-->更换IP重连（控制器被重启）
# 重启控制器后，连接被断开，如果没有备用IP，则工具无法与阵列建立连接，工具只有等待阵列起来后重新建连接。
# 上电检查有可能处于无法连接状态，等待5分钟

def checkCliResult(cliResult):
    """将错误转为异常
        cliUtil.RESULT_NOSUPPORT-->no license异常
        cliUtil.RESULT_NOCHECK-->no support异常(当前模式不支持该命令)
        False-->cli命令执行不符合预期异常(为空，在黑名单之中)
        其他-->正常返回

    :param cliResult: cli结果
    :return:
    """
    status = cliResult[0]
    cliRet = cliResult[1]
    if status == cliUtil.RESULT_NOSUPPORT:
        raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_NO_LICENSE))
    if status == cliUtil.RESULT_NOCHECK:
        raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_NO_SUPPORT_CMD))
    if status is False:
        raise CliCmdException(ErrorCode(
            ErrorCodeSet.CLI_EXECUTE_CMD_FAILED), cliRet)
    if "internal error" in cliRet.lower():
        raise CliCmdException(ErrorCode(
            ErrorCodeSet.CLI_EXECUTE_INTERNAL_ERROR), cliRet)
    return cliResult[1]


def handException(retry=False, reCon=False, toMaster=False):
    """
    处理执行命令过程中的异常
    :param retry: 是否异常后重试
    :param reCon: 是否异常后重建连接
    :param toMaster: 是否异常后切换到主控节点
    :return: 执行命令的结果
    """

    def wrapper(func):
        @functools.wraps(func)
        def oneHand(*params):
            cliService = params[0]
            cliCon = cliService.cliCon
            logger = cliService.logger

            # 看是否需要连接主控
            if toMaster:
                logger.info("begin to switch to master.")
                logger.info("succeed to switch to master.")
                # 无法连接到主控，直接抛出异常，可以连接主控，重新执行下命令
                # 检查连接是否可用，如果不可用
                result = func(*params)
                return result

            # 不需要切主控执行
            # noinspection PyBroadException
            try:
                result = func(*params)
                logger.info("wrapper result succeed.")
                return result
            except Exception:
                logger.error("exception in wrapper in first.")
                try:
                    # 先重连排除是否连接失效、模式不正确的问题，看能否解决
                    cliCon.close()
                    cliCon.reConnect()

                    result = func(*params)
                    return result
                except (Exception, JException) as ex:
                    logger.error(
                        "exception in wrapper after reconnect:%s" % str(ex))
                    # 无法建立连接，切换IP后重试一次
                    if reCon:
                        logger.info("begin to switch available ip.")
                        # 无法切换到可用IP，直接抛出异常，切换到可用IP后，重新执行下命令
                        logger.info("cliCon in oneHandle before=%s"
                                    % cliService.cliCon)
                        logger.info("cliCon in oneHandle after=%s"
                                    % cliService.cliCon)

                        logger.info("begin to retry cmd use available ip.")
                        result = func(*params)
                        return result

                    raise

        def inner(*params):
            # 根据参数设置重试次数
            retryTimes = 1
            if retry:
                retryTimes = 3
            while retryTimes > 0:
                # noinspection PyBroadException
                try:
                    return oneHand(*params)
                except CliCmdException:
                    retryTimes -= 1
                    if retryTimes == 0:
                        cliService = params[0]
                        logger = cliService.logger
                        logger.info("handle CliCmdException inner")
                        raise
                except Exception:
                    retryTimes -= 1
                    if retryTimes == 0:
                        cliService = params[0]
                        logger = cliService.logger
                        logger.error("handle exception inner")
                        raise CliCmdException(
                            ErrorCode(ErrorCodeSet.CLI_CONNECTION_EXCEPTION))

        return inner

    return wrapper


class CliService:
    def __init__(self, dataDict):
        self.dataDict = dataDict
        # noinspection PyBroadException
        try:
            self.cliCon = contextUtil.getCli(dataDict)
        except Exception:
            self.cliCon = None
        self.lang = contextUtil.getLang(dataDict)
        self.logger = dataDict.get("logger")
        self.lastMasterCtrl = ""

    def updateCliCon(self):
        """ 更新CLI连接 """
        try:
            self.cliCon = contextUtil.getCli(self.dataDict)
        except Exception as e:
            self.logger.error(
                "[cliService]update cli connection error:%s" % unicode(e))
            self.cliCon = None

    def checkCliResult(self, cliResult):
        """本方法的职责：将错误转为异常
            cliUtil.RESULT_NOSUPPORT-->no license异常
            cliUtil.RESULT_NOCHECK-->no support异常(当前模式不支持该命令)
            False-->cli命令执行不符合预期异常(为空，在黑名单之中)
            其他-->正常返回

        :param cliResult: cli结果
        :return:
        """
        status = cliResult[0]
        cliRet = cliResult[1]
        if status == cliUtil.RESULT_NOSUPPORT:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_NO_LICENSE))
        if status == cliUtil.RESULT_NOCHECK:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_NO_SUPPORT_CMD))

        if "TOOLKIT_SEND_CMD_TIME_OUT" in cliRet \
                or "messages timed out" in cliRet:
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_EXECUTE_CMD_TIMEOUT))

        inWait, errMsg = cliUtil.checkLineInWaiteWhiteList(cliRet, self.lang)
        if inWait:
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_NO_AVAILABLE_MODE))

        if status is False:
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_EXECUTE_CMD_FAILED), cliRet)
        if "internal error" in cliRet.lower():
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_EXECUTE_INTERNAL_ERROR), cliRet)
        return cliResult[1]

    @handException(retry=False, reCon=True, toMaster=False)
    def showControllers(self):
        """ 查询控制器信息 """
        cmd = "show controller general"
        cliResult = cliUtil.excuteCmdInCliMode(
            self.cliCon, cmd, True, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getVerticalCliRet(cliRet)
        return datas

    @handException(retry=False, reCon=True, toMaster=False)
    def showSysStatus(self):
        """
        查询系统状态
        @:returns: datas：解析后的回显；localNodeId：节点ID
        """
        cmd = "showsysstatus"
        cliResult = cliUtil.excuteCmdInMinisystemModel(
            self.cliCon, cmd, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getHorizontalCliRet(cliRet)
        verticalDatas = cliUtil.getVerticalCliRet(cliRet)
        localNodeId = ""
        if len(verticalDatas) > 0:
            localNodeId = verticalDatas[0].get("local node id", "")
        self.logger.info("showSysStatus datas=%s, "
                         "localNodeId=%s" % (datas, localNodeId))
        return datas, localNodeId

    def showSysStatusVertical(self):
        """
        查询系统状态
        @:return: datas：解析后的回显
        """
        cmd = "showsysstatus"
        cliResult = cliUtil.excuteCmdInMinisystemModel(
            self.cliCon, cmd, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getVerticalCliRet(cliRet)
        self.logger.info("showSysStatus datas=%s" % datas)
        return datas

    @handException(retry=False, reCon=True, toMaster=False)
    def existAlarmById(self, alarmId):
        """
        查询特定ID的告警
        :param alarmId: 告警ID
        :return: True：存在该告警；False：不存在该告警
        """
        cmd = "show alarm |filterRow column=ID predict=equal_to " \
              "value=%s" % alarmId
        cliResult = cliUtil.excuteCmdInCliMode(self.cliCon,
                                               cmd, True, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getHorizontalCliRet(cliRet)
        if cliUtil.queryResultWithNoRecord(cliRet):
            return True
        if len(datas) == 0:
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_EXECUTE_ECHO_ERROR))
        return False

    # 这里要加异常处理，是因为1批次里面有多个节点，如果工具连接的节点（或者心跳的目标节点）被重启了，则无法重启其他节点
    @handException(retry=False, reCon=True, toMaster=False)
    def rebootController(self, controllId):
        """重启控制器。如果无法连接，尝试切换控制器

        :param controllId: 控制器ID
        :return: True：重启成功；
        """
        try:
            self.logger.info("rebootController self.cliCon=%s" % self.cliCon)
            cmd = "reboot controller controller=%s" % controllId
            cliResult = cliUtil.excuteCmdInDeveloperMode(
                self.cliCon, cmd, True, self.lang)
            cliRet = self.checkCliResult(cliResult)

            # 需要确认直接输入y
            cnt = 1
            while "y/n" in cliRet and cnt <= 3:
                cliRet = self.cliCon.execCmdWithTimout("y", 2 * 60)
                cnt += 1

            if "Password:" in cliRet:
                cliRet = self.cliCon.execCmdNoLogTimout(
                    self.dataDict.get("dev").get("pawd"), 2 * 60)

            self.logger.info("rebootController cli ret=%s" % str(cliRet))
            if cliUtil.queryResultWithNoRecord(cliRet):
                return True

            # 命令执行y或密码输入超时也认为执行成功
            if "TOOLKIT_SEND_CMD_TIME_OUT" in cliRet or \
                    "messages timed out" in cliRet:
                raise CliCmdException(ErrorCode(
                    ErrorCodeSet.CLI_EXECUTE_CMD_TIMEOUT))

        except CliCmdException, cliEx:
            self.logger.error(
                "[RebootController]execute reboot cmd CliCmdException, "
                "exception:%s" % traceback.format_exc())
            errorCode = cliEx.getId()
            # 执行命令超时则查看连接是否存在，无法连接则认为重启成功
            if errorCode not in [ErrorCodeSet.CLI_EXECUTE_CMD_TIMEOUT]:
                raise cliEx
            else:
                baseUtil.safeSleep(2 * 60)
        # noinspection PyBroadException
        try:
            # 检查连接，如果连接不可用，认为系统被重启，无法执行命令
            conn = contextUtil.getItem(self.dataDict,
                                       contextUtil.CLI_CONNECTION)
            conn.checkConnect()
            ctrlIds = self.showControllerId()
            if controllId not in ctrlIds:
                return True

        except Exception:
            self.logger.error("input y exception.the system is reboot.")
            contextUtil.destroyCliConnection(self.dataDict)
            return True

        raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_ECHO_ERROR))

    @handException(retry=False, reCon=True, toMaster=True)
    def checkPowerOnByShowTrace(self):
        """ V3R3之前版本, sys showtrace只有在主控才能执行，如果本控无法执行，则登陆到主控去执行
        :return: 上电命令的回显
        """
        cmd = "sys showtrace"
        cliResult = cliUtil.executeCmdInDebugMode(
            self.cliCon, cmd, True, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getHorizontalCliRet(cliRet)
        return datas

    @handException(retry=False, reCon=True, toMaster=False)
    def checkPowerOnByShowSystemTrace(self):
        """ V3R3及之后版本，如果无法连接，尝试切换控制器

        :return: 上电命令的回显
        """
        cmd = "show system trace"
        cliResult = cliUtil.excuteCmdInDeveloperMode(
            self.cliCon, cmd, True, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getHorizontalCliRet(cliRet)
        return datas

    @handException(retry=False, reCon=True, toMaster=False)
    def checkSystemStatusByShowobjstatus(self):
        """V3R3之前版本

        :return: 系统状态的回显
        """
        cmd = "sys showobjstatus"
        cliResult = cliUtil.executeCmdInDebugMode(
            self.cliCon, cmd, True, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getVerticalCliRet(cliRet)
        return datas

    @handException(retry=False, reCon=True, toMaster=False)
    def checkSystemStatusByShowvnode(self):
        """V3R3之后版本

        :return: 系统状态的回显
        """
        cmd = "sys.sh showvnode"
        cliResult = cliUtil.excuteCmdInMinisystemModel(
            self.cliCon, cmd, self.lang)
        cliRet = self.checkCliResult(cliResult)
        datas = cliUtil.getHorizontalCliRet(cliRet)
        return datas

    def createCliConnection(self, ip=None):
        """初始化cli连接

        :param ip: ip
        :return: cli连接
        """

        connectorFactoryObj = contextUtil.getConnectorFactoryObj(self.dataDict)
        devObj = contextUtil.getDevObj(self.dataDict)

        if ip is None:
            ip = devObj.get("ip")
        sshPort = int(devObj.get("sshPort"))
        conn = CliConnection(connectorFactoryObj, ip, sshPort)
        conn.close()
        contextUtil.setItem(self.dataDict, contextUtil.CLI_CONNECTION, conn)
        cliCon = conn.getCli(devObj.get("user"), devObj.get("pawd"))
        self.logger.info("cliCon in createcliConnection=%s" % cliCon)
        self.cliCon = cliCon
        return cliCon

    @handException(retry=False, reCon=False, toMaster=False)
    def isSingleModel(self):
        """ 控制器模式检查：
              执行show system config_model命令，查看返回结果：
              Configuration Model为Multi-Controller时多多控，
              为Single-Controller时为单控
        :return: True：单控；False：多控
        """
        cmd = "show system config_model"
        checkRet = cliUtil.excuteCmdInCliMode(self.cliCon,
                                              cmd, True, self.lang)
        if "single" in checkRet[1].lower():
            return True
        else:
            return False

    def executeHorizontalCliCmd(self, cmd, isHasLog):
        """执行水平回显的命令

        :param cmd: 执行的命令
        :param isHasLog: 是否记录日志
        :return: cli回显信息
        """
        ret = cliUtil.excuteCmdInCliMode(self.cliCon, cmd, isHasLog, self.lang)
        cliRet = self.checkCliResult(ret)
        infoList = cliUtil.getHorizontalCliRet(cliRet)
        return infoList

    def executeHorizontalDevloperCmd(self, cmd, isHasLog):
        """developer下执行水平回显的命令

        :param cmd: 执行的命令
        :param isHasLog: 是否记录日志
        :return: cli 回显信息
        """
        ret = cliUtil.excuteCmdInDeveloperMode(self.cliCon,
                                               cmd, isHasLog, self.lang)
        cliRet = self.checkCliResult(ret)
        infoList = cliUtil.getHorizontalCliRet(cliRet)
        return infoList

    def executeVerticalDebugCmd(self, cmd, isHasLog):
        """ debug模式下执行水平回显的命令

        :param cmd:执行的命令
        :param isHasLog:是否记录日志
        :return:cli 回显信息
        """
        ret = cliUtil.executeCmdInDebugMode(self.cliCon,
                                            cmd, isHasLog, self.lang)
        cliRet = self.checkCliResult(ret)
        infoList = cliUtil.getVerticalCliRet(cliRet)
        return infoList

    def executeVerticalCliCmd(self, cmd, isHasLog):
        """ 执行水平回显的命令

        :param cmd:执行的命令
        :param isHasLog:是否记录日志
        :return:cli 回显信息
        """
        ret = cliUtil.excuteCmdInCliMode(self.cliCon, cmd, isHasLog, self.lang)
        cliRet = self.checkCliResult(ret)
        infoList = cliUtil.getVerticalCliRet(cliRet)
        return infoList

    def showAlarms(self):
        """获取告警的Sequence,ID,Level

        :return: cli 回显信息
        """
        cmd = "show alarm|filterColumn include columnList=Sequence,ID,Level"
        return self.executeHorizontalCliCmd(cmd, False)

    def showMgmtPortIp(self):
        """ 获取控制器的管理ip

        :return:cli 回显信息
        """
        cmd = "show port general logic_type=Management_Port physical_type=ETH"
        return self.executeHorizontalCliCmd(cmd, True)

    def showFcPort(self):
        """获取FC端口

        :return:cli 回显信息
        """
        cmd = "show port general physical_type=FC"
        return self.executeHorizontalCliCmd(cmd, True)

    def showFCoEPort(self):
        """获取FCoe端口

        :return:cli 回显信息
        """
        cmd = "show port general physical_type=FCoE"
        return self.executeHorizontalCliCmd(cmd, True)

    def showETHPort(self):
        """获取ETH端口

        :return:cli 回显信息
        """
        cmd = "show port general physical_type=ETH"
        return self.executeHorizontalCliCmd(cmd, True)

    @handException(retry=False, reCon=False, toMaster=False)
    def showSystemGeneral(self):
        """获取系统信息

        :return:cli 回显信息
        """
        cmd = "show system general"
        return self.executeVerticalCliCmd(cmd, True)

    def showIBPort(self):
        """获取IB端口

        :return:cli 回显信息
        """
        try:
            cmd = "show ib_port general"
            return self.executeHorizontalCliCmd(cmd, True)
        except CliCmdException, e:
            if e.getId() == ErrorCodeSet.CLI_NO_SUPPORT_CMD:
                return []
            raise

    def showFCInitiator(self):
        """
        查询FC启动器信息
        :return: 解析后的cli回显
        """
        cmd = "show initiator initiator_type=FC"
        return self.executeHorizontalCliCmd(cmd, True)

    def showiScsiInitiator(self):
        """
        查询iSCSI启动器信息
        :return: 解析后的CLI回显
        """
        cmd = "show initiator initiator_type=iSCSI"
        return self.executeHorizontalCliCmd(cmd, True)

    def showIBInitiator(self):
        """
        查询ib启动器信息
        :return: 解析后的cli回显
        """
        try:
            cmd = "show ib_initiator general"
            return self.executeHorizontalCliCmd(cmd, True)
        except CliCmdException, e:
            if e.getId() == ErrorCodeSet.CLI_NO_SUPPORT_CMD:
                return []
            raise

    @handException(retry=False, reCon=True, toMaster=False)
    def getMasterNodeReCon(self):
        """
        查询主控节点信息，并处理异常
        :return: 控制器ID和位置信息；如：{'0A':'CTE0.A'}
        """
        return self.getMasterNode()

    def getMasterNode(self):
        """
        查询主控节点信息
        :return: 控制器ID和位置信息；如：{'0A':'CTE0.A'}
        """
        cmd = "show controller general"
        ctrlInfos = self.executeVerticalCliCmd(cmd, True)
        for ctrl in ctrlInfos:
            ctrlId = ctrl.get("Controller")
            ctrlLocation = ctrl.get("Location")
            role = ctrl.get("Role")
            if role.upper() == "MASTER":
                return dict(Controller=ctrlId, Location=ctrlLocation)
        return None

    @handException(retry=False, reCon=True, toMaster=False)
    def getCtrlIpDictReCon(self):
        """
        查询控制器的IP信息，并处理异常
        :return: 控制器和IP信息，如{'0A':'10.23.21.11'}
        """
        return self.getCtrlIpDict()

    def getCtrlIpDict(self):
        """
        查询控制器的IP信息
        :return: 控制器和IP信息，如{'0A':'10.23.21.11'}
        """
        cmd = "show upgrade package"
        checkRet = cliUtil.excuteCmdInCliMode(self.cliCon,
                                              cmd, True, self.lang)
        if checkRet[0] is not True:
            return None

        cliRet = checkRet[1]
        beginIndex = cliRet.find('HotPatch Version')
        if beginIndex == -1:
            return None

        ipInfo = {}
        cliRetDictList = cliUtil.getHorizontalCliRet(cliRet[:beginIndex])
        for info in cliRetDictList:
            ctrlId = info.get('Name')
            ip = info.get('IP')
            ipInfo[ctrlId] = ip
        return ipInfo

    def sshtoRemoteExt(self, ctrlId):
        """
        心跳到指定控制器
        :param ctrlId: 控制器id，如0A
        :return: cli 回显信息
        """
        ctrlId = ctrlId.strip().upper()[-1]
        ctrlIdDict = dict(A=0, B=1, C=2, D=3)
        nodeId = ctrlIdDict.get(ctrlId)
        devNode = contextUtil.getDevObj(self.dataDict)
        cmd = "sshtoremoteExt %s" % nodeId
        return cliUtil.sshToRemoteContr(self.cliCon,
                                        cmd, devNode.get("pawd"), self.lang)

    def sshtoRemote(self):
        """
        心跳到对端控制器
        :return: cli 回显信息
        """
        devNode = contextUtil.getDevObj(self.dataDict)
        cmd = "sshtoremote"
        return cliUtil.sshToRemoteContr(self.cliCon,
                                        cmd, devNode.get("pawd"), self.lang)

    def showControllerId(self):
        """
        查询控制器节点ID
        :return: 控制器节点ID
        """
        ctrls = []
        cmd = "show controller general|filterColumn include " \
              "columnList=Controller"
        infoList = self.executeVerticalCliCmd(cmd, True)
        for info in infoList:
            ctrls.append(info.get("Controller"))
        ctrls.sort()
        return ctrls

    def showEnclosure(self):
        """
        执行查询控制器命令
        :return: cli回显信息
        """
        cmd = "show enclosure"
        return self.executeHorizontalCliCmd(cmd, True)

    def showInterIp(self):
        """
        查询系统内部IP
        :return: cli 回显信息
        """
        cmd = "show system inter_ip"
        return self.executeHorizontalDevloperCmd(cmd, True)

    def getEngIds(self):
        """
        查询引擎ID
        :return: 引擎ID列表
        """
        engList = []
        infoList = self.showEnclosure()
        self.logger.info("infoList=%s" % infoList)
        for info in infoList:
            encType = info.get("Logic Type")
            if encType == "Engine":
                encId = info.get("ID")
                engList.append(encId)
                self.logger.info("encId=%s" % encId)
        return engList

    def showEnclosureById(self, enclosureId):
        """
        查询指定控制器的详细信息
        :param enclosureId: 控制器ID
        :return: 解析后的cli 回显信息
        """
        cmd = "show enclosure enclosure_id=%s" % enclosureId
        return self.executeVerticalCliCmd(cmd, True)

    def getEnclosureHeight(self, enclosureId):
        """
        查询控制框高度
        :param enclosureId: 控制器ID
        :return:  控制器高度
        """
        height = None
        infoList = self.showEnclosureById(enclosureId)
        for info in infoList:
            if info is None or len(info) == 0:
                continue
            height = int(info.get("Height(U)"))
            break
        return height

    def showSysCls(self):
        """
        执行‘sys showcls’命令
        :return: 解析后的cli回显
        """
        cmd = "sys showcls"
        return self.executeVerticalDebugCmd(cmd, True)

    def exitMinisystemToCliMode(self):
        """
        退出小系统模式到正常模式
        :return: None
        """
        cliUtil.exitMinisystemToCliMode(self.cliCon)
        return

    def enterCliModeFromSomeModel(self):
        """
        从其他模式进入正常CLI模式
        :return: None
        """
        # noinspection PyBroadException
        try:
            # 退出到正常模式
            cliUtil.enterCliModeFromSomeModel(self.cliCon, self.lang)
        except Exception:
            self.logger.error("enter cli mode from minisystem caught error")
