# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import config
from cbb.business.operate.expansion.enclosureCommon import ERR_SAS_PORTS_FOR_DC

from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil
from cbb.common.dev.utils import skipPowerCableCheck

# 管理口更新linkup/linkdown例测时间（继承V1）
MGMT_PORT_ROUTINE_TIME = 50
MGMT_PORT_ROUTINE_INTERVAL = 5
# SAS端口更新linkup/linkdown例测时间
SAS_PORT_ROUTINE_TIME = 15
# PCIe端口更新linkup/linkdown例测时间
PCIE_PORT_ROUTINE_TIME = 20

MGMT_INTF_MDL_CONFIG = {
    # 1引擎场景：
    "1": [
        "CTE0.SMM0",
        "CTE0.SMM1",
    ],
    # 2引擎场景：
    "2": [
        "CTE0.SMM0",
        "CTE0.SMM1",

        "CTE1.SMM0",
        "CTE1.SMM1",
    ],
    # 3引擎场景：
    "3": [
        "CTE0.SMM0",
        "CTE0.SMM1",

        "CTE1.SMM0",
        "CTE1.SMM1",

        "CTE2.SMM0",
        "CTE2.SMM1",
    ],
    # 4引擎场景：
    "4": [
        "CTE0.SMM0",
        "CTE0.SMM1",

        "CTE1.SMM0",
        "CTE1.SMM1",

        "CTE2.SMM0",
        "CTE2.SMM1",

        "CTE3.SMM0",
        "CTE3.SMM1",
    ]
}

PCIE_INTF_MDL_CONFIG = {
    # 2个节点：
    "2": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
    ],
    # 4个节点：
    "4": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",
    ],
    # 6个节点：
    "6": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
    ],
    # 8个节点：
    "8": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
        "CTE1.L3.IOM1",
        "CTE1.R3.IOM1",
    ],
    # 10个节点：
    "10": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
        "CTE1.L3.IOM1",
        "CTE1.R3.IOM1",

        "CTE2.L3.IOM0",
        "CTE2.R3.IOM0",
    ],
    # 12个节点：
    "12": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
        "CTE1.L3.IOM1",
        "CTE1.R3.IOM1",

        "CTE2.L3.IOM0",
        "CTE2.R3.IOM0",
        "CTE2.L3.IOM1",
        "CTE2.R3.IOM1",
    ],
    # 14个节点：
    "14": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
        "CTE1.L3.IOM1",
        "CTE1.R3.IOM1",

        "CTE2.L3.IOM0",
        "CTE2.R3.IOM0",
        "CTE2.L3.IOM1",
        "CTE2.R3.IOM1",

        "CTE3.L3.IOM0",
        "CTE3.R3.IOM0",
    ],
    # 16个节点：
    "16": [
        "CTE0.L3.IOM0",
        "CTE0.R3.IOM0",
        "CTE0.L3.IOM1",
        "CTE0.R3.IOM1",

        "CTE1.L3.IOM0",
        "CTE1.R3.IOM0",
        "CTE1.L3.IOM1",
        "CTE1.R3.IOM1",

        "CTE2.L3.IOM0",
        "CTE2.R3.IOM0",
        "CTE2.L3.IOM1",
        "CTE2.R3.IOM1",

        "CTE3.L3.IOM0",
        "CTE3.R3.IOM0",
        "CTE3.L3.IOM1",
        "CTE3.R3.IOM1",
    ]
}

PCIE_PORT_CONFIG = {
    # 2个节点：
    "2": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
    ],
    # 4个节点：
    "4": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",
    ],
    # 6个节点：
    "6": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
    ],
    # 8个节点：
    "8": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
        "CTE1.L3.IOM1.P0",
        "CTE1.L3.IOM1.P1",
        "CTE1.R3.IOM1.P0",
        "CTE1.R3.IOM1.P1",
    ],
    # 10个节点：
    "10": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
        "CTE1.L3.IOM1.P0",
        "CTE1.L3.IOM1.P1",
        "CTE1.R3.IOM1.P0",
        "CTE1.R3.IOM1.P1",

        "CTE2.L3.IOM0.P0",
        "CTE2.L3.IOM0.P1",
        "CTE2.R3.IOM0.P0",
        "CTE2.R3.IOM0.P1",
    ],
    # 12个节点：
    "12": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
        "CTE1.L3.IOM1.P0",
        "CTE1.L3.IOM1.P1",
        "CTE1.R3.IOM1.P0",
        "CTE1.R3.IOM1.P1",

        "CTE2.L3.IOM0.P0",
        "CTE2.L3.IOM0.P1",
        "CTE2.R3.IOM0.P0",
        "CTE2.R3.IOM0.P1",
        "CTE2.L3.IOM1.P0",
        "CTE2.L3.IOM1.P1",
        "CTE2.R3.IOM1.P0",
        "CTE2.R3.IOM1.P1",
    ],
    # 14个节点：
    "14": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
        "CTE1.L3.IOM1.P0",
        "CTE1.L3.IOM1.P1",
        "CTE1.R3.IOM1.P0",
        "CTE1.R3.IOM1.P1",

        "CTE2.L3.IOM0.P0",
        "CTE2.L3.IOM0.P1",
        "CTE2.R3.IOM0.P0",
        "CTE2.R3.IOM0.P1",
        "CTE2.L3.IOM1.P0",
        "CTE2.L3.IOM1.P1",
        "CTE2.R3.IOM1.P0",
        "CTE2.R3.IOM1.P1",

        "CTE3.L3.IOM0.P0",
        "CTE3.L3.IOM0.P1",
        "CTE3.R3.IOM0.P0",
        "CTE3.R3.IOM0.P1",
    ],
    # 16个节点：
    "16": [
        "CTE0.L3.IOM0.P0",
        "CTE0.L3.IOM0.P1",
        "CTE0.R3.IOM0.P0",
        "CTE0.R3.IOM0.P1",
        "CTE0.L3.IOM1.P0",
        "CTE0.L3.IOM1.P1",
        "CTE0.R3.IOM1.P0",
        "CTE0.R3.IOM1.P1",

        "CTE1.L3.IOM0.P0",
        "CTE1.L3.IOM0.P1",
        "CTE1.R3.IOM0.P0",
        "CTE1.R3.IOM0.P1",
        "CTE1.L3.IOM1.P0",
        "CTE1.L3.IOM1.P1",
        "CTE1.R3.IOM1.P0",
        "CTE1.R3.IOM1.P1",

        "CTE2.L3.IOM0.P0",
        "CTE2.L3.IOM0.P1",
        "CTE2.R3.IOM0.P0",
        "CTE2.R3.IOM0.P1",
        "CTE2.L3.IOM1.P0",
        "CTE2.L3.IOM1.P1",
        "CTE2.R3.IOM1.P0",
        "CTE2.R3.IOM1.P1",

        "CTE3.L3.IOM0.P0",
        "CTE3.L3.IOM0.P1",
        "CTE3.R3.IOM0.P0",
        "CTE3.R3.IOM0.P1",
        "CTE3.L3.IOM1.P0",
        "CTE3.L3.IOM1.P1",
        "CTE3.R3.IOM1.P0",
        "CTE3.R3.IOM1.P1",
    ]
}

MGMT_PORT_CONFIG_FOR_DORADO = {

    "2U": {
        "2": ["CTE0.A.MAINTENANCE",
              "CTE0.B.MGMT", "CTE0.B.MAINTENANCE",
              "DSW0.MGMT.0", "DSW0.MGMT.1", "DSW1.MGMT.0", "DSW1.MGMT.1",
              "CTE1.A.MGMT", "CTE1.A.MAINTENANCE", "CTE1.B.MAINTENANCE"],
        "3": ["CTE0.A.MAINTENANCE",
              "CTE0.B.MGMT", "CTE0.B.MAINTENANCE",
              "CTE1.A.MGMT", "CTE1.A.MAINTENANCE",
              "CTE1.B.MGMT", "CTE1.B.MAINTENANCE",
              "CTE2.A.MGMT", "CTE2.A.MAINTENANCE",
              "CTE2.B.MGMT", "CTE2.B.MAINTENANCE",
              "DSW0.MGMT.0", "DSW0.MGMT.1", "DSW1.MGMT.0", "DSW1.MGMT.1"],
    },
    "2U_NVMe": {"2": ["CTE0.A.MGMT1", "CTE0.B.MGMT0", "CTE0.B.MGMT1", "DSW0.MGMT.0", "DSW0.MGMT.1", "DSW1.MGMT.0",
                      "DSW1.MGMT.1", "CTE1.A.MGMT0", "CTE1.A.MGMT1", "CTE1.B.MGMT1"], },
    "3U": {
        "2": ["CTE0.SMM1.MGMT", "CTE1.SMM0.MGMT", "CTE1.SMM1.MAINTENANCE", "DSW0.MGMT.0", "DSW0.MGMT.1", "DSW1.MGMT.0",
              "DSW1.MGMT.1", "CTE0.SMM0.MAINTENANCE", "CTE0.SMM1.MAINTENANCE", "CTE1.SMM0.MAINTENANCE"], },

    # 2引擎场景：
    # from fru
    "SWITCH": ["DSW0.MGMT.0", "DSW0.MGMT.1", "DSW1.MGMT.0", "DSW1.MGMT.1"],
    # 3U2E indicates the engines are 3U and there are 2 engines  in the system
    "3U2E": ["CTE0.SMM1.MGMT", "CTE1.SMM0.MGMT", "CTE1.SMM1.MGMT",
             "CTE0.SMM0.MAINTENANCE", "CTE0.SMM1.MAINTENANCE", "CTE1.SMM0.MAINTENANCE"]

}
MGMT_PORT_CONFIG = {
    # 1引擎场景：
    "1": {
        "CTE0.SMM0.MGMT1": "SVP.P4",
        "CTE0.SMM1.MGMT1": "SVP.P3",
        "SVP.P3": "CTE0.SMM1.MGMT1",
        "SVP.P4": "CTE0.SMM0.MGMT1",
    },
    # 2引擎场景：
    "2": {
        "CTE0.SMM0.MGMT1": "CTE1.SMM0.MGMT2",
        "CTE0.SMM0.MGMT2": "DSW1.MGMT.1",
        "CTE1.SMM0.MGMT1": "SVP.P4",
        "CTE1.SMM0.MGMT2": "CTE0.SMM0.MGMT1",

        "CTE0.SMM1.MGMT1": "SVP.P3",
        "CTE0.SMM1.MGMT2": "CTE1.SMM1.MGMT1",
        "CTE1.SMM1.MGMT1": "CTE0.SMM1.MGMT2",
        "CTE1.SMM1.MGMT2": "DSW0.MGMT.0",

        "DSW0.MGMT.0": "CTE1.SMM1.MGMT2",
        "DSW0.MGMT.1": "DSW1.MGMT.0",
        "DSW1.MGMT.0": "DSW0.MGMT.1",
        "DSW1.MGMT.1": "CTE0.SMM0.MGMT2",

        "SVP.P3": "CTE0.SMM1.MGMT1",
        "SVP.P4": "CTE1.SMM0.MGMT1",
    },
    # 3引擎场景：
    "3": {
        "CTE0.SMM0.MGMT1": "CTE1.SMM0.MGMT2",
        "CTE0.SMM0.MGMT2": "DSW1.MGMT.1",
        "CTE1.SMM0.MGMT1": "CTE2.SMM0.MGMT2",
        "CTE1.SMM0.MGMT2": "CTE0.SMM0.MGMT1",
        "CTE2.SMM0.MGMT1": "SVP.P4",
        "CTE2.SMM0.MGMT2": "CTE1.SMM0.MGMT1",

        "CTE0.SMM1.MGMT1": "SVP.P3",
        "CTE0.SMM1.MGMT2": "CTE1.SMM1.MGMT1",
        "CTE1.SMM1.MGMT1": "CTE0.SMM1.MGMT2",
        "CTE1.SMM1.MGMT2": "CTE2.SMM1.MGMT1",
        "CTE2.SMM1.MGMT1": "CTE1.SMM1.MGMT2",
        "CTE2.SMM1.MGMT2": "DSW0.MGMT.0",

        "DSW0.MGMT.0": "CTE2.SMM1.MGMT2",
        "DSW0.MGMT.1": "DSW1.MGMT.0",
        "DSW1.MGMT.0": "DSW0.MGMT.1",
        "DSW1.MGMT.1": "CTE0.SMM0.MGMT2",

        "SVP.P3": "CTE0.SMM1.MGMT1",
        "SVP.P4": "CTE2.SMM0.MGMT1",
    },
    # 4引擎场景：
    "4": {
        "CTE0.SMM0.MGMT1": "CTE1.SMM0.MGMT2",
        "CTE0.SMM0.MGMT2": "DSW1.MGMT.1",
        "CTE1.SMM0.MGMT1": "CTE2.SMM0.MGMT2",
        "CTE1.SMM0.MGMT2": "CTE0.SMM0.MGMT1",
        "CTE2.SMM0.MGMT1": "CTE3.SMM0.MGMT2",
        "CTE2.SMM0.MGMT2": "CTE1.SMM0.MGMT1",
        "CTE3.SMM0.MGMT1": "SVP.P4",
        "CTE3.SMM0.MGMT2": "CTE2.SMM0.MGMT1",

        "CTE0.SMM1.MGMT1": "SVP.P3",
        "CTE0.SMM1.MGMT2": "CTE1.SMM1.MGMT1",
        "CTE1.SMM1.MGMT1": "CTE0.SMM1.MGMT2",
        "CTE1.SMM1.MGMT2": "CTE2.SMM1.MGMT1",
        "CTE2.SMM1.MGMT1": "CTE1.SMM1.MGMT2",
        "CTE2.SMM1.MGMT2": "CTE3.SMM1.MGMT1",
        "CTE3.SMM1.MGMT1": "CTE2.SMM1.MGMT2",
        "CTE3.SMM1.MGMT2": "DSW0.MGMT.0",

        "DSW0.MGMT.0": "CTE3.SMM1.MGMT2",
        "DSW0.MGMT.1": "DSW1.MGMT.0",
        "DSW1.MGMT.0": "DSW0.MGMT.1",
        "DSW1.MGMT.1": "CTE0.SMM0.MGMT2",

        "SVP.P3": "CTE0.SMM1.MGMT1",
        "SVP.P4": "CTE3.SMM0.MGMT1",
    }
}


def getPortIdByLocation(portRecords, location):
    """根据端口location获取端口id

    :param portRecords: 端口记录
    :param location: 位置
    :return:
    """
    for record in portRecords:
        if location == tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"]):
            return tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
    return None


def detectNoInputPower(tlv, tlvSvp, logger, **kwargs):
    """获取未插入电源线的电源Id列表

    :param tlv: tlv对象
    :param tlvSvp: svp tlv对象
    :param logger: 日志对象
    :param kwargs: 可变参数
    :return:
    """
    if skipPowerCableCheck():
        return []

    noInputPowerList = []
    powerRecordList = []
    powerRecords = tlvUtil.getHardwareRecords(tlv, "POWER")
    powerRecordList.extend(powerRecords)
    logger.logInfo("getPowerRecords:%s" % powerRecords)

    if tlvSvp:
        svpPowerRecords = tlvUtil.getHardwareRecords(tlvSvp, "POWER")
        powerRecordList.extend(svpPowerRecords)
        logger.logInfo("getSvpPowerRecords:%s" % svpPowerRecords)

    for record in powerRecordList:
        healthStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["healthStatus"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        if healthStatus != tlvData.HEALTH_STATUS_E["NORMAL"] or \
                runningStatus != tlvData.RUNNING_STATUS_E["ONLINE"]:
            noInputPowerLocation = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
            noInputPowerId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
            noInputPowerList.append({"id": noInputPowerId, "location": noInputPowerLocation})
    return noInputPowerList


def detectPowerModule(tlv, tlvSvp, logger, **kwargs):
    """检测电源模块是否在位

    :param tlv: tlv对象
    :param tlvSvp: svp tlv对象
    :param logger: 日志对象
    :param kwargs: 可变参数
    :return:
    """
    if skipPowerCableCheck():
        return []

    powerInfoList = []
    powerRecordList = []

    powerRecords = tlvUtil.getHardwareRecords(tlv, "POWER")
    powerRecordList.extend(powerRecords)
    logger.logInfo("getPowerRecords:%s" % powerRecords)
    if tlvSvp:
        svpPowerRecords = tlvUtil.getHardwareRecords(tlvSvp, "POWER")
        powerRecordList.extend(svpPowerRecords)
        logger.logInfo("getSvpPowerRecords:%s" % svpPowerRecords)

    for record in powerRecordList:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        powerInfoList.append(location)

    infoList = []
    enclosureRecordList = []

    enclosureRecords = tlvUtil.getHardwareRecords(tlv, "ENCLOSURE")
    enclosureRecordList.extend(enclosureRecords)
    logger.logInfo("getEnclosureRecords:%s" % enclosureRecords)
    if tlvSvp:
        svpEnclosureRecords = tlvUtil.getHardwareRecords(tlvSvp, "ENCLOSURE")
        enclosureRecordList.extend(svpEnclosureRecords)
        logger.logInfo("getSvpEnclosureRecords:%s" % svpEnclosureRecords)

    for record in enclosureRecordList:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        encId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])

        is_computing_dev = kwargs.get('is_computing_dev', False)
        if is_computing_dev and "CTE" in name:
            continue

        # 获取电源数量
        currentNum = 0
        for powerLocation in powerInfoList:
            if name in powerLocation:
                currentNum += 1

        need_num = 0
        is_dorado18000 = kwargs.get('isDorado18000', False)
        is_v5_v6_high_end = kwargs.get('isV5V6HighEnd', False)
        need_num = get_need_num(is_dorado18000, is_v5_v6_high_end, name, need_num)

        if currentNum < need_num:
            infoList.append(get_info(currentNum, encId, location, name, need_num))

    return infoList


def get_info(currentNum, encId, location, name, need_num):
    info = {
        "id": encId,
        "location": location,
        "name": name,
        "currentNum": currentNum,
        "needNum": need_num,
    }
    return info.copy()


def get_need_num(is_dorado_18000, is_v5_v6_high_end, name, need_num):
    if "DAE" in name:
        need_num = 2
    elif "CTE" in name and (is_dorado_18000 or is_v5_v6_high_end):
        need_num = 4
    elif "CTE" in name and not is_dorado_18000:
        need_num = 2
    elif "DSW" in name:
        need_num = 2
    elif "SVP" in name:
        need_num = 2
    return need_num


def getBayControllersNum(tlv):
    """获取配置文件中的控制器数量

    :param tlv: tlv对象
    :return:
    """
    return tlvUtil.getBayConfigCtrlNum(tlv, "0", isIpScaleOut=True)


def getDetectingPciePortRecords(tlv, logger):
    """获取需要检测的PCIe端口信息集合

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    num = str(getBayControllersNum(tlv))
    requiredLcoationList = PCIE_PORT_CONFIG.get(num, {})
    logger.logInfo("requiredLcoationList:%s" % requiredLcoationList)

    records = tlvUtil.getPCIePortRecords(tlv)
    logger.logInfo("pciePortRecords:%s" % records)
    newRecords = []
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        if location in requiredLcoationList:
            newRecords.append(record)
    return newRecords


def getPCIeIntfInfoDict(tlv, logger):
    """获取PCIe接口卡信息

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    portInfoDict = {}
    records = tlvUtil.getHardwareRecords(tlv, "INTF_MODULE")
    for record in records:
        model = tlvUtil.getRecordValue(record, tlvData.INTF_MODULE["model"])
        if model not in [tlvData.INTF_MODEL_E["PCIe_2X5G"], tlvData.INTF_MODEL_E["PCIe_2X"]]:
            continue

        intfId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        healthStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["healthStatus"])
        portInfoDict[location] = {
            "id": intfId,
            "runningStatus": runningStatus,
            "healthStatus": healthStatus,
        }
    return portInfoDict


def detectPCIeInterfaceModuleNum(tlv, logger):
    """检测PCIe接口卡数量

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    pcieIntfInfoDict = getPCIeIntfInfoDict(tlv, logger)
    logger.logInfo("pcieIntfInfoDict:%s" % pcieIntfInfoDict)
    currIntfLocations = pcieIntfInfoDict.keys()
    num = str(getBayControllersNum(tlv))
    logger.logInfo("bayControllersNum:%s" % str(num))
    requiredLcoationList = PCIE_INTF_MDL_CONFIG.get(num, {})
    logger.logInfo("requiredLcoationList:%s" % requiredLcoationList)

    offLineIntfList = []
    for requiredLocation in requiredLcoationList:
        # PCIe接口卡未插入场景
        logger.logInfo("requiredLocation:%s" % requiredLocation)
        if requiredLocation not in currIntfLocations:
            offLineIntfList.append({"location": requiredLocation})

    return offLineIntfList


def detectPCIeInterfaceModuleStatus(tlv, logger):
    """检测PCIe接口卡状态

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    pcieIntfInfoDict = getPCIeIntfInfoDict(tlv, logger)
    logger.logInfo("pcieIntfInfoDict:%s" % pcieIntfInfoDict)
    num = str(getBayControllersNum(tlv))
    logger.logInfo("bayControllersNum:%s" % str(num))
    requiredLcoationList = PCIE_INTF_MDL_CONFIG.get(num, {})
    logger.logInfo("requiredLcoationList:%s" % requiredLcoationList)

    poweredOffIntfList = []
    for requiredLocation in requiredLcoationList:
        # PCIe接口卡运行状态不正常场景
        pcieIntfInfo = pcieIntfInfoDict.get(requiredLocation, {})
        runningStatus = pcieIntfInfo.get("runningStatus", None)
        intfId = pcieIntfInfo.get("id", None)
        if runningStatus != tlvData.RUNNING_STATUS_E["RUNNING"]:
            poweredOffIntfList.append({"id": intfId, "location": requiredLocation})

    return poweredOffIntfList


def detectPCIeLinkStatus(tlv, logger):
    """检测PCIe端口状态

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    baseUtil.safeSleep(PCIE_PORT_ROUTINE_TIME)
    linkDownPortList = []
    records = getDetectingPciePortRecords(tlv, logger)
    logger.logInfo("getDetectingPciePortRecords:%s" % records)
    for record in records:
        # PCIe口运行状态不正常场景
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        if runningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
            portId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
            location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
            sugId = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["suggest_peer_port_id"])
            sugLocation = tlvUtil.getPciePortLocation(tlv, sugId)
            portInfoDict = {
                "id": portId,
                "location": location,
                "sugId": sugId,
                "sugLocation": sugLocation,
            }
            linkDownPortList.append(portInfoDict.copy())
    return linkDownPortList


def detectPCIeInterConnections(tlv, logger):
    """根据当前连接PCIe端口的当前连接id与建议连接端口id比对，检测PCIe线缆连接是否正确

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    wrongInterConnectionList = []
    records = getDetectingPciePortRecords(tlv, logger)
    logger.logInfo("getDetectingPciePortRecords:%s" % records)
    for record in records:
        curId = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["current_peer_port_id"])
        sugId = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["suggest_peer_port_id"])
        logger.logInfo("curId:%s" % curId)
        logger.logInfo("sugId:%s" % sugId)
        if curId != sugId:
            portId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
            location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
            curLocation = tlvUtil.getPciePortLocation(tlv, curId)
            sugLocation = tlvUtil.getPciePortLocation(tlv, sugId)
            wrongInfoDict = {
                "id": portId,
                "location": location,
                "curId": curId,
                "sugId": sugId,
                "curLocation": curLocation,
                "sugLocation": sugLocation,
            }
            wrongInterConnectionList.append(wrongInfoDict.copy())
    return wrongInterConnectionList


def getSugMgmtPortLocation(enginesNum, location):
    """根据配置字典，获取指定的管理网口应该连接的对端端口位置

    :param enginesNum: 引擎数量
    :param location: 端口位置
    :return:
    """
    config = MGMT_PORT_CONFIG.get(enginesNum, {})
    return config.get(location, None)


def getDetectingEthPortRecrods(tlv, tlvSvp):
    """获取所有ETH口信息（包含交换机和SVP信息）

    :param tlv: tlv对象
    :param tlvSvp: svp tlv对象
    :return:
    """
    ethPortsRecords = tlvUtil.getHardwareRecords(tlv, "ETH_PORT")
    svpPortsRecords = tlvUtil.getHardwareRecords(tlvSvp, "ETH_PORT")

    portInfoDict = getPortInfo(ethPortsRecords)
    newRecords = []
    newRecords.extend(ethPortsRecords)
    for record in svpPortsRecords:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        if location not in portInfoDict:
            newRecords.append(record)

    return newRecords


def getDetectingMgmtPortRecords(tlv, tlvSvp, logger):
    """获取需要检测的管理口信息集合

    :param tlv: tlv对象
    :param tlvSvp: tlv svp对象
    :param logger: 日志对象
    :return:
    """
    records = getDetectingEthPortRecrods(tlv, tlvSvp)
    newRecords = []
    num = str(len(tlvUtil.getEngineNameList(tlv)))
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        if getSugMgmtPortLocation(num, location) is not None:
            newRecords.append(record)
    return newRecords


def getMgmtPortSugInfoDict(tlv, tlvSvp, logger):
    """获取管理口建议连接端口信息

    :param tlv: tlv对象
    :param tlvSvp: tlv svp对象
    :param logger: 日志对象
    :return:
    """
    linkInfoDict = {}
    records = getDetectingMgmtPortRecords(tlv, tlvSvp, logger)
    num = str(len(tlvUtil.getEngineNameList(tlv)))
    for record in records:
        portId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        sugPortLocation = getSugMgmtPortLocation(num, location)
        sugPortId = getPortIdByLocation(records, sugPortLocation)
        linkInfoDict[portId] = {
            "id": portId,
            "location": location,
            "runningStatus": runningStatus,
            "sugPortLocation": sugPortLocation,
            "sugPortId": sugPortId,
        }
    return linkInfoDict


def getMgmtPortInfoDict(tlv, tlvSvp, logger):
    """获取管理口信息

    :param tlv: tlv对象
    :param tlvSvp: tlv svp对象
    :param logger: 日志对象
    :return:
    """
    portInfoDict = {}
    records = getDetectingMgmtPortRecords(tlv, tlvSvp, logger)
    logger.logInfo("mgmtPortSugInfoDict:%s" % records)
    for record in records:
        portId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        portInfoDict[portId] = {
            "id": portId,
            "runningStatus": runningStatus,
            "location": location,
        }
    return portInfoDict


def getMgmtIntfInfoDict(tlv, logger):
    """获取管理板信息

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    portInfoDict = {}
    records = tlvUtil.getHardwareRecords(tlv, "INTF_MODULE")
    for record in records:
        model = tlvUtil.getRecordValue(record, tlvData.INTF_MODULE["model"])
        if model != tlvData.INTF_MODEL_E["ManagerBoard"]:
            continue

        intfId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        healthStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["healthStatus"])
        portInfoDict[location] = {
            "id": intfId,
            "runningStatus": runningStatus,
            "healthStatus": healthStatus,
        }
    return portInfoDict


def detectMgmtInterfaceModuleNum(tlv, logger):
    """检测管理板数量

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    mgmtIntfInfoDict = getMgmtIntfInfoDict(tlv, logger)
    logger.logInfo("mgmtIntfInfoDict:%s" % mgmtIntfInfoDict)
    currIntfLocations = mgmtIntfInfoDict.keys()
    num = str(len(tlvUtil.getEngineNameList(tlv)))
    requiredLcoationList = MGMT_INTF_MDL_CONFIG.get(num, {})
    logger.logInfo("requiredLcoationList:%s" % requiredLcoationList)

    offLineIntfList = []
    for requiredLocation in requiredLcoationList:
        # 管理板未插入场景
        if requiredLocation not in currIntfLocations:
            offLineIntfList.append({"location": requiredLocation})

    return offLineIntfList


def detectDSWNum(tlv, logger):
    """检测PCIe交换机数量

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    records = tlvUtil.getHardwareRecords(tlv, "ENCLOSURE")
    logger.logInfo("enclosureInfoRecords:%s" % records)
    dswEncInfoList = []
    for record in records:
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        if "DSW" in name:
            dswEncInfoList.append(name)
    return dswEncInfoList


def detectDSWStatus(tlv, logger):
    """检测PCIe交换机状态

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    records = tlvUtil.getHardwareRecords(tlv, "ENCLOSURE")
    logger.logInfo("enclosureInfoRecords:%s" % records)
    dswEncInfoList = []
    for record in records:
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        if "DSW" in name:
            runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
            if runningStatus != tlvData.RUNNING_STATUS_E["ONLINE"]:
                encId = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
                dswEncInfoList.append({"name": name, "id": encId})
    return dswEncInfoList


def detectMgmtInterfaceModuleStatus(tlv, logger):
    """检测管理板状态

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    mgmtIntfInfoDict = getMgmtIntfInfoDict(tlv, logger)
    logger.logInfo("mgmtIntfInfoDict:%s" % mgmtIntfInfoDict)
    num = str(len(tlvUtil.getEngineNameList(tlv)))
    requiredLcoationList = MGMT_INTF_MDL_CONFIG.get(num, {})
    logger.logInfo("requiredLcoationList:%s" % requiredLcoationList)

    poweredOffIntfList = []
    for requiredLocation in requiredLcoationList:
        # 管理板运行状态不正常场景
        mgmtIntfInfo = mgmtIntfInfoDict.get(requiredLocation, {})
        runningStatus = mgmtIntfInfo.get("runningStatus", None)
        intfId = mgmtIntfInfo.get("id", None)
        if runningStatus != tlvData.RUNNING_STATUS_E["RUNNING"]:
            poweredOffIntfList.append({"id": intfId, "location": requiredLocation})

    return poweredOffIntfList


def detectMgmtPortLinkStatus(tlv, tlvSvp, logger):
    """检测管理口是否都link up（管理口列表为配置字典中配置的所有管理口）

    :param tlv: tlv对象
    :param tlvSvp: tlv svp对象
    :param logger: 日志对象
    :return:
    """
    baseUtil.safeSleep(MGMT_PORT_ROUTINE_TIME)
    portInfoDict = getMgmtPortInfoDict(tlv, tlvSvp, logger)
    logger.logInfo("portInfoDict:%s" % portInfoDict)
    portIds = portInfoDict.keys()
    portIds.sort()

    linkDownList = []
    locationList = []
    sugLocationList = []
    num = str(len(tlvUtil.getEngineNameList(tlv)))
    for portId in portIds:
        portInfo = portInfoDict.get(portId, {})
        location = portInfo.get("location", None)
        runningStatus = portInfo.get("runningStatus", None)
        sugLocation = getSugMgmtPortLocation(num, location)
        logger.logInfo("sugLocation:%s" % sugLocation)

        # 过滤不在配置字典中的管理口
        if sugLocation is None:
            continue

        # 过滤已经处理过的端口,避免重复报错
        if location in sugLocationList and sugLocation in locationList:
            continue

        # 过滤端口状态正常的管理口
        if runningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
            info = portInfo.copy()
            info["sugLocation"] = sugLocation
            linkDownList.append(info.copy())
            locationList.append(location)
            sugLocationList.append(sugLocation)
    return linkDownList


def getPortInfo(records):
    """以location为key，转换端口信息

    :param records: 端口记录
    :return:
    """
    infoDict = {}
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        infoDict[location] = {"runningStatus": runningStatus}
    return infoDict


def detectMgmtInterConnections(tlv, tlvSvp, logger):
    """根据设置一个端口linkDown后观察另外端口是否linkDown来检测管理网线是否正确连接

    :param tlv: tlv对象
    :param tlvSvp: tlv svp对象
    :param logger: 日志对象
    :return:
    """
    linkInfoDict = getMgmtPortSugInfoDict(tlv, tlvSvp, logger)
    logger.logInfo("linkInfoDict:%s" % linkInfoDict)
    mgmtPortIds = linkInfoDict.keys()
    mgmtPortIds.sort()

    wrongInfoList = []
    wrongPortLocationList = []
    for mgmtPortId in mgmtPortIds:
        portInfo = linkInfoDict.get(mgmtPortId)
        logger.logInfo("portInfo:%s" % portInfo)
        portLocation = portInfo["location"]
        peerLocation = portInfo["sugPortLocation"]

        # 不报重复错误
        if portLocation in wrongPortLocationList:
            continue

        # 交换机连接管理口的场景通过管理口连接交换机的处理流程覆盖
        # 交换机连接交换机的场景通过检测交换机的端口正常来覆盖（避免处理交换机端口id动态变化）
        if "DSW" in portLocation:
            continue
        # SVP连接管理口的场景通过管理口连接SVP的处理流程覆盖
        elif "SVP" in portLocation or "SVP" in peerLocation:
            continue
        else:
            # 将该端口link down
            tlvUtil.setEthPortLinkStatus(tlv, mgmtPortId, tlvData.RUNNING_STATUS_E["LINK_DOWN"])
            logger.logInfo("set port [%s] status link down successfully" % portLocation)
            try:
                # 检测建议对端端口状态
                isConnected = False
                for i in range(0, MGMT_PORT_ROUTINE_TIME / MGMT_PORT_ROUTINE_INTERVAL):
                    baseUtil.safeSleep(MGMT_PORT_ROUTINE_INTERVAL)
                    # 交换机端口id会动态变化,此处不采用id查找,采用获取所有的管理口信息,根据对应的location找到对应的状态
                    mgmtPortsRecords = getDetectingEthPortRecrods(tlv, tlvSvp)
                    portInfoDict = getPortInfo(mgmtPortsRecords)
                    logger.logInfo("latestPortInfoDict:%s" % portInfoDict)
                    if peerLocation in portInfoDict:
                        peerPortRunningStatus = portInfoDict.get(peerLocation).get("runningStatus")
                        if peerPortRunningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
                            isConnected = True
                            break

                if not isConnected:
                    wrongInfoList.append(portInfo.copy())
                    wrongPortLocationList.append(peerLocation)
            except Exception as exception:
                logger.logException(exception)

            # 恢复端口状态
            tlvUtil.setEthPortLinkStatus(tlv, mgmtPortId, tlvData.RUNNING_STATUS_E["LINK_UP"])
            logger.logInfo("set port [%s] status link up successfully" % portLocation)
    return wrongInfoList


def detectSASLinkConnections(tlv, logger):
    """检测硬盘框链路连接。级联模块A与控制器A上的级联接口模块之间的链路未连接时，硬盘框环路号为DAE999

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    enclosureRecords = tlvUtil.getHardwareRecords(tlv, "ENCLOSURE")
    logger.logInfo("enclosureRecords:%s" % enclosureRecords)

    wrongInfoList = []
    for record in enclosureRecords:
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        # 级联模块A与控制器A上的级联接口模块之间的链路连接错误
        if "999" in name:
            location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
            serialNum = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["serialNum"])
            wrongInfo = {
                "name": name,
                "location": location,
                "serialNum": serialNum,
            }
            wrongInfoList.append(wrongInfo.copy())
    return wrongInfoList


def getSASCardSlot(cardNum, side="R"):
    if cardNum == 0:
        return "%s%s" % (side, 5)
    elif cardNum == 1:
        return "%s%s" % (side, 4)
    else:
        raise Exception('SAS card number wrong.')


def getDetectingSASPortRecords(tlv, logger):
    """ 获取需要检测的SAS端口信息集合
        SAS线缆连接规则
            环路0的A级联板上连线规则：
                "DAEX00.A.PRI":"CTEX.R5.P0"
                "DAEX01.A.PRI":"DAEX00.A.EXP"
                "DAEX02.A.PRI":"DAEX01.A.EXP"
                "DAEX03.A.PRI":"DAEX02.A.EXP"
            环路0的B级联板上连线规则：
                "DAEX00.B.PRI":"CTEX.L5.P0"
                "DAEX01.B.PRI":"DAEX02.B.EXP"
                "DAEX02.B.PRI":"DAEX03.B.EXP"
                "DAEX03.B.PRI":"DAEX00.B.EXP"
            环路0之外的其他环路的A级联板上连线规则：
                "DAEX10.A.PRI":"CTEX.R5.P1"
                "DAEX11.A.PRI":"DAEX10.A.EXP"
                "DAEX12.A.PRI":"DAEX11.A.EXP"
                "DAEX13.A.PRI":"DAEX12.A.EXP"
            环路0之外的其他环路的B级联板上连线规则：
                "DAEX10.B.PRI":"DAEX11.B.EXP"
                "DAEX11.B.PRI":"DAEX12.B.EXP"
                "DAEX12.B.PRI":"DAEX13.B.EXP"
                "DAEX13.B.PRI":"CTEX.L5.P1"

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    # from exp
    diskEncInfoList = getAllDiskEnclosures(tlv)
    logger.logInfo("diskEncInfoList:%s" % diskEncInfoList)

    requiredSasPortDict = {}
    daeList = [diskEncInfo["name"] for diskEncInfo in diskEncInfoList]

    for diskEncInfo in diskEncInfoList:
        name = diskEncInfo["name"]
        engNum = int(name[3])  # 引擎号

        portNum = name[4]  # SAS卡端口号
        if not portNum.isdigit():
            portNum = ord(portNum) - ord("A") + 10
        portNum = int(portNum)

        slotNum = int(name[5])
        cardNum = slotNum / 4  # 第几张卡
        depthNum = slotNum % 4  # 当前位于的级联深度

        aSideSrcPort = "%s.A.PRI" % name
        bSideSrcPort = "%s.B.PRI" % name
        aSideSrcPortHighPower = "%s.A.P0" % name
        bSideSrcPortHighPower = "%s.B.P0" % name

        aSideDesPort = ""
        bSideDesPort = ""
        aSideDesPortHighPower = ""
        bSideDesPortHighPower = ""

        # 0号环路正正接
        if portNum == 0 and cardNum == 0:
            if slotNum == 0:
                aSideDesPort = "CTE%s.R5.P0" % engNum
                bSideDesPort = "CTE%s.L5.P0" % engNum
                aSideDesPortHighPower = aSideDesPort
                bSideDesPortHighPower = bSideDesPort

            else:
                aSideDesPort = "DAE%s%s%s.A.EXP" % (engNum, name[4], slotNum - 1)
                aSideDesPortHighPower = "DAE%s%s%s.A.P1" % (engNum, name[4], slotNum - 1)
                currentLoopLastDAE = "DAE%s%s%s" % (engNum, name[4], slotNum + 1)
                if currentLoopLastDAE not in daeList or (slotNum + 1) / 4 != slotNum / 4:
                    currentLoopLastDAE = "DAE%s%s%s" % (engNum, name[4], 0)
                bSideDesPort = "%s.B.EXP" % currentLoopLastDAE
                bSideDesPortHighPower = "%s.B.P1" % currentLoopLastDAE
                # 0号环路以外正反接
        else:
            if depthNum == 0:
                aSideDesPort = "CTE%s.%s.P%s" % (engNum, getSASCardSlot(cardNum, side="R"), portNum)
                aSideDesPortHighPower = aSideDesPort

            else:
                aSideDesPort = "DAE%s%s%s.A.EXP" % (engNum, name[4], slotNum - 1)
                aSideDesPortHighPower = "DAE%s%s%s.A.P1" % (engNum, name[4], slotNum - 1)
            currentLoopLastDAE = "DAE%s%s%s" % (engNum, name[4], slotNum + 1)
            if currentLoopLastDAE not in daeList or (slotNum + 1) / 4 != slotNum / 4:
                bSideDesPort = "CTE%s.%s.P%s" % (engNum, getSASCardSlot(cardNum, side="L"), portNum)
                bSideDesPortHighPower = bSideDesPort
            else:
                bSideDesPort = "DAE%s%s%s.B.EXP" % (engNum, name[4], slotNum + 1)
                bSideDesPortHighPower = "DAE%s%s%s.B.P1" % (engNum, name[4], slotNum + 1)
        requiredSasPortDict[aSideSrcPort] = aSideDesPort
        requiredSasPortDict[bSideSrcPort] = bSideDesPort
        requiredSasPortDict[aSideSrcPortHighPower] = aSideDesPortHighPower
        requiredSasPortDict[bSideSrcPortHighPower] = bSideDesPortHighPower
    return requiredSasPortDict


# from exp
def getAllDiskEnclosures(tlv):
    """获取系统所有的硬盘框信息字典的列表.

    :param tlv:
    :return:
    """
    records = tlvUtil.getHardwareRecords(tlv, "ENCLOSURE")
    diskEncInfoList = []
    for record in records:
        logicType = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["logicType"])
        if logicType != tlvData.ENCLOSURE_TYPE_E["EXP"]:
            continue
        diskEncInfo = {
            "id": tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"]),
            "name": tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"]),
            "location": tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"]),
            "logicType": tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["logicType"]),
            "expanderDepth": tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["expanderDepth"]),
        }
        diskEncInfoList.append(diskEncInfo.copy())
    return diskEncInfoList


def detectSASLinkStatus(tlv, logger):
    """检测SAS端口状态

    :param tlv:
    :param logger:
    :return:
    """
    requiredSasPortDict = getDetectingSASPortRecords(tlv, logger)
    logger.logInfo("requiredSasPortDict:%s" % requiredSasPortDict)
    requiredSasPorts = requiredSasPortDict.keys()

    # 更新当前端口连接的对端端口ID
    baseUtil.safeSleep(SAS_PORT_ROUTINE_TIME)
    records = tlvUtil.getSASPortRecords(tlv)
    pcieRecords = tlvUtil.getPciePortRecords(tlv)
    if records and pcieRecords:
        records.extend(pcieRecords)

    newRecords = []
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])

        logger.logInfo("location:%s" % location)
        if location not in requiredSasPorts:
            continue

        portType = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["type"])
        if portType == tlvData.OM_OBJ_E["PCIE_PORT"]:
            logger.logInfo("get pcie port current peer port id, type:%s" % portType)
            curentPeerPortId = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["current_peer_port_id"])
        else:
            logger.logInfo("get sas port current peer port id, type:%s" % portType)
            curentPeerPortId = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["current_peer_port_id"])

        suggestPeerLocation = requiredSasPortDict[location]
        # noinspection PyBroadException
        try:
            record = tlvUtil.getSASPortsById(tlv, curentPeerPortId)
            curentPeerLocation = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        except Exception:
            curentPeerLocation = ""
        if curentPeerLocation != requiredSasPortDict[location] or runningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
            infoDict = {
                "location": location,
                "runningStatus": runningStatus,
                "curentPeerLocation": curentPeerLocation,
                "suggestPeerLocation": suggestPeerLocation,
            }
            newRecords.append(infoDict.copy())
    return newRecords


def getDiskEnclosureStandardSASPortDict(daeName):
    """获取指定的Dorado18000 V3R1C30 的硬盘框对应的SAS端口标准连线字典.

    :param daeName:
    :return:
    """
    engineSASPortDict = {}
    engineNum = daeName[3]
    baseSasPortDict = config.DORADO_18000V3_SMB0_SAS_PORT_DICT
    if '0' == engineNum:
        engineSASPortDict = baseSasPortDict.copy()
    else:
        for daeSasPort in baseSasPortDict:
            if daeSasPort.replace('0', engineNum, 1).split('.')[0] == daeName:
                engineSASPortDict[daeSasPort.replace('0', engineNum, 1)] = baseSasPortDict.get(daeSasPort).replace(
                    '0', engineNum, 1)
    daeSASPortDict = {}
    for daePort in engineSASPortDict:
        if daePort.startswith(daeName):
            daeSASPortDict[daePort] = engineSASPortDict.get(daePort)
    return daeSASPortDict


def getDetectingSASPortRecordsForDorado18000(context, isExchangeKeyValue=False):
    """获取Dorado18000 V3 的新扩容的所有硬盘标准组网信息字典: { 硬盘框上的端口号：控制框上的端口号 }

    :param context:
    :param isExchangeKeyValue:
    :return:
    """
    diskEncInfoList = contextUtil.getItem(context, 'newEnc2UList', [])
    logger = common.getLogger(context.get("logger"), __file__)
    logger.logInfo("new diskEncInfoList:%s" % diskEncInfoList)
    requiredSasPortDict = {}
    daeList = [diskEncInfo[0] for diskEncInfo in diskEncInfoList]
    for daeName in daeList:
        requiredSasPortDict.update(getDiskEnclosureStandardSASPortDict(daeName))

    retRequiredSasDictPort = {}
    if isExchangeKeyValue:
        for (key, value) in requiredSasPortDict.items():
            retRequiredSasDictPort.setdefault(value, key)
        logger.logInfo("requiredSasPortDict:%s" % str(retRequiredSasDictPort))
        return retRequiredSasDictPort
    else:
        logger.logInfo("requiredSasPortDict:%s" % str(requiredSasPortDict))
        return requiredSasPortDict


def detectDorado18000SASLinkStatus(context):
    """检测Dorado18000 V3 的SAS端口状态

    :param context:
    :return:
    """
    tlv = contextUtil.getTlv(context)
    logger = common.getLogger(context.get("logger"), __file__)
    requiredSasPortDict = getDetectingSASPortRecordsForDorado18000(context, False)
    baseUtil.safeSleep(SAS_PORT_ROUTINE_TIME)
    records = tlvUtil.getSASPortRecords(tlv)
    newRecords = []
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        runningStatus = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        logger.logInfo("location:%s" % location)
        if location not in requiredSasPortDict:
            continue
        currentPeerPortId = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["current_peer_port_id"])
        suggestPeerLocation = requiredSasPortDict[location]
        try:
            record = tlvUtil.getSASPortsById(tlv, currentPeerPortId)
            curentPeerLocation = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        except Exception as e:
            logger.logInfo('Query SAS port by port ID failed:%s' % e)
            curentPeerLocation = ""
        if curentPeerLocation != requiredSasPortDict[location] or runningStatus != tlvData.RUNNING_STATUS_E["LINK_UP"]:
            infoDict = {
                "location": location,
                "runningStatus": runningStatus,
                "curentPeerLocation": curentPeerLocation,
                "suggestPeerLocation": suggestPeerLocation,
            }
            newRecords.append(infoDict.copy())
    return newRecords


def detectDcErrNet(cli, deepth, powerType, lang):
    """判断直流电源场景是否连接了不允许接入的SAS口，二级级联：6、7环路，四级级联：3号环路。

    :param cli: cli
    :param deepth: 级联深度
    :param powerType: 电源类型
    :param lang: 语言
    :return: errPorts异常端口
    """
    errPorts = []

    if powerType == tlvData.POWER_TYPE_E['AC']:
        return errPorts

    confErrPorts = ERR_SAS_PORTS_FOR_DC.get(deepth, [])
    linkupSasPorts = cliUtil.getLinkupSasPorts(cli, lang)
    for linkupSasPort in linkupSasPorts:
        if 'CTE' not in linkupSasPort:
            continue
        location = linkupSasPort[5:]
        if location.upper() in confErrPorts:
            infoDict = {
                "port": linkupSasPort,
            }
            errPorts.append(infoDict.copy())

    return errPorts
