# -*- coding: UTF-8 -*-
import traceback

from cbb.frame.faulttree.common import NodeStatus, NodeResult
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException


class FaultAnalyseResult:
    def __init__(self):
        self.status = NodeStatus.NO_START
        self.message = ""
        self.suggestion = ""
        self.origin_info = ""


class FaultTreeAnalyserEngine:
    """
    故障树分析引擎
    对故障树执行一次遍历分析

    """

    def __init__(self, context):
        self.context = context
        self.resource = context.get("resource")
        self.observer = context.get("observer")
        self.logger = context.get("logger")

        # 故障树脚本实例对象
        self.tree_obj = None

        # 故障树本次所有节点的分析结果
        self.all_results = {}

        # 每次故障树的配置
        self.tree_analyser = context.get("treeAnalyser")

    def init_data(self, start_node, tree):
        node_file = __import__(start_node.getId())
        node_class = getattr(node_file, "CheckItem")
        if tree:
            self.tree_obj = tree
        else:
            self.tree_obj = node_class(self.context)

    def run(self, start_node):
        self.analyse_node_try(start_node)

    def analyse_node_try(self, node):
        try:
            self.context.get('processingItems').add(node.getId())
            self.analyse_node(node)
        except (Exception, JException):
            self.logger.error("analyse node exception: {}".format(traceback.format_exc()))
        finally:
            # 每个节点执行完成后更新进度
            self.notify_node_result()

    def analyse_node(self, node):
        """
        状态设置为HUNG-通过返回值判断、从缓存那结果、合并、
        :param node:
        :return:
        """
        # 设置到上下文中
        node_id = node.getId()
        children = node.getChildren()
        cause_ids = node.getScriptCauseId()
        init_fun_name = node_id + "_init"
        run_fun_name = node_id + "_check"
        after_fun_name = node_id + "_after"
        next_fun_name = node_id + "_next"
        node_obj = self.tree_obj.create_check_node(node_id)
        node_obj.init_result(self.all_results)

        # 日志分析未完成，整个节点状态设置为挂起，当前节点及子节点暂不分析
        if cause_ids and not self.tree_analyser.isLogAnalyseFinish():
            self.logger.info("log analyse not over.")
            node_obj.set_node_hung()
            return

        if node_obj.is_not_finish():
            node_obj.exec_fun(self.get_fun_obj(init_fun_name))
            node_obj.exec_check(self.get_fun_obj(run_fun_name))
            # 如果检查项不为不通过，则不再检查子项
            if node_obj.is_no_hit():
                self.update_children_status(node)
                return
            # 执行下一步的初始化工作
            node_obj.exec_fun(self.get_fun_obj(next_fun_name))
        else:
            self.logger.info("the node  {} has executed".format(node_id))

        # 根据状态，可以在此处做一些处理
        self.analyse_children(children)
        node_obj.exec_fun(self.get_fun_obj(after_fun_name))
        return

    def get_fun_obj(self, fun_name):
        return getattr(self.tree_obj, fun_name, "")

    def update_children_status(self, node):
        children = node.getChildren()
        if not children:
            return
        for child in children:
            node_id = child.getId()
            node_result = NodeResult()
            node_result.id = node_id
            node_result.status = NodeStatus.SUCCESS
            node_result.detail = "NA"
            self.all_results[node_id] = node_result
            self.update_children_status(child)

    def analyse_children(self, children):
        if children:
            for child in children:
                self.analyse_node_try(child)

    def notify_node_result(self):
        try:
            detail = map(lambda item: dict(item), self.all_results.values())
            self.tree_analyser.updateResults({"detail": detail})
        except (Exception, JException):
            self.logger.error("notify error.{}".format(traceback.format_exc()))
