# -*- coding: UTF-8 -*-
import os
import time
import traceback
import ast

from com.huawei.ism.tlv.lang import UnsignedInt64
from cbb.frame.base import config, jsonUtil
from cbb.frame.rest import commonRestData as restData
from cbb.frame.rest import restDataType
from cbb.business.operate.expansion import config as exp_config

REST_CAN_NOT_EXECUTE = "1077936891"  # 命令执行失败


def convertRecord2Dict(jsonRecord):
    """
    @summary: 将Json格式的回显转换成字典
    @param jsonRecord: REST返回的数据. 
    """
    return jsonUtil.jsonStr2Dict(jsonRecord)


def getParamsJsonStr(paramDict):
    """
    @summary: 将参数字典转化为json字符串
    @param paramDict: 参数字典. 
    """
    if paramDict is None or len(paramDict) == 0:
        return ""
    return jsonUtil.dict2JsonStr(paramDict)


def getRecordValue(data, field):
    """
    @summary: 获取指定属性Key值转化后成指定类型并返回
    @param data: REST返回的单个DATA数据（类型为字典）. 
    @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
    """
    fieldKey = field.get("key")
    fieldType = field.get("type")
    paramValue = data.get(fieldKey, "")

    if fieldType == restDataType.DATA_TYPE_UNSIGN_INT or \
            fieldType == restDataType.DATA_TYPE_SIGN_INT or \
            fieldType == restDataType.DATA_TYPE_BOOL or \
            fieldType == restDataType.DATA_TYPE_ENUM or \
            fieldType == restDataType.DATA_TYPE_TIME:
        return int(paramValue)

    elif fieldType == restDataType.DATA_TYPE_U64LONG_TYPE:
        return UnsignedInt64(paramValue)
    else:
        return unicode(paramValue)


def getData(record):
    """
    @summary: 获取回显中的data数据
    @param record: rest回显
    @return : 返回data数据
    """
    if record is None or len(record) == 0:
        return None

    return record.get("data", None)


def getErrInfo(record):
    """
    @summary: 获取回显中的error数据
    @param record: rest回显
    @return : 返回error数据
    """
    if record is None or len(record) == 0:
        return None

    return record.get("error", None)


def getParamDict(objUri, uriParams=[]):
    """
    @summary: 获取指定属性Key值转化后成指定类型并返回
    @param data: REST返回的单个DATA数据（类型为字典）. 
    @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
    """
    uriParamDict = {}
    uriParamDict["objUri"] = objUri
    uriParamDict["paramList"] = uriParams
    return uriParamDict


def getUriParamDict(objUri, uriParams=[]):
    """
    @summary: 获取指定属性Key值转化后成指定类型并返回
    @param data: REST返回的单个DATA数据（类型为字典）. 
    @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
    """
    uriParamDict = {}
    uriParamDict["objUri"] = objUri
    uriParamDict["paramList"] = uriParams
    return uriParamDict


def getRestUri(baseUri, objUri, uriParams=[]):
    """
    @summary: 获取REST接口使用的uri连接
    @param objUri: objUri为各对象差异的URI
    """
    if len(uriParams) > 0:
        objUri = objUri + '/' + '/'.join(uriParams)
    return os.path.join(baseUri, objUri)


def execCmd(rest, uriParamDict, paramDict, restMethod, timeOut=restData.REST_CMD_TIMEOUT, **kwargs):
    '''
    @summary: 执行rest命令
    @param rest: rest对象
    @param cmd: 命令字
    @param paramDict: 参数字典
    @param restMethod: rest接口方法
    @param timeOut: 超时时间
    '''
    records = None
    restConn = rest.getRest()
    baseUri = rest.getBaseUri()
    uri = getRestUri(baseUri, uriParamDict["objUri"], uriParamDict.get("paramList"))  # {'objUri':'offline_controller',

    if kwargs and kwargs.get('useNewJsonMethod'):
        paramsStr = jsonUtil.dumpsObj(paramDict)
        logger = kwargs.get('logger')
        if logger:
            logger.logInfo("Para dict val:%s" % paramsStr)
            logger.logInfo("uri val:%s" % uri)
    else:
        paramsStr = getParamsJsonStr(paramDict)  # {"ID": ctrlId}

    for i in range(config.RETRY_TIMES + 1):
        try:
            if restMethod == restData.RestMethod.GET:
                records = convertRecord2Dict(restConn.execGet(uri, paramsStr).getContent())

            elif restMethod == restData.RestMethod.POST:
                records = convertRecord2Dict(restConn.execPost(uri, paramsStr).getContent())

            elif restMethod == restData.RestMethod.PUT:
                records = convertRecord2Dict(restConn.execPut(uri, paramsStr).getContent())

            elif restMethod == restData.RestMethod.DELETE:
                records = convertRecord2Dict(restConn.execDelete(uri, paramsStr).getContent())

            else:
                # 无此方法返回空
                return None

        except:
            raise Exception(config.ERR_CODE_COMMUNICATION_ABNOEMAL, traceback.format_exc())

        errInfo = getErrInfo(records)
        if errInfo is not None:
            errCode = getRecordValue(errInfo, restData.ErrorInfo.CODE)

            # 为默认错误码则为正常情况
            if errCode == config.REST_DEFAULT_ERR_CODE:
                return records

            # 往从节点下发消息失败，可以忽略。
            if errCode == config.ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE:
                return None

            # 通信异常进行重试
            if i < config.RETRY_TIMES and errCode in config.COMMUNICATION_FAIL_ERROR_CODE:
                time.sleep(30)  # 间隔30s重新下发
                continue

            # 无法处理的错误码，以异常形式抛出。
            description = getRecordValue(errInfo, restData.ErrorInfo.DESCRIPTION)
            raise Exception(errCode, description)

        return records

    return records


def getSysRecord(rest):
    '''
    @summary: 查询系统信息
    @param rest: rest连接
    @param baseUri: rest基础链接
    @return: 系统回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.SYSTEM)
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)


def getIBCRecord(rest):
    '''
    @summary: 检查ibc密码是否为默认密码
    @param rest: rest连接
    @param baseUri: rest基础链接
    @return: ibc回显
    '''
    params = {}
    uriParam = getUriParamDict(restData.OBJ.IBC)
    return execCmd(rest, uriParam, params, restData.RestMethod.GET)


def getCtrlRecords(rest):
    '''
    @summary: 批量查询控制器
    @param rest: rest连接
    @param baseUri: rest基础链接
    @return: rest回显
    '''
    params = {}
    uriParam = getUriParamDict(restData.OBJ.CONTROLLER)
    return execCmd(rest, uriParam, params, restData.RestMethod.GET)


def getCtrlIds(ctrlRecords):
    '''
    @summary: 获取控制器id与父id的映射字典{id:parentId}
    @param ctrlRecords: 控制器rest回显
    @return: 控制器id字典
    '''
    ctrlIds = []
    datas = getData(ctrlRecords)
    for ctrlData in datas:
        ctrlId = getRecordValue(ctrlData, restData.Controller.ID)
        ctrlIds.append(ctrlId)
    return ctrlIds


def getCtrlIdDict(ctrlRecords):
    '''
    @summary: 获取控制器id与父id的映射字典{id:parentId}
    @param ctrlRecords: 控制器rest回显
    @return: 控制器id字典
    '''
    ctrlIdDict = {}
    datas = getData(ctrlRecords)
    for ctrlData in datas:
        ctrlId = getRecordValue(ctrlData, restData.Controller.ID)
        parentId = getRecordValue(ctrlData, restData.Controller.PARENT_ID)
        ctrlIdDict[ctrlId] = parentId
    return ctrlIdDict


def getCtrlRecord(rest, ctrlId):
    '''
    @summary: 单个查询控制器
    @param rest: rest连接
    @param ctrlId: 控制器ID
    @return: rest回显
    '''
    params = {}
    uriParamDict = getUriParamDict(restData.OBJ.CONTROLLER, [ctrlId])
    return execCmd(rest, uriParamDict, params, restData.RestMethod.GET)


def getCtrlIoRecord(rest, ctrlId):
    '''
    @summary: 查询控制器Io
    @param rest: rest连接
    @param baseUri: rest基础链接
    @return: rest回显
    '''
    params = {restData.ControllerIo.ID["key"]: ctrlId}
    uriParam = getUriParamDict(restData.OBJ.CONTROLLER_IO)
    return execCmd(rest, uriParam, params, restData.RestMethod.GET)


def getEthPortRecords(rest):
    '''
    @summary: 批量查询eth端口
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.ETH_PORT)
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)


def getCtrlEnclosureIds(rest):
    '''
    @summary: 获取控制框的id列表
    @param rest: rest连接
    @return: rest回显
    '''
    encIds = []
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.ENCLOSURE)
    record = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    datas = getData(record)
    for data in datas:
        encId = getRecordValue(data, restData.Enclosure.ID)
        logicType = getRecordValue(data, restData.Enclosure.LOGIC_TYPE)
        if logicType == restData.EncLogicTypeEnum.CTRL:
            encIds.append(encId)

    return encIds


def getBackupPowerRecords(rest):
    '''
    @summary: 获取备电模块信息
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.BACKUP_POWER)
    record = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return record


def backUpConfigData(rest):
    """
    @summary:下载后删除阵列上的临时DB信息
    @param rest: rest对象
    """

    uriParams = getUriParamDict(restData.OBJ.EXPORT_DB_CONFIG)
    params = {restData.ExportData.CMO_EXPORT_DB_TYPE["key"]: restData.ExportTypeEnum.MEMORY}
    return execCmd(rest, uriParams, params, restData.RestMethod.PUT)


def clearDb(rest, ctrlId):
    """
    @summary:下载后删除阵列上的临时DB信息
    @param rest: rest对象
    """

    uriParam = getUriParamDict(restData.OBJ.CLEAR_EXPORT_DATA)
    paramDict = {restData.ExportData.EXPORT_DATA_TYPE["key"]: restData.DataTypeEnum.CONFIG_DATA
        , restData.ExportData.NODE_ID["key"]: ctrlId}
    return execCmd(rest, uriParam, paramDict, restData.RestMethod.PUT)


def offLineCtrl(rest, ctrlId):
    """
    @summary:离线控制器（多控模式）
    @param rest: rest对象
    @param ctrlId: 控制器id
    """
    paramDict = {restData.Controller.ID["key"]: ctrlId}  # {"ID": ctrlId}
    uriParamDict = getUriParamDict(restData.SpecialUri.OFFLINE_CONTROLLER)  # {'objUri':'offline_controller',
    # 'paramList': []}
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.PUT)


def expandCtrl(rest, newCtrlNum, networkMode, newBayIdToEncSnJson, logger,
               internalModel=""):
    """执行扩容控制器流程。

    :param rest: rest连接
    :param newCtrlNum: 扩容后控制器数量
    :param networkMode: 组网方式
    :param newBayIdToEncSnJson: 新引擎信息json字符串
    :param logger: 日志
    :param internalModel: 原控内部型号
    :return:
    """
    paramDict = {restData.ScaleOut.TOTAL_CTRL_NUM["key"]: newCtrlNum,
                 restData.ScaleOut.NET_WORK_MODE["key"]: networkMode,
                 restData.ScaleOut.NEW_CTR_SN["key"]: newBayIdToEncSnJson,
                 }
    if internalModel:
        paramDict[restData.ScaleOut.INTERNAL_MODEL["key"]] = internalModel
    uriParamDict = getUriParamDict(restData.SpecialUri.EXP_CTRL_FLOW)
    logger.logInfo('paramDict is: %s' % paramDict)
    try:
        exeRet = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.PUT,
                         **dict(useNewJsonMethod=True, logger=logger))
    except Exception as e:
        if len(e.args) >= 1:
            return int(e.args[0])
        else:
            raise e
    else:
        errInfo = getErrInfo(exeRet)
        return int(getRecordValue(errInfo, restData.ErrorInfo.CODE))


def queryExpandCtrlFlow(rest):
    """查询扩容流程。

    :param rest:
    :return:
    """
    paramDict = getUriParamDict(restData.SpecialUri.QUERY_EXP_CTRL_INFO)
    resp = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(resp), getErrInfo(resp)


def expandDiSkPoolController(rest, logger):
    """触发pool扩容

    :param rest: rest对象
    :param logger: 日志对象
    :return: True-执行成功，False-执行失败
    """
    paramDict = getUriParamDict(restData.SpecialUri.EXPAND_DISK_POOL_CONTROLLER)
    try:
        resp = execCmd(rest, paramDict, None, restData.RestMethod.POST)
    except Exception as e:
        logger.logException(e)
        return False
    return True


def offLineCtrlForSingle(rest, ctrlId):
    """
    @summary:离线控制器（单控模式）
    @param rest: rest对象
    @param cntrId: 控制器id
    """
    paramDict = {restData.Controller.ID["key"]: ctrlId}
    uriParamDict = getUriParamDict(restData.SpecialUri.OFFLINE_CONTROLLER_SIGNLE)
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.PUT)


def lightCtrl(rest, ctrlId):
    """
    @summary:点灯控制器
    @param rest: rest对象
    @param cntrId: 控制器id
    """
    paramDict = {restData.Controller.ID["key"]: ctrlId,
                 restData.Controller.LIGHT_STATUS["key"]: restData.LightStatusEnum.ON,
                 }
    uriParamDict = getUriParamDict(restData.OBJ.CONTROLLER)
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.PUT)


def getMgmtPortRecords(rest):
    """
    @summary:查询管理eth端口
    @param rest: rest对象
    """
    paramDict = {restData.EthPort.LOGIC_TYPE["key"]: restData.EthPortLogicTypeEnum.MNGT}
    uriParamDict = getUriParamDict(restData.OBJ.ETH_PORT)
    return execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)


def getAllSasPorts(rest):
    """
    @summary: 获取所有的SAS端口
    @param rest: rest对象
    """
    paramDict = getUriParamDict(restData.OBJ.SAS_PORT)
    data = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(data)


def getInterfaceById(rest, intfId):
    """
    @summary: 根据接口卡ID，查询接口卡
    """
    paramDict = getUriParamDict(restData.OBJ.INTERFACE_MODULE, [intfId])
    data = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(data)


def getCurrentAlarm(rest):
    '''
    @summary: 获取系统中当前告警
    @param rest: rest连接
    '''
    paramDict = getUriParamDict(restData.SpecialUri.QUERY_CURRENT_ALARM_INFO)
    data = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(data)


def getReplicationPair(rest):
    '''
    @summary: 获取远程复制pair， 出现异常认为不存在远程复制项目License。
    @param rest: rest连接
    '''
    paramDict = getUriParamDict(restData.OBJ.REPLICATIONPAIR)
    data = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(data)


def getFcPortRecords(rest):
    '''
    @summary: 批量查询eth端口
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.FC_PORT)
    data = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return getData(data)


def getFcoePortRecords(rest):
    '''
    @summary: 批量查询eth端口
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.FCoE_PORT)
    data = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return getData(data)


def getEthPortsRecord(rest):
    '''
    @summary: 批量查询eth端口
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.FCoE_PORT)
    data = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return getData(data)


def getRelicationPairById(rest, id):
    """
    @summary: 根据类型和ID获取端口
    """
    paramDict = getUriParamDict(restData.OBJ.REPLICATIONPAIR, [id])
    data = execCmd(rest, paramDict, None, restData.RestMethod.GET)
    return getData(data)


def getEnclosureRecords(rest):
    '''
    @summary: 批量查询框信息
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.ENCLOSURE)
    record = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return getData(record)


def getBayConfigBaseIpAddr(rest):
    '''
    @summary: 查询PCIe组网基地址IP，子网掩码，网关
    @param rest: rest连接
    @return: rest回显
    '''
    paramDict = {}
    uriParamDict = getUriParamDict(restData.OBJ.BASEIPADDR)
    record = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.GET)
    return getData(record)


def enableMgmtPortForward(rest, oldNodeNum):
    '''
    @summary: 打开管理网口转发功能
    @param rest: rest连接
    @param oldNodeNum: 原集群控制器数量 
    @return: rest回显
    '''
    try:
        paramDict = {restData.Scaleout.OLD_NODE_NUM["key"]: oldNodeNum}
        uriParamDict = getUriParamDict(restData.OBJ.ENABLE_MGMT_PORT_FORWARD)
        record = execCmd(rest, uriParamDict, paramDict, restData.RestMethod.POST)
        return (True, getData(record))
    except:
        return (False, None)


def get_performance_layer_disk_info_by_rest(rest, performance_layer_id=None):
    """
    获取性能层硬盘信息
    :param rest: rest连接
    :param performance_layer_id: 性能层ID
    :return:
    """
    param_dict = {}
    rest_uri = restData.SpecialUri.GET_PERFORMANCE_LAYER_DISK_INFO
    if performance_layer_id is not None:
        rest_uri = rest_uri + "?ID=" + performance_layer_id
    uri_param_dict = getUriParamDict(rest_uri)
    return execCmd(rest, uri_param_dict, param_dict, restData.RestMethod.GET)


def get_performance_layer_free_disk_info(rest, logger):
    """
    获取性能层空闲盘信息
    :param rest: rest连接
    :param logger: 日志
    :return:
    """
    free_disk_list = []
    ret_dict = {}
    try:
        tlv_ret = get_performance_layer_disk_info_by_rest(rest)
        if tlv_ret and tlv_ret.get("data"):
            all_engine_disk_info_str = tlv_ret.get("data").\
                get("allEngineDiskInfo").replace("true", "True").\
                replace("false", "False")
            ret_dict = ast.literal_eval(all_engine_disk_info_str)
    except Exception as ex:
        logger.error("can not get performance layer disk info:{}".format(ex))
        return free_disk_list

    if ret_dict:
        free_disk_list = ret_dict.get("freeDisk")
        free_disk_list = get_formatted_disk_info("", "", free_disk_list)

        logger.info("Performance Layer Free Disk Info is: {}".format(
            str(free_disk_list)))

    return free_disk_list


def get_all_engine_disk_info_by_rest(rest, disk_domain_id=None):
    """获取所有引擎的空闲盘信息 和硬盘域下的硬盘信息

    :param rest: rest连接
    :param disk_domain_id: 可以为空 为空时不返回硬盘域下的硬盘信息
    :return:
    """
    param_dict = {}
    rest_uri = restData.SpecialUri.GET_ALL_ENGINE_DISK_INFO
    if disk_domain_id is not None:
        rest_uri = rest_uri + "?ID=" + disk_domain_id
    uri_param_dict = getUriParamDict(rest_uri)
    return execCmd(rest, uri_param_dict, param_dict, restData.RestMethod.GET)


def get_formatted_engine_disk_info(rest, pool_id,
                                   disk_domain_id, logger, lang):
    """通过rest接口获取硬盘域盘信息并转换为java类结构

    :param rest: rest连接
    :param pool_id: 存储池ID
    :param disk_domain_id: 硬盘域ID
    :param logger: 日志句柄
    :param lang: 语言
    :return flag: True(获取成功)/False(获取失败)
    :return free_disk_info: 空闲盘信息
    :return member_disk_info: 成员盘信息
    """
    free_disk_list = []
    member_disk_list = []
    ret_dict = {}
    try:
        tlv_ret = get_all_engine_disk_info_by_rest(rest, disk_domain_id)
        if tlv_ret and tlv_ret.get("data"):
            all_engine_disk_info_str = \
                tlv_ret.get("data").get("allEngineDiskInfo")
            all_engine_disk_info_str = \
                all_engine_disk_info_str.replace("true", "True")
            all_engine_disk_info_str = \
                all_engine_disk_info_str.replace("false", "False")
            ret_dict = ast.literal_eval(all_engine_disk_info_str)
    except Exception as ex:
        logger.error("can not get engine disk info:{}".format(ex))
        return False, free_disk_list, member_disk_list

    if ret_dict:
        free_disk_list = ret_dict.get("freeDisk")
        member_disk_list = ret_dict.get("memberDisk")

        free_disk_list = get_formatted_disk_info("", "", free_disk_list)
        if pool_id and disk_domain_id:
            member_disk_list = get_formatted_disk_info(pool_id,
                                                       disk_domain_id,
                                                       member_disk_list)

        logger.info("freeDiskInfo is: %s" % str(free_disk_list))
        logger.info("memberDiskInfo is: %s" % str(member_disk_list))

    return True, free_disk_list, member_disk_list


def get_formatted_disk_info(pool_id, disk_domain_id, disk_info_list):
    """将rest接口的数据结构转换为java类数据结构

    :param pool_id: 存储池ID
    :param disk_domain_id: 硬盘域ID
    :param disk_info_list: 硬盘信息列表
    :return:
    """
    ret_list = []

    if len(disk_info_list) == 0:
        return ret_list

    for disk_info in disk_info_list:
        keys = disk_info.keys()
        engine_str = keys[0]
        disk_list = disk_info.get(engine_str)
        engine_ids = get_formatted_engine_ids(engine_str)
        disks = get_formatted_disks(disk_list)

        formatted_disk_info = {}
        formatted_disk_info["poolId"] = pool_id
        formatted_disk_info["diskDomainId"] = disk_domain_id
        formatted_disk_info["engineIds"] = engine_ids
        formatted_disk_info["disks"] = disks
        ret_list.append(formatted_disk_info)

    return ret_list


def get_formatted_engine_ids(str):
    """从字符串中提取引擎ID列表

    :param str: 例子 "EngineId:0;Enclosure:CTE0"
    :return:
    """
    tmp_list = str.split(";")
    if len(tmp_list) < 2:
        return []
    tmp_str = tmp_list[1]
    tmp_list = tmp_str.split(":")
    if len(tmp_list) < 2:
        return []
    tmp_str = tmp_list[1]
    return tmp_str.split(",")


def get_formatted_disks(disk_list):
    """获取格式化的硬盘信息列表

    :param disk_list:
    :return:
    """
    retList = []

    if not disk_list:
        return retList

    for disk in disk_list:
        ret = {}
        enclosure_info_list = []
        disk_type_str = disk.get("diskType").replace("_", " ").replace("-", " ")
        ret["type"] = exp_config.DISK_TYPE_TOOL_PRODUCT.get(disk_type_str) if exp_config.DISK_TYPE_TOOL_PRODUCT.get(
            disk_type_str) else disk_type_str
        # 设备返回的容量单位是扇区 每个扇区的大小固定为512B
        # 折算为GB返回
        ret["capacity"] = float(disk.get("cap")) * 512 / (1024 * 1024 * 1024)
        ret["count"] = disk.get("diskNum")
        disk_enclosures = disk.get("enclosureInfo")
        if disk_enclosures:
            for enclosure_info in disk_enclosures:
                info_map = {"id": enclosure_info.get("id"),
                            "name": enclosure_info.get("name"),
                            "count": enclosure_info.get("count")}
                enclosure_info_list.append(info_map)
        ret["enclosureInfo"] = enclosure_info_list
        retList.append(ret)

    return retList


def get_disk_type_enum(disk_model, disk_bom):
    """
    获取对应的硬盘类型的枚举。
    :param disk_model: 硬盘类型。
    :param disk_bom: 硬盘bom编码。
    :return:  枚举
    """
    # 获取QLC的盘类型枚举
    if disk_model in exp_config.PACKED_QLC_DISK_TYPE:
        return get_qlc_disk_type_enum(disk_model, disk_bom)
    if disk_model in exp_config.HDD_DISK_TYPE:
        return exp_config.HDD_DISK_TYPE.get(disk_model)
    else:
        return exp_config.SSD_DISK_TYPE.get(disk_model)


def get_qlc_disk_type_enum(disk_model, disk_bom):
    """
    获取对应QLC硬盘类型的枚举。
    @param disk_model: 硬盘类型
    @param disk_bom: 硬盘编码
    @return: 枚举
    """
    # 如果是包装的QLC盘编码("02356CKH", "02356CKJ", "02356CVN")，则盘枚举是19/20,否则盘枚举为21/22
    if disk_bom in ("02356CKH", "02356CKJ", "02356CVN"):
        return exp_config.PACKED_QLC_DISK_TYPE.get(disk_model)
    return exp_config.REALLY_QLC_DISK_TYPE.get(disk_model)
