# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.
import traceback
import re

from psdk.dsl.dsl import Dsl, DslContext
from psdk.dsl.fault_mode import ReturnIfHandler
from psdk.platform.util.base_util import format_msg, get_common_msg
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.base.exception import MessageException
from psdk.dsl.common import DslException
from psdk.dsl.dsl_common import get_version_info


class BaseCheckItem(object):
    def __init__(self, context):
        self.logger = context.get_logger()
        self.lang = context.get_lang()
        self.context = context
        self.sn = context.get_dev_node().get_sn()
        self.dsl_obj = Dsl(DslContext(self.context))
        self.origin_info = []
        self.item_config = {}

    def check(self):
        """
        对外的检查入口，捕获所有将异常，并转为了NO_CHECK
        :return: status, err_msg
        """
        try:
            return self.execute()
        except MessageException as e:
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, e.get_msg())
        except DslException:
            if self.context.get_dev_node().origin_dev.getPriKey():
                # 不支持PublicKey鉴权方式进行心跳控制器
                err_msg = "Publickey does not support log in to other controllers through heartbeat."
                self.logger.error("{} {}".format(err_msg, traceback.format_exc()))
                return CheckStatus.NOT_PASS, get_common_msg(self.lang, "no.support.publickey.forensics")
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")

        # 此处要捕获所有异常（python和java异常）
        except:
            self.logger.error("execute check error{}".format(traceback.format_exc()))
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        finally:
            try:
                self.dsl_obj.run("exec_cli 'show system general'")
                self.close_svp_ctrl_conn()
            except:
                self.logger.error(traceback.format_exc())

    def close_svp_ctrl_conn(self):
        try:
            dev_obj = self.context.get_dev_node()
            ip = dev_obj.get_ip()
            sn = dev_obj.get_sn()
            conn_key = "SVP_CTRL_CONNECTION" + ip + sn
            conn = self.context.get_item(conn_key)
            self.logger.info("base dsl start close svp ctrl conn:{}".format(conn))
            if conn and conn.isConnected():
                conn.close()
            self.logger.info("base dsl close svp ctrl conn success:{}".format(conn))
        except:
            self.logger.error(traceback.format_exc())

    def execute(self):
        """
        该方法由业务侧实现
        :return:
        """
        raise NotImplementedError

    def set_item_config(self, item_config):
        self.item_config = item_config

    def dsl(self, *args, **kwargs):
        """
        执行dsl命令
        :param args:
        :param kwargs:
        :return:
        @rtype: object
        """
        try:
            self.dsl_obj = Dsl(DslContext(self.context))
            ret = self.dsl_obj.run(*args, **kwargs)
        except MessageException as ex:
            raise ex
        except Exception as ex:
            ret = ReturnIfHandler(kwargs.get("return_if")).process_exception(ex)
        finally:
            if self.dsl_obj.origin_info:
                self.origin_info.extend(self.dsl_obj.origin_info)
                self.dsl_obj.origin_info = []
        return ret

    def get_err_msg(self, err_key, sug_key, err_args="", sug_args=""):
        """
        按场景决定是否拼接修复建议
        """
        if self._is_not_need_suggestion_msg():
            return self.get_mes_by_param_type(err_key, err_args)
        return self.get_msg_with_suggestion(err_key, sug_key, err_args, sug_args)

    def get_msg(self, key, *args):
        origin_msg = self.item_config.get("resource_{}".format(self.lang), {}).get(key)
        return format_msg(origin_msg, *args)

    def get_msg_with_suggestion(self, err_key, sug_key, err_args="", sug_args=""):
        err_msg = self.get_mes_by_param_type(err_key, err_args)
        suggestion = self.get_mes_by_param_type(sug_key, sug_args)
        sug_keyword = self.get_msg("check.suggestion.keyword")
        return "{}{}{}".format(err_msg, sug_keyword, suggestion)

    def get_mes_by_param_type(self, err_key, err_args=""):
        """
        按照参数类型拼接异常提示信息
            在备件和巡检不同场景下err_args可能会存在传递元组和字符串等多种场景；
            *err_args 会将字符串处理为字符列表，如果同时在巡检和备件中体现会存在显示问题，需要按类型处理
        :param err_key:
        :param err_args:
        :return:
        """
        if isinstance(err_args, tuple) or isinstance(err_args, list):
            return self.get_msg(err_key, *err_args)
        return self.get_msg(err_key, err_args)

    def add_origin_info(self, info):
        self.origin_info.append(info)

    def get_origin_info(self):
        return "\n".join(self.origin_info)

    def _is_not_need_suggestion_msg(self):
        """
        判断当前场景是否不需要拼接修复建议，涉及巡检、升级前检查、扩容评估
        """
        execute = self.context.execute_env
        return execute.tool_name in ["inspect_daily", "inspect_bureau", "upgrade_eval"] \
            or ('exp_' in execute.tool_name and execute.step == 'eval_check')

    def is_supper_admin(self):
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        version = version_info.get("base_version").get("Current Version")
        res = self.dsl("exec_cli 'show user user_name={}' | horizontal_parser".format(self.get_user_name()))
        if not res:
            return False
        for line in res:
            if any([
                line.get("Level", "") and line.get("Level", "").lower() == "super_admin",
                line.get("Role ID", "") and line.get("Role ID", "") == "1"
            ]):
                return True
            if self.can_enter_debug_mode_for_read_only_user(version, patch_version, line.get("Level", ""),
                                                            line.get("Role ID", "")):
                return True

        return False

    def get_user_name(self):
        user_name = self.context.dev_node.user_name
        # 域用户登录的设备，需要处理用户名，去掉 domain/ 标识
        return user_name.replace("domain/", "") if user_name.startswith("domain/") else user_name

    def can_enter_debug_mode_for_read_only_user(self, p_version, p_patch, level, role):
        """
        判断是否支持只读用户采集
        :param p_version: 版本信息
        :param p_patch: 补丁信息
        :param level: 账户level
        :param role: 账户角色
        :return: True 非超级管理员并且支持只读用户采集
        """
        if not self.is_support_read_only_user_enter_debug(p_version, p_patch):
            return False
        if level == "Admin" and role == "10":
            return True
        return False

    def is_support_read_only_user_enter_debug(self, product_version, hot_patch):
        """
        判断是否支持只读用户进去debug,如果支持返回true
        :param product_version: 版本信息
        :param hot_patch: 补丁信息
        :return: True 非超级管理员并且支持只读用户采集
        """
        # 支持的版本列表，后续可以在此列表中新增
        support_read_only_version = {'V300R006C20': 30, 'V500R007C61': 30, }
        if product_version in support_read_only_version:
            support_patch = support_read_only_version.get(product_version)

            match_hot_path = re.compile(r"SPH(\d+)", flags=re.IGNORECASE).search(hot_patch)
            if match_hot_path and support_patch <= int(match_hot_path.group(1)):
                return True
        return False

    def get_upgrade_target_version(self):
        """
         获取其他升级环境变量中的信息时也可直接从java_env中取值，支持的API如下：
         例子：self.context.execute_env.java_env.getUpgradeMode()
         java_env支持的API：'getOriEnv', 'getScenario', 'getSceneValue', 'getStep',
         'getSysPath', 'getTargetVersion', 'getToolName', 'getToolVersion', 'getUpgradeMode',
          'getUpgradePkgPath', 'getUpgradePkgVersion',
        :return: 升级目标版本
        """
        return self.context.execute_env.target_version

    def get_upgrade_mode(self):
        return self.context.execute_env.upgrade_mode

    def get_upgrade_pkg_path(self):
        return self.context.execute_env.upgrade_pkg_path

    def get_upgrade_pkg_version(self):
        return self.context.execute_env.upgrade_pkg_version
