# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.
import os
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.checkitem.check_item_mgr import read_json_file
from psdk.dsl.dsl_common import get_engine_height
from psdk.platform.util.product_util import get_ctrl_id_by_node_id

# 系统盘盘符为/dev/sda, 或者/dev/sda + /dev/sdb
SYS_DISK_LIST = ['/dev/sda', '/dev/sdb']
LIFE_NOT_PASS_VALUE = 80
# 用于标识巡检项是否通过, 2: NO_PASS
RESULT_NOT_PASS_VALUE = 2
# 用于标识巡检项是否通过, 1: WARNING
RESULT_WARNING_VALUE = 1
# 2023.8.31增加3款系统盘model：MD619HXCLDE3TC、ME619HXELDF3TE、ME619DXFNEC6CF
# 240G容量点盘PE不准
CAPACITY_240 = ('ME619GXEHDE3TE', 'ME619GXEHDE3AE', 'ME619HXELDF3TE')


class CheckSystemDiskLifeItem(BaseCheckItem):

    def __init__(self, context):
        super(CheckSystemDiskLifeItem, self).__init__(context)
        self.max_program_fail = 0
        self.config_map = read_json_file(os.path.join(os.path.dirname(__file__), "system_disk_risk.json"))
        # 系统盘厂商列表及SMART解析项：id为smartID，value_column为取值的列(VALUE为3，RAW_VALUE为-1)
        self.vendor_map = self.config_map.get('vendor_map')
        # 未Trim的系统盘SN
        self.un_trim_list = self.config_map.get('un_trim_list')
        self.not_pass_ctrl_list = []

    def execute(self):
        raise NotImplementedError

    def get_drive_life(self):
        """
        flag: 用于标识巡检项是否通过，0: PASS 1: WARNING 2: NO_PASS
        """
        flag = 0
        """
        获取工具版本信息
        """
        tool_name = ""
        check_tool_version = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                             "get_index(0)|get_index(0)".format("disk_repair.sh")
        tool_version = self.dsl(check_tool_version)
        if tool_version:
            check_tool_version_new = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                                     "get_index(0)|get_index(0)".format("disktool")
            tool_version_new = self.dsl(check_tool_version_new)
            if not tool_version_new:
                tool_name = "disktool"
        else:
            tool_name = "disk_repair.sh"
        if not tool_name:
            return flag
        node_id_cmd = "exec_mini_system 'showsysstatus' |regex 'local node id\s*:\s*(\S+)' |get_index(0)"
        node_id = self.dsl(node_id_cmd)
        dsl_cmd_info = "exec_mini_system '{} -s' |splitlines".format(tool_name)
        disk_info_list = self.dsl(dsl_cmd_info)
        if not disk_info_list:
            return flag
        for sys_letter in SYS_DISK_LIST:
            for disk_info in disk_info_list:
                if sys_letter in disk_info:
                    vendor = self._get_drive_vendor(disk_info)
                    dsl_cmd_smart = "exec_mini_system '{} -f a {}'|splitlines".format(tool_name, sys_letter)
                    flag = self._get_drive_smart(dsl_cmd_smart, vendor, flag)
        if flag != 0:
            self._get_not_pass_ctrl_info(node_id, flag)
        return flag

    def _get_drive_vendor(self, disk_info):
        for vendor_key in self.vendor_map.keys():
            if vendor_key in disk_info:
                return vendor_key
        return ""

    def _get_drive_smart(self, dsl_cmd_smart, vendor, res_flag):
        if not vendor:
            return res_flag
        dsl_smart_res = self.dsl(dsl_cmd_smart)
        smart = self._get_vendor_smart_info(dsl_smart_res, vendor)
        return self._check_smart_res_pass(res_flag, smart, vendor)

    def _get_not_pass_ctrl_info(self, node_id, res_flag):
        if not node_id:
            self.not_pass_ctrl_list.append("Get controller info failed")
            return
        # 获取node id对应ctrl id
        engine_height = get_engine_height(self.dsl, self.context)
        ctrl_id = get_ctrl_id_by_node_id(node_id, engine_height)
        # 获取ctrl id对应的条码
        ctrl_info_cmd = "exec_cli 'show controller general controller={}' | regex 'Electronic Label" \
                        "[\\s\\S]+BarCode=(?P<barcode>[\\w]+)'".format(ctrl_id)
        ctrl_info = self.dsl(ctrl_info_cmd)
        err_msg_format = ""
        # 根据当前控制器结果，判断错误信息打印格式
        if res_flag == RESULT_NOT_PASS_VALUE:
            err_msg_format = "ctrl.info.not.pass"
        elif res_flag == RESULT_WARNING_VALUE:
            err_msg_format = "ctrl.info.warning"
        # 拼接控制器错误信息：  控制器ID: {}, 条码: {} 不通过;\n控制器ID: {}, 条码: {} 建议优化;
        if ctrl_info:
            self.not_pass_ctrl_list.append(self.get_msg(err_msg_format, ctrl_id, ctrl_info.get("barcode")))
        else:
            self.not_pass_ctrl_list.append(self.get_msg(err_msg_format, ctrl_id, "NULL"))

    def _get_vendor_smart_info(self, dsl_smart_res, vendor):
        """
        pe准、pe不准盘的smart获取
        """
        vendor_smart_map = self.vendor_map[vendor]
        smart = {
            'PE_Times': -1, 'Spare_Block_Life': -1, 'Reallocated': -1, 'Program_Fail': 0,
            'Erase_Fail': 0, 'Host_Write': -1, 'unc': -1, 'Uncorrectable_Sector_Count': -1,
            'Offline_Uncorrectable_Sector_Count': -1, 'SN': ''
        }
        for smart_info in dsl_smart_res:
            # smart中取SN
            sn_reg = re.compile(r"Serial Number:\s+(\S+)", re.I)
            sn_ret = sn_reg.findall(smart_info)
            if sn_ret:
                smart.update({'SN': sn_ret[0]})
                continue
            # smart中取数字开头的行
            smart_id_reg = re.compile(r"^\s*(\d+)\s+.*")
            smart_id_ret = re.match(smart_id_reg, smart_info)
            if not smart_id_ret:
                continue
            self._update_smart_by_vendor_map(smart, smart_id_ret, vendor, vendor_smart_map)
            # 部分model号有多个Program fail值，取最大
            self._update_max_program_fail(smart, smart_id_ret, vendor_smart_map)
        return smart

    def _update_smart_by_vendor_map(self, smart, smart_id_ret, vendor, vendor_smart_map):
        smart_id = smart_id_ret.group(1)
        for smart_key in smart.keys():
            if smart_key not in vendor_smart_map.keys():
                continue
            if smart_id != vendor_smart_map[smart_key]['id']:
                continue
            smart_value = int(smart_id_ret.group().split()[vendor_smart_map[smart_key]["value_column"]])
            if smart_key == 'PE_Times' and vendor.startswith("RTMM"):
                smart.update({smart_key: self._get_rtmm_petimes_value(smart_value)})
                break
            if smart_key in ['Uncorrectable_Sector_Count', 'Offline_Uncorrectable_Sector_Count']:
                unc = smart.get('unc')
                smart.update({smart_key: smart_value})
                smart.update({'unc': smart_value if unc == -1 else smart_value + unc})
                break
            smart.update({smart_key: smart_value})
            break

    def _update_max_program_fail(self, smart, smart_id_ret, vendor_smart_map):
        if "Program_Fail_One" in vendor_smart_map.keys():
            smart_id = smart_id_ret.group(1)
            smart_value = -1
            for key in ["One", "Two", "Three"]:
                value = vendor_smart_map.get("Program_Fail_{}".format(key))
                if smart_id == value["id"]:
                    smart_value = int(
                        smart_id_ret.group().split()[value["value_column"]])
                    break
            if smart_value > self.max_program_fail:
                self.max_program_fail = smart_value
                smart.update({"Program_Fail": self.max_program_fail})

    def _get_rtmm_petimes_value(self, smart_value):
        # RAW_VALUE值转化成2进制数
        binary = str(bin(int(smart_value))).replace('0b', '')
        # 高位补0,补为48位
        if len(binary) < 48:
            temp_num = 48 - len(binary)
            for j in range(temp_num):
                binary = '0' + binary
        # 比较3组数据大小，取中间值作为PE次数
        nums = [int(binary[0:16], 2), int(binary[16:32], 2), int(binary[32:48], 2)]
        nums.sort()
        return nums[1]

    def _check_smart_res_pass(self, res_flag, smart, vendor):
        temp_flag = 0
        # 基于Spare Block的剩余寿命<=80,建议更换
        if -1 < smart.get('Spare_Block_Life') <= 80:
            return RESULT_NOT_PASS_VALUE
        # 获取PE准确盘、PE不准盘的PE次数
        if vendor in CAPACITY_240 and smart.get("Host_Write") != -1:
            if smart.get("SN") in self.un_trim_list:
                # PE不准盘，未Trim：Host写入量 * 20 为估算PE次数
                pe_times = smart.get("Host_Write") * 32 / 1024 / 240 * 20
            else:
                # PE不准盘，已Trim：Host写入量 * 8 为估算PE次数
                pe_times = smart.get("Host_Write") * 32 / 1024 / 240 * 8
        else:
            pe_times = smart.get("PE_Times")
        # 根据PE次数判断是否建议优化
        if 5000 <= pe_times < 6000:
            temp_flag = RESULT_WARNING_VALUE
        # 根据PE次数、UNC、Program fail、Erase fail判断是否建议更换
        if pe_times >= 6000:
            # PE次数大于6000，建议更换
            temp_flag = RESULT_NOT_PASS_VALUE
        elif pe_times >= 3000:
            # PE次数大于3000小于6000，unc大于1，建议更换
            if smart.get("unc") > 1:
                temp_flag = RESULT_NOT_PASS_VALUE
            elif smart.get('Program_Fail') + smart.get('Erase_Fail') > 1:
                # PE次数大于3000小于6000，unc小于2，program fail + erase fail大于1，建议更换
                temp_flag = RESULT_NOT_PASS_VALUE
        else:
            # PE次数小于3000，unc大于1，建议更换
            if smart.get("unc") > 1:
                temp_flag = RESULT_NOT_PASS_VALUE
        return max(res_flag, temp_flag)
